#include "mal.h"
#include "lcd_driverhd44780.h"
#include "c_lcd_driverhd44780.h"
#include "int_lcd_driverhd44780.h"
#include "tmr.h"
#include "io_states.h"

#define BUSY	1
#define NOTBUSY	0

#define		READ	1
#define 	WRITE	0

#define 	INSTRUCTION_REG 0
#define 	DATA_REG	1

#ifdef LCD_ASYNC_INIT
	typedef enum _Lcd_Driver_Init_States {
		Lcd_Driver_Init_State_1 = 0,
		Lcd_Driver_Init_State_2,
		Lcd_Driver_Init_State_3,
		Lcd_Driver_Init_State_4,
		Lcd_Driver_Init_State_5,
		Lcd_Driver_Init_State_6,
		Lcd_Driver_Init_State_7,
		Lcd_Driver_Init_State_8,
		Lcd_Driver_Init_State_9,
		Lcd_Driver_Init_State_10,
		Lcd_Driver_Init_State_Idle,
	} Lcd_Driver_Init_States;
#endif

#ifdef LCD_ASYNC_INTERFACE
	typedef enum _Lcd_Driver_States {
		//Check Busy Flag
		Lcd_Driver_State_0_R_W_READ = 0,
		Lcd_Driver_State_1_RS_INSTRUCTION,
		Lcd_Driver_State_2_TRIS_IN,
		Lcd_Driver_State_3_HIGH_E,
		Lcd_Driver_State_4_READ_BUSY_FLAG,
		Lcd_Driver_State_5_LOW_E,
		Lcd_Driver_State_6_HIGH_E,
		Lcd_Driver_State_7_LOW_E,

		//exchange data
		Lcd_Driver_State_8_R_W_WRITE,
		Lcd_Driver_State_9_RS_INSTRUCTIONORDATA,
		Lcd_Driver_State_10_TRIS_OUT,
		Lcd_Driver_State_11_HIGH_E,
		Lcd_Driver_State_12_UPPER_NIBBLE,
		Lcd_Driver_State_13_LOW_E,
		Lcd_Driver_State_14_HIGH_E,
		Lcd_Driver_State_15_LOWER_NIBBLE,
		Lcd_Driver_State_16_LOW_E,
		
		Lcd_Driver_State_17_Finished,

		Lcd_Driver_State_Idle,
	} Lcd_Driver_States;

	typedef enum _Lcd_Driver_IsCommandOrData {
		Lcd_Driver_Empty = 0,
		Lcd_Driver_Command = 1,
		Lcd_Driver_Data = 2,
	} Lcd_Driver_IsCommandOrData;

	typedef struct _Lcd_Driver_Fifo {
		Lcd_Driver_IsCommandOrData lcd_Driver_IsCommandOrData;
		uint8 lcd_Driver_Payload;
	} Lcd_Driver_Fifo;
#endif

uint16 delayCnt2 = 0;
Timer lcd_driver_busyTimer;
#ifdef LCD_ASYNC_INIT
	Timer lcd_driver_async_init_timer;
	Lcd_Driver_Init_States Lcd_Driver_Init_State = Lcd_Driver_Init_State_1;
#endif
#ifdef LCD_ASYNC_INTERFACE
	uint8 lcd_Driver_busyFlag = 0;
	Lcd_Driver_States lcd_Driver_States = Lcd_Driver_State_Idle;
	uint32 lcd_Driver_Fifo_write = 0;
	uint32 lcd_Driver_Fifo_read = 0;
	Lcd_Driver_Fifo lcd_Driver_Fifo[LCD_ASYNC_INTERFACE_ITEMS];
#endif

static void putDataToLCDPort(uint8 data);
static void lcdTrisOut(void);
static void lcdTrisIn(void);
static void lcd_driver_high_e(void);
static void lcd_driver_low_e(void);
static void delay100us(void);
static uint8 Check_Busy(void);
static uint8 Wait_Busy(void);
#ifdef LCD_ASYNC_INTERFACE
	static void start100us(void);
	static uint8 check100us(void);
#endif

void init_lcd_driver(void) {
	removeTimer(&lcd_driver_busyTimer);
	addTimer(&lcd_driver_busyTimer);
	#ifdef LCD_ASYNC_INTERFACE
		lcd_Driver_States = Lcd_Driver_State_Idle;
		lcd_Driver_Fifo_write = 0;
		lcd_Driver_Fifo_read = 0;
		memset(lcd_Driver_Fifo, 0x00, sizeof(lcd_Driver_Fifo));
	#endif
	#ifdef LCD_ASYNC_INIT
		removeTimer(&lcd_driver_async_init_timer);
		addTimer(&lcd_driver_async_init_timer);
		Lcd_Driver_Init_State = Lcd_Driver_Init_State_1;
	#else
		delayms(LCD_DELAY_TIME); //TODO rework as state machine
		putDataToLCDPort(0x00);
		lcdTrisOut();

		E_LAT = 0;
		LCD_PORT_SYNC();
		RS_LAT = INSTRUCTION_REG; //0
		LCD_PORT_SYNC();
		R_W_LAT = WRITE; //0
		LCD_PORT_SYNC();
		E_TRIS = OUT;
		LCD_PORT_SYNC();
		R_W_TRIS = OUT;
		LCD_PORT_SYNC();
		RS_TRIS = OUT;
		LCD_PORT_SYNC();

		//resync
		RS_LAT = DATA_REG; //1
		LCD_PORT_SYNC();
		delayms(5);
		RS_LAT = INSTRUCTION_REG; //0
		LCD_PORT_SYNC();
		delayms(5);
		R_W_LAT = READ; //1
		LCD_PORT_SYNC();
		delayms(5);
		R_W_LAT = WRITE; //0
		LCD_PORT_SYNC();
		delayms(5);

		#ifdef LCD_USE_BACKLIGHT
			#ifndef LCD_USE_USER_BACKLIGHT
				BACKLIGHT_TRIS = OUT;
			#endif
			lcd_backlight_on();
		#endif

		putDataToLCDPort(0x30);		//8bit
		lcd_driver_high_e();
		lcd_driver_low_e();
		delayms(5);
		lcd_driver_high_e();
		lcd_driver_low_e();
		delayms(5);
		lcd_driver_high_e();
		lcd_driver_low_e();
		delayms(5);
		putDataToLCDPort(0x20);		//4bit
		lcd_driver_high_e();
		lcd_driver_low_e();
		delayms(5);

		#ifdef LCD_1_LINE
			Send_Cmd_LCD(FUNCTION_SET | 0x00); //0x10:8bit, 0x08: 2lines, 0x04 5*10fonts.
		#endif
		#ifdef LCD_2_LINE
			Send_Cmd_LCD(FUNCTION_SET | 0x08);
		#endif
		Send_Cmd_LCD(ENTRY_INC);
		Send_Cmd_LCD(CLR_DISP);
		Send_Cmd_LCD(DISP_ON);
		Send_Cmd_LCD(HOME);
	#endif
}

void do_lcd_driver(void) {
	#ifdef LCD_ASYNC_INTERFACE
		switch (lcd_Driver_States) {
			//Check Busy Flag
			case Lcd_Driver_State_0_R_W_READ : {
				R_W_LAT = READ; //1
				//LCD_PORT_SYNC();
				lcd_Driver_States = Lcd_Driver_State_1_RS_INSTRUCTION;
				break;
			}
			case Lcd_Driver_State_1_RS_INSTRUCTION : {
				RS_LAT = INSTRUCTION_REG; //0
				//LCD_PORT_SYNC();
				lcd_Driver_States = Lcd_Driver_State_2_TRIS_IN;
				break;
			}
			case Lcd_Driver_State_2_TRIS_IN : {
				lcdTrisIn();
				start100us();
				lcd_Driver_States = Lcd_Driver_State_3_HIGH_E;
				break;
			}
			case Lcd_Driver_State_3_HIGH_E : {
				if (check100us()) {
					E_LAT = 1;
					//LCD_PORT_SYNC();
					start100us();
					lcd_Driver_States = Lcd_Driver_State_4_READ_BUSY_FLAG;
				}
				break;
			}
			case Lcd_Driver_State_4_READ_BUSY_FLAG : {
				if (check100us()) {
					lcd_Driver_busyFlag = LCD_PORT7; //busy flag is needed sometimes
					start100us();
					lcd_Driver_States = Lcd_Driver_State_5_LOW_E;
				}
				break;
			}
			case Lcd_Driver_State_5_LOW_E : {
				if (check100us()) {
					E_LAT = 0;
					//LCD_PORT_SYNC();
					start100us();
					lcd_Driver_States = Lcd_Driver_State_6_HIGH_E;
				}
				break;
			}
			case Lcd_Driver_State_6_HIGH_E : {
				if (check100us()) {
					E_LAT = 1;
					//LCD_PORT_SYNC();
					start100us();
					lcd_Driver_States = Lcd_Driver_State_7_LOW_E;
				}
				break;
			}
			case Lcd_Driver_State_7_LOW_E : {
				if (check100us()) {
					E_LAT = 0;
					//LCD_PORT_SYNC();
					start100us();
					if (lcd_Driver_busyFlag == 0) {
						lcd_Driver_States = Lcd_Driver_State_8_R_W_WRITE;
					} else {
						if (readTimer(&lcd_driver_busyTimer) == 0) {
							lcd_Driver_States = Lcd_Driver_State_17_Finished;
						} else {
							lcd_Driver_States = Lcd_Driver_State_0_R_W_READ;
						}
					}
				}
				break;
			}
			//exchange data
			case Lcd_Driver_State_8_R_W_WRITE : {
				if (check100us()) {
					R_W_LAT = WRITE; //0
					//LCD_PORT_SYNC();
					lcd_Driver_States = Lcd_Driver_State_9_RS_INSTRUCTIONORDATA;
				}
				break;
			}
			case Lcd_Driver_State_9_RS_INSTRUCTIONORDATA : {
				if (lcd_Driver_Fifo[lcd_Driver_Fifo_read].lcd_Driver_IsCommandOrData == Lcd_Driver_Command) {
					RS_LAT = INSTRUCTION_REG; //0
				} else {
					RS_LAT = DATA_REG; //1
				}
				//LCD_PORT_SYNC();
				lcd_Driver_States = Lcd_Driver_State_10_TRIS_OUT;
				break;
			}
			case Lcd_Driver_State_10_TRIS_OUT : {
				lcdTrisOut();
				start100us();
				lcd_Driver_States = Lcd_Driver_State_11_HIGH_E;
				break;
			}
			case Lcd_Driver_State_11_HIGH_E : {
				if (check100us()) {
					E_LAT = 1;
					//LCD_PORT_SYNC();
					start100us();
					lcd_Driver_States = Lcd_Driver_State_12_UPPER_NIBBLE;
				}
				break;
			}
			case Lcd_Driver_State_12_UPPER_NIBBLE : {
				if (check100us()) {
					uint8 x = (lcd_Driver_Fifo[lcd_Driver_Fifo_read].lcd_Driver_Payload);
					uint8 temp = 0;
					temp = (x & 0xF0);											
					putDataToLCDPort(temp);
					start100us();
					lcd_Driver_States = Lcd_Driver_State_13_LOW_E;
				}
				break;
			}
			case Lcd_Driver_State_13_LOW_E : {
				if (check100us()) {
					E_LAT = 0;
					//LCD_PORT_SYNC();
					start100us();
					lcd_Driver_States = Lcd_Driver_State_14_HIGH_E;
				}
				break;
			}
			case Lcd_Driver_State_14_HIGH_E : {
				if (check100us()) {
					E_LAT = 1;
					//LCD_PORT_SYNC();
					start100us();
					lcd_Driver_States = Lcd_Driver_State_15_LOWER_NIBBLE;
				}
				break;
			}
			case Lcd_Driver_State_15_LOWER_NIBBLE : {
				if (check100us()) {
					uint8 x = (lcd_Driver_Fifo[lcd_Driver_Fifo_read].lcd_Driver_Payload);
					uint8 temp = 0;
					temp = ((x & 0x0F) << 4);											
					putDataToLCDPort(temp);											
					start100us();
					lcd_Driver_States = Lcd_Driver_State_16_LOW_E;
				}
				break;
			}
			case Lcd_Driver_State_16_LOW_E : {
				if (check100us()) {
					E_LAT = 0;
					//LCD_PORT_SYNC();
					start100us();
					lcd_Driver_States = Lcd_Driver_State_17_Finished;
				}
				break;
			}
			case Lcd_Driver_State_17_Finished : {
				lcd_Driver_States = Lcd_Driver_State_Idle;

				lcd_Driver_Fifo_read++;
				if (lcd_Driver_Fifo_read >= LCD_ASYNC_INTERFACE_ITEMS) {
					lcd_Driver_Fifo_read = 0;
				}
			}
			case Lcd_Driver_State_Idle : {
				if (is_lcd_driver_inited() == 1) {
					if (lcd_Driver_Fifo_write != lcd_Driver_Fifo_read) {
						if (lcd_Driver_Fifo[lcd_Driver_Fifo_read].lcd_Driver_IsCommandOrData != Lcd_Driver_Empty) {
							writeTimer(&lcd_driver_busyTimer, 4);
							lcd_Driver_States = Lcd_Driver_State_0_R_W_READ;
						}
					}
				}
				break;
			}
			default : {
				lcd_Driver_States = Lcd_Driver_State_Idle;
			}
		}
	#endif
	#ifdef LCD_ASYNC_INIT
		if (is_lcd_driver_inited() == 0) {
			switch (Lcd_Driver_Init_State) {
				case Lcd_Driver_Init_State_1 : {
					writeTimer(&lcd_driver_async_init_timer, LCD_DELAY_TIME);
					Lcd_Driver_Init_State = Lcd_Driver_Init_State_2;
					break;
				}
				case Lcd_Driver_Init_State_2 : {
					if (readTimer(&lcd_driver_async_init_timer) == 0) {

						putDataToLCDPort(0x00);
						lcdTrisOut();

						E_LAT = 0;
						LCD_PORT_SYNC();
						RS_LAT = INSTRUCTION_REG; //0
						LCD_PORT_SYNC();
						R_W_LAT = WRITE; //0
						LCD_PORT_SYNC();
						E_TRIS = OUT;
						LCD_PORT_SYNC();
						R_W_TRIS = OUT;
						LCD_PORT_SYNC();
						RS_TRIS = OUT;
						LCD_PORT_SYNC();

						//resync
						RS_LAT = DATA_REG; //1
						LCD_PORT_SYNC();
						writeTimer(&lcd_driver_async_init_timer, 5);
						Lcd_Driver_Init_State = Lcd_Driver_Init_State_3;
					}
					break;
				}
				case Lcd_Driver_Init_State_3 : {
					if (readTimer(&lcd_driver_async_init_timer) == 0) {
						RS_LAT = INSTRUCTION_REG; //0
						LCD_PORT_SYNC();
						writeTimer(&lcd_driver_async_init_timer, 5);
						Lcd_Driver_Init_State = Lcd_Driver_Init_State_4;
					}
					break;
				}
				case Lcd_Driver_Init_State_4 : {
					if (readTimer(&lcd_driver_async_init_timer) == 0) {
						R_W_LAT = READ; //1
						LCD_PORT_SYNC();
						writeTimer(&lcd_driver_async_init_timer, 5);
						Lcd_Driver_Init_State = Lcd_Driver_Init_State_5;
					}
					break;
				}
				case Lcd_Driver_Init_State_5 : {
					if (readTimer(&lcd_driver_async_init_timer) == 0) {
						R_W_LAT = WRITE; //0
						LCD_PORT_SYNC();
						writeTimer(&lcd_driver_async_init_timer, 5);
						Lcd_Driver_Init_State = Lcd_Driver_Init_State_6;
					}
				break;
				}
				case Lcd_Driver_Init_State_6 : {
					if (readTimer(&lcd_driver_async_init_timer) == 0) {
						#ifdef LCD_USE_BACKLIGHT
							#ifndef LCD_USE_USER_BACKLIGHT
								BACKLIGHT_TRIS = OUT;
							#endif
							lcd_backlight_on();
						#endif

						putDataToLCDPort(0x30);		//8bit
						lcd_driver_high_e();
						lcd_driver_low_e();
						writeTimer(&lcd_driver_async_init_timer, 5);
						Lcd_Driver_Init_State = Lcd_Driver_Init_State_7;
					}
					break;
				}
				case Lcd_Driver_Init_State_7 : {
					if (readTimer(&lcd_driver_async_init_timer) == 0) {
						lcd_driver_high_e();
						lcd_driver_low_e();
						writeTimer(&lcd_driver_async_init_timer, 5);
						Lcd_Driver_Init_State = Lcd_Driver_Init_State_8;
					}
					break;
				}
				case Lcd_Driver_Init_State_8 : {
					if (readTimer(&lcd_driver_async_init_timer) == 0) {
						lcd_driver_high_e();
						lcd_driver_low_e();
						writeTimer(&lcd_driver_async_init_timer, 5);
						Lcd_Driver_Init_State = Lcd_Driver_Init_State_9;
					}
					break;
				}
				case Lcd_Driver_Init_State_9 : {
					if (readTimer(&lcd_driver_async_init_timer) == 0) {
						putDataToLCDPort(0x20);		//4bit
						lcd_driver_high_e();
						lcd_driver_low_e();
						writeTimer(&lcd_driver_async_init_timer, 5);
						Lcd_Driver_Init_State = Lcd_Driver_Init_State_10;
					}
					break;
				}
				case Lcd_Driver_Init_State_10 : {
					if (readTimer(&lcd_driver_async_init_timer) == 0) {
						#ifdef LCD_1_LINE
							Send_Cmd_LCD(FUNCTION_SET | 0x00); //0x10:8bit, 0x08: 2lines, 0x04 5*10fonts.
						#endif
						#ifdef LCD_2_LINE
							Send_Cmd_LCD(FUNCTION_SET | 0x08);
						#endif
						Send_Cmd_LCD(ENTRY_INC);
						Send_Cmd_LCD(CLR_DISP);
						Send_Cmd_LCD(DISP_ON);
						Send_Cmd_LCD(HOME);
						Lcd_Driver_Init_State = Lcd_Driver_Init_State_Idle;
					}
					break;
				}
				case Lcd_Driver_Init_State_Idle :
				default : {
					break;
				}
			}
		}
	#endif
}

void isr_lcd_driver100us(void) {
	if (delayCnt2 != 0) {
		delayCnt2--;
	}
}

uint8 is_lcd_driver_inited(void) {
	uint8 result = 0;
	#ifdef LCD_ASYNC_INIT
		if (Lcd_Driver_Init_State == Lcd_Driver_Init_State_Idle) {
			result = 1;
		}
	#else
		result = 1;
	#endif
	return result;
}

#ifdef LCD_ASYNC_INTERFACE
uint8 Send_Async_Cmd_LCD(uint8 x) {
	uint8 result = 1;
	lcd_Driver_Fifo[lcd_Driver_Fifo_write].lcd_Driver_IsCommandOrData = Lcd_Driver_Command;
	lcd_Driver_Fifo[lcd_Driver_Fifo_write].lcd_Driver_Payload = x;
	
	lcd_Driver_Fifo_write++;
	if (lcd_Driver_Fifo_write >= LCD_ASYNC_INTERFACE_ITEMS) {
		lcd_Driver_Fifo_write = 0;
	}

	result = 0;
	
	return result;
}

uint8 Send_Async_Data_LCD(uint8 x) {
	uint8 result = 1;
	lcd_Driver_Fifo[lcd_Driver_Fifo_write].lcd_Driver_IsCommandOrData = Lcd_Driver_Data;
	lcd_Driver_Fifo[lcd_Driver_Fifo_write].lcd_Driver_Payload = x;
	lcd_Driver_Fifo_write++;

	if (lcd_Driver_Fifo_write >= LCD_ASYNC_INTERFACE_ITEMS) {
		lcd_Driver_Fifo_write = 0;
	}
	
	result = 0;
	
	return result;
}
#endif

uint8 Send_Cmd_LCD(uint8 x) {
	uint8 result = 1;
	if (Wait_Busy() == NOTBUSY) {
		uint8 temp;
		R_W_LAT = WRITE; //0
		LCD_PORT_SYNC();
		RS_LAT = INSTRUCTION_REG; //0
		LCD_PORT_SYNC();
		lcd_driver_high_e();
		temp = (x & 0xF0);
		putDataToLCDPort(temp);
		lcd_driver_low_e();
		temp = ((x & 0x0F) << 4);
		lcd_driver_high_e();
		putDataToLCDPort(temp);
		lcd_driver_low_e();
	} else {
		result = 0;
	}
	return result;
}

uint8 Send_Data_LCD(uint8 x) {
	uint8 result = 1;
	if (Wait_Busy() == NOTBUSY) {
		uint8 temp;
		R_W_LAT = WRITE; //0
		LCD_PORT_SYNC();
		RS_LAT = DATA_REG; //1
		LCD_PORT_SYNC();
		lcd_driver_high_e();
		temp = (x & 0xF0);
		putDataToLCDPort(temp);
		lcd_driver_low_e();
		temp = ((x & 0x0F) << 4);
		lcd_driver_high_e();
		putDataToLCDPort(temp);
		lcd_driver_low_e();
	} else {
		result = 0;
	}
	return result;
}

static uint8 Wait_Busy(void) {
	uint8 result = BUSY;
	writeTimer(&lcd_driver_busyTimer, 4);
	while (1) {
		if (Check_Busy() == BUSY) {
			if (readTimer(&lcd_driver_busyTimer) == 0) {
				break;
			}
		} else {
			result = NOTBUSY;
			break;
		}	
	}
	return result;
}

static uint8 Check_Busy(void) {
	uint8 result = BUSY;
	uint8 busyFlag = 1;
	
	R_W_LAT = READ; //1
	LCD_PORT_SYNC();
	RS_LAT = INSTRUCTION_REG; //0
	LCD_PORT_SYNC();
	
	lcdTrisIn();

	lcd_driver_high_e();
	
	busyFlag = LCD_PORT7; //busy flag is needed sometimes
	
	lcd_driver_low_e();
	lcd_driver_high_e();
	lcd_driver_low_e();

	R_W_LAT = WRITE; //0
	LCD_PORT_SYNC();
	RS_LAT = DATA_REG; //1
	LCD_PORT_SYNC();

	lcdTrisOut();

	if (busyFlag) {
		result = BUSY;
	} else {
		result = NOTBUSY;
	}
	return result;
}

#ifdef LCD_USE_BACKLIGHT
	#ifndef LCD_USE_USER_BACKLIGHT
		void lcd_backlight_on(void) {
			BACKLIGHT = 1;
		}

		void lcd_backlight_off(void) {
			BACKLIGHT = 0;
		}
	#endif
#endif

static void lcd_driver_high_e(void) {
	delay100us();//should be 60us
	E_LAT = 1;
	LCD_PORT_SYNC();
}

static void lcd_driver_low_e(void) {
	delay100us();//should be 10us
	E_LAT = 0;
	LCD_PORT_SYNC();
	delay100us();//should be 10us
}

static void putDataToLCDPort(uint8 data) {
	LCD_LAT7 = (data & 0b10000000) && 0x01;
	LCD_PORT_SYNC();
	LCD_LAT6 = (data & 0b01000000) && 0x01;
	LCD_PORT_SYNC();
	LCD_LAT5 = (data & 0b00100000) && 0x01;
	LCD_PORT_SYNC();
	LCD_LAT4 = (data & 0b00010000) && 0x01;
	LCD_PORT_SYNC();
}

static void lcdTrisOut(void) {
	LCD_DATA7_TRIS = OUT;
	LCD_PORT_SYNC();
	LCD_DATA6_TRIS = OUT;
	LCD_PORT_SYNC();
	LCD_DATA5_TRIS = OUT;
	LCD_PORT_SYNC();
	LCD_DATA4_TRIS = OUT;
	LCD_PORT_SYNC();
}

static void lcdTrisIn(void) {
	LCD_DATA7_TRIS = IN;
	LCD_PORT_SYNC();
	LCD_DATA6_TRIS = IN;
	LCD_PORT_SYNC();
	LCD_DATA5_TRIS = IN;
	LCD_PORT_SYNC();
	LCD_DATA4_TRIS = IN;
	LCD_PORT_SYNC();
}

static void delay100us(void) {
	uint16 temp = 0;
	lock_isr();
	delayCnt2 = 2;
	unlock_isr();
	while (1) {
		lock_isr();
		temp = delayCnt2;
		unlock_isr();

		if (temp == 0) {
			break;
		}
	}
}

#ifdef LCD_ASYNC_INTERFACE
	static void start100us(void) {
		lock_isr();
		delayCnt2 = 2;
		unlock_isr();
	}

	static uint8 check100us(void) {
		uint8 result = 0;
		uint16 temp = 0;
		lock_isr();
		temp = delayCnt2;
		unlock_isr();

		if (temp == 0) {
			result = 1;
		}
		return result;
	}
#endif

