#include "oneWire.h"
#include "ds18b20.h"
#include <stdio.h>
#include "oneWireCRC.h"

volatile uint32 ds18b20cnt = 0;

DS18B20_POWERING ds18b20_powering = DS18B20_EXTERNAL_SUPPPLY;

static float ds18b20_convert_temperature(uint16 rawTemp);
static void ds18b20_read_scratchpad(OW_device ow_device, DS18B20_device *result);
uint8 conversion_state = 0;

void ds18b20_init(OW_device ow_device) {
	ds18b20_set_powering(ds18b20_detect_powering(ow_device));
}

DS18B20_POWERING ds18b20_detect_powering(OW_device ow_device) {
	DS18B20_POWERING result = DS18B20_EXTERNAL_SUPPPLY;
	if (!OW_reset_pulse()) {
		OW_address_device(ow_device);
		OW_write_byte(DS18B20_READ_POWER_SUPPLY);
		result = OW_read_bit();
		if (result) {
			//printf("Power 1\r\n");
		} else {
			//printf("Power 0\r\n");
		}
	} else {
		result = DS18B20_NO_POWER;
	}
	return result;
}

void ds18b20_set_powering(DS18B20_POWERING new_ds18b20_powering) {
	ds18b20_powering = new_ds18b20_powering;
}

void isr_ds18b201ms(void) {
	if (ds18b20cnt != 0) {
		ds18b20cnt--;
	}
}

int ds18b20_get_temperature_nb_restart(OW_device ow_device, float *temperature, uint8 restart) {
	if (restart) {
		conversion_state = 0;
	}
	return ds18b20_get_temperature_nb(ow_device, temperature);
}

int ds18b20_get_temperature_nb(OW_device ow_device, float *temperature) {
	int result = 0;
	DS18B20_device ds18b20_device;
	if (conversion_state == 0) {
		if (temperature != NULL) {
			if (!OW_reset_pulse()) {
				//present
				OW_address_device(ow_device);
				OW_write_byte(DS18B20_CONVERT_T);
				{
					switch (ds18b20_powering) {
						case DS18B20_PARASITIC : {
							drive_OW_high();
							ds18b20cnt = 1500;
							conversion_state = 1;
							break;
						}
						case DS18B20_EXTERNAL_SUPPPLY : {
							conversion_state = 1;
							break;
						}
						default : {
							ds18b20_set_powering(ds18b20_detect_powering(ow_device));
							conversion_state = 0;
							result = -1;
							break;
						}
					}
				}
			} else {
				conversion_state = 0;
				result = -1;
			}
		} else {
			conversion_state = 0;
			result = -1;
		}
	} else if (conversion_state == 1) {
		switch (ds18b20_powering) {
			case DS18B20_PARASITIC : {
				if (ds18b20cnt == 0) {
					read_OW();
					conversion_state = 2;
				} else {
				}
				break;
			}
			case DS18B20_EXTERNAL_SUPPPLY : {
				if (OW_read_bit() == 1) {
					//printf("Conversion done.\r\n");
					conversion_state = 2;
				}
				break;
			}
			default : {
				conversion_state = 0;
				result = -1;
				break;
			}
		}
	} else if (conversion_state == 2) {
		ds18b20_read_scratchpad(ow_device, &ds18b20_device);
		{
			uint8 myCRC = 0;
			uint8 receivedCRC = ds18b20_device.value.crc;
			myCRC =  calculateOneWireCRC(ds18b20_device.byte, (sizeof(ds18b20_device.byte) / sizeof(*ds18b20_device.byte)));
			if (myCRC == receivedCRC) {
				if (ds18b20_device.value.reserved3 == 0x10) {
					*temperature = ds18b20_convert_temperature(ds18b20_device.value.temperature);
					conversion_state = 0;
					result = 1;
				} else {
					result = -1;
					conversion_state = 0;
				}
			} else {
				result = -1;
				conversion_state = 0;
			}
		}
	} else {
		result = -1;
		conversion_state = 0;
	}
	return result;
}

void ds18b20_set_resolution(OW_device ow_device, DS18B20_resolution resolution) {
	DS18B20_device ds18b20_device;
	uint8 config = 0x7F;
	switch (resolution) {
		case DS18B20_9bit : {
			config = 0b00011111;
			break;
		}
		case DS18B20_10bit : {
			config = 0b00111111;
			break;
		}
		case DS18B20_11bit : {
			config = 0b01011111;
			break;
		}
		case DS18B20_12bit : {
			config = 0b01111111;
			break;
		}
		default : {
			config = 0b01111111;
			break;
		}
	}
	ds18b20_read_scratchpad(ow_device, &ds18b20_device);
	if (ds18b20_device.value.reserved3 == 0x10) {
		if (!OW_reset_pulse()) {
			DS18B20_device ds18b20_device;
			//present
			OW_address_device(ow_device);
			OW_write_byte(DS18B20_WRITE_SCRATCHPAD);
			OW_write_byte(ds18b20_device.value.thigh);
			OW_write_byte(ds18b20_device.value.tlow);
			OW_write_byte(config);
		} else {
		}	
	} else {
	}
}

static void ds18b20_read_scratchpad(OW_device ow_device, DS18B20_device *result) {
	uint8 tempData[9];
	uint16 i = 0;
	if (result != NULL) {
		result->longint[0] = 0;
		result->longint[1] = 0;
		if (!OW_reset_pulse()) {
			uint16 tempInt = 0;
			//present
			OW_address_device(ow_device);
			OW_write_byte(DS18B20_READ_SCRATCHPAD);
			//printf("------------\r\n");
			for (i = 0; i < (sizeof(tempData) / sizeof(*tempData)); i++) {
				tempData[i] = OW_read_byte();
				//result.byte[i] = OW_read_byte();
				//printf("Data[%u]:0x%2.2X\r\n", i, result.byte[i]);
			}
			tempInt = tempData[1];
			tempInt <<= 8;
			tempInt += tempData[0];
			result->value.temperature = tempInt;
			result->value.thigh = tempData[2];
			result->value.tlow = tempData[3];
			result->value.configuration = tempData[4];
			result->value.reserved1 = tempData[5];
			result->value.reserved2 = tempData[6];
			result->value.reserved3 = tempData[7];
			result->value.crc = tempData[8];
			//printf("Temperature:0x%4.4X %u\r\n", result.value.temperature, result.value.temperature / 16);
		} else {
			//not present
		}
	}
}

static float ds18b20_convert_temperature(uint16 rawTemp) {
	float result = 0;
	int temp = 0;
	temp = (int)rawTemp;
	result = temp;
	//printf("Raw:%u\r\n", rawTemp);
	//printf("Raw:%d\r\n", temp);
	result /= 16.0;
	//printf("Raw:%d\r\n", (int)result);
	return result;
}
