/*
     IEC-ATA, a hard drive controller for the CBM IEC bus 
     Copyright (C) 2002  Asbjrn Djupdal
     
     This program is free software; you can redistribute it and/or
     modify it under the terms of the GNU General Public License
     as published by the Free Software Foundation; either version 2
     of the License, or (at your option) any later version.
     
     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.
     
     You should have received a copy of the GNU General Public License along
     with this program; if not, write to the Free Software Foundation, Inc.,
     59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.

     ----

     The author of IEC-ATA may be reached by electronic mail:
     
       djupdal@idi.ntnu.no

     or if the email address no longer is valid, by paper mail:

       Asbjrn Djupdal
       Grnngjelet 50
       4640 SGNE
       NORWAY
*/

/*
  iec.c V1.0
  Contains the IEC serial bus protocol code
*/

#include "iecata.h"

#define JUMPER  PB0

#define IEC_CLK_OUT  PD0
#define IEC_DATA_OUT PD1
#define IEC_ATN_IN   PD2
#define IEC_CLK_IN   PD3
#define IEC_DATA_IN  PD4

#define DEVICENUMBER  10

#define LISTEN    0x20
#define TALK      0x40
#define UNLISTEN  0x3f
#define UNTALK    0x5f
#define OPEN      0xf0
#define CLOSE     0xe0
#define DATA      0x60

/* protos ****************************************************************/

static bool_t iecGetByte (uint8_t *data);
static void iecPutByte (uint8_t data, bool_t eoi);
static void iecDelay (void);

/* variables *************************************************************/

volatile bool_t attention;

/* functions *************************************************************/

bool_t iecListen (uint8_t *data, bufferSize_t maxBytes,
                  bufferSize_t *bytesReceived) {
  bool_t eoi = FALSE;
      
  /* init bytes received */
  *bytesReceived = 0;

  while (!eoi && (*bytesReceived < maxBytes)) {
    /* wait for talker ready to send */
    loop_until_bit_is_set (PIND, IEC_CLK_IN);
    /* fetch next byte */
    eoi = iecGetByte (data + (*bytesReceived)++);
  }

  /* Release line after a suitable (?) pause */
  /* TODO: find the correct delay.
     This seems to cause errors sometimes */
  if (eoi) {
    iecDelay();
    cbi (PORTD, IEC_DATA_OUT);
  }    

  return eoi;
}

/* bytesToSend must be > 0 */
inline extern void iecTalk (uint8_t *data, bufferSize_t bytesToSend,
              bufferSize_t *bytesSendt, bool_t eoi) {

  /* init bytes sendt */
  *bytesSendt = 0;

  while (!attention && (*bytesSendt < bytesToSend)) {
    /* signal ready to send */
    cbi (PORTD, IEC_CLK_OUT);
    /* wait for listener */
    while (!attention && bit_is_clear (PIND, IEC_DATA_IN));
    if (!attention) {
      if (*bytesSendt == (bytesToSend - 1)) {
        /* send last byte */
        iecPutByte (data[*bytesSendt], eoi);
      } else {
        /* send byte */
        iecPutByte (data[*bytesSendt], FALSE);
      }
      /* increase byte counter */
      (*bytesSendt)++;
    }
  }

  /* Release line after a suitable (?) pause */
  /* TODO: find the correct delay.
     This seems to cause errors sometimes */
  if (eoi) {
    iecDelay();
    cbi (PORTD, IEC_CLK_OUT);
  }    
}

inline extern void iecAttention (void) {
  cli();
  if (attention) {
    static uint8_t data[2];
    uint8_t *dataPtr = &data[0];

    /* LED off */
    sbi (PORTD, LED);

    /* reset attention flag */
    attention = FALSE;

    /* default command */
    command = IDLE;

    /* fetch control bytes from iec talker */
    while (TRUE) {
      /* quit interrupt when IEC ATN is released */
      if (bit_is_set (PIND, IEC_ATN_IN)) {
        /* decode received control data */
        if ((data[0] & 0x1f) == DEVICENUMBER) {
          /* get secondary address */
          channelNumber = data[1] & 0x0f;

          if ((data[0] == UNLISTEN) || (data[0] == UNTALK)) {
            /* release IEC DATA */
            cbi (PORTD, IEC_DATA_OUT);
          } else if ((data[0] & 0xe0) == LISTEN) {
            if ((data[1] & 0xf0) == OPEN) {
              command = LISTEN_OPEN;
              error = 0;
            } else if ((data[1] & 0xf0) == CLOSE) {
              if (!error) {
                command = LISTEN_CLOSE;
              }
              /* release IEC DATA */
              cbi (PORTD, IEC_DATA_OUT);
            } else if ((data[1] & 0xf0) == DATA) {
              if (error) {
                /* release IEC DATA */
                cbi (PORTD, IEC_DATA_OUT);
              } else {
                command = LISTEN_DATA;
              }
            }
          } else if ((data[0] & 0xe0) == TALK) {
            /* do turn-around sequence */
            /* wait for IEC CLK */
            loop_until_bit_is_set (PIND, IEC_CLK_IN);
            /* release IEC DATA */
            cbi (PORTD, IEC_DATA_OUT);
            /* delay */
            iecDelay();
            /* take IEC CLK */
            sbi (PORTD, IEC_CLK_OUT);
            /* delay */
            iecDelay();
            if (!error || (channelNumber == COMMAND_CHANNEL)) {
              command = TALK_DATA;
            } else {
              /* release IEC CLK */
              cbi (PORTD, IEC_CLK_OUT);
            }
          }
        } else {
          /* release IEC DATA */
          cbi (PORTD, IEC_DATA_OUT);
        }
        break;
      }
      /* fetch new byte if IEC CLK is released */
      if (bit_is_set (PIND, IEC_CLK_IN)) {
        iecGetByte (dataPtr++);
      }
    }
  }
  sei();
}

bool_t iecGetByte (uint8_t *data) {
  bool_t eoi = FALSE;

  /* signal ready to listen */
  cbi (PORTD, IEC_DATA_OUT);
    
  /* wait for all other devices */
  loop_until_bit_is_set (PIND, IEC_DATA_IN);

  /* wait for talker, maximum 200us */
  outp (0x00, TCNT0);
  while (!eoi) {
    if (inp (TCNT0) > 100) {
      eoi = TRUE;
    }
    if (bit_is_clear (PIND, IEC_CLK_IN)) {
      break;
    }
  }
  
  /* acknowledge if eoi */
  if (eoi) {
    /* set IEC DATA */
    sbi (PORTD, IEC_DATA_OUT);
    /* delay */
    iecDelay();
    /* clear IEC DATA */
    cbi (PORTD, IEC_DATA_OUT);
    /* wait for talker */
    loop_until_bit_is_clear (PIND, IEC_CLK_IN);
  }

  { /* fetch byte */
    uint8_t bitnumber;
    
    for (bitnumber = 0; bitnumber < 8; bitnumber++) {
      /* shift result right */
      *data >>= 1;

      /* wait for next bit */
      loop_until_bit_is_set (PIND, IEC_CLK_IN);
      /* fetch bit */
      if (bit_is_set (PIND, IEC_DATA_IN)) {
        *data |= 0x80;
      }

      /* wait for IEC CLK */
      loop_until_bit_is_clear (PIND, IEC_CLK_IN);
    }

    /* acknowledge byte received */
    sbi (PORTD, IEC_DATA_OUT);
  }

  return eoi;
}

void iecPutByte (uint8_t data, bool_t eoi) {
  if (eoi) {
    /* wait for listener to acknowledge EOI */
    loop_until_bit_is_clear (PIND, IEC_DATA_IN);
    /* wait for listener */
    loop_until_bit_is_set (PIND, IEC_DATA_IN);
  }

  /* wait for listener to detect signal changes */
  iecDelay();

  /* set IEC CLK */
  sbi (PORTD, IEC_CLK_OUT);

  { /* send byte */
    uint8_t bitnumber;

    for (bitnumber = 0; bitnumber < 8; bitnumber++) {
      /* delay */
      iecDelay();
      /* put data on line */
      if (data & 0x01) {
        cbi (PORTD, IEC_DATA_OUT);
      } else {
        sbi (PORTD, IEC_DATA_OUT);
      }
      /* signal data ready */
      cbi (PORTD, IEC_CLK_OUT);
      /* delay */
      iecDelay();
      /* set IEC CLK */
      sbi (PORTD, IEC_CLK_OUT);
      /* release IEC DATA */
      cbi (PORTD, IEC_DATA_OUT);
      /* shift data right */
      data >>= 1;
    }
    /* wait for listener to ackowledge */
    /* TODO: timeout */
    loop_until_bit_is_clear (PIND, IEC_DATA_IN);
  }
}

static void iecDelay (void) {
  outp (0x00, TCNT0);

  if (bit_is_set (PINB, JUMPER)) {
    /* slow (C64) timing: > 60us */
    while (inp (TCNT0) < 35);
  } else {
    /* fast (VIC20) timing: > 20us */
    while (inp (TCNT0) < 15);
  }
}

INTERRUPT (SIG_INTERRUPT0) {
  /* wait a short delay to ensure host has all lines ready */
  outp (0x00, TCNT0);
  while (inp (TCNT0) < 100);
  
  /* release IEC CLK */
  cbi (PORTD, IEC_CLK_OUT);

  /* set IEC DATA */
  sbi (PORTD, IEC_DATA_OUT);

  /* set attention flag */
  attention = TRUE;
}

