/*
     IEC-ATA, a hard drive controller for the CBM IEC bus 
     Copyright (C) 2002  Asbjrn Djupdal
     
     This program is free software; you can redistribute it and/or
     modify it under the terms of the GNU General Public License
     as published by the Free Software Foundation; either version 2
     of the License, or (at your option) any later version.
     
     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.
     
     You should have received a copy of the GNU General Public License along
     with this program; if not, write to the Free Software Foundation, Inc.,
     59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.

     ----

     The author of IEC-ATA may be reached by electronic mail:
     
       djupdal@idi.ntnu.no

     or if the email address no longer is valid, by paper mail:

       Asbjrn Djupdal
       Grnngjelet 50
       4640 SGNE
       NORWAY
*/

/*
  iecata.c V1.0
  The main source file for the IEC-ATA software, contains main()
*/

#include "iecata.h"

/* include c-files instead of linking; saves program space because
   functions may be declared inline and extern */
#include "ata.c"
#include "iec.c"
#include "dos-file.c"
#include "dos-dir.c"
#include "dos-init.c"

#define BASIC_LINE_LENGTH    33 /* TODO: check this number */
#define CDOS_DIRENTRY_LENGTH 32

/* error numbers, given in bcd */
#define NO_ERROR         0x00
#define INIT_ERROR       0x20
#define CREATE_ERROR     0x21
#define NOT_OPEN_ERROR   0x22
#define NOT_FOUND_ERROR  0x23
#define SYNTAX_ERROR     0x24
#define VERSION_ERROR    0x25

/* protos ****************************************************************/

void init (void);
void updateLED (void);
bool_t readStatus (struct channelTableStruct *channel);
bool_t readDir (struct channelTableStruct *channel);
static void parseCommand (void);
void parseName (struct channelTableStruct *channel);
uint8_t filenamelen (char *filename);

int main (void);

/* variables *************************************************************/

uint8_t command;
uint8_t error;
uint8_t channelNumber;

/* functions *************************************************************/

inline extern void init (void) {
  /* enable external SRAM */
  sbi (MCUCR, SRE);

  /* set up I/O pins */
  outp (0x00, DDRA);
  outp (0x00, PORTA);
  outp (0xff, DDRB);
  outp (0xff, PORTB);
  outp (0x00, DDRC);
  outp (0x00, PORTC);
  outp (0xe3, DDRD);
  outp (0xc0, PORTD);

  /* enable timer */
  outp (0x02, TCCR0);

  /* setup interrupt */
  sbi (MCUCR, ISC01);
  cbi (MCUCR, ISC00);
  sbi (GIMSK, INT0);

  /* init variables */
  command = IDLE;
  attention = FALSE;

  error = INIT_ERROR;
  /* init submodules */
  if (ataInit()) {
    if (dosInit()) {
      error = VERSION_ERROR;
    }
  }

  /* turn off LED */
  sbi (PORTD, LED);

  /* Enable interrupts */
  sei();
}

inline extern void updateLED (void) {
  /* blink LED if error */
  if (error && (error != VERSION_ERROR)) {
    static uint8_t cnt;
      
    if (cnt > 250) {
      if (bit_is_set (PORTD, LED)) {
        /* LED on */
        cbi (PORTD, LED);
      } else {
        /* LED off */
        sbi (PORTD, LED);
      }
      cnt = 0;
    }
    /* timer delay */
    outp (0x00, TCNT0);
    while (inp (TCNT0) < 200);
    /* update counter */
    cnt++;
  }
}

inline extern bool_t readStatus (struct channelTableStruct *channel) {
  uint8_t *buffer = channel->buffer;

  /* error number */
  *(buffer++) = (error >> 4) | '0';
  *(buffer++) = (error & 0x0f) | '0';

  /* space */
  *(buffer++) = ',';
  *(buffer++) = ' ';

  { /* error message */
    uint8_t bufferAdd;
    char *errorMessage;
    
    switch (error) {
      case NO_ERROR:
        errorMessage = PSTR ("OK");
        bufferAdd = 2;
        break;
      case INIT_ERROR:
        errorMessage = PSTR ("INIT ERROR");
        bufferAdd = 10;
        break;
      case CREATE_ERROR:
        errorMessage = PSTR ("CREATE ERROR");
        bufferAdd = 12;
        break;
      case NOT_OPEN_ERROR:
        errorMessage = PSTR ("NOT OPEN");
        bufferAdd = 8;
        break;
      case NOT_FOUND_ERROR:
        errorMessage = PSTR ("NOT FOUND");
        bufferAdd = 9;
        break;
      case VERSION_ERROR:
        errorMessage = PSTR ("IEC-ATA V1.0");
        bufferAdd = 12;
        break;
      default:
      case SYNTAX_ERROR:
        errorMessage = PSTR ("SYNTAX ERROR");
        bufferAdd = 12;
        break;
    }
    memcpy_P (buffer, errorMessage, bufferAdd);

    buffer += bufferAdd;

    /* clear error */
    error = NO_ERROR;
  }

  /* track and sector (not used; always 0,0) */
  memcpy_P (buffer, PSTR (",00,00\x0d"), 7);
  buffer += 7;

  /* record buffer length */
  channel->endOfBuffer = buffer - channel->buffer;

  return TRUE;
}

inline extern bool_t readDir (struct channelTableStruct *channel) {
  bool_t eof = FALSE;
  uint8_t *buffer = channel->buffer;
  static entryIndex_t entryIndex;
  static uint8_t writtenEntries;

  if (channel->readDirState == READ_DIR_BEGIN) {
    if (channelNumber == 0) {
      memcpy_P (buffer, PSTR ("\x01\x04\x01\x01\x00\x00"
                              "\x12\"IEC-ATA V1.0    \" AD 2A\x00"), 32);
      buffer += 32;
    } else {
      memset (buffer, 255, 142);
      memcpy_P (buffer, PSTR ("\x41\x00"), 2);
      buffer += 142;
      memset (buffer, 0, 112);
      memcpy_P (buffer, PSTR ("IEC-ATA V1.0\xa0\xa0\xa0\xa0\xa0\xa0"
                              "AD\xa0""2A\xa0\xa0\xa0\xa0"), 27);
      buffer += 112;
    }

    /* not begin anymore */
    channel->readDirState = READ_DIR_PROGRESS;
    /* start at dir index 0 */
    entryIndex = 0;
    writtenEntries = 0;
  }

  /* put directory */
  while ((channel->readDirState == READ_DIR_PROGRESS) &&
         (buffer < (channel->buffer + BLOCKSIZE - (BASIC_LINE_LENGTH + 20)))) {

    struct dirEntryStruct *entry;

    /* get entry */
    if ((entry = getEntry (entryIndex))) {
      /* only process non-deleted files */
      if (entry->startBlock) {
        /* only show files that match pattern */
        if (filenameMatch (entry->fileName, channel->dirEntry.fileName) ||
            !(channel->dirEntry.fileName)) {
          fileSize_t fileSize = entry->fileSize;

          { /* convert fileSize to number of 254 byte blocks (like 1541) */
            fileSize *= 2;
            uint16_t extraBytes = entry->bytesInLastBlock + (fileSize * 2);

            if (fileSize) {
              fileSize--;
            }

            while (extraBytes >= 254) {
              fileSize++;
              extraBytes -= 254;
            }
          }

          if (channelNumber == 0) {
            /* pointer to next line */
            *(buffer++) = 1;
            *(buffer++) = 1;
            /* linenumber */
            *(buffer++) = (uint8_t)fileSize;
            *(buffer++) = (uint8_t)(fileSize >> 8);
            /* clear line */
            memset (buffer, ' ', BASIC_LINE_LENGTH - 5);
            /* space in beginning of line */
            if (fileSize < 1000) buffer++;
            if (fileSize < 100) buffer++;
            if (fileSize < 10) buffer++;
            { /* filename */
              uint8_t fileNameSize = filenamelen (entry->fileName);
              /* quotes */
              *(buffer++) = '"';
              /* name */
              memcpy (buffer, entry->fileName, fileNameSize);
              buffer += fileNameSize;
              /* quotes */
              *(buffer++) = '"';
              /* spaces */
              buffer += FILE_NAME_SIZE - fileNameSize;
            }
            /* splat */
            if (entry->splat) *buffer = '*';
            buffer++;
            /* filetype */
            switch (entry->fileType) {
              case PRG:
                memcpy_P (buffer, PSTR ("PRG"), 3);
                break;
              case SEQ:
                memcpy_P (buffer, PSTR ("SEQ"), 3);
                break;
              case DIR:
                memcpy_P (buffer, PSTR ("DIR"), 3);
                break;
            }
            buffer += 3;
            /* TODO: locked */
            /* end of line */
            *(buffer++) = 0;
          } else {
            /* clear direntry */
            memset (buffer, 0, CDOS_DIRENTRY_LENGTH);
            /* file type */
            *buffer = entry->fileType;
            if (!entry->splat) *buffer |= 0x80;
            /* todo: locked */
            buffer += 3;
            /* name */
            memset (buffer, 160, FILE_NAME_SIZE);
            memcpy (buffer, entry->fileName, filenamelen (entry->fileName));
            buffer += (FILE_NAME_SIZE + 9);
            /* file size */
            *(buffer++) = (uint8_t)fileSize;
            *(buffer++) = (uint8_t)(fileSize >> 8);
            /* write to bytes of 0 if not at end of 1541 block */
            writtenEntries++;
            if (writtenEntries == 8) {
              writtenEntries = 0;
            } else {
              buffer += 2;
            }
          }
        }
      }
      entryIndex++;
    } else {
      if (channelNumber == 0) {
        /* put blocks free */
        memcpy_P (buffer,
                  PSTR ("\x01\x01\xff\xf9""BLOCKS FREE.\x00\x00\x00\x00"),
                  20);
        buffer += 20;
      }
      /* read dir finished */
      channel->readDirState = READ_DIR_FINISHED;
    }
  }

  if (channel->readDirState == READ_DIR_FINISHED) {
    eof = TRUE;
  }

  /* number of bytes to save */
  channel->endOfBuffer = buffer - channel->buffer;

  return eof;
}

void parseCommand (void) {
  static uint8_t command[255];
  uint8_t *cmdArg1;
  uint8_t *cmdArg2;

  { /* get message */
    bufferSize_t bytesReceived;

    iecListen (command, 255, &bytesReceived);

    /* make message a proper string */
    command[bytesReceived] = '\0';
  }

  /* get arg1 */
  if ((cmdArg1 = strchr (command, ':'))) {
    *(cmdArg1++) = '\0';
  }

  /* get arg2 */
  if ((cmdArg2 = strchr (cmdArg1, '='))) {
    *(cmdArg2++) = '\0';
  } else {
    cmdArg2 = cmdArg1;
  }

  { /* erase possible CR at end of arg2 */
    uint8_t *cr;

    if ((cr = strchr (cmdArg2, 0x0d))) {
      *cr = '\0';
    }
  }

  { /* interpret and execute command */
    char c1 = *command;
    char c2 = *(command + 1);

    if ((c1 == 'C') && (c2 == 'D')) {
      /* change directory */
      if (!setCurrentDir (cmdArg1)) {
        error = NOT_FOUND_ERROR;
      }
    } else if ((c1 == 'M') && (c2 == 'D')) {
      /* create directory */
      if (!createDir (cmdArg1)) {
        error = CREATE_ERROR;
      }
    } else if ((c1 == 'S') && (c2 == 'D')) {
      /* delete directory */
      deleteDir (cmdArg1);
    } else if (c1 == 'S') {
      /* delete file */
      deleteFile (cmdArg1);
    } else if (c1 == 'R') {
      /* rename entry */
      if (!renameEntry (cmdArg1, cmdArg2)) {
        error = NOT_FOUND_ERROR;
      }
    } else if (c1 == 'N') {
      /* format drive */
      if (!formatDrive()) {
        error = INIT_ERROR;
      }
    } else if (c1 == 'I') {
      /* initialize */
    } else {
      /* not a valid command */
      error = SYNTAX_ERROR;
    }
  }
}

inline extern void parseName (struct channelTableStruct *channel) {
  static uint8_t commandBuffer[255];
  uint8_t *bufferPtr = commandBuffer;

  bool_t overwrite = FALSE;
  char *filename;
  uint8_t filetype;
  bool_t read;

  { /* get string */
    bufferSize_t bytesReceived;

    iecListen (commandBuffer, 255, &bytesReceived);

    /* make buffer a proper string */
    commandBuffer[bytesReceived] = '\0';
  }

  channel->readDirState = NOT_READ_DIR;

  switch (*bufferPtr) {
    case '@':
      /* overwrite */
      overwrite = TRUE;
      bufferPtr++;
      break;
    case '$':
      /* directory */
      channel->readDirState = READ_DIR_BEGIN;
      bufferPtr++;
      break;
  }

  { /* skip drive specifier if present */
    char *ptr;

    if ((ptr = strchr (bufferPtr, ':'))) {
      bufferPtr = ptr + 1;
    }
  }

  filename = bufferPtr;

  /* file type and direction */
  switch (channelNumber) {
    case 0:
      /* read PRG */
      filetype = PRG;
      read = TRUE;
      break;
    case 1:
      /* write PRG */
      filetype = PRG;
      read = FALSE;
      break;
    default:
      filetype = ANY; /* read anything, write SEQ */
      read = TRUE;
      break;
  }

  { /* override file type and direction with data extracted from file name */
    char *ptr = NULL;

    do {
      if ((ptr = strchr (bufferPtr, ','))) {
        *ptr = '\0'; /* to make sure filename is properly ended */
        bufferPtr = ptr + 1;

        switch (*bufferPtr) {
          case 'S':
            filetype = SEQ;
            break;
          case 'P':
            filetype = PRG;
            break;
          case 'W':
            read = FALSE;
            if (filetype == ANY) {
              filetype = SEQ;
            }
            break;
          case 'R':
            read = TRUE;
            break;
        }
      }
    } while (ptr);
  }

  if (channel->readDirState) {
    /* directory */
    if (read) {
      channel->fileState = READ_FILE;
      /* load "$" or load "$0" ==> filename = "*" */
      if (((*filename == 0) && (*(filename - 1) != ':')) ||
          ((*filename == '0') && (*(filename + 1) == 0))) {
        *filename = '*';
        *(filename + 1) = 0;
      }
      /* copy filename (to be used by pattern matching) */
      memcpy (channel->dirEntry.fileName, filename, FILE_NAME_SIZE);
    } else {
      error = CREATE_ERROR;
    }
  } else {
    /* normal file */
    if (read) {
      /* open file */
      if (!openRead (filename, filetype, channelNumber)) {
        error = NOT_FOUND_ERROR;
      } else {
        channel->fileState = READ_FILE;
      }
    } else {
      /* delete old file */
      if (overwrite) {
        deleteFile (filename);
      }
      /* open file */
      if (!openWrite (filename, filetype, channelNumber)) {
        error = CREATE_ERROR;
      } else {
        channel->fileState = WRITE_FILE;
      }
    }
  }
}

int main (void) {
  /* I/O register setup, etc. */
  init();

  /* main loop */
  while (TRUE) {
    struct channelTableStruct *channel;

    /* blink LED if error condition */
    updateLED();

    /* service commanding device */
    iecAttention();

    /* get pointer to channel structure */
    channel = &channelTable[channelNumber];

    /* execute command */
    switch (command) {

      case LISTEN_OPEN: {
        /* reset variables */
        channel->bufferPtr = 0;
        channel->endOfBuffer = 0;

        if (channelNumber == COMMAND_CHANNEL) {
          /* get command and execute it */
          parseCommand();
        } else {
          /* normal data channel, get file name and open file */
          parseName (channel);
        }
        break;
      }

      case LISTEN_CLOSE:
        if (channelNumber == COMMAND_CHANNEL) {
          /* close all files */
          uint8_t i;
          for (i = 0; i < 15; i++) {
            closeFile (i);
          }
        } else {
          /* close requested file */
          closeFile (channelNumber);
        }
        break;

      case LISTEN_DATA: {
        /* TODO: attention check may be necessary */
        if (channelNumber == COMMAND_CHANNEL) {
          /* status channel must be reset before each command */
          channel->bufferPtr = 0;
          channel->endOfBuffer = 0;
          parseCommand();
        } else {
          if (channel->fileState == WRITE_FILE) {
            bool_t eoi;

            do {
              bufferSize_t bytesReceived;
              bufferSize_t *bufPtr = &(channel->bufferPtr);
        
              /* receive bytes */
              eoi = iecListen (channel->buffer + *bufPtr,
                               BLOCKSIZE - *bufPtr, &bytesReceived);

              /* update bufferPtr */
              *bufPtr += bytesReceived;

              /* save to disk */
              if (*bufPtr == BLOCKSIZE) { 
                writeFile (channelNumber);
                *bufPtr = 0;
              }
            } while (!eoi);

          } else {
            error = NOT_OPEN_ERROR;
          }
        }
        break;
      } 

      case TALK_DATA: {
        if ((channel->fileState == READ_FILE) ||
            (channelNumber == COMMAND_CHANNEL)) {
          bool_t done = FALSE;

          while (!done && !attention) {
            static bool_t eof;

            /* get new block of data */
            if (channel->bufferPtr == channel->endOfBuffer) {

              /* start new block at beginning */
              channel->bufferPtr = 0;
              eof = FALSE;

              if (channelNumber == COMMAND_CHANNEL) {
                eof = readStatus (channel);
              } else if (channel->readDirState) {
                eof = readDir (channel);
              } else {
                readFile (channelNumber, &eof);
                if (eof) {
                  channel->endOfBuffer = channel->dirEntry.bytesInLastBlock;
                } else {
                  channel->endOfBuffer = BLOCKSIZE;
                }
              }
            }

            { /* send data block */
              bufferSize_t bytesSendt;

              iecTalk (channel->buffer + channel->bufferPtr,
                       channel->endOfBuffer - channel->bufferPtr,
                       &bytesSendt, eof);
              channel->bufferPtr += bytesSendt;
            }

            done = eof;

          }
        } else {
          error = NOT_OPEN_ERROR;
        }
        break;
      }

    }
    command = IDLE;

  }

  return 0;
}

uint8_t filenamelen (char *filename) {
  uint8_t len = 0;
  
  while (*filename && (len < 16)) {
    filename++;
    len++;
  }

  return len;
}
