	list	b=4
;************************************************************************
; Soubry Henk wrote:													*
; "Following the example of Bob Blick's original propeler clock I made 	*
; my own version of this fancy clock and have re-written the code from	*
; scratch. 																*
; This 16f628 code is for the stationary part of the clock and will 	*
; drive the primary transformer coil, and use RC5 to switch the motor	*
; on and off (and adjust the speed)"									*
;																		*
; - Rewritten, extended to an alarm clock:								*
;	- Deap sleep mode - No power on coil								*
;	- Auto standby timer - Can be disabled								*
;	- Wake up in RUN or SLEED mode										*
;	- Settings can be stored in on chip EEProm							*
;	- Fixed PR2 values .49, improuved time counting using TMR2			*
;																		*
;************************************************************************
;                                                        		       	*
;    Filename:	    base683.asm                                        	*
;    Date:          20/08/2002                                        	*
#define LastUpdate	"2010-10-02";										*
#define	VersionMajor	0x02	;										*
#define	VersionMinor	0x02	;										*
;                                                                     	*
;  Based on:                                                           	*
;    Author:        Soubry Henk                                       	*
;    Company:       Soubry Software Service                           	*
;                                                                     	*
;  Rewritten - Extended                                                	*
;		Hp41C		www.hobbielektronika.hu								*
;                                                                     	*
;************************************************************************
;                                                                    	*
;    Files required:                                                  	*
;           keys683.asm                                                	*
;                                                                     	*
;************************************************************************
;                                                                     	*
;    Notes:                                                           	*
;    	Pin assignment                                                	*
;      		Port GPIO         								       	  	*
;				GP0 = Index LED output									*
;				GP1 = Motor Shutdown output								*
;               GP2 = CCP1 PWM output - FET GATE for coil				*
;               GP3 = Ir decoder input use 10k pullup to Vdd		  	*
;               GP5.4 = 20 MHz quartz									*
;																		*
;************************************************************************

; Disabling messages
		errorlevel	-302

	ifdef	__12F683
		list	p=12f683              ; 12F683 can be used
		#include <p12f683.inc>        ; processor specific variable definitions
EEPROM_SIZE	EQU	128
#define ProcType	"12F683"
    endif
	ifndef	EEPROM_SIZE
		error	"Incompatible processor type selected"
	endif

	#include "keys683.asm"

#define RC5AddrCheck			; Comment this line if RC5 Address checking not required
; Auto standby time in seconds
AutoStbyTime	EQU	.3600

; Default mode settings at first power up
#define	AutoStandby				; Comment this line to disable auto standby
;#define	WakeUpRun				; Comment this line if wake up in sleep mode
#define	DeapSleeping			; Comment this line if coil has to be powered in sleep mode

; PWM settings for Sleep mode
CCPR1LSleep	EQU	0x0C

; PWM settings for Run mode
CCPR1LRUN	EQU	0x28

	__CONFIG _IESO_OFF & _CP_OFF & _CPD_OFF & _WDT_OFF & _BOD_ON & _PWRTE_ON & _HS_OSC & _MCLRE_OFF

	__idlocs		0xB000 | (VersionMajor << 8) | (VersionMinor)

;***** VARIABLE DEFINITIONS

;Vars in shared memory. This memory is available from any bank

  cblock	0x070
	Scratch		; memory locations for general use
	Scratch2	;
	Scratch3	; Alarm index
	BCD			; BCD routine temp var

	flags		;
	Mode		;

	w_temp		; variable used for context saving
	status_temp	; variable used for context saving
	fsr_temp	; variable used for context saving

LastCommon:
  endc
	IF LastCommon > 0x7F
		ERROR "To many variables used in Common RAM"
	ENDIF

  cblock	0x20
;Vars for time-keeping
  ;!! Keep variable order - Timecheck use indirect access to them v
	Second2			; 0..119 counting half seconds
	Minute			; 0.. 59
	Hour			; 0.. 23
	Day				; 1.. Dmon-1
	Month			; 1.. 12
	Year			; 0.. 99
	Century			; 0.. 99
  ;!! Keep variable order - Timecheck use indirect access to them ^

	DMon			; days in current Month + 1
	WDay			; Day of week 0..6

	SubSec_L		; Sub second timer
	SubSec_H

	SleepTimer		; Down counter in second, started at standby.

	StandbyTimerL	; Down counter for auto standby
	StandbyTimerH	;

	AutoStbyTimL	; Auto Standby time in seconds
	AutoStbyTimH	;

	CCP1_Sleep		; PWM pulse width for sleep mode
	CCP1_Run		; PWM pulse width for run   mode

	RC5_flags		; RC5 decoding state flags
	RC5_Tmr			; RC5 time counter
	RC5_BitCnt		; RC5 decoders bit counter
	RC5_Addr		; Address field of RC5 message just received
	RC5_Cmd			; Command field of RC5 message just received
	RC5_Cmd2		; storage for previous cmd
	RC5_ShrA
	RC5_ShrC

LastBank0:
  endc
	IF LastBank0 >= 0x70
		ERROR "To many variables used in Bank0"
	ENDIF

;Vars on Bank1
  cblock	0x0A0	; Only 32 locations on 12F683
	; Not to use memory 0x0F0..0x0FF  -  it will overwrite 0x070..0x07F
LastBank1:
  endc
	IF LastBank1 >= 0xC0
		ERROR "To many variables used in Bank1"
	ENDIF

;**********************************************************************

; GPIO bits
bIndexLed	EQU	0				; Index led output
bMotorOFF	EQU	1				; Motor control output
bTrafo		EQU	2
bRC5inp		EQU	3				; IR receiver's output

; Flags
bStandBy	EQU	0				; Standby mode
bSleeping	EQU	1				; Sleep mode
bCoilPower	EQU	2				; High power on coil
bIndexOn	EQU	3				; Index LED on
;			EQU	4
;			EQU	5
bNewTime	EQU	6				; Check the time
;bTimeInv	EQU	7				; Time invalid

; RC5_flags
bRC5_WaitStart	EQU	0
bRC5_DataReady	EQU	1
bRC5_Idle		EQU	2
bRC5_prev_inp	EQU	bRC5inp		; Has to be on the same bit position as bRC5inp
bRC5_HalfBit	EQU	4
;				EQU	5
bRC5_ReSynced	EQU	6
;				EQU	7

; Mode bits
bAutoStby		EQU	0			; Auto standby timer enable
bWakeRun		EQU	1			; Wake Run afther power up
bDeapSleep		EQU	2			; No coil drive in standby mode
;				EQU	3
;				EQU	4
;				EQU	5
;				EQU	6
;				EQU	7

;**********************************************************************

; RC5_flags
#define		RC5_WaitStart	RC5_flags,bRC5_WaitStart
#define		RC5_DataReady	RC5_flags,bRC5_DataReady
#define		RC5_prev_inp	RC5_flags,bRC5inp		; Has to be on the same bit position as bRC5inp
#define		RC5_Idle		RC5_flags,bRC5_Idle
#define		RC5_HalfBit		RC5_flags,bRC5_HalfBit
;							RC5_flags,5
#define		RC5_ReSynced	RC5_flags,bRC5_ReSynced
;							RC5_flags,7

; flags
#define 	StandBy		flags,bStandBy
#define		Sleeping	flags,bSleeping
#define		CoilPower	flags,bCoilPower
#define		IndexOn		flags,bIndexOn
#define		NewTime		flags,bNewTime
;#define		TimeInv		flags,bTimeInv

; Mode
#define		AutoStby	Mode,bAutoStby
#define		WakeRun		Mode,bWakeRun
#define		DeapSleep	Mode,bDeapSleep

;******************************************************************************
;	Define macro's for a Select - Case
;******************************************************************************
select_w	macro
sel_last	set 0                     ;setup variable only
         	endm
case     	macro   val
	        xorlw   val ^ sel_last
			btfsc	STATUS,Z
sel_last set val
         	endm
;
;**********************************************************************
		org     0x0000           	; processor reset vector
		clrf	GPIO				; Clear outputs
		movlw	7					;
		movwf	CMCON0				; Turn off comparator
		goto    main            	; go to beginning of program

;**********************************************************************
		org     0x0004           	; interrupt vector location

		movwf	w_temp				; context saveing
		swapf	w_temp, f
		swapf	STATUS, w

		clrf 	STATUS				; select Bank0, IRP=0 for interrupt stuff
		movwf	status_temp


INT_TMR2
		btfss	PIR1,TMR2IF			; Interrupt on TMR2? PWM uses 40uS period
		goto	lTMR2EXIT			; nope

		bcf		PIR1,TMR2IF			; Clear TMR2 interrupt flag

CountSubSec
		incfsz	SubSec_L,f			; Increment 16-bit Sub Second counter
		goto	lTime_1				; stil counting
		incfsz	SubSec_H,f			;
		goto	lTime_1				; stil counting

		call	InitSubSec			; load counter SubSecond = 0x10000 - 2500 = 0xF63C

		incf	Second2,f			; Move time
		bsf		NewTime				; Sign new time for TimeCheck

lChkStby
		btfss	StandBy				; In standby?
		goto	lTime_2				; no, count down StandbyTimer
		movf	SleepTimer,f		;
		btfsc	STATUS,Z			; SleepTimer = 0 ?
		goto	lTime_1				; yes, stop countdown
		decf	SleepTimer,f		; count down sleeptimer in seconds steps
		goto	lTime_1

lTime_2
		movf	StandbyTimerL,w		; If 16 bit StandbyTimer != 0
		iorwf	StandbyTimerH,w
		btfsc	STATUS,Z
		goto	lTime_1

		decf	StandbyTimerL,f		; Decrement it
		incf	StandbyTimerL,w
		btfss	STATUS,Z
		goto	lTime_1
		decf	StandbyTimerH,f

lTime_1

lRC5_1								; start RC5 stuff here
		btfsc	RC5_DataReady
		goto	lRC5_Exit
		btfss	RC5_Idle
		goto	lRC5_Not_Idle		;
		decfsz	RC5_Tmr,f			;
		goto	lRC5_Exit			;
		btfsc	GPIO,bRC5inp		; test input
		bcf		RC5_Idle			; input = high, cancel Idle state
		incf	RC5_Tmr,f			; continue Idle state until input = high
		goto	lRC5_Exit			;

lRC5_Not_Idle
		btfss	RC5_WaitStart		;
		goto	lRC5_ReSync			;

lRC5WaitStart
		btfsc	GPIO,bRC5inp		; test input
		goto	lRC5_Exit			; no startbit
		bcf		RC5_WaitStart		; start received
		movlw	.13					; 13 bits to receive
		movwf	RC5_BitCnt			;
		clrf 	RC5_ShrA			;
		clrf	RC5_ShrC			;
		movlw	.6					;
		goto	lRC5_Reload			;

lRC5_ReSync							;
		movf	GPIO,w				;
		xorwf	RC5_flags,w			;
		andlw	1<<bRC5inp			;
		btfsc	STATUS,Z			;
		goto	lRC5_no_sync		;
		bsf		RC5_ReSynced		;
		xorwf	RC5_flags,f			; Save new input in RC5_flags
		movlw	.6					; re-sync the timer
		btfss	RC5_HalfBit			;
		movlw	.2					;
		movwf	RC5_Tmr				;

lRC5_no_sync
		btfsc	RC5_HalfBit			;
		goto	lRC5_2nd_Half		;

lRC5_1st_Half
		decfsz	RC5_Tmr,f			;
		goto	lRC5_Exit			;
		bcf		STATUS,C			;
		btfsc	GPIO,bRC5inp		;
		bsf		STATUS,C			; C = RC5inp
		rlf		RC5_ShrC,f			;
		rlf		RC5_ShrA,f			;
		bsf		RC5_HalfBit			; indicate that the first half bit is received
		bcf		RC5_ReSynced		;
		movlw	.4					;
		goto	lRC5_Reload

lRC5_2nd_Half
		btfsc	RC5_ReSynced		;
		goto	lReSyncOK			;
		decfsz	RC5_Tmr,f			;
		goto	lRC5_Exit			;

lRC5_Error
		movf	RC5_flags,w
		andlw	(1<<7) | (1<<5) | (1<<bRC5_DataReady)
		iorlw	(1<<bRC5_WaitStart) | (1<<bRC5_Idle)
		movwf	RC5_flags
		movlw	.128				;
lRC5_Reload
		movwf	RC5_Tmr				;
		goto	lRC5_Exit			;

lReSyncOK							; test second bit half
		bcf		RC5_HalfBit			;
		decfsz	RC5_BitCnt,f		;
		goto	lRC5_Exit			;
		rlf		RC5_ShrC,f			; Shift left to complete Addr
		rlf		RC5_ShrA,f			;
		rlf		RC5_ShrC,w			; RC5_Cmd remains unchanged
		rlf		RC5_ShrA,f			; Complete address
		bcf		RC5_ShrC,7			;
		btfss	RC5_ShrA,6			;
		bsf		RC5_ShrC,7			; Bit 7 of RC5_Cmd = Inv Bit6 in RC5_Addr
									; RC5_Addr and RC5_Cmd was cleared and shifted left 13+2 times, so C = 0
		btfsc	RC5_ShrA,5			;
		bsf		STATUS,C			; C = ToggleBit
		rrf		RC5_ShrC,w			; ToggleBit in bit 7 of RC5_Cmd
		movwf	RC5_Cmd
		movf	RC5_ShrA,w
		andlw	0x1F
		movwf	RC5_Addr
		bsf		RC5_DataReady		; Only lower 5 bits of RC5_Addr are valid

lRC5_Exit

lTMR2EXIT

;--------
INT_EXIT
		swapf	status_temp,w		; Restore context for main program
		movwf	STATUS				; This will also restore bank selection and IRP
		swapf	w_temp,w
		retfie                  	; Return from interrupt

;******************************************************************************
;	Check correct progress of sec, min, hour, day, day of week, month, year

TimeCheck
		movlw	Second2
		movwf	FSR
									; FSR -> Second2
		movlw	.120				; Check for second overflow
		call	CheckAndIncNext
									; FSR -> Minute
		movlw	.60					; Check for minute overflow
		call	CheckAndIncNext
									; FSR -> Hour
		movlw	.24					; Check for hour overflow
		call	CheckAndIncNext
									; FSR -> Day
		btfsc	STATUS,C        	; if Day incremented increment WDay as well
		incf	WDay,f

		movlw	.7					; if Day of week == 7
		subwf	WDay,w
		btfsc	STATUS,C        	; if Day of week < 7 then c=0 (borrow)
		movwf	WDay				; Day of week = 0

		movlw	.32					; DMon = 32
		movwf	DMon
		movf	Month,w				;
		movwf	Scratch				; For months <8 odd  months have 31 days
		btfsc	Month,3
		incf	Scratch,f			; For months >7 even months have 31 days
		btfss	Scratch,0
		decf	DMon,f
		xorlw	.2					; Month = February ?
		btfss	STATUS,Z			;
		goto	lNotFeb2			; continue
		decf	DMon,f				; February has max 29 days
		movf	Year,w
		btfsc	STATUS,Z			; Year == 0 check Century instead
		movf	Century,w
		andlw	0x03
		btfss	STATUS,Z
		decf	DMon,f				; Year is not a LeapYear, dec DMon for Feb

lNotFeb2
		movf	DMon,w
		call	CheckAndIncNext		; Check for day overflow
		btfsc	STATUS,C        	; if day cleared, correct day
		incf	Day,f
									; FSR -> Month
		movlw	.13					; Check for month overflow
		call	CheckAndIncNext
		btfsc	STATUS,C 	    	; if month cleared, correct month
		incf	Month,f
									; FSR -> Year
		movlw	.100				;
		call	CheckAndIncNext
									; FSR -> Century
		movlw	.100				;
		subwf	Century,w			; Century < 100 ?
		btfsc	STATUS,C			;
		clrf	Century				;

		bcf		NewTime				; Clear new time flag
		return

CheckAndIncNext
		subwf	INDF,w				; test INDF value to w
		btfsc	STATUS,C    	    ; if INDF < w then c=0 (borrow)
		movwf	INDF				; Reset value
		incf	FSR,f				; Move to next unit
		btfsc	STATUS,C    	    ; If reseted, increment next unit
		incf	INDF,f
		return						; C = 1, if carry was to next unit

;******************************************************************************
;	Process RC5 commands

ProcessRC5

	ifdef	RC5AddrCheck
		movf	RC5_Addr,w			;
		xorlw	RemoteAddr			; test if RC5_Addr = RemoteAddr
		btfss	STATUS,Z			;
		goto	ProcessRC5Done		;
	endif

		movf	RC5_Cmd,w			; Get command code
		xorwf	RC5_Cmd2,f			; Compare with previous one
		btfsc	STATUS,Z			;
		goto	ProcessRC5Done		; Exit if the same code

		andlw	0x7F				; Mask toggle bit

		select_w
		case	WIDTH_DN			;
		  goto	DecWidth			; Decrement pulse width
		case	WIDTH_UP			;
		  goto	IncWidth			; Increment pulse width
		case	POWUPCOIL			;
		  goto	PowerUpCoil			; Power up coil
		case	POWDNCOIL			;
		  goto	PowerDownCoil		; Power down coil
		case	TINDEX
		  goto	ToggleIndex	 		; Toggle index LED
		case	STORE_SETTING
		  goto	Store				; Store settings to EEprom
		case	STANDBY				;
		  goto	ToggleStndBy		; StandBy
;		default
		call	ReloadASTimer		; Reload timer if command sent to clock

		goto	ProcessRC5Done		;

ToggleIndex							; Toggle index led
		movlw	1 << bIndexLed		;
		xorwf	GPIO,f				;
		movlw	1 << bIndexOn		; Toggle in flags too
		xorwf	flags,f
		goto	ProcessRC5Done		;

IncWidth							; Increment pulse width
		movlw	-2

DecWidth							; Decrement pulse width
		addlw	1
SetWidth
		addwf	CCPR1L,f
		movlw	CCP1_Run			; Update in memory	use run variable
		btfsc	CoilPower			; Coil powered up?
		movlw	CCP1_Sleep			; No - Use sleep variable
		movwf	FSR
		movf	CCPR1L,w			; Get pulse width
		movwf	INDF				; Write to INDF
		goto	ProcessRC5Done		;

PowerUpCoil							; High power drive to trafo coil
		call	InitPWMHighPower	;
		goto	ProcessRC5Done		;

PowerDownCoil						; Low power drive to trafo coil
		call	InitPWMLowPower		;
		goto	ProcessRC5Done		;

Store								; Store settings
		call	StoreSettings
		goto	ProcessRC5Done		;


ToggleStndBy						; Toggle mode
		call	TgStandBy

ProcessRC5Done
		movf	RC5_Cmd,w			;
		movwf	RC5_Cmd2			; Store new command code
		bcf		RC5_DataReady		;
		return						;

;******************************************************************************
;	Init PWM

InitPWM
		movlw	b'00100101'			; Timer2=ON, Prescaler = 4, Postscaler = 5
		movwf	T2CON
		movlw	0x0C				; Set CCP in PWM mode
		movwf	CCP1CON				;

; Low power

InitPWMLowPower
		bsf		STATUS,RP0			; Bank1
		btfsc	DeapSleep			; If deap sleep enabled
		bsf		TRISIO,bTrafo		; Disable trafo FET
		btfss	DeapSleep			; If deap sleep disabled
		bcf		TRISIO,bTrafo		; Enable  trafo FET
		bcf		STATUS,RP0			; Bank0
		movf	CCP1_Sleep,w		; Set dutycycle for sleep mode
		movwf	CCPR1L				;
		bcf		CoilPower
		return

;--------
; Wake up from sleep / standby

NotStandbyPb
		bcf		GPIO,bIndexLed		; turn on index led
		bsf		IndexOn
		bcf		StandBy				; spin motor
		bcf		Sleeping			; stop sleeping
		bcf		GPIO,bMotorOFF		; startup MIC 2940A

; High power

InitPWMHighPower
		bsf 	STATUS,RP0			; Bank1
		bcf		TRISIO,bTrafo		; Enable trafo FET
		bcf 	STATUS,RP0			; Bank0
		movf	CCP1_Run,w			;
		movwf	CCPR1L				; CCPR1L = Run
		bsf		CoilPower

; Reload Auto standby timer

ReloadASTimer
		movf	AutoStbyTimH,w		; Count is in seconds
		movwf	StandbyTimerH
		movf	AutoStbyTimL,w
		movwf	StandbyTimerL
		return

TgStandBy
		btfsc	StandBy				;
		goto	NotStandbyPb		;

;--------
; Go to standby

ToStandby							; Go to standby
		movlw	.5					;
		movwf	SleepTimer			; SleepTimer = 5 seconds

		bsf		GPIO,bIndexLed		; turn off index led
		bcf		IndexOn
		bsf		StandBy				; motor in standby
		bsf		GPIO,bMotorOFF		; shutdown MIC 2940A

ClearASTimer
		clrf	StandbyTimerH
		clrf	StandbyTimerL
		return

InitSubSec
		movlw	0x3C				; load counter SubSecond = 0x10000 - .2500 = 0xF63C
		movwf	SubSec_L			;
		movlw	0xF6				;
		movwf	SubSec_H			;
		return

;**********************************************************************
; Initialization

main
InitIO
;;;		clrf	GPIO				; Done at reset vector
;;;		movlw	7					;
;;;		movwf	CMCON0


		movlw	(1 << bIndexLed) | (1 << bMotorOFF)		; shutdown MIC 2940A
		movwf	GPIO
		bsf		STATUS,RP0			; Bank1
		clrf	ANSEL				; All pins are digital
		movlw	~ ( (1<<bMotorOFF) | (1<<bIndexLed) )
 		movwf	TRISIO				; Disable trafo drive

		clrf	PIE1				; disable all possible interrupt flag

		bsf		PIE1,TMR2IE			; Enable Peripheral interrupt from TMR2
		movlw	.49					; set frequency for 40uS interrupt
		movwf	PR2					;

		clrf	STATUS				; Bank0

		call	InitSubSec

		clrf	RC5_flags			; clear flags
		clrf	RC5_Tmr				;
		bsf		RC5_WaitStart		; RC5 is waiting for startbit

		movlw	(1 << bStandBy) | (1 << bSleeping)
		movwf	flags				; Init flags: Sleeping

		call 	InitTimeDef			; Init data from RTC

		call	InitPWM				; Init the low power PWM output to drive primary coil

		clrf	SleepTimer			; SleepTimer = 0 seconds
		call	ClearASTimer		; Clear auto standby timer

		btfsc	WakeRun				; Wake up in run mode
		call	NotStandbyPb		; Start in Active mode

		movlw	(1<<GIE) | (1<<PEIE); Enable global, extended peripheral interrupt
		movwf	INTCON

;================================================================================
;	Main loop

MainLoop
		btfsc	RC5_DataReady		; Decode Command received from RC5 receiver
		call	ProcessRC5			;

		btfsc	NewTime				; test new time flag
 		call	TimeCheck			; half second past, do TimeCheck

		btfsc	Sleeping			; yes, in standby: already sleeping?
		goto	MainLoop			; yes, sleeping, continue main loop

		btfsc	StandBy				; in standby?
		goto	CheckToSleep		; no, continue main loop

		movf	StandbyTimerH,w		; Check for StandbyTimer(16 bit)==0
		iorwf	StandbyTimerL,w
		btfss	STATUS,Z
		goto	MainLoop

		btfsc	AutoStby			; If AutoStandby enabled
		call	ToStandby			; Go to standby
		goto	MainLoop

CheckToSleep
		movf	SleepTimer,w		;
		btfss	STATUS,Z			; SleepTimer == 0
		goto	MainLoop			; no, continue main loop

		call	InitPWMLowPower		; Low power drive to trafo coil

		bsf		Sleeping			; Sleeping
		goto	MainLoop			;

;******************************************************************************
; Init time with default value: 2001-01-01 12:00:00, day of week: 1 ,set time invalid
; Init CCPR1L for Run and Sleep mode with default values
; Init Auto Standby Timer with default value
; Init mode with default value

InitTimeDef
;;		clrf	Hour 				; May start from 00:00:00
		movlw	.12					; why do clocks always start
		movwf	Hour				; at 12:00 ?
;;
		clrf	Minute
        clrf	Second2
		movlw	.1
		movwf	Day
		clrf	WDay
		movwf	Month
		movwf	Year
		movlw	.20
		movwf	Century

		movlw	high(AutoStbyTime)
		movwf	AutoStbyTimH		; Auto Standby time

		movlw	low(AutoStbyTime)
		movwf	AutoStbyTimL		; Auto Standby time

		movlw	low(Settings)
		bsf		STATUS,RP0			; Bank1
		movwf	EEADR
		bcf		STATUS,RP0			; Bank0

		call	EpromRead
		movwf	Scratch2
		xorlw	0x55
		movwf	Scratch

		call	EpromRead
		addwf	Scratch2,f
		xorlw	0xAA
		iorwf	Scratch,f
		btfss	STATUS,Z
		bsf		Scratch,0

		call	EpromRead
		addwf	Scratch2,f
		movwf	CCP1_Run			; Set pulse width for run mode

		call	EpromRead
		addwf	Scratch2,f
		movwf	CCP1_Sleep			; Set pulse width for sleep mode

		call	EpromRead
		addwf	Scratch2,f
		movwf	Mode

		call	EpromRead
		xorwf	Scratch2,w
		addlw	1
		btfss	STATUS,Z
		goto	InitDef
	
		btfss	Scratch,0
		return

InitDef
		movlw	CCPR1LRUN
		movwf	CCP1_Run			; Set pulse width for run mode

		movlw	CCPR1LSleep
		movwf	CCP1_Sleep			; Set pulse width for sleep mode


ModeCode	set	0x00
	ifdef	AutoStandby
ModeCode	set	ModeCode | (1<<bAutoStby)
	endif
	ifdef	WakeUpRun
ModeCode	set	ModeCode | (1<<bWakeRun)
	endif
	ifdef	DeapSleeping
ModeCode	set ModeCode | (1<<bDeapSleep)
	endif

		movlw	ModeCode
		movwf	Mode
		return

StoreSettings
		movlw	low(Settings)
		bsf		STATUS,RP0			; Bank1
		movwf	EEADR
		bcf		STATUS,RP0			; Bank0

		movlw	0x55
		movwf	Scratch2
		call	EpromWrite
		movlw	0xAA
		addwf	Scratch2,f
		call	EpromWrite

		movf	CCP1_Run,w			; Save pulse width for run mode
		addwf	Scratch2,f
		call	EpromWrite
		movf	CCP1_Sleep,w		; Save pulse width for sleep mode
		addwf	Scratch2,f
		call	EpromWrite
		movf	Mode,w				; Save mode
		addwf	Scratch2,f
		call	EpromWrite
		comf	Scratch2,w

EpromWrite
		clrf	FSR
		bsf		STATUS,RP0			; Bank1
		movwf	EEDATA				; Data to be programmed
		call	WaitWr0
		bsf		EECON1,WREN
		btfsc	INTCON,GIE
		bsf		FSR,GIE
		bcf		INTCON,GIE
WaitGie0
		btfsc	INTCON,GIE
		goto	WaitGie0
		movlw	0x55
		movwf	EECON2
		movlw	0xAA
		movwf	EECON2
		bsf		EECON1,WR			; Write data to EEProm
		btfsc	FSR,GIE
		bsf		INTCON,GIE
		call	WaitWr0
		incf	EEADR,f				; inc address for next write
		bcf		STATUS,RP0			; Bank0
		return						; Returns with Bank0 selected

EpromRead
		bsf		STATUS,RP0			; Bank1
		bsf		EECON1,RD			; read data from EEProm
		incf	EEADR,f				; inc address for next read
		movf	EEDATA,w			; Get next data
		bcf		STATUS,RP0			; Bank0
		return						; Returns with Bank0 selected

WaitWr0
		btfsc	EECON1,WR
		goto	WaitWr0
		return

;******************************************************************************
;		Define EEProm content
;******************************************************************************


		ORG		0x2100			; Start of message 1 in EEPROM
		de		"Propeller Clock Base ",LastUpdate," ",ProcType," V",VersionMajor+.48,".",((VersionMinor/.16)+.48),((VersionMinor%.16)+.48)

		ORG		0x21F0			; Settings
Settings:

		de		0x55,  0xAA,  CCPR1LRUN,  CCPR1LSleep,  ModeCode
ChkSum	set		0x55 + 0xAA + CCPR1LRUN + CCPR1LSleep + ModeCode
		de		low(~ ChkSum)
	END                     		; directive 'end of program'
