#include <stdio.h>
#include "stm32f10x_conf.h"
#include "uart.h"

#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wunused-parameter"
#pragma GCC diagnostic ignored "-Wmissing-field-initializers"

//----------------------------------------------------------------------------
#if TBUF_SIZE < 2
#error TBUF_SIZE is too small.  It must be larger than 1.
#elif ((TBUF_SIZE & (TBUF_SIZE-1)) != 0)
#error TBUF_SIZE must be a power of 2.
#endif

#if RBUF_SIZE < 2
#error RBUF_SIZE is too small.  It must be larger than 1.
#elif ((RBUF_SIZE & (RBUF_SIZE-1)) != 0)
#error RBUF_SIZE must be a power of 2.
#endif

//----------------------------------------------------------------------------
struct buf_r {
  unsigned int in;                      // Next In Index
  unsigned int out;                     // Next Out Index
  char buf [RBUF_SIZE];                 // Buffer
};

struct buf_t {
  unsigned int in;                      // Next In Index
  unsigned int out;                     // Next Out Index
  char buf [TBUF_SIZE];                 // Buffer
};

static struct buf_r rbuf = { 0, 0, };
#define SIO_RBUFLEN ((unsigned short)(rbuf.in - rbuf.out))

static struct buf_t tbuf = { 0, 0, };
#define SIO_TBUFLEN ((unsigned short)(tbuf.in - tbuf.out))

static unsigned int tx_restart = 1;               // NZ if TX restart is required

/*----------------------------------------------------------------------------
  USART1_IRQHandler
  Handles USART1 global interrupt request.
 *----------------------------------------------------------------------------*/
void USART1_IRQHandler(void)
{
  volatile unsigned int IIR;

  IIR = USART1->SR;
  if (IIR & USART_FLAG_RXNE) {                // RX ?
    USART1->SR &= ~USART_FLAG_RXNE;           // clear interrupt

    if (((rbuf.in - rbuf.out) & ~(RBUF_SIZE-1)) == 0) {
      rbuf.buf [rbuf.in & (RBUF_SIZE-1)] = USART1->DR;
      rbuf.in++;
    }
  }

  if (IIR & USART_FLAG_TXE) {                 // TX ?
    USART1->SR &= ~USART_FLAG_TXE;            // clear interrupt

    if (tbuf.in != tbuf.out) {
      USART1->DR = tbuf.buf [tbuf.out & (TBUF_SIZE-1)] & 0x00FF;
      tbuf.out++;
      tx_restart = 0;
    }
    else {
      tx_restart = 1;
      USART1->CR1 &= ~USART_FLAG_TXE;         // disable TX interrupt if nothing to send
    }
  }
}

/*------------------------------------------------------------------------------
  SenChar
  transmit a character
 *------------------------------------------------------------------------------*/
int send_char (int c)
{
  if (SIO_TBUFLEN >= TBUF_SIZE)
    return (-1);

  tbuf.buf[tbuf.in & (TBUF_SIZE - 1)] = c; // Add data to the transmit buffer.
  tbuf.in++;

  if (tx_restart) {                     // If transmit interrupt is disabled, enable it
    tx_restart = 0;
    USART1->CR1 |= USART_FLAG_TXE;      // enable TX interrupt
  }

  return (0);
}

/*------------------------------------------------------------------------------
  GetKey
  receive a character
 *------------------------------------------------------------------------------*/
int get_char(void)
{
  if (SIO_RBUFLEN == 0)
    return (-1);

  return (rbuf.buf [(rbuf.out++) & (RBUF_SIZE - 1)]);
}

/*------------------------------------------------------------------------------
  buffer_Init
  initialize the buffers
 *------------------------------------------------------------------------------*/
void uart_init(int br)
{
  tbuf.in = 0;                                    // Clear com buffer indexes
  tbuf.out = 0;
  tx_restart = 1;

  rbuf.in = 0;
  rbuf.out = 0;

  RCC_APB2PeriphClockCmd(RCC_APB2Periph_USART1 | RCC_APB2Periph_GPIOA, ENABLE);

  /* Enable the USARTx Interrupt */
  NVIC_InitTypeDef NVIC_InitStructure;
  NVIC_InitStructure.NVIC_IRQChannel = USART1_IRQn;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = 0;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0;
  NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
  NVIC_Init(&NVIC_InitStructure);

  GPIO_InitTypeDef GPIO_InitStructure;
  GPIO_InitStructure.GPIO_Pin = GPIO_Pin_9;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF_PP;
  GPIO_Init(GPIOA, &GPIO_InitStructure);
  GPIO_InitStructure.GPIO_Pin = GPIO_Pin_10;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_IN_FLOATING;
  GPIO_Init(GPIOA, &GPIO_InitStructure);

  USART_InitTypeDef USART_InitStructure;
  USART_InitStructure.USART_BaudRate = br;
  USART_InitStructure.USART_WordLength = USART_WordLength_8b;
  USART_InitStructure.USART_StopBits = USART_StopBits_1;
  USART_InitStructure.USART_Parity = USART_Parity_No;
  USART_InitStructure.USART_HardwareFlowControl =USART_HardwareFlowControl_None;
  USART_InitStructure.USART_Mode = USART_Mode_Rx | USART_Mode_Tx;
  USART_Init(USART1, &USART_InitStructure);
  USART_Cmd(USART1, ENABLE);

  USART_ITConfig(USART1, USART_IT_RXNE, ENABLE);
  USART_ITConfig(USART1, USART_IT_TXE, ENABLE);
}

#ifdef PRINTF_REDIRECT
int _write (int fd, char *ptr, int len)
{
  int i = 0;
  while (*ptr && (i < len)) {
    send_char(*ptr);
    if (*ptr == '\n')
      send_char('\r');
    i++;
    ptr++;
  }
  return i;
}
#endif

#pragma GCC diagnostic pop
