#include <avr/io.h>
#include <avr/interrupt.h>
#include <stdlib.h>
#include <stdio.h>
#include <avr/pgmspace.h>
#include <avr/sleep.h>
#include <avr/wdt.h>
#include "hd44780.h"
#include "adelay.h"
#include "uart.h"
#include "hardware.h"
#include "eestruct.h"
#include "menu.h"


#define TRUE  1
#define FALSE 0


/*

        Vcc
        ___
         |
         |___________
         |           |
         /           /
         \           \ <<
         /  R1       / <<  LDR (Flame detector)
         \           \ <<
         /           /
         |           |
ADC _____|___________|
         |
         |
         /
         \
         /  R2 (47 kOhm)
         \
         /
         |
         |
       /////


LDR value in Ohms. LDR = ((1024*R2)/ADC - R2).
R1 is 1 MOhm and we can safely ignore it.

*/

/* 
    This resistor should be less than 10k. Don't remember why it is 47k.
    Although it doesn't really mater. You only have to associate a
    value with the amount of the flame presence.
 */
#define R2 46440.0



#define room_thermo       PD2 // House thermostat input.
#define boiler_thermo     PD3 // Water temperature thermostat input.

#define auger             PD4 // Auger motor output
#define igniter           PD5 // Igniter output.
#define air_motor         PD6 // Air flow motor output.
#define air_shutter       PD7 // Air shutter output. Optional to reduces air flow when in stand by mode if necessary.

#define buzzer            PC5 // Buzzer Output. 



uint16_t normal_time_on;  // Feed time when in normal operation.
uint16_t normal_time_off; // No feed time when in normal operation.

uint32_t warm_time;       // Warming up time.
uint32_t warm_time_on;    // Feed time when in warming up operation.
uint32_t warm_time_off;   // No feed time when in warming up operation.

uint32_t stb_time_on;     // Feed time when in stand by operation.
uint32_t stb_time_off;    // No feed time when in stand by operation.

uint32_t feed_time;       // Time for the initial fuel feeding when starting. 
uint32_t ign_time;        // Time for the initial fuel ignition.
uint32_t ign_air_time;    // Time to start the air flow when igniting.

uint32_t clean_time;      // Time that the cleaning must last.

//Set DEBUG to 0 to disable Debug messages and save space. 
#define DEBUG 1

//Set LANG to 0 to enable English language.
#define LANG  1

#define delay(_ms) Delay_ms(_ms)
#define delay_us(_us) Delay_us(_us)

#define BAUD 38400UL                          // Baud rate.

/* Baud rate calculations. */
#define UBRR_VAL ((F_CPU+BAUD*8)/(BAUD*16)-1) // Smart roundup.
#define BAUD_REAL (F_CPU/(16*(UBRR_VAL+1)))   // Real Baud.
#define BAUD_ERROR ((BAUD_REAL*1000)/BAUD)    // Fault per 1000 parts.

#if ((BAUD_ERROR<990) || (BAUD_ERROR>1010))
    #error To sfalma ston ry8mo Baud einai megalytero toy 1%, mataiosh!
#endif


#if DEBUG == 1
//adjust UARTsendChar() function for stream
static int UARTsendstream(char c, FILE *stream);
//----set output stream to UART----
static FILE uart_str = FDEV_SETUP_STREAM(UARTsendstream, NULL, _FDEV_SETUP_WRITE);

static int UARTsendstream(char c , FILE *stream){
    uart_putc(c);
    return 0;
}
#endif


struct {

    uint16_t n_time_on,n_time_off;
    uint32_t s_time_on, s_time_off;
    uint32_t w_time,w_time_on,w_time_off;
    uint32_t f_time, i_time, i_air_time;
    uint32_t c_time;

    unsigned int ver,crc16;

} vars;

volatile uint32_t msecs; // The variable that keeps track of the time in milliseconds. 

uint32_t secs(void);
uint32_t millis(void);
void eeprom_init(void);
uint16_t read_LDR(uint8_t);


uint8_t pelletfeed(void); // Initial fuel feeding function.
uint8_t ignition(void);   // Fuel ignition function. 
uint8_t warmup(void);     // Warming up function.
uint8_t normal(void);     // Normal operation function.

uint8_t clean_process(uint32_t,uint16_t,uint8_t); // Cleaning process function.
void fault(const char *s);


uint8_t normal(){

    uint32_t time1,time2;
    uint8_t ret=1,stand_by=0,counter=0,status,flag=0;
    uint32_t t_on=(uint32_t)normal_time_on, t_off=(uint32_t)normal_time_off, LDR_threshold=200;

    lcd_clrscr();
    lcd_home();
#if LANG == 1
    lcd_puts_P("KANONIKH LEIT.");
#else
    lcd_puts_P("NORMAL OPER.");
#endif
    
#if DEBUG == 1
    printf_P(PSTR("Normal operation. t_on: %lu, t_off %lu, stand_by: %d, flag: %d, LDR_threshold %u\n"), t_on, t_off, stand_by,flag, LDR_threshold);
#endif

    time1 = millis();
    time2 = millis();
    //time3 = millis();

#if DEBUG == 1
    uint32_t time4;
    time4 = millis();
#endif

    PORTD |= (1<<air_shutter);  // Open air intake.
    PORTD |= (1<<air_motor);    // Make sure air flow is on.

    while(1){

        if ((millis()-time1) >= t_off && bit_is_clear(PORTD,auger)) {
            PORTD ^= (1<<auger);
            time1=millis();
            time2=millis();

            if (flag == 1){                        // This "if" condition gradually reduces the t_off
                t_off -= 4000;                     // time when returning from stand by operation.

                LDR_threshold = 200;               // Restore the flame detection threshold for normal mode.

                if (t_off <= normal_time_off || t_off >= stb_time_off){
                    flag = 0;
                    t_off = normal_time_off;
                };
            };

            if (stand_by == 1) PORTD &= ~(1<<air_shutter);  /* Air intake close. When entering stand by mode
                                                               wait some time (t_off) before closing the airflow.
                                                               This reduces the smoke on the exhaust pipe.*/
        };

        if ((millis()-time2) >= t_on && bit_is_set(PORTD,auger)) {
            PORTD &= ~(1<<auger);
            time1=millis();
            time2=millis();
        };

        /* If LDR value reached (flame off) check again
           after some time to avoid false/random readings.
           If flame is detected as off call clean process
           for 60 secs. Maybe pellets block the flame light.*/
        if (read_LDR(0) >= LDR_threshold){
            delay(100);
            counter++;
            if (counter >= 50) {
#if DEBUG ==1
                    printf_P(PSTR("Cleaning"));
#endif
                    PORTD &= ~(1<<auger);
                    lcd_gotoxy(15,0);
                    lcd_putc('*');
                    status = clean_process(60000,LDR_threshold,1);
                    lcd_gotoxy(15,0);
                    lcd_putc(' ');

#if DEBUG ==1
                    printf_P(PSTR("Clean proc status %d\n"),status);
#endif
                    if (status == 0){
                        ret = 2;
                        break;
                    };
            };
        } else {
            counter = 0;
        };

        if (bit_is_set(PIND,room_thermo)){
            delay(25);
            if (bit_is_set(PIND,room_thermo)) {
                ret = 0;
                PORTD &= ~(1<<auger);  // Feeding off.

#if DEBUG == 1
                printf_P(PSTR("Switch off, feeder off, air flow off\n"));
#endif
                break;
            };
        };

        if(bit_is_set(PIND,boiler_thermo)){
            delay(25);
            if (bit_is_set(PIND,boiler_thermo)){
                if (stand_by == 0){
                    t_off = stb_time_off;
                    t_on = stb_time_on;
                    flag = 0;
                    LDR_threshold = 300;           // When in stand by mode change the flame detection threshold.

                    lcd_clrscr();
                    lcd_home();
#if LANG == 1		    
                    lcd_puts_P("ANAMONH"); 
#else
		    lcd_puts_P("STAND BY"); 
#endif
                    stand_by = 1;
                    time1 = millis();
#if DEBUG == 1
                    printf_P(PSTR("Stand by\n"));
#endif
                };
            } else {
                if (stand_by == 1){
                    t_on = normal_time_on;
                    flag = 1;

                    PORTD |= (1<<air_shutter);     // Air intake open.
                    lcd_clrscr();
                    lcd_home();
#if LANG == 1		    
                    lcd_puts_P("KANONIKH LEIT.");
#else		    
		    lcd_puts_P("NORMAL OPER.");
#endif
                    stand_by = 0;
#if DEBUG == 1
                    printf_P(PSTR("Normal\n"));
#endif
                };
            };
        };


#if DEBUG == 1
        if (millis()-time4 >= 2000) {
            printf_P(PSTR("Norm: t_on %lu, t_off %lu, flag %d, stb %d, stb_t_off %lu, LDR_threshold %u, LDR: %u kOhm \n"),t_on,t_off, flag, stand_by, stb_time_off, LDR_threshold, read_LDR(0));
            time4 = millis();
        };
#endif

    };

    return ret;
}


uint8_t warmup(){

    uint32_t time, time1,time2, time3;
    uint8_t ret=1,counter=0,status;
    uint32_t t_on=warm_time_on, t_off=warm_time_off;
#if DEBUG == 1
    uint8_t tmp=0;
#endif

    lcd_clrscr();
    lcd_home();
#if LANG == 1    
    lcd_puts_P("ZESTAMA");
#else    
    lcd_puts_P("WARMING UP");
#endif

#if DEBUG == 1
    printf_P(PSTR("Warming up\n"));
#endif

    PORTD |= (1<<air_shutter);  // Air intake open.
    PORTD |= (1<<air_motor);    // Make sure air flow on.

    time = millis();
    time1 = millis();
    time2 = millis();
    time3 = millis();

    while(millis()-time <= warm_time){

        if ((millis()-time1) >= t_off && bit_is_clear(PORTD,auger)) {
            PORTD ^= (1<<auger);
            time1=millis();
            time2=millis();
        };

        if ((millis()-time2) >= t_on && bit_is_set(PORTD,auger)) {
            PORTD &= ~(1<<auger);
            time1=millis();
            time2=millis();
        };

        /* If LDR value reached (flame off) check again
           after some time to avoid false/random readings.
           If flame is detected as off call clean process
           for 60 secs. Maybe pellets block the flame light.*/
        if (read_LDR(0) >= 200){
            delay(100);
            counter++;
            if (counter >= 30) {
                    PORTD &= ~(1<<auger);

                    lcd_gotoxy(15,0);
                    lcd_putc('*');
                    status = clean_process(60000,200,1);
                    lcd_gotoxy(15,0);
                    lcd_putc(' ');
#if DEBUG ==1
                    printf_P(PSTR("Clean proc status %d\n"),status);
#endif
                    if (status == 0){
                        ret = 2;
                        break;
                    };
            };
        } else {
            counter = 0;
        };

        if (bit_is_set(PIND,room_thermo)){
            delay(25);
            if (bit_is_set(PIND,room_thermo)) {
                ret = 0;
                PORTD &= ~(1<<auger);          // Feeding off.
                PORTD &= ~(1<<air_shutter);    // Air intake close.
                //PORTD &= ~(1<<air_motor);    // Air flow off.
#if DEBUG == 1
                printf_P(PSTR("Switch off, feeder off, air flow off\n"));
#endif
                break;
            };
        };

        if (bit_is_set(PIND,boiler_thermo)){
            delay(25);
            if (bit_is_set(PIND,boiler_thermo)) {
                ret = 1;
                PORTD &= ~(1<<auger);          // Feeding off.
                PORTD &= ~(1<<air_shutter);    // Air intake close.
                //PORTD &= ~(1<<air_motor);    // Air flow off.
#if DEBUG == 1
                printf_P(PSTR("Thermostat off\n"));
#endif
                break;
            };
        };


        if (millis()-time3 >= 10000) {
            t_off -= 1000;
            time3 = millis();
#if DEBUG == 1
            tmp++;
            printf_P(PSTR("Warm UP: t_on %lu, t_off %lu, count %d, LDR:%u kOhms\n"),t_on,t_off,tmp,read_LDR(0));
#endif
        };


    };

    return ret;
}

uint8_t pelletfeed(){

/*  
               WARNING 
I have NOT implemented yet any checks 
to make sure that a previous feeding 
operation didn't complete as expected.
There is danger clogging the auger and the burner.
 
 */
    uint32_t time,counter=0;
    uint8_t ret=1;

    lcd_clrscr();
    lcd_home();
#if LANG == 1    
    lcd_puts_P("KATHARISMOS");
#else    
    lcd_puts_P("CLEANING");
#endif    

#if DEBUG == 1
    printf_P(PSTR("Cleaning before start, pellet_feed_time: %lu, LDR:%u kOhms\n"),feed_time,read_LDR(0));
#endif

    PORTD |= (1<<igniter);       // Turing igniter on.

    clean_process(60000,200,1);  // Blow some air with igniter on (extra air flow) to remove left over ash.

    PORTD &= ~(1<<igniter);      // Turing igniter  off.

    lcd_clrscr();
    lcd_home();
#if LANG == 1    
    lcd_puts_P("EISAGOGH PELLET");
#else    
    lcd_puts_P("FEEDING PELLET");
#endif

    PORTD |= (1<<air_motor);               // Start air flow.

    delay(1000);

    PORTD |= (1<<auger);                   // Start feeding Pellets.

#if DEBUG == 1
    printf_P(PSTR("Pellet feeding, feed_time: %lu, LDR:%u kOhms\n"),feed_time,read_LDR(0));
#endif

    time=millis();

    while (millis()-time <= feed_time){

        // Searching for flame. Left overs could ignite pellet while feeding.
        if (read_LDR(0) <= 200){
            delay(100);
            counter++;

#if DEBUG == 1
    printf_P(PSTR("counter: %d\n"),counter);
#endif
	    // If a stable flame is detected stop feeding.
            if (counter >= 30) {
                PORTD &= ~(1<<auger);        // Stop feeding Pellets.
                return ret;
            };
        };

        if (bit_is_set(PIND,room_thermo)){
            delay(25);
            if (bit_is_set(PIND,room_thermo)) {
                    ret = 0;
                    break;
            };
        };
    };

    PORTD &= ~(1<<auger);                   // Stop feeding Pellets.
    PORTD &= ~(1<<air_motor);               // Stop air flow.

    return ret;
}


uint8_t ignition (void) {
    uint32_t time;
    uint8_t ret=1,counter=0;
    uint8_t temp=0;

    lcd_clrscr();
    lcd_home();
#if LANG == 1    
    lcd_puts_P("ANAFLEXH PELLET");
#else    
    lcd_puts_P("IGNITING PELLET");
#endif

    time = millis();

    PORTD |= (1<<igniter);                  // Turing igniter on.

#if DEBUG == 1
    printf_P(PSTR("Igniting Pellet, ignition time: %lu , air time: %lu, LDR: %u\n"),ign_time, ign_air_time, read_LDR(0));
#endif

    while ((millis()-time) <= ign_time){

        // Star air flow after some time.
        if (millis()-time >= ign_air_time && temp == 0) {
            PORTD |= (1<<air_motor);
            temp = 1;
#if DEBUG == 1
            printf_P(PSTR("Air flow started\n"));
#endif
        };

        // If LDR value reached (flame present) check again
        // after some time to avoid false/random readings.
        if (read_LDR(0) <= 200){
            delay(100);
            counter++;
            if (counter >= 250) {
                ret = 2;
#if DEBUG == 1
                printf_P(PSTR("Flame on\n"));
#endif
                break;
            };
        } else {
            counter=0;
        };

        if (bit_is_set(PIND,room_thermo)){
            delay(25);
            if (bit_is_set(PIND,room_thermo)) {
                ret = 0;

#if DEBUG == 1
                printf_P(PSTR("Switch off, air flow off\n"));
#endif
                break;
            };
        };


    };

    PORTD &= ~(1<<igniter);  // Turing igniter off.

    return ret;
}


int main(void){

#if LANG == 1
    static const char mem_err[] PROGMEM = "SFALMA MNHMHS";
    static const char ign_err[] PROGMEM = "SFALMA ANAFLEXH";
    static const char warm_err[] PROGMEM = "SFALMA ZESTAMA";
    static const char norm_err[] PROGMEM = "SFALMA KAN LEIT";
#else
    static const char mem_err[] PROGMEM = "MEMORY ERROR";
    static const char ign_err[] PROGMEM = "IGNITION ERROR.";
    static const char warm_err[] PROGMEM = "WARMING ERROR";
    static const char norm_err[] PROGMEM = "NORM OPER ERROR";
#endif
    
    uint8_t mem_set, mem_load, status,temp1=1,temp2=0;

    mem_set = SetEEPROMStructureParameters(0,0,512,sizeof(vars));
    mem_load = LoadEEPROMStructure(0,&vars);


    normal_time_on = vars.n_time_on*(uint16_t)100;
    normal_time_off = vars.n_time_off*(uint16_t)100;
    warm_time = vars.w_time*(uint32_t)1000;
    warm_time_on = vars.w_time_on*(uint32_t)1000;
    warm_time_off= vars.w_time_off*(uint32_t)1000;
    stb_time_on = vars.s_time_on*(uint32_t)100;
    stb_time_off = vars.s_time_off*(uint32_t)1000;
    feed_time = vars.f_time*(uint32_t)1000;
    ign_time = vars.i_time*(uint32_t)1000;
    ign_air_time = vars.i_air_time*(uint32_t)1000;
    clean_time = vars.c_time*(uint32_t)1000;



    DDRD |= (1<<auger)|(1<<igniter)|(1<<air_motor)|(1<<air_shutter);            // PD7, PD6, PD5, PD4 as output.
    DDRD &= ~((1<<room_thermo)|(1<<boiler_thermo));                             // room_thermo, boiler_thermo as input.

    PORTD &= ~((1<<auger)|(1<<igniter)|(1<<air_motor)|(1<<air_shutter));        // PD7, PD6, PD5,PD4 LOW.
    PORTD |= (1<<room_thermo)|(1<<boiler_thermo);                               // room_thermo, boiler_thermo enable pull up resistors.

#if DEBUG == 1
    stdout = &uart_str;                           // Stream initialization.
    uart_init( UART_BAUD_SELECT(BAUD,F_CPU) );    // UART initialization.
#endif

    adc_init();                                   // ADC initialization.
    millis_init();                                // Millisecond counter initialization.
    lcd_init();                                   // LCD initialization.

    sei();                                        // Enable Global Interrupts.


    lcd_clrscr();
    lcd_home();
#if LANG == 1    
    lcd_puts_P("KAYSTHPAS PELLET");
#else    
    lcd_puts_P("PELLET BURNER");
#endif    
    delay(1000);
    lcd_clrscr();
    lcd_home();

//    step = 0;

#if DEBUG == 1
    printf_P(PSTR("\033[2J"));   // Clear terminal screen.
    printf_P(PSTR("\033[H"));    // Go to upper left corner.
    printf_P(PSTR("\033[1m"));   // Set bold.

    printf_P(PSTR("Debug Interface:\n\n"));


    printf_P(PSTR("Reading times (in msecs) from EEPROM (size in bytes:%d)\n\n"),sizeof(vars));

    printf_P(PSTR("\tNormal time on: %u\n"),normal_time_on);
    printf_P(PSTR("\tNormal time of: %u\n"),normal_time_off);
    printf_P(PSTR("\tWarm time:      %lu\n"),warm_time);
    printf_P(PSTR("\tWarm time on:   %lu\n"),warm_time_on);
    printf_P(PSTR("\tWarm time off:  %lu\n"),warm_time_off);
    printf_P(PSTR("\tStby time on:   %lu\n"),stb_time_on);
    printf_P(PSTR("\tStby time off:  %lu\n"),stb_time_off);
    printf_P(PSTR("\tFeed time:      %lu\n"),feed_time);
    printf_P(PSTR("\tIgnition time:  %lu\n"),ign_time);
    printf_P(PSTR("\tIgn air time:   %lu\n"),ign_air_time);
    printf_P(PSTR("\tCleaning time:  %lu\n\n\n"),clean_time);

    printf_P(PSTR("mem_set: %d, mem_load: %d\n\n"),mem_set, mem_load);

#endif

    DDRB &= ~(1<<PB5);
    PORTB |= (1<<PB5);
    
   
    // If menu button is pressed during start-up go to Menu options.
    if(bit_is_clear(PINB,PB5)){
        delay(25);
        if(bit_is_clear(PINB,PB5)) menu();
    };

    if (mem_set==0 || mem_load==0) fault(mem_err);      //If there was an error reading from the EEPROM go to fault.

    // Main loop.
    while(1) {


        while (bit_is_clear(PIND,room_thermo)) {

            delay(25);
            if(!bit_is_clear(PIND,room_thermo)) break;

	    // Temporary variables
            temp1 = 1;
            temp2 = 1;

            status = pelletfeed();
            if (status == 0) break;

            status = ignition();
            if (status == 0) {
                break;
            } else if (status == 1){
                fault(ign_err);
            };

            status = warmup();
            if (status == 0) {
                break;
            } else if (status == 2){
                fault(warm_err);
            };

            status = normal();
            if (status == 0) {
                break;
            } else if (status == 2){
                fault(norm_err);
            };

        };

        if (temp1 == 1){
            lcd_clrscr();
            lcd_home();
            lcd_puts_P("THERMOSTAT OFF");
            if (temp2 == 1){
                clean_process(clean_time,200,0);
                lcd_clrscr();
                lcd_home();
                lcd_puts_P("THERMOSTAT OFF");
                temp2 = 0;
            };

            temp1 = 0;
        };

    };

    // Never reached.

    return 0;
}

uint8_t clean_process(uint32_t run_time, uint16_t threshold, uint8_t LDR_chk){


    uint8_t ret=0, counter=0,temp;
    uint32_t time;

    temp = PORTD;

    if (LDR_chk == 0){
        lcd_gotoxy(0,1);
#if LANG == 1	
        lcd_puts_P("KATHARISMA ");
#else	
	lcd_puts_P("CLEANING ");
#endif	
    };

    PORTD |= (1<<air_motor);
    PORTD |= (1<<air_shutter);

    time = millis();

    while(millis() - time <= run_time){
        if(!bit_is_set(PIND,room_thermo)) {
           delay(25);
           if(!bit_is_set(PIND,room_thermo) && LDR_chk == 0) break;
           };


        if (LDR_chk == 1){
            if (read_LDR(0) <= threshold){
                delay(100);
                counter++;
                if (counter >= 50) {
                    ret = 1;
                    break;
                };
            } else {
                counter = 0;
            };
        };
    };

    if (LDR_chk == 0){
        PORTD &= ~(1<<air_motor); // Air flow off.
        PORTD &= ~(1<<air_shutter); // Air intake close.
    } else {

        PORTD = temp;
    };

    return ret;
}

void fault(const char *s){


    uint32_t time,time1;

    lcd_clrscr();
    lcd_home();
#if LANG == 1    
    lcd_puts_P("SFALMA...");
#else    
    lcd_puts_P("ERROR...");
#endif    
    lcd_gotoxy(0,1);
    lcd_puts_p(s);

    // Turning off pellet feed and igniter.
    PORTD &= ~(1<<auger);
    PORTD &= ~(1<<igniter);

    PORTD |= (1<<air_motor)|(1<<air_shutter);  // Air flow on and air intake open
					       // in order to burn any left overs.

#if DEBUG == 1
    printf_P(PSTR("System fault. Everything off\n"));
#endif
    DDRC |= (1<<buzzer);

    time1 = millis();

    // This loop is a crude way to produce an intermittent beep as alarm indication that something is wrong.
    while(1){
        time=millis();
        while (millis()-time <= 500){
            PORTC ^= (1<<buzzer);
            delay_us(250);
        };
        delay(500);
        time=millis();
        while (millis()-time <= 500){
            PORTC ^= (1<<buzzer);
            delay_us(250);
        };
        delay(1000);
        delay(1000);

        if (millis()-time1 >= 180000) {
            PORTD &= ~((1<<air_motor)|(1<<air_shutter));  // Air flow off, air intake off. Clean process completed.
        };

    };

}


uint32_t millis(void) {

    uint32_t temp;
    uint8_t oldSREG = SREG;

    cli();

    temp=msecs;
    SREG=oldSREG;

    return temp;
}

uint32_t secs(void){

    uint32_t temp;
    uint8_t oldSREG = SREG;

    cli();

    temp=msecs/1000;
    SREG=oldSREG;

    return temp;
}

uint16_t read_LDR(uint8_t ADC_channel){
    int16_t adc_val;
    uint16_t ldr_val;
    char s[10];
    static uint32_t time=0;

    ADMUX = (ADMUX & 0xF0) | (ADC_channel & 0x0F);

    ADCSRA |= (1<<ADSC);    // Start ADC conversion.

    while( ADCSRA & (1<<ADSC) );   // Wait until ADC conversion is complete.

    adc_val = ADC;
    
/* 
    Can't remember the reason I am dividing with 1000.
    Although it doesn't really mater. You only have to associate a
    value with the amount of the flame's presence.
 */
    ldr_val = ((1024*R2)/adc_val - R2)/1000;

    if (millis() - time >= 500){
        lcd_gotoxy(10,1);
        utoa(ldr_val,s,10);
        lcd_puts_P("R:");
        lcd_puts(s);
        lcd_puts_P("   ");
        time=millis();
    };

    return ldr_val;
}

ISR(TIMER0_OVF_vect){

    uint32_t temp;
    temp = msecs;
    temp++;
    msecs = temp;

    TCNT0 += 6;  // Pre-load Timer with the calculated value for 1 msec.
}

