/* Copyright (C) 2017, Attila Kovs
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation  and/or other materials provided with the distribution.
 * 3. Neither the names of the copyright holders nor the names of any
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/* This file is part of LwOS (Lightweight runtime Operating System).
 */

#ifndef LWOS_H_INCLUDED
#define LWOS_H_INCLUDED

#define LWOS                                  // LwOS present

#if defined (STM32F10X_LD) || defined (STM32F10X_LD_VL) || defined (STM32F10X_MD) || defined (STM32F10X_MD_VL) || defined (STM32F10X_HD) || defined (STM32F10X_HD_VL) || defined (STM32F10X_XL) || defined (STM32F10X_CL)
#include "stm32f10x.h"
#define LWOS_CM3
#else
#error "Platform not supported!"
#endif

#include "lwos_config.h"
#include "heap.h"
#include "macro_common.h"
#ifdef LWOS_CM3
#include "macro_cm3.h"
#endif

typedef enum
{
  LOW_PRIORITY,                               // One time-slice per cycle
  NORMAL_PRIORITY,                            // Two time-slices per cycle
  ELEVATED_PRIORITY,                          // Four time-slices per cycle
  HIGH_PRIORITY,                              // Exclusive tasks blocking lower level tasks until all exits/sleeps/resource blocked
  RUNTIME_PRIORITY                            // Runs to completion (no IRQ or task switching until it exits), it uses main stack space, stack size setting is ignored
} LwOS_TaskPriority_t;

typedef enum
{
  EXCLUSIVE_MODE_OFF,                         // Enable task switching (normal operation)
  EXCLUSIVE_MODE_ON                           // Disable task switching (IRQs are still enabled)
} LwOS_ExclusiveMode_t;

typedef enum
{                                             // Time reference unit:
  TIMEREF_IN_SECONDS,                         // 1s
  TIMEREF_IN_TENTHSOFASECOND,                 // 0.1s
  TIMEREF_IN_TENSOFMILLISECOND,               // 10ms
  TIMEREF_IN_MILLISECONDS                     // 1ms
} LwOS_TimeRefFormat_t;

typedef volatile uint32_t LwOS_Resource_t;    // Simple resource handle type, resource is available if handle is zero, holds pointer to blocker task if locked

typedef volatile struct
{
	uint16_t locked;                            // Locking ring counter
	uint16_t freed;                             // Unlocking ring counter
	uint32_t blocker;                           // Current blocker task (pointer to task header)
} LwOS_Resource_RR_t;                         // Resource handle type to be allocated by round robin method (first come first served)

typedef struct LwOS_Task_t volatile LwOS_Task_t;
typedef void (*TaskFunc_t)(void* arg);

typedef union {
  LwOS_Task_t* blockedBy;
  LwOS_TimeRef_t sleepUntil;
} LwOS_TaskData_t;

typedef enum {
  TASK_INVALID = 0,                           // Invalid task state
  TASK_READY,                                 // Task is ready to run
  TASK_SLEEPING,                              // Task is sleeping
  TASK_ENDING,                                // Task is ending (waiting to be destroyed)
  TASK_BLOCKED,                               // Task is blocked by unavailable resource
  TASK_INACTIVE                               // Task is inactive
} LwOS_TaskState_t;

struct LwOS_Task_t {
  void *sp;                                   // Stack pointer
  LwOS_TaskState_t state;                     // Task state
  LwOS_Task_t* next;                          // Next task
  LwOS_TaskPriority_t priority;               // Task priority
  LwOS_TaskData_t data;                       // Task state dependent data
#ifdef TASK_DEBUG
  const char* name;                           // Task name
  uint32_t id;                                // Task ID
  uint64_t runTime;                           // Task CPU use
#endif
#ifdef STACK_DEBUG
  void *spTop;                                // Stack pointer to task area
  uint32_t maxUsed;                           // Maximum stack used (statistical, measured at context switches and service calls)
  uint32_t canary;                            // Canary to check for stack overflow
#endif
} MEM_ALIGNED;

#ifdef NESTED_IRQ_LOCK
extern volatile uint16_t __IRQLock;
#endif

LwOS_Task_t* LwOS_CreateTask(TaskFunc_t taskFunc, void *arg, size_t stackSize, LwOS_TaskPriority_t priority);
void LwOS_WakeUpTask(LwOS_Task_t* task);
void LwOS_Sleep(void);
void LwOS_SleepFor(uint32_t time);
void LwOS_SetPriority(LwOS_TaskPriority_t priority);
void LwOS_Yield(void);
void LwOS_Init(void);
void LwOS_Start(void);
LwOS_Task_t* LwOS_GetCurrentTask(void);
LwOS_Task_t* LwOS_GetHeadOfTaskList(void);
void LwOS_SetExclusiveMode(LwOS_ExclusiveMode_t mode);
LwOS_TimeRef_t LwOS_GetRawTimeRef(void);
LwOS_TimeRef_t LwOS_GetTimeRef(LwOS_TimeRefFormat_t format);

#ifdef TASK_DEBUG
void LwOS_AssigTaskName(LwOS_Task_t* task, const char* name);
LwOS_TimeRef_t LwOS_GetTaskRunningTime(LwOS_Task_t* task);
#endif
#ifdef STACK_DEBUG
void LwOS_MeasureStackUsage(void);
size_t LwOS_GetMaxStackUsage(void);
#else
#define LwOS_MeasureStackUsage()    ((void) 0)
#endif

void Lock(void);
void Unlock(void);
void RESOURCE_LOCK(LwOS_Resource_t *resource);
void RESOURCE_UNLOCK(LwOS_Resource_t *resource);
void RESOURCE_RR_LOCK(LwOS_Resource_RR_t *resourceRR);
void RESOURCE_RR_UNLOCK(LwOS_Resource_RR_t *resourceRR);

#endif /* LWOS_H_INCLUDED */
