/* Copyright (C) 2017, Attila Kovs
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation  and/or other materials provided with the distribution.
 * 3. Neither the names of the copyright holders nor the names of any
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/* This file is part of LwOS (Lightweight runtime Operating System).
 */

#include "heap.h"
#include "lwos.h"

extern uint32_t STACK_LIMIT, HEAP_LIMIT, STACK_TOP;

typedef struct Area_t Area_t;

typedef enum
{
	AREA_FREE,
	AREA_USED
} AreaState_t;

struct Area_t
{
  AreaState_t state;
	Area_t *next;
} MEM_ALIGNED;

#define AREAHEADER_SIZE   (sizeof(Area_t))

static Area_t *_head;

static size_t _getAreaSize(Area_t *area)
{
	void *areaStart, *areaEnd;

	areaStart = (void*)area;

	if(area->next == NULL) areaEnd = (void*)&STACK_LIMIT;
	else areaEnd = (void*)(area->next);

	return (areaEnd - areaStart);
}

static void _assignRemainingMemoryToNewArea(Area_t *area, size_t areaSize)
{
	Area_t *newArea;

	if((_getAreaSize(area) - areaSize) <= AREAHEADER_SIZE * 2) return;

	newArea = (Area_t *)((void *)area + AREAHEADER_SIZE + areaSize);
	newArea->state = AREA_FREE;

	newArea->next = area->next;
	area->next = newArea;
}

static void _defragHeap(void)
{
	Area_t *area;

	area = _head;
	while(area)
	{
    if ((area->state == AREA_FREE) && (area->next != NULL))
    {
      if (area->next->state == AREA_FREE) {
        area->next = area->next->next;
      }
      else area = area->next->next;
    }
    else area = area->next;
	}
}

void LwOS_HeapInit(void)
{
  uint32_t heapStart;

  heapStart = (uint32_t)&HEAP_LIMIT;
  if (heapStart % MEM_ALIGN) heapStart += MEM_ALIGN - (heapStart % MEM_ALIGN);

	_head = (Area_t *)heapStart;
	_head->next = NULL;
	_head->state = AREA_FREE;
}

void* LwOS_Malloc(size_t memSize)
{
  Area_t *area, *selected;

  if (!memSize) return NULL;

  if (memSize % MEM_ALIGN) memSize += MEM_ALIGN - (memSize % MEM_ALIGN);

  area = _head;
  selected = NULL;
  while(area)
	{
    if ((area->state == AREA_FREE) && (_getAreaSize(area) - AREAHEADER_SIZE >= memSize)) {
      if (selected == NULL) selected = area;
      else if (_getAreaSize(area) < _getAreaSize(selected)) selected = area;
    }
    area = area->next;
	}
	_assignRemainingMemoryToNewArea(selected, memSize);
  selected->state = AREA_USED;
  return (selected + 1);
}

void LwOS_Free(void* memP)
{
  Area_t *area;

  if (memP == NULL) return;

  area = memP;
  area--;
  area->state = AREA_FREE;
  _defragHeap();
}

size_t LwOS_BlockSize(void* memP)
{
  Area_t *area;

  if (memP == NULL) return 0;

  area = memP;
  area--;

  return (_getAreaSize(area) - AREAHEADER_SIZE);
}

