/* Copyright (C) 2017, Attila Kovs
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation  and/or other materials provided with the distribution.
 * 3. Neither the names of the copyright holders nor the names of any
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/* This file is part of LwOS (Lightweight runtime Operating System).
 */

#include "lwos.h"
#include "lwos_priv.h"
#ifdef LWOS_CM3
#include "platform_cm3.h"
#include "macro_cm3.h"
#endif

#ifdef TASK_DEBUG
static __IO uint32_t _nextID;
static const char _mainTaskName[] = "MAIN";
#endif

#ifdef STACK_DEBUG
static const uint32_t _CANARY = 0xCEA12F6B;
#endif

#define _TASKHEADER_SIZE   (sizeof(LwOS_Task_t))

extern uint32_t STACK_LIMIT, HEAP_LIMIT, STACK_TOP;

static LwOS_Task_t *_main, *_current, *_idle, *_pos;

#ifdef NESTED_IRQ_LOCK
uint16_t volatile __IRQLock;
#endif

/******************************* PRIVATE / INTERNAL ******************************/

/** \brief Task end function (called when task exits), sets ending state, requests context switch and waits to be destroyed
 *
 * \param void
 * \return NAKED  void NAKED attribute prevents memory leaks by not allocating stack space at function entry for context data (which the compiler would normally do)
 *
 */
NAKED static void _taskEnd(void)
{
  _current->state = TASK_ENDING;
  RequestContextSwitch();
  while(1) {}
}

/** \brief Idle task to be ran when no other task is ready to process information
 *
 * \param arg void*
 * \return void
 *
 */
static void _idleTask(void *arg)
{
  (void)arg;

  while(1) {
#ifndef DEBUG
    __WFI(); // CPU sleep
#endif
  }
}

#ifdef STACK_DEBUG
/** \brief Calculate current stack use from provided stack pointer, updates maximum value if necessary
 *
 * \param stackP uint32_t
 * \return void
 *
 */
void _calculateStackUsage(uint32_t stackP)
{
  if (_current == NULL) return;

  uint32_t val = (uint32_t)_current->spTop - stackP;

  if (val > _current->maxUsed) _current->maxUsed = val;
}
#endif

/** \brief Creates new task by allocating stack space for header, context and user data
 *         Also generates starting context and sets up initial task header data
 *
 * \param void
 * \return void
 *
 */
LwOS_Task_t* _createTask(TaskFunc_t taskFunc, void *arg, size_t stackSize, LwOS_TaskPriority_t priority)
{
  LwOS_Task_t *task;

  if (priority == RUNTIME_PRIORITY) {
      taskFunc(arg);
      return NULL;
  }

  task = LwOS_Malloc(_TASKHEADER_SIZE + stackSize + LwOS_ContextSize);
  if (task == NULL) return NULL;

  task->sp = (uint8_t*)task + _TASKHEADER_SIZE + stackSize + LwOS_ContextSize;
  task->state = TASK_READY;
  task->next = NULL;
  task->priority = priority;
#ifdef TASK_DEBUG
  task->id = _nextID++;
#endif
#ifdef STACK_DEBUG
  task->canary = _CANARY;
  task->spTop = task->sp;
  task->maxUsed = 0;
#endif

  task->sp = CreateContext(task->sp, (void *)taskFunc, arg, _taskEnd);

  if (_main == NULL) _main = task;
  else {
    LwOS_Task_t *t = _main;
    while (t->next) {
      t = t->next;
    }
    t->next = task;
  }

  return task;
}

/** \brief Creates task header for main task by allocating stack space for header data and sets up initial values
 *
 * \param void
 * \return void
 *
 */
void _createMainTask(void)
{
  LwOS_Task_t *task;

  task = LwOS_Malloc(_TASKHEADER_SIZE);
  if (task == NULL) return NULL;

  task->sp = NULL;
  task->state = TASK_READY;
  task->next = NULL;
  task->priority = NORMAL_PRIORITY;
#ifdef TASK_DEBUG
  task->id = _nextID++;
  task->name = _mainTaskName;
#endif
#ifdef STACK_DEBUG
  task->canary = _CANARY;
  task->spTop = (void*)&STACK_TOP;
  task->maxUsed = 0;
#endif

  _main = task;
}

/** \brief Task scheduler
 *         Selects next task to run during context switch
 *
 * \param mainTaskActive uint32_t If true (1), current task is the main task which uses MSP (main stack pointer)
 *                                If false (0), current task is using PSP as stack pointer (process stack pointer)
 * \return uint32_t Returns if newly selected task is main task (1) or not (0)
 *
 */
uint32_t _scheduler(uint32_t mainTaskActive)
{
  LwOS_Task_t *best = NULL;
  LwOS_Task_t *t;
#ifdef TASK_DEBUG
  uint32_t startRef = 0;

  _current->runTime += SysTick_GetReference(0) - startRef;
#endif

#ifdef HIGH_PRIORITY_OVER_OTHERS
  // If high priority task returned as blocked, it will try to give control to blocking lower priority task
  if ((_current->priority == HIGH_PRIORITY) && (_current->state == TASK_BLOCKED)) {
    if (_current->data.blockedBy) {
      if (((LwOS_Task_t*)(_current->data.blockedBy))->state == TASK_READY) best = _current->data.blockedBy; // Give timeslice to blocking task
    }
  }
#endif

  // Electing new task to run
  if (!best) {
#ifdef HIGH_PRIORITY_OVER_OTHERS
    // If high priority task returned and ready to run select it as best temporarily
    if ((_pos->priority == HIGH_PRIORITY) && (_pos->state == TASK_READY)) best = _pos;
#endif
    // Checks if current position in scheduling is at the end of the task list, selects next task to be examined
    if (_pos->next) t = _pos->next;
    else t = _main;

    // Try to elect new task
    while (t != _pos) {
      // If task is asleep, check if it can be waken up
      if (t->state == TASK_SLEEPING) {
        if ((t->data.sleepUntil > 0) && (t->data.sleepUntil <= _timeTick)) t->state = TASK_READY;
      }
#ifdef HIGH_PRIORITY_OVER_OTHERS
      // If task is high priority, check if it can be elected to run
      if (t->priority == HIGH_PRIORITY) {
        if ((t->state == TASK_BLOCKED) || (t->state == TASK_READY)) {
          best = t;
          break;
        }
      }

      // If task is ready/blocked, elect it temporarily (maybe a high priority task is found in the list)
      if (((t->state == TASK_BLOCKED) || (t->state == TASK_READY)) && (!best)) best = t;
#else
      // If task is ready/blocked, elect it as next to be ran
      if (((t->state == TASK_BLOCKED) || (t->state == TASK_READY)) && (!best)) {
        best = t;
        break;
      }
#endif /* HIGH_PRIORITY_OVER_OTHERS */
      // Go to next task in list
      t = t->next;
      // If end of list is reached, start from main task
      if (t == NULL) t = _main;
    }

    // If new task is elected to run, move scheduler position to it, otherwise elect idle task to run
    if (best) _pos = best;
    else best = _idle;
  }

  //if ((best->state != TASK_READY) && (best->priority != HIGH_PRIORITY)) best = _idle;

  // If current task is ending, destroy its stack and header
  if(_current->state == TASK_ENDING) {
    // Search for the previous task in list
    t = _main;
    while (t->next != _current) {
      t = t->next;
    }
    // Connect previous task to next task (remove current task from list)
    t->next = _current->next;
    // If scheduler position is at current task, move position to main task
    if (_pos == _current) _pos = _main;
    // Free memory
    LwOS_Free((void*)_current);
  } else {
    // Store current stack pointer in task header
    if (_current != _main) _current->sp = (void*)GET_PSP();
    else  _current->sp = (void*)GET_MSP();
  }

#ifdef STACK_DEBUG
  // If canary data is corrupted that means stack overflow fault
  if (best->canary != _CANARY) {
    while (1) { }    // Stack overflow!
  }
#endif

  // Next task is elected
  _current = best;

#ifdef STACK_DEBUG
  // Calculate current stack usage and check if it is higher than previous maximum
  _calculateStackUsage((uint32_t)_current->sp + LwOS_ContextSize);
#endif

  // Check next task type
  if (_current != _main) {
    // If not main (uses process stack pointer), set stack pointer to task's stack
    SET_PSP((uint32_t)_current->sp);
    mainTaskActive = 0;
  } else {
    // Otherwise signal context switch handler to load main stack pointer instead of process stack pointer
    mainTaskActive = 1;
  }

  // Set number of time slices for next task
  switch (_current->priority) {
    case LOW_PRIORITY:      SetTaskTimeSliceNum(TIMESLICES_LOW_PRIORITY);       break;
    case NORMAL_PRIORITY:   SetTaskTimeSliceNum(TIMESLICES_NORMAL_PRIORITY);    break;
    case ELEVATED_PRIORITY: SetTaskTimeSliceNum(TIMESLICES_ELEVATED_PRIORITY);  break;
    case HIGH_PRIORITY:     SetTaskTimeSliceNum(TIMESLICES_HIGH_PRIORITY);      break;
    case RUNTIME_PRIORITY: break;
  }

#ifdef TASK_DEBUG
  startRef = SysTick_GetReference(1);
#endif

  return mainTaskActive;
}


/******************************* PUBLIC ******************************/


/** \brief Calls service function to create new task
 *
 * \param taskFunc TaskFunc_t
 * \param arg void*
 * \param stackSize size_t
 * \param priority LwOS_TaskPriority_t
 * \return NOINLINE Task_t*
 *
 */
NOINLINE LwOS_Task_t* LwOS_CreateTask(TaskFunc_t taskFunc, void *arg, size_t stackSize, LwOS_TaskPriority_t priority)
{
  LwOS_Task_t* task;
  // Supervisor will use input arguments
  __SVC_RET(SVC_CREATETASK, task);
  return task;
}

#ifdef TASK_DEBUG
void LwOS_AssigTaskName(LwOS_Task_t* task, const char* name)
{
  if (task) task->name = name;
}

/** \brief Get task's CPU use
 *
 * \param task LwOS_Task_t* Pointer to task header
 * \return LwOS_TimeRef_t Task CPU use in milliseconds
 *
 */
LwOS_TimeRef_t LwOS_GetTaskRunningTime(LwOS_Task_t* task)
{
  LwOS_TimeRef_t val;
  if (!task) return 0;
  ATOMIC(
    val = task->runTime;
  );
  val /= (_minorTickPerSecond / 1000);
  return val;
}
#endif

#ifdef STACK_DEBUG
/** \brief Calls service function to measure current stack usage
 *
 * \param void
 * \return void
 *
 */
void LwOS_MeasureStackUsage(void)
{
  __SVC(SVC_MEASURESTACKUSE);
}

/** \brief Returns maximum measured stack usage (measured ate context switches and external service calls only)
 *
 * \param void
 * \return size_t Maximum stack use
 *
 */
size_t LwOS_GetMaxStackUsage(void)
{
  return _current->maxUsed;
}
#endif

/** \brief Wakes up task (if it is suspended)
 *
 * \param task Task_t* Pointer to task header
 * \return void
 *
 */
void LwOS_WakeUpTask(LwOS_Task_t* task)
{
  if (!task) return;
  ATOMIC(
    if (task->state == TASK_SLEEPING) task->state = TASK_READY;
  );
}

/** \brief Suspends task (task have to be waken up by user application)
 *
 * \param void
 * \return void
 *
 */
void LwOS_Sleep(void)
{
  ATOMIC(
    if (_current) {
      _current->state = TASK_SLEEPING;
      _current->data.sleepUntil = 0;
    }
    RequestContextSwitch();
    //__SVC(SVC_CONTEXTSWITCHSLEEP);
  );
  while (_current->state == TASK_SLEEPING) {}
}

/** \brief Suspends task for set amount of time in milliseconds
 *
 * \param time uint32_t Sleep time in milliseconds
 * \return void
 *
 */
void LwOS_SleepFor(uint32_t time)
{
  ATOMIC(
    if (_current) {
      _current->state = TASK_SLEEPING;
      uint32_t tval = _timeTick + ((time * 10) / TIMESLICE);
      _current->data.sleepUntil = tval;
    }
    RequestContextSwitch();
    //__SVC(SVC_CONTEXTSWITCHSLEEP);
  );
  while (_current->state == TASK_SLEEPING) {}
}

/** \brief Yields processor to next task
 *
 * \param void
 * \return void
 *
 */
void LwOS_Yield(void)
{
  if (_taskSwitchingEnabled) RequestContextSwitch();
  //__SVC(SVC_CONTEXTSWITCH);
}

/** \brief Sets current task's priority
 *
 * \param priority LwOS_TaskPriority_t New task priority
 * \return void
 *
 */
void LwOS_SetPriority(LwOS_TaskPriority_t priority)
{
  ATOMIC(
    if (_current) _current->priority = priority;
  );
}

/** \brief Initializes runtime OS
 *         Runs platform dependent initialization, sets up heap, creates idle and main tasks
 *
 * \param void
 * \return void
 *
 */
void LwOS_Init(void)
{
  Platform_Init();
  LwOS_HeapInit();
  _createMainTask();
#ifndef MAIN_TASK_CONTINUES
  _main->state = TASK_INACTIVE;
#endif
  _idle = _createTask(_idleTask, NULL, 128, LOW_PRIORITY);
  _main->next = NULL;
  _current = _main;
  _pos = _main;
}

/** \brief Starts runtime OS by configuring and enabling SysTick timer
 *         Systick timer is responsible for initiating context switches after allocated time slices are used up
 *
 * \param void
 * \return void
 *
 */
void LwOS_Start(void)
{
  SysTick_Init();

#ifndef MAIN_TASK_CONTINUES
  while (1) { }
#endif
}

/** \brief Returns pointer to currently running task's header
 *
 * \param void
 * \return Task_t* Pointer to running task's header structure
 *
 */
LwOS_Task_t* LwOS_GetCurrentTask(void)
{
  return _current;
}

/** \brief Returns pointer to head of task list
 *
 * \param void
 * \return Task_t* Pointer to head of task list's header structure
 *
 */
LwOS_Task_t* LwOS_GetHeadOfTaskList(void)
{
  return _main;
}

/** \brief Enables/disables task switching
 *         No resource locking or task state change are allowed (sleep, yield, etc.) until task switching is re-enabled
 *
 * \param mode LwOS_ExclusiveMode_t
 * \return void
 *
 */
void LwOS_SetExclusiveMode(LwOS_ExclusiveMode_t mode)
{
  if (mode == EXCLUSIVE_MODE_ON) __SVC(SVC_EXCLUSIVE_MODE_ON);
  else __SVC(SVC_EXCLUSIVE_MODE_OFF);
}


/** \brief Get raw time reference generated by SysTick
 *
 * \param void
 * \return LwOS_TimeRef_t Time reference (unit depends on LWOS_TICK configuration)
 *
 */
LwOS_TimeRef_t LwOS_GetRawTimeRef(void)
{
  return _timeTick;
}

/** \brief Get time reference
 *
 * \param format LwOS_TimeRefFormat_t Format of time reference
 * \return LwOS_TimeRef_t Time reference
 *
 */
LwOS_TimeRef_t LwOS_GetTimeRef(LwOS_TimeRefFormat_t format)
{
  LwOS_TimeRef_t timeRef;

  switch(format)
  {
    case TIMEREF_IN_MILLISECONDS: timeRef = (LwOS_TimeRef_t)((_timeTick * TIMESLICE) / 10); break;
    case TIMEREF_IN_TENSOFMILLISECOND: timeRef = (LwOS_TimeRef_t)((_timeTick * TIMESLICE) / 100); break;
    case TIMEREF_IN_TENTHSOFASECOND: timeRef = (LwOS_TimeRef_t)((_timeTick * TIMESLICE) / 1000); break;
    case TIMEREF_IN_SECONDS: timeRef = (LwOS_TimeRef_t)((_timeTick * TIMESLICE) / 1000); break;
  }

  return timeRef;
}


/******************************* RESOURCES ******************************/


/** \brief Calls LOCK macro to disable IRQs (except kernel level SVC)
 *
 * \param void
 * \return void
 *
 */
void Lock(void)
{
  LOCK();
}

/** \brief Calls UNLOCK macro to enable IRQs
 *
 * \param void
 * \return void
 *
 */
void Unlock(void)
{
  UNLOCK();
}

/** \brief Waits (and releases processor) until resource is available then locks it
 *
 * \param resource LwOS_Resource_t* Pointer to resource handle (declared by user application)
 * \return void
 *
 */
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wunused-value"
void RESOURCE_LOCK(LwOS_Resource_t *resource)
{
  do {
    LOCK();
    if (!*resource) {
      *resource = (LwOS_Resource_t)_current;
      _current->state = TASK_READY;
      UNLOCK();
      break;
    }
    _current->data.blockedBy = (LwOS_Task_t*)*resource;
    _current->state = TASK_BLOCKED;
    LwOS_Yield();
    UNLOCK();
#if (defined(LOCKING_DEBUG) && defined (NESTED_IRQ_LOCK))
    if (__IRQLock) {
        while (1) { __NOP; }  // Locking fault!
    }
#endif
  } while (1);
}
#pragma GCC diagnostic pop

/** \brief Releases resource
 *
 * \param resource LwOS_Resource_t* Pointer to resource handle (declared by user application)
 * \return void
 *
 */
void RESOURCE_UNLOCK(LwOS_Resource_t *resource)
{
  ATOMIC( if (*resource == (LwOS_Resource_t)_current) *resource = 0; );
}

/** \brief Waits (and releases processor) until resource is available then locks it, resource is allocated by round robin method
 *
 * \param resourceRR LwOS_Resource_RR_t* Pointer to resource handle (declared by user application)
 * \return void
 *
 */
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wunused-value"
void RESOURCE_RR_LOCK(LwOS_Resource_RR_t *resourceRR)
{
  LOCK();
  uint16_t myPos = resourceRR->locked;
  resourceRR->locked++;
  UNLOCK();
  do {
    LOCK();
    if (resourceRR->freed == myPos) {
      resourceRR->blocker = (uint32_t)_current;
      _current->state = TASK_READY;
      UNLOCK();
      break;
    }
    _current->data.blockedBy = (LwOS_Task_t*)resourceRR->blocker;
    _current->state = TASK_BLOCKED;
    LwOS_Yield();
    UNLOCK();
#if (defined(LOCKING_DEBUG) && defined (NESTED_IRQ_LOCK))
    if (__IRQLock) {
        while (1) { __NOP; }  // Locking fault!
    }
#endif
  } while(1);
}
#pragma GCC diagnostic pop

/** \brief Releases resource (allocated by round robin method)
 *
 * \param resourceRR LwOS_Resource_RR_t* Pointer to resource handle (declared by user application)
 * \return void
 *
 */
void RESOURCE_RR_UNLOCK(LwOS_Resource_RR_t *resourceRR)
{
  ATOMIC(
    resourceRR->freed++;
    resourceRR->blocker = 0;
  );
}
