/* Copyright (C) 2017, Attila Kovs
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation  and/or other materials provided with the distribution.
 * 3. Neither the names of the copyright holders nor the names of any
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/* This file is part of LwOS (Lightweight runtime Operating System).
 */

#include "platform_cm3.h"
#include <string.h>

// Control register bits
#define CONTROL_TPL               (1UL << 0)        // Thread mode privilege level
#define CONTROL_ASPSEL            (1UL << 1)        // Active stack pointer selection
// PSR register bits
#define APSR_N                    (1UL << 31)       // Negative or less than flag
#define APSR_Z                    (1UL << 30)       // Zero flag
#define APSR_C                    (1UL << 29)       // Carry or borrow flag
#define APSR_V                    (1UL << 28)       // Overflow flag
#define APSR_Q                    (1UL << 27)       // Sticky saturation flag
#define EPSR_T                    (1UL << 24)       // Thumb set (always 1)

// Callee saved registers (Context)
typedef struct
{
	uint32_t r4;
	uint32_t r5;
	uint32_t r6;
	uint32_t r7;
	uint32_t r8;
	uint32_t r9;
	uint32_t r10;
	uint32_t r11;
} Context_t;

// Exception stack frame
typedef struct
{
	uint32_t r0;               // R0
	uint32_t r1;               // R1
	uint32_t r2;               // R2
	uint32_t r3;               // R3
	uint32_t r12;              // R12
	uint32_t lr;               // Link register
	uint32_t pc;               // Program counter
	uint32_t psr;              // Program status register
} StackFrame_t;


// Total context size
const uint32_t LwOS_ContextSize = sizeof(StackFrame_t) + sizeof(Context_t);

// Timekeeping
__IO LwOS_TimeRef_t _timeTick;
#ifdef TASK_DEBUG
static __IO uint32_t _majorTick;
uint32_t _minorTickPerSecond;
#endif

// Timeslice
__IO uint16_t _timeSlices;

// Task switching enable
__IO uint8_t _taskSwitchingEnabled = 1;

/** \brief Create new task's context at the top of its stack
 *
 * \param stackP void* Top of new task's stack
 * \param taskFunc void* Task entry point
 * \param arg void* Optional parameters for the task
 * \param returnFunc void* Task end function entry point
 * \return void*
 *
 */
void* CreateContext(void *stackP, void *taskFunc, void *arg, void *returnFunc)
{
	StackFrame_t *stackFrame;
	uint8_t *sp;

	sp = stackP;
	sp -= sizeof(StackFrame_t);
	stackFrame = (StackFrame_t *)sp;
	stackFrame->r0 = (uint32_t)arg;
	stackFrame->r1 = 0;
	stackFrame->r2 = 0;
	stackFrame->r3 = 0;
	stackFrame->r12 = 0;
	stackFrame->lr = (uint32_t)returnFunc;
	stackFrame->pc = (uint32_t)taskFunc;
	stackFrame->psr = APSR_C | EPSR_T;

	sp -= sizeof(Context_t);
	memset(sp, 0x00, sizeof(Context_t));

	return sp;
}

/** \brief Initialization for Cortex-M3 platform
 *
 * \param void
 * \return void
 *
 */
void Platform_Init(void)
{
  NVIC_SetPriorityGrouping(7 - __NVIC_PRIO_BITS);
  NVIC_SetPriority(SVCall_IRQn, PRIORITY_SVCALL);
  NVIC_SetPriority(PendSV_IRQn, PRIORITY_PENDSV);
}

/** \brief Initialize SysTick interrupt
 *
 * \param void
 * \return void
 *
 */
void SysTick_Init(void)
{
  LOCK();
  uint32_t cnt = (SystemCoreClock / 8) / LWOS_TICK;

#ifdef TASK_DEBUG
  _minorTickPerSecond = SystemCoreClock / 8;
#endif

  SysTick_Config(cnt);
  NVIC_SetPriority(SysTick_IRQn, PRIORITY_SYSTICK);
  SysTick_CLKSourceConfig(SysTick_CLKSource_HCLK_Div8);
  _timeSlices = 1;
  UNLOCK();
}

/** \brief Sets the allocated time slices for the current task
 *
 * \param num uint16_t Time slice number
 * \return void
 *
 */
void SetTaskTimeSliceNum(uint16_t num)
{
  _timeSlices = num;
}

#ifdef TASK_DEBUG
/** \brief Returns high resolution elapsed time counter (for CPU usage calculation)
 *
 * \param reset uint8_t Resets the main counter
 * \return uint32_t Elapsed time
 *
 */
uint32_t SysTick_GetReference(uint8_t reset)
{
  uint32_t mtick0 , tick, mtick1;

  if (reset) ATOMIC ( _majorTick = 0; );

  do
  {
    mtick0 = _majorTick;
    tick = SysTick->VAL;
    mtick1 = _majorTick;
  } while (mtick0 != mtick1);

  return (_majorTick * (SysTick->LOAD + 1)) - tick;
}
#endif

/** \brief Interrupt handler for SysTick
 *         Handling timekeeping for OS and application purposes
 *         Initiating context switch
 *
 * \param void
 * \return void
 *
 */
void SysTick_Handler(void)
{
  _timeTick++;
#ifdef TASK_DEBUG
  _majorTick++;
#endif

  if (_taskSwitchingEnabled) {
    if (--_timeSlices) {
      LwOS_Yield();
      _timeSlices = 1;
    }
  }
}

/** \brief Interrupt handler for pending services
 *         Handles context switching between tasks
 *
 * \param void
 * \return NAKED void
 *
 */
NAKED void PendSV_Handler(void)
{
  __ASM volatile(
    "tst lr, #4               \n\t"     // Test bit 2 of EXC_RETURN code
    "ite eq                   \n\t"
    "mrseq r0, msp            \n\t"     // If 0, stacking used MSP, copy to R0 (main task)
    "mrsne r0, psp            \n\t"     // If 1, stacking used PSP, copy to R0 (any other task)
    "stmdb	r0!, {r4-r11}     \n\t"     // Store context registers in stack, write new stack pointer back to R0
    "ittee eq                 \n\t"     // New "if then" block with previous result
    "msreq	msp, r0           \n\t"     // If 0, store new stack pointer in MSP
    "moveq  r0, #1            \n\t"     // If 0, set R0 to 1
    "msrne	psp, r0           \n\t"     // If 1, store new stack pointer in PSP
    "movne  r0, #0            \n\t"     // If 1, set R0 to 0
    "bl _scheduler            \n\t"     // Call scheduler with input argument R0
    "teq r0, #1               \n\t"     // Check if return argument is equal to 1
    "ite eq                   \n\t"
    "mrseq r0, msp            \n\t"     // If 0, copy MSP to R0
    "mrsne r0, psp            \n\t"     // If 1, copy PSP to R0
    "ldmia	r0!, {r4-r11}     \n\t"     // Restore context registers from stack
    "ittee eq                 \n\t"     // New "if then" block with previous result
    "msreq	msp, r0           \n\t"     // If 0, store new stack pointer in MSP
    "moveq lr, #0xFFFFFFF9    \n\t"     // If 0, store 0xFFFFFFF9 as EXC_RETURN code in link register (return to thread mode and use main stack pointer for stacking)
    "msrne	psp, r0           \n\t"     // If 1, store new stack pointer in PSP
    "movne lr, #0xFFFFFFFD    \n\t"     // If 1, store 0xFFFFFFFD as EXC_RETURN code in link register (return to thread mode and use process stack pointer for stacking)
    "bx lr"                             // Return to task
  );
}

/** \brief Handles supervisor calls
 *
 * \param svcArgs unsigned int* Pointer to argument array received from caller
 * \return void
 *
 */
void _SVC_HandlerC(unsigned int * svcArgs)
{
  uint8_t svcNumber;

  svcNumber = ((uint8_t *) svcArgs[6])[-2]; // Memory[(Stacked PC)-2]

  switch(svcNumber)
  {
    // Creating new task
    case SVC_CREATETASK:
      svcArgs[0] = (unsigned int)_createTask((TaskFunc_t)svcArgs[0], (void*)svcArgs[1], (size_t)svcArgs[2], (size_t)svcArgs[3]);
      break;
    // Requesting context switch
    case SVC_CONTEXTSWITCHSLEEP:
      _taskSwitchingEnabled = 1;
    case SVC_CONTEXTSWITCH:
      if (_taskSwitchingEnabled) RequestContextSwitch();
      break;
#ifdef STACK_DEBUG
    case SVC_MEASURESTACKUSE:
      _calculateStackUsage((uint32_t)svcArgs + sizeof(StackFrame_t));
      break;
#endif
    case SVC_EXCLUSIVE_MODE_ON:
      _taskSwitchingEnabled = 0;
      SCB->ICSR |= SCB_ICSR_PENDSVCLR_Msk;
      break;
    case SVC_EXCLUSIVE_MODE_OFF:
      _taskSwitchingEnabled = 1;
      break;
  }
}

/** \brief Handler for supervisor calls
 *         Checks which stack pointer were used at call (main / process), stores stack pointer as argument for C handler function
 *
 * \param void
 * \return NAKED void
 *
 */
NAKED void SVC_Handler(void)
{
  __ASM volatile(
    "tst lr, #4       \n\t"     // Test bit 2 of EXC_RETURN code
    "ite eq           \n\t"
    "mrseq r0, msp    \n\t"     // If 0, stacking used MSP, copy to R0 (main task, IRQ or kernel function)
    "mrsne r0, psp    \n\t"     // If 1, stacking used PSP, copy to R0 (any other task)
    "b _SVC_HandlerC  \n\t"     // Call C handler function
  );
}


/** \brief Stop point for hard faults
 *         Registers are available in stack frame for debugging
 *
 * \param stackFrameAddress uint32_t*
 * \return void
 *
 */
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wunused-but-set-variable"
void _HardFault_HandlerC(uint32_t *stackFrameAddress)
{
    StackFrame_t *stackFrame;

    stackFrame = (StackFrame_t*)stackFrameAddress;

    while (1) {}
}
#pragma GCC diagnostic pop

/** \brief Handler for hard faults
 *         Checks which stack pointer were used at point of fault (main / process), stores stack pointer as argument for C handler function
 *
 * \param void
 * \return NAKED void
 *
 */
NAKED void HardFault_Handler(void)
{
    __ASM volatile(
      "tst lr, #4             \n\t"     // Test bit 2 of EXC_RETURN code
      "ite eq                 \n\t"
      "mrseq r0, msp          \n\t"     // If 0, stacking used MSP, copy to R0 (main task, IRQ or kernel function)
      "mrsne r0, psp          \n\t"     // If 1, stacking used PSP, copy to R0 (any other task)
      "b _HardFault_HandlerC  \n\t"     // Call C hard fault handler function
    );
}
