#include "message.h"
#include "timer.h"
#include "lwos.h"
#include <string.h>

uint8_t _silent;
static __IO uint16_t _usartBufferPos;
static __IO uint16_t _usartSending;
static __IO uint8_t _usartBuffer;

static char __IO _debugBuffer1[USART_DEBUG_BUFFERSIZE / 2];
static char __IO _debugBuffer2[USART_DEBUG_BUFFERSIZE / 2];


#ifdef USARTDEBUG
static void _usartSend(const char *str)
{
  char *target;

  LOCK();
  size_t l = strlen(str);
  if (!l) {
    UNLOCK();
    return;
  }

#ifdef LWOS
  while ((l + _usartBufferPos + 1) > (USART_DEBUG_BUFFERSIZE/2))
  {
    LwOS_Yield();
    UNLOCK();
    __NOP();
    LOCK();
  }
#else
  if ((l + _usartBufferPos + 1) > (USART_DEBUG_BUFFERSIZE/2)) {
    UNLOCK();
    return; // Drop message
  }
#endif


  if (_usartBuffer) target = (char*)_debugBuffer2;
  else target = (char*)_debugBuffer1;
  target += _usartBufferPos;

  LwOS_MeasureStackUsage();

  strcpy(target, str);

  _usartBufferPos += l;
  if (_usartSending) {
    UNLOCK();
    return;
  }
  DMA_ClearFlag(USART_DEBUG_DMA_FLAG);
  DMA_Cmd(USART_DEBUG_DMA_Channel, DISABLE);
  DMA_SetCurrDataCounter(USART_DEBUG_DMA_Channel, l);
  if (_usartBuffer) {
    USART_DEBUG_DMA_Channel->CMAR = (uint32_t)_debugBuffer2;
  } else {
    USART_DEBUG_DMA_Channel->CMAR = (uint32_t)_debugBuffer1;
  }
  _usartSending = l;
  _usartBufferPos = 0;
  _usartBuffer = !_usartBuffer;
  DMA_ITConfig(USART_DEBUG_DMA_Channel, DMA_IT_TC, ENABLE);
  DMA_Cmd(USART_DEBUG_DMA_Channel, ENABLE);
  UNLOCK();
}

void DBG_BlockPrint(const char *str)
{
  DisableIRQ();
  size_t l = strlen(str);
  int i;
  for (i = 0; i < l; i++) {
    USART_SendData(USART1,*(char *)(str + i));
    while (USART_GetFlagStatus(USART1, USART_FLAG_TXE) == RESET);
  }
  EnableIRQ();
}

ISR(USART_DEBUG_DMA_IRQHandler)
{
  DISABLE_IRQ();
  DMA_ClearFlag(USART_DEBUG_DMA_FLAG);
  DMA_Cmd(USART_DEBUG_DMA_Channel, DISABLE);
  if (!_usartBufferPos) {
    _usartSending = 0;
    DMA_ITConfig(USART_DEBUG_DMA_Channel, DMA_IT_TC, DISABLE);
    ENABLE_IRQ();
    __DSB();
    return;
  }
  DMA_SetCurrDataCounter(USART_DEBUG_DMA_Channel, _usartBufferPos);
  if (_usartBuffer) {
    USART_DEBUG_DMA_Channel->CMAR = (uint32_t)_debugBuffer2;
  } else {
    USART_DEBUG_DMA_Channel->CMAR = (uint32_t)_debugBuffer1;
  }
  _usartSending = _usartBufferPos;
  _usartBufferPos = 0;
  _usartBuffer = !_usartBuffer;
  DMA_Cmd(USART_DEBUG_DMA_Channel, ENABLE);
  ENABLE_IRQ();
  __DSB();
}

void DBG_Init(void)
{
  USART_DEBUG_Config((uint32_t)_debugBuffer1);
  _usartSend(TXT("*** LOG STARTED ***\r\n"));
}

void DBG_PrintC(const char *str, uint16_t color)
{
  char escstr[8] = "\x1B[36;1m";
  switch (color) {
    case LCD_BLACK:
      escstr[3] = '0';
      break;
    case LCD_RED:
      escstr[3] = '1';
      break;
    case LCD_GREEN:
      escstr[3] = '2';
      break;
    case LCD_YELLOW:
      escstr[3] = '3';
      break;
    case LCD_BLUE:
      escstr[3] = '4';
      break;
    case LCD_CYAN:
      escstr[3] = '6';
      break;
    case LCD_WHITE:
      escstr[3] = '7';
      break;
    default:
      escstr[3] = '5';
      break;
  }
  _usartSend(escstr);
  _usartSend(str);
  _usartSend(TXT("\x1B[0m"));
}

void DBG_Print(const char *str)
{
  _usartSend(str);
}

void DBG_PrintLn(const char *str)
{
  _usartSend(str);
  _usartSend(TXT("\r\n"));
}

void DBG_PrintLnC(const char *str, uint16_t color)
{
  DBG_PrintC(str, color);
  _usartSend(TXT("\r\n"));
}

void DBG_TimeRef(void)
{
  char str[20];

  //ltoa(str, SysTick_GetReference(), 10);
  DBG_Print(TXT("["));
  DBG_PrintC(str, LCD_YELLOW);
  DBG_Print(TXT("]:"));
#ifdef SCHEDULER_NAMEDTASKS
  DBG_PrintC(Scheduler_GetActiveTaskName(), LCD_WHITE);
  DBG_Print(TXT(":"));
#endif
}

void DBG_PrintChar(char c)
{
  char s[2];

  s[0] = c;
  s[1] = 0;
  _usartSend(s);
}

void DBG_Ok(void)
{
  _usartSend(TXT("\x1B[32;1mOk!\x1B[0m\r\n"));
}

void DBG_Fail(void)
{
  _usartSend(TXT("\x1B[31;1mFailed!\x1B[0m\r\n"));
}
#endif

/*void MSG_PrintC(const char *str, uint16_t color)
{
  DBG_PrintC(str, color);
  if (_silent) return;
  LCD_SetColor(color, LCD_BLACK);
  LCD_Print(str);
}

void MSG_Print(const char *str)
{
  DBG_Print(str);
  if (_silent) return;
  LCD_SetColor(LCD_LIGHTGRAY, LCD_BLACK);
  LCD_Print(str);
}

void MSG_PrintLnC(const char *str, uint16_t color)
{
  if (_silent) return;
  MSG_PrintC(str, color);
  MSG_PrintLn("");
}

void MSG_PrintLn(const char *str)
{
  DBG_PrintLn(str);
  if (_silent) return;
  LCD_SetColor(LCD_LIGHTGRAY, LCD_BLACK);
  LCD_PrintLn(str);
}

void MSG_Ok(void)
{
#ifdef USARTDEBUG
  _usartSend(TXT("\x1B[32;1mOk!\x1B[0m\r\n"));
#endif
  if (_silent) return;
  LCD_SetColor(LCD_GREEN, LCD_BLACK);
  LCD_PrintLn(TXT("Ok!"));
}

void MSG_Fail(void)
{
#ifdef USARTDEBUG
  _usartSend(TXT("\x1B[31;1mFailed!\x1B[0m\r\n"));
#endif
  if (_silent) return;
  LCD_SetColor(LCD_RED, LCD_BLACK);
  LCD_PrintLn(TXT("Failed!"));
}*/
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wunused-but-set-variable"
ISR(USART_DEBUG_IRQHandler)
{
  uint16_t data;
  if (USART_DEBUG->SR & USART_SR_RXNE) {
    data = USART_DEBUG->DR;
  }
  else
  {
    data = USART_DEBUG->DR; // Flush data
  }
  __DSB();
}
#pragma GCC diagnostic pop

#ifdef MONITOR
void Monitoring_Init(void)
{
  Monitoring_Config();
}
#endif
