;*************************************************************************
;
;	Software for Frequency Counter model PIC-FC1.
;	Version 1.00.
;
;	Copyright (c) 1999 by 32 Technical Laboratory. (JG6DFK/QRP)
;	All rights reserved.
;
;*************************************************************************

; I/O assignment:
;	RA4 (pin  3) Input.				(I)
;	RA3 (pin  2) LCD Data #3.			(O)
;	RA2 (pin  1) LCD Data #2.			(O)
;	RA1 (pin 18) LCD Data #1.			(O)
;	RA0 (pin 17) LCD Data #0.			(O)
;	RB7 (pin 13) LCD Load Enable.			(O)
;	RB6 (pin 12) LCD Register Select.		(O)
;	RB5 (pin 11) Display Mode.			(I)
;	RB4 (pin 10) External prescaler mode. 		(I)
;	RB3 (pin  9) Internal prescaler mode #1.	(I)
;	RB2 (pin  8) Internal prescaler mode.#0.	(I)
;	RB1 (pin  7) Gate time #1.			(I)
;	RB0 (pin  6) Gate time #0.			(I)

	list b=.4

	ifdef	__16F84
		list	p=16f84			; 16F84 can be used
		#include <p16f84.inc>	; processor specific variable definitions

	__config _FOSC_XT & _WDTE_OFF & _PWRTE_ON & _CP_OFF
	endif

	ifdef	__16F84A
		list	p=16f84A		; 16F84A can be used
		#include <p16f84A.inc>	; processor specific variable definitions

	__config _FOSC_XT & _WDTE_OFF & _PWRTE_ON & _CP_OFF
	endif

;		.osc		xt		; Clock = 10 MHz.
;		.wdt		off
;		.pwrt		on
;		.protect	off

	__idlocs		0100h			; = Version No.

	errorlevel -302

;-------------------------------------------------------------------------
; Equations.
;-------------------------------------------------------------------------

; ----- I/O difinitions.

#define	LCD_LE		RB7			; LCD load enable.
#define	LCD_RS		RB6			; LCD register select.

#define	I_SELDSP	RB5			; Display mode       select SW.
#define	I_EXTPSC	RB4			; External prescaler select SW.
#define	I_INTPS1	RB3			; External prescaler select SW #1.
#define	I_INTPS0	RB2			; External prescaler select SW #0.
#define	I_SELRS1	RB1			; Resolution         select SW #1.
#define	I_SELRS0	RB0			; Resolution         select SW #0.

; ----- Constants.

FRES_1		equ	0			; Resolution mode =   1 Hz.
FRES_10		equ	1			;                    10 Hz.
FRES_100	equ	2			;                   100 Hz.
FRES_1000	equ	3			;                     1 kHz.

DSP_FREQ	equ	0			; Display mode = frequency.
DSP_LENG	equ	1			;                wave length. (X.XXX m)

IPSC_1		equ	0			; Internal prescaler = 1:1.
IPSC_4		equ	1			;                      1:4.
IPSC_8		equ	2			;                      1:8.
IPSC_16		equ	3			;                      1:16.

TMR0_1		equ	b'00101000'		; Prescaler rate for TMR0 = 1:1.
TMR0_4		equ	b'00100001'		;                           1:4.
TMR0_8		equ	b'00100010'		;                           1:8.
TMR0_16		equ	b'00100011'		;                           1:16.

XPSC_OFF	equ	0			; External prescaler = disabled.
XPSC_ON		equ	1			;                      enabled.

XPSC_RATE	equ	64			; External prescaler rate.

MAXDIGIT	equ	10			; Max digit = "XXXXXXX.XXX".

; ----- LCD functions.

L_SELDR		equ	0			; LCD data register select.

L_FNCSET	equ	b'00101000'		; LCD function set.
L_ENTSET	equ	b'00000110'		; LCD entry mode set.
L_DSPCLR	equ	b'00000001'		; LCD display clear.
L_DSPOFF	equ	b'00001000'		; LCD display OFF.
L_DSPON		equ	b'00001100'		; LCD display ON.
L_CSHOME	equ	b'00000010'		; LCD cursor position reset.
L_DDFREQ	equ	b'10000000'+01h		; LCD "Frequency" DD RAM addr.
L_DDMODE	equ	b'10000000'+4Ch		; LCD "Mode"      DD RAM addr.
L_DDINP1	equ	b'10000000'+40h		; LCD "Input"     DD RAM addr1.
L_DDINP2	equ	b'10000000'+46h		; LCD "Input"     DD RAM addr2.
L_DDGATE	equ	b'10000000'+4Fh		; LCD "Gate"      DD RAM addr.

;-------------------------------------------------------------------------
; Register file (= Work area) definitions.
;
; Notes: Lower byte is located first on long length variables.
;                                     $  $+1 $+2
;        (Intel format. Ex: 186A0h -> A0  86  01)
;-------------------------------------------------------------------------

		cblock	0Ch

; ----- General.

I:	.4			; General loop counter.
X:	.4			; Source       register for long word calc.
EX:	.4			; Extra source register for mul/div.
Y:	.4			; Destination  register for long word calc.
EY:	.4			; Extra dest.  register for mul/div.

; ----- For count procedure.

CNTMODE			; Temporary input buffer.

DSPMODE			; Display mode.
EXT_PSC			; External prescaler mode.
INT_PSC			; Internal prescaler mode.
RESMODE			; Resolution mode.

GATETIME:	.2	; Gate time in milliseconds.
RTC_OLD			; RTCC previous value.
RTC_NEW			; RTCC new value.
CNT:		.3	; Count value.

; ----- For LCD procedure.

LCD_RSEL		; LCD register select.
LCD_DATA		; LCD command or character data.
LCD_STRBUF:		MAXDIGIT		; LCD string buffer.
LCD_DPOINT		; LCD decimal point location.
LCD_FXMPTR		; LCD fixed message pointer.

; ----- For calculation procedure.

MDLOOPCT		; Loop counter for mul/div.
CALCTMP1:	.5	; Temporary buffer for calculation.
CALCTMP2:	.5	; Temporary buffer for calculation.

; ----- For wait procedure.

WAIT1			; For large loop.
WAIT2			; For small loop.

;-------------------------------------------------------------------------
; Program entry.
;-------------------------------------------------------------------------
	endc

		org	0
		goto	main
		nop
		nop
		nop
		goto	main

;-------------------------------------------------------------------------
; String tables.
;-------------------------------------------------------------------------

s_kHz
		retlw	' '			; "kHz".
		retlw	'k'			;
		retlw	'H'			;
		retlw	'z'			;
		retlw	0			;

s_MHz
		retlw	' '			; "MHz".
		retlw	'M'			;
		retlw	'H'			;
		retlw	'z'			;
		retlw	0			;

s_m
		retlw	' '			; "m  ".
		retlw	'm'			;
		retlw	' '			;
		retlw	' '			;
		retlw	0			;

s_LFmode
		retlw	'L'			; "LF".
		retlw	'F'			;
		retlw	0			;

s_MFmode
		retlw	'M'			; "MF".
		retlw	'F'			;
		retlw	0			;

s_MHmode
		retlw	'M'			; "MH".
		retlw	'H'			;
		retlw	0			;

s_HFmode
		retlw	'H'			; "HF".
		retlw	'F'			;
		retlw	0			;

s_xpshdr
		retlw	'I'			; "INPUT".
		retlw	'N'			;
		retlw	'P'			;
		retlw	'U'			;
		retlw	'T'			;
		retlw	':'			;
		retlw	0			;

s_xpsOFF
		retlw	'A'			; "A".
		retlw	0			; (External prescaler:OFF)

s_xpsON
		retlw	'B'			; "B".
		retlw	0			;(External prescaler:ON)

s_over
		retlw	'T'			; "Too Long !!"
		retlw	'o'			;
		retlw	'o'			;
		retlw	' '			;
		retlw	'L'			;
		retlw	'o'			;
		retlw	'n'			;
		retlw	'g'			;
		retlw	' '			;
		retlw	'!'			;
		retlw	'!'			;
		retlw	0			;

;-------------------------------------------------------------------------
; Data tables.
;-------------------------------------------------------------------------

; ----- Get Internal prescaler rate. -------------------------------------

get_iprate
		addwf	PCL,f
		retlw	1			; 1:1.
		retlw	4			; 1:4.
		retlw	8			; 1:8.
		retlw	16			; 1:16.

; ----- Get Internal prescaler rate for TMR0. ----------------------------

get_iprtm0
		addwf	PCL, f
		retlw	TMR0_1			; 1:1.
		retlw	TMR0_4			; 1:4.
		retlw	TMR0_8			; 1:8.
		retlw	TMR0_16			; 1:16.

; ----- Get message address of internal prescaler mode. ------------------

get_ipmptr
		addwf	PCL, f
		retlw	s_LFmode		; "LF".
		retlw	s_MFmode		; "MF".
		retlw	s_MHmode		; "MH".
		retlw	s_HFmode		; "HF".

; ----- Get decimal point location. --------------------------------------

get_dploc
		addwf	PCL, f
		retlw	3			;   X.XXX kHz.
		retlw	5			; X.XXXXX MHz.
		retlw	4			;  X.XXXX MHz.
		retlw	3			;   X.XXX MHz.

;-------------------------------------------------------------------------
; Interrupt procedure.
;-------------------------------------------------------------------------

;		No interrupt procedure.

;-------------------------------------------------------------------------
; Main procedure.
;-------------------------------------------------------------------------

main

;	*** Initialize registers ***

		clrf	PCLATH			; Select program memory bank #0.

		movlw	00000000b		; Disabled all interrupt
		movwf	INTCON			; sources.

		clrf	PORTA			; Initialize I/O ports.
		clrf	PORTB			;

		bsf		STATUS, RP0		; Select bank 1.

		movlw	b'00101000'		; Pull-up PORTB, T0CS = EXT &
		movwf	OPTION_REG		; set TMR0 rate. (1:1)

		movlw	b'00010000'		; Set data directions.
		movwf	TRISA			;   1: as input.
		movlw	b'00111111'		;
		movwf	TRISB			;

		bcf		STATUS, RP0		; Select bank 0. (Default)

		movlw	b'11110000'		; Set all output ports to '0'.
		andwf	PORTA, f		;
		movlw	b'00111111'		;
		andwf	PORTB, f		;

;	*** Initialize variable ***

		clrf	CNT+0			; Clear counter for
		clrf	CNT+1			; dummy display.
		clrf	CNT+2			;

;	*** Initialize LCD ***

		call	init_LCD		; Contains function set & clear screen.
		call	dsp_fixmsg		; Display fixed message.

;	*** Main Loop ***

mainloop
		call	set_cntmode		; Set count mode. (Gate time etc.)
		call	dsp_status		; Display status.

		movf	DSPMODE, f		; Check display mode.
		btfss	STATUS, Z		;
		goto	len_mode		;

		call	dsp_freq		; Display frequency.
		goto	cntstart
len_mode
		call	dsp_leng		; Display wave length.
cntstart
		call	dsp_goff		; Display "Gate OFF".
		call	dsp_gon			; Display "Gate ON".
		call	count			; Count core.

		goto	mainloop

;-------------------------------------------------------------------------
; Set count mode.
;   Input : Port B   (Data input port of count mode)
;   Output: DSPMODE  (Display mode)
;           EXT_PSC  (External prescaler mode)
;           INT_PSC  (Internal prescaler mode)
;           RESMODE  (Resolution mode)
;           GATETIME (Gate time in milliseconds)
;           W register & variable I will be broken.
;
;   Notes: All input ports are negative logic.
;-------------------------------------------------------------------------

set_cntmode

;	*** Get count mode from input port ***

		movf	PORTB, w		; Read input port &
		andlw	b'00111111'		; negate logic.
		xorlw	b'00111111'		;
		movwf	CNTMODE			;

		andlw	b'00000011'		; Get resolution mode.
		movwf	RESMODE			;

		rrf		CNTMODE, f		; Get internal prescaler mode.
		rrf		CNTMODE, f		;
		movf	CNTMODE, w		;
		andlw	b'00000011'		;
		movwf	INT_PSC			;

		rrf		CNTMODE, f		; Get external prescaler mode.
		rrf		CNTMODE, f		;
		movf	CNTMODE, w		;
		andlw	b'00000001'		;
		movwf	EXT_PSC			;

		rrf		CNTMODE, f		; Get display mode.
		movf	CNTMODE, w		;
		andlw	b'00000001'		;
		movwf	DSPMODE			;

;	*** Set gate time ***

		movlw	FRES_100		; Set gate time soon if
		subwf	RESMODE, w		; resolution >= 100 Hz.
		btfsc	STATUS, C		;
		goto	set_gtime		;

		movf	EXT_PSC, f		; Check other mode
		btfsc	STATUS, Z		; if external prescaler is enabled &
		goto	set_gtime		; internal prescaler rate <> 1:1.
						;
		movf	INT_PSC, f		;
		btfsc	STATUS, Z		;
		goto	set_gtime		;

		movlw	IPSC_16			; Resolution = 100 Hz if
		subwf	INT_PSC, w		; internal prescaler rate = 1:16,
		movlw	FRES_10			; else resolution = 10 Hz.
		btfsc	STATUS, C		;
		movlw	FRES_100		;
						;
		movwf	RESMODE			;
set_gtime
		movlw	1			; X <- external prescaler rate
		movf	EXT_PSC, f		;       * internal prescaler rate
		btfss	STATUS, Z		;       * 1000.
		movlw	XPSC_RATE		;
						;
		movwf	X+0			;
		clrf	X+1			;
		clrf	X+2			;
		clrf	X+3			;
							;
		movf	INT_PSC, w		;
		call	get_iprate		;
		movwf	Y+0			;
		clrf	Y+1			;
		clrf	Y+2			;
		clrf	Y+3			;
						;
		call	mul_xy			;
						;
		movlw	0e8h			;
		movwf	Y+0			;
		movlw	03h			;
		movwf	Y+1			;
		call	mul_xy			;

		movlw	10			; X <- X / 10 while I > 0.
		movwf	Y+0			;(Skip soon if RESMODE = FRES_1)
		clrf	Y+1			;
						;
		movf	RESMODE, w		;
		btfsc	STATUS, Z		;
		goto	end_sgt			;
								;
		movwf	I			;
sgt_loop					;
		call	div_xy			;
								;
		decfsz	I, f			;
		goto	sgt_loop		;
end_sgt
		movf	X+0, w			; GATETIME <- X.
		movwf	GATETIME+0		;
		movf	X+1, w			;
		movwf	GATETIME+1		;

;	*** Set Internal prescaler rate ***

		movf	INT_PSC, w		; Get TMR0 rate.
		call	get_iprtm0		;

		bsf		STATUS, RP0		; Select bank 1.
		movwf	OPTION_REG		; set new TMR0 rate.
		bcf		STATUS, RP0		; Select bank 0. (Default)

;	*** Return from subroutine ***

		return

;-------------------------------------------------------------------------
; Count procedure.
;   Input : GATETIME (Gate time in milliseconds)
;   Output: CNT (Value)
;           W register & variable I, X, Y will be broken.
;-------------------------------------------------------------------------

count

;	*** Prologue ***

		movf	GATETIME+0, w		; I <- GATETIME * 100.
		movwf	X+0			; (Convert milliseconds to
		movf	GATETIME+1, w		;  10 microseconds)
		movwf	X+1			;
		clrf	X+2			; I: Count loop counter.
		clrf	X+3			; (Use lower 24 bits)
						;
		movlw	64h			;
		movwf	Y+0			;
		clrf	Y+1			;
		clrf	Y+2			;
		clrf	Y+3			;
						;
		call	mul_xy			;
						;
		movf	X+0, w			;
		movwf	I+0			;
		movf	X+1, w			;
		movwf	I+1			;
		movf	X+2, w			;
		movwf	I+2			;

		clrf	CNT+0			; Clear pulse counter.
		clrf	CNT+1			;
		clrf	CNT+2			;

						; Clear RTCC & init OLD value of RTCC.
		clrw				;(But you may not clear it)
		movwf	TMR0			; RTCC is disabled to increment
		movwf	RTC_OLD			; for 2 instruction cycles after
		nop				; it is modified.

		movlw	7			; Wait 10 microseconds.
		movwf	I+3			;
						;
		decfsz	I+3, f		;
		goto	$-1			;
		nop				;
		nop				;
		nop				;

;	*** Pulse counter (Gate time = 10 microseconds * I) ***

pulse_cnt
		movf	TMR0, w			; Increment pulse counter.
		movwf	RTC_NEW			;
		movf	RTC_OLD, w		;
		subwf	RTC_NEW, w		;
						;
		addwf	CNT+0, f		;
		movlw	1			;
		btfsc	STATUS, C		;
		addwf	CNT+1, f		;
		btfsc	STATUS, C		;
		addwf	CNT+2, f		;

		movf	RTC_NEW, w		; OLD <- NEW.
		movwf	RTC_OLD			;

		movlw	1			; Decrement counter for sub loop.
		subwf	I+0, f			;
		btfss	STATUS, C		;
		subwf	I+1, f			;
		btfss	STATUS, C		;
		subwf	I+2, f			;

		nop				; Trim.

		movf	I+0, w			; Continue if I > 0.
		iorwf	I+1, w			;
		iorwf	I+2, w			;
		btfss	STATUS, Z		;
		goto	pulse_cnt		;

;	*** Return from subroutine ***

		return

;-------------------------------------------------------------------------
; Init LCD.
;   Input : None.
;   Output: Port A, Port B (Data output port for LCD)
;           LCD_RSEL (LCD register select which always 0)
;           W register & variable LCD_DATA, I will be broken.
;
; Notes: Waiting time is about execution time * 2.7.
;-------------------------------------------------------------------------

init_LCD

;	*** Power ON wait ***

		movlw	.100			; Power ON wait for LCD.
		call	wait_ms			; (>= 15 milliseconds)

;	*** Function set (Dummy #1) ***

		clrf	LCD_RSEL		; RS = '0'.
		bcf	PORTB, LCD_RS		; (Default)

		movlw	b'11110000'		; Function set.
		andwf	PORTA, f		; (Data bus = 8 bits)
		movlw	b'00000011'		;
		iorwf	PORTA, f		;

		movlw	3			; Repeat 3 times.
		movwf	I			;
Li_fsloop
		bsf	PORTB, LCD_LE		; Strobe.
		nop				;
		nop				;
		bcf	PORTB, LCD_LE		;

		movlw	5			; Wait >= 4.1 milliseconds.
		call	wait_ms			;

		decfsz	I, f			; Continue if I > 0.
		goto	Li_fsloop		;

;	*** Function set (Dummy #2) ***

		movlw	b'11110000'		; Function set.
		andwf	PORTA, f		; (Data bus = 4 bits)
		movlw	b'00000010'		;
		iorwf	PORTA, f		;

		bsf	PORTB, LCD_LE		; Strobe.
		nop				;
		nop				;
		bcf	PORTB, LCD_LE		;

		call	wait_100us		; Wait >= 100 microseconds.

;	*** Function set (Real) ***

		movlw	L_FNCSET		; Set real function.
		call	set_LCD			;

;	*** Others ***

		movlw	L_DSPOFF		; Display OFF.
		call	set_LCD			;

		call	cls			; Clear display.

		movlw	L_ENTSET		; Set entry mode.
		call	set_LCD			;

		movlw	L_DSPON			; Display ON.
		call	set_LCD			;

;	*** Return from subroutine ***

		return

;-------------------------------------------------------------------------
; Display fixed message.
;   Input : None.
;   Output: None.
;           W register, variable LCD_DATA & LCD_FXMPTR will be broken.
;-------------------------------------------------------------------------

dsp_fixmsg

;	*** Display message ***

		movlw	L_DDINP1		; Set cursor to "Input".
		call	set_LCD			; (Header message)

		movlw	s_xpshdr		; Display header message of
		movwf	LCD_FXMPTR		; input channel.
		call	dsp_fixstr		; (External prescaler)

;	*** Return from subroutine ***

		return

;-------------------------------------------------------------------------
; Display status.
;   Input : INT_PSC  (Internal prescaler mode)
;           EXT_PSC  (External prescaler mode)
;   Output: None.
;           W register, variable LCD_FXMPTR will be broken.
;-------------------------------------------------------------------------

dsp_status

;	*** Display internal prescaler mode ***

		movlw	L_DDMODE		; Set cursor to "Mode".
		call	set_LCD			;

		movf	INT_PSC, w		; Get message address.
		call	get_ipmptr		;

		movwf	LCD_FXMPTR		; Display internal prescaler mode.
		call	dsp_fixstr		;

;	*** Display Input channel (Ext. prescaler mode) ***

		movlw	L_DDINP2		; Set cursor to "Input".
		call	set_LCD			; (Core)

		movlw	s_xpsOFF		; Set message address.
		movf	EXT_PSC, f		;
		btfss	STATUS, Z		;
		movlw	s_xpsON			;

		movwf	LCD_FXMPTR		; Display input channel.
		call	dsp_fixstr		;

;	*** Return from subroutine ***

		return

;-------------------------------------------------------------------------
; Display "Gate Open".
;   Input : None.
;   Output: None.
;           W register will be broken.
;-------------------------------------------------------------------------

dsp_gon

;	*** Display message ***

		movlw	L_DDGATE		; Set cursor to "Gate".
		call	set_LCD			;

		bsf	LCD_RSEL, L_SELDR	; Select data register.

		movlw	'*'			; Set gate sign.
		call	set_LCD			;

		clrf	LCD_RSEL		; Set to default.

;	*** Return from subroutine ***

		return

;-------------------------------------------------------------------------
; Display "Gate Close".
;   Input : None.
;   Output: None.
;           W register & variable LCD_DATA will be broken.
;-------------------------------------------------------------------------

dsp_goff

;	*** Display message ***

		movlw	L_DDGATE		; Set cursor to "Gate".
		call	set_LCD			;

		bsf	LCD_RSEL, L_SELDR	; Select data register.

		movlw	' '			; Clear gate sign &
		movwf	LCD_DATA		; wait.
		call	set_LCDcmd		;
		movlw	.100			;
		call	wait_ms			;

		clrf	LCD_RSEL		; Set to default.

;	*** Return from subroutine ***

		return

;-------------------------------------------------------------------------
; Display frequency.
;   Input : CNT     (Count value)
;           RESMODE (Resolution mode)
;   Output: None.
;           W register, variable X, EX, Y, EY, LCD_DPOINT & LCD_FXMPTR
;           will be broken.
;-------------------------------------------------------------------------

dsp_freq

;	*** Set cursor location of LCD & convert count data to string ***

		movlw	L_DDFREQ		; Set cursor to "frequency".
		call	set_LCD			;

		movf	CNT+0, w		; Change count data to
		movwf	X+0			; string.
		movf	CNT+1, w		;
		movwf	X+1			;
		movf	CNT+2, w		;
		movwf	X+2			;
		clrf	X+3			;
		clrf	EX+0			;
		clrf	EX+1			;
		clrf	EX+2			;
		clrf	EX+3			;
		call	x_to_str		;

;	*** Display frequency ***

		movf	RESMODE, w		; Set decimal point location.
		call	get_dploc		;
		movwf	LCD_DPOINT		;

		call	dsp_value		; Display value.

		movlw	s_kHz			; Add unit. (kHz or MHz)
		movf	RESMODE, f		;
		btfss	STATUS, Z		;
		movlw	s_MHz			;
						;
		movwf	LCD_FXMPTR		;
		call	dsp_fixstr		;

;	*** Return from subroutine ***

		return

;-------------------------------------------------------------------------
; Display wave length.
;   Input : CNT     (Count value)
;           RESMODE (Resolution mode)
;   Output: None.
;           W register, variable I, X, EX, Y, EY, LCD_DPOINT & LCD_FXMPTR
;           will be broken.
;-------------------------------------------------------------------------

dsp_leng

;	*** Set cursor location of LCD ***

		movlw	L_DDFREQ		; Set cursor to "frequency".
		call	set_LCD			;

;	*** Trim count data ***

		movf	CNT+0, w		; X <- Count value.
		movwf	X+0			; Y <- 1000 / Resolution.
		movf	CNT+1, w		; X <- X / Y.
		movwf	X+1			;
		movf	CNT+2, w		;
		movwf	X+2			;
		clrf	X+3			;
		clrf	EX+0			;
		clrf	EX+1			;
		clrf	EX+2			;
		clrf	EX+3			;
						;
		movlw	10			;
		movwf	Y+0			;
		clrf	Y+1			;
		clrf	Y+2			;
		clrf	Y+3			;
						;
		movf	RESMODE, w		;
		sublw	FRES_1000		;
		btfsc	STATUS, Z		;
		goto	dpln_cv0		;
						;
		movwf	I			;
dpln_trim					;
		call	div_xy			;
						;
		decfsz	I, f			;
		goto	dpln_trim		;

;	*** Convert data to string ***

dpln_cv0
		movf	X+0, w			; Y <- X.
		movwf	Y+0			; (Y: frequency in kHz)
		movf	X+1, w			;
		movwf	Y+1			;
		movf	X+2, w			;
		movwf	Y+2			;
		movf	X+3, w			;
		movwf	Y+3			;

		iorwf	Y+2, w			; Display "Too Long !!"
		iorwf	Y+1, w			; if value = 0.
		iorwf	Y+0, w			; (Less than 1 kHz)
		btfss	STATUS, Z		;
		goto	dpln_cv1		;
						;
		movlw	s_over			;
		movwf	LCD_FXMPTR		;
		call	dsp_fixstr		;
		goto	end_dpln		;
dpln_cv1
		movlw	00h			; X <- 300000000 / Y
		movwf	X+0			; (X: wave length in mm)
		movlw	0A3h			; & convert to string.
		movwf	X+1			;
		movlw	0E1h			;
		movwf	X+2			;
		movlw	11h			;
		movwf	X+3			;
		call	div_xy			;
		call	x_to_str		;

;	*** Display wave length ***

		movlw	3			; Set decimal point location.
		movwf	LCD_DPOINT		; (X.XXX m)

		call	dsp_value		; Display value.
end_dpln
		movlw	s_m			; Add unit. (m)
		movwf	LCD_FXMPTR		;
		call	dsp_fixstr		;

;	*** Return from subroutine ***

		return

;-------------------------------------------------------------------------
; Display value. (With 0 suppless & decimal point)
;   Input : LCD_STRBUF (String buffer)
;           LCD_DPOINT (Decimal point from right side)
;   Output: None.
;           W, Index register & variable I will be broken.
;-------------------------------------------------------------------------

dsp_value

;	*** Initialize ***

		bsf	LCD_RSEL, L_SELDR	; Select data register.

		movlw	LCD_STRBUF		; Set pointer of string.
		movwf	FSR			;

;	*** Display space (Zero suppless) ***

		movlw	MAXDIGIT-1		; Set max space count.
		movwf	I			; (MAXDIGIT - 1)
dspvl_l1
		movf	I, w			; Exit this loop if
		subwf	LCD_DPOINT, w		; this loc. = decimal point or
		btfsc	STATUS, Z		; not zero.
		goto	dspvl_2s		;
						;
		movf	INDF, w			;
		sublw	'0'			;
		btfss	STATUS, Z		;
		goto	dspvl_2s		;

		movlw	' '			; Put ' '.
		call	set_LCD			;

		incf	FSR, f			; Set location to next.
		decfsz	I, f			;
		goto	dspvl_l1		;

;	*** Display number & decimal point ***

dspvl_2s
		incf	I, f			; Increment counter.
dspvl_l2
		movf	INDF, w			; Display number.
		call	set_LCD			;

		incf	FSR, f			; Decrement counter &
		decf	I, f			; skip below if this loc
		movf	I, w			; is not decimal point loc.
		subwf	LCD_DPOINT, w		;
		btfss	STATUS, Z		;
		goto	dspvl_2e		;

		movlw	'.'			; Display decimal point.
		call	set_LCD			;
dspvl_2e
		movf	I, f			; Continue if I > 0.
		btfss	STATUS, Z		;
		goto	dspvl_l2		;

;	*** Return from subroutine ***

		clrf	LCD_RSEL		; Set to default.
		return

;-------------------------------------------------------------------------
; Convert value of X to string. (Fixed 10 digits)
;   Input :  X[0]- X[3] (Value)
;           EX[0]-EX[3]
;   Output: LCD_STRBUF  (Buffer of string for LCD)
;           W, Index register & variable I, Y, EY will be broken.
;-------------------------------------------------------------------------

x_to_str

;	*** Initialize ***

		movlw	LCD_STRBUF		; Set pointer of string.
		addlw	9			; (To bottom)
		movwf	FSR

		movlw	10			; Set loop counter &
		movwf	I			; Y.
		movwf	Y+0			;
		clrf	Y+1			;
		clrf	Y+2			;
		clrf	Y+3			;

;	*** Conversion loop (32 bits = 10 digits max) ***

xts_loop
		call	div_xy			; X <- X / 10.
		movf	EY+0, w			; EY(Lowest digit) <- X mod 10.

		addlw	'0'			; Change to ASCII &
		movwf	INDF			; store.

		decf	FSR, f			; Set location to next.
		decfsz	I, f			;
		goto	xts_loop		;

;	*** Return from subroutine ***

		return

;-------------------------------------------------------------------------
; Clear screen. (display)
;   Input : None.
;   Output: None.
;           W register & variable LCD_DATA will be broken.
;-------------------------------------------------------------------------

cls

;	*** Clear display ***

		movlw	L_DSPCLR		; Clear display.
		movwf	LCD_DATA		;
		call	set_LCDcmd		;

		movlw	5			; Wait.
		call	wait_ms			;

;	*** Return from subroutine ***

		return

;-------------------------------------------------------------------------
; Display fixed string. (ASCIZ string on ROM)
;   Input : LCD_FXMPTR (Pointer of fixed ASCIZ string table)
;   Output: None.
;           W, Index register, variable & LCD_FXMPTR will be broken.
;-------------------------------------------------------------------------

dsp_fixstr

;	*** Initialize ***

		bsf	LCD_RSEL, L_SELDR	; Select data register.

;	*** Display loop ***

dfs_loop
		call	get_fixchr		; Exit if character = NULL.
		addlw	0			;
		btfsc	STATUS, Z		;
		goto	end_dfs			;

		call	set_LCD			; Display character.

		incf	LCD_FXMPTR, f		; Continue.
		goto	dfs_loop		;

;	*** Return from subroutine ***

end_dfs
		clrf	LCD_RSEL		; Set to default.
		return

;-------------------------------------------------------------------------
; Get character from fixed message table.
;   Input : LCD_FXMPTR (Pointer of fixed ASCIZ string table)
;   Output: None.
;           W register will be broken.
;-------------------------------------------------------------------------

get_fixchr
		movf	LCD_FXMPTR, w		; Caution: Upper byte of PCL
		movwf	PCL			; is always 0.

;-------------------------------------------------------------------------
; Set LCD command for standard operation.
;   Input : W register (Command)
;   Output: None.
;           W register & variable LCD_DATA will be broken.
;-------------------------------------------------------------------------

set_LCD

;	*** Set command ***

		movwf	LCD_DATA
		call	set_LCDcmd
		call	wait_100us

;	*** Return from subroutine ***

		return

;-------------------------------------------------------------------------
; Set LCD command (Instruction).
;   Input : LCD_RSEL (LCD register select)
;           LCD_DATA (LCD data)
;   Output: Port A, Port B (Data output port for LCD)
;           W register will be broken.
;-------------------------------------------------------------------------

set_LCDcmd

;	*** Select register ***

		bcf	PORTB, LCD_RS		; Select register.
		btfsc	LCD_RSEL, L_SELDR	;
		bsf	PORTB, LCD_RS		;

;	*** Send upper nibbles ***

		movlw	b'11110000'		; Set upper nibbles.
		andwf	PORTA, f		;
		swapf	LCD_DATA, w		;
		andlw	b'00001111'		;
		iorwf	PORTA, f		;

		bsf	PORTB, LCD_LE		; Strobe.
		nop				;
		nop				;
		bcf	PORTB, LCD_LE		;
		nop				;
		nop				;

;	*** Send lower nibbles ***

		movlw	b'11110000'		; Set lower nibbles.
		andwf	PORTA, f		;
		movf	LCD_DATA, w		;
		andlw	b'00001111'		;
		iorwf	PORTA, f		;

		bsf	PORTB, LCD_LE		; Strobe.
		nop				;
		nop				;
		bcf	PORTB, LCD_LE		;

;	*** Return from subroutine ***

		return

;-------------------------------------------------------------------------
; General: Add. (32 bits source + 32 bits destination)
;   Input : X[0]-X[3] (Source     : 32 bits)
;           Y[0]-Y[3] (Destination: 32 bits)
;   Output: X[0]-X[3] (Answer     : 32 bits)
;           W register will be broken.
;
; Allocations (MSB <-> LSB):
;	 X[3] X[2] X[1] X[0]
;	 Y[3] Y[2] Y[1] Y[0]
;-------------------------------------------------------------------------

add_xy

;	*** Add Y to X ***

		movf	Y+0, w			; Lower.
		addwf	X+0, f			;
		movlw	1			;
		btfsc	STATUS, C		;
		addwf	X+1, f			;
		btfsc	STATUS, C		;
		addwf	X+2, f			;
		btfsc	STATUS, C		;
		addwf	X+3, f			;

		movf	Y+1, w			; Middle lower.
		addwf	X+1, f			;
		movlw	1			;
		btfsc	STATUS, C		;
		addwf	X+2, f			;
		btfsc	STATUS, C		;
		addwf	X+3, f			;

		movf	Y+2, w			; Middle upper.
		addwf	X+2, f			;
		movlw	1			;
		btfsc	STATUS, C		;
		addwf	X+3, f			;

		movf	Y+3, w			; Upper.
		addwf	X+3, f			;

;	*** Return from subroutine ***

		return

;-------------------------------------------------------------------------
; General: Subtract. (32 bits source - 32 bits destination)
;   Input : X[0]-X[3] (Source     : 32 bits)
;           Y[0]-Y[3] (Destination: 32 bits)
;   Output: X[0]-X[3] (Answer     : 32 bits)
;           W register will be broken.
;
; Allocations (MSB <-> LSB):
;	 X[3] X[2] X[1] X[0]
;	 Y[3] Y[2] Y[1] Y[0]
;-------------------------------------------------------------------------

sub_xy

;	*** Subtract Y from X ***

		movf	Y+0, w			; Lower.
		subwf	X+0, f			;
		movlw	1			;
		btfss	STATUS, C		; <- Positive if CF = 1.
		subwf	X+1, f			;
		btfss	STATUS, C		;
		subwf	X+2, f			;
		btfss	STATUS, C		;
		subwf	X+3, f			;

		movf	Y+1, w			; Middle lower.
		subwf	X+1, f			;
		movlw	1			;
		btfss	STATUS, C		;
		subwf	X+2, f			;
		btfss	STATUS, C		;
		subwf	X+3, f			;

		movf	Y+2, w			; Middle upper.
		subwf	X+2, f			;
		movlw	1			;
		btfss	STATUS, C		;
		subwf	X+3, f			;

		movf	Y+3, w			; Upper.
		subwf	X+3, f			;

;	*** Return from subroutine ***

		return

;-------------------------------------------------------------------------
; General: Multiply. (32 by 32 bits)
;   Input :  X[0]- X[3] (Source     : 32 bits)
;            Y[0]- Y[3] (Destination: 32 bits)
;   Output: EX[0]-EX[3] (Answer     : 64 bits)
;            X[0]- X[3]
;           EY[0]-EY[3] (As same as X[0]-X[3])
;
;           W register will be broken.
;
; Notes: Referred to AKIZUKI's PIC sample software.
;        This routine uses the same technic of manual-calculation.
;
; Example:	 00010010 (12h)
;		x00110100 (34h)
;		----------
;		 00000000
;		00000000
;	       00010010
;	      00000000
;	     00010010
;	    00010010
;	   00000000
;	  00000000
;	------------------
;	 0000001110101000 (03A8h)
;
; Allocations (MSB <-> LSB):
;	EX[3] EX[2] EX[1] EX[0] X[3] X[2] X[1] X[0]
;	EY[3] EY[2] EY[1] EY[0] Y[3] Y[2] Y[1] Y[0]
;-------------------------------------------------------------------------

mul_xy

;	*** Initialize ***

		movf	X+0, w			; EY <- X.
		movwf	EY+0			;
		movf	X+1, w			;
		movwf	EY+1			;
		movf	X+2, w			;
		movwf	EY+2			;
		movf	X+3, w			;
		movwf	EY+3			;

		movf	Y+0, w			; X <- Y.
		movwf	X+0			;
		movf	Y+1, w			;
		movwf	X+1			;
		movf	Y+2, w			;
		movwf	X+2			;
		movf	Y+3, w			;
		movwf	X+3			;

		clrf	EX+0			; Clear EX.
		clrf	EX+1			;
		clrf	EX+2			;
		clrf	EX+3			;

		movlw	32			; Set data length.
		movwf	MDLOOPCT		;

;	*** Multiply loop (32 times) ***

mul_loop
		btfss	X+0, 0			; Skip below if LSB of XL = '0'.
		goto	mul_next		; (EY * '0' = 0)

		bcf	X+0, 0			; Clear always LSB of X.

		movf	EY+0, w			; EX <- EX + EY.
		addwf	EX+0, f			;
		movlw	1			; Set LSB of X to '1'
		btfsc	STATUS, C		; if overflow. (Set carry)
		addwf	EX+1, f			;
		btfsc	STATUS, C		;
		addwf	EX+2, f			;
		btfsc	STATUS, C		;
		addwf	EX+3, f			;
		btfsc	STATUS, C		;
		bsf	X+0, 0			;
						;
		movf	EY+1, w			;
		addwf	EX+1, f			;
		movlw	1			;
		btfsc	STATUS, C		;
		addwf	EX+2, f			;
		btfsc	STATUS, C		;
		addwf	EX+3, f			;
		btfsc	STATUS, C		;
		bsf	X+0, 0			;
						;
		movf	EY+2, w			;
		addwf	EX+2, f			;
		movlw	1			;
		btfsc	STATUS, C		;
		addwf	EX+3, f			;
		btfsc	STATUS, C		;
		bsf	X+0, 0			;
						;
		movf	EY+3, w			;
		addwf	EX+3, f			;
		btfsc	STATUS, C		;
		bsf	X+0, 0			;
mul_next
		bcf	STATUS, C		; Rotate EX + X to right.
		rrf	EX+3, f			; (LSB of X -> MSB of EX)
		rrf	EX+2, f			;
		rrf	EX+1, f			;
		rrf	EX+0, f			;
		rrf	X+3, f			;
		rrf	X+2, f			;
		rrf	X+1, f			;
		rrf	X+0, f			;
		btfsc	STATUS, C		;
		bsf	EX+3, 7			;

		decfsz	MDLOOPCT, f		; Continue if loop > 0.
		goto	mul_loop		;

;	*** Return from subroutine ***

		return

;-------------------------------------------------------------------------
; General: Divide. (64 bits source by 32 bits destination)
;   Input : EX[0]-EX[3] (Source     : 64 bits)
;            X[0]- X[3]
;            Y[0]- Y[3] (Destination: 32 bits)
;   Output: EX[0]-EX[3] (Answer     : 64 bits)
;            X[0]- X[3]
;            Y[0]- Y[3] (Mod        : 32 bits)
;
;           W register will be broken.
;
; Notes: Referred to AKIZUKI's PIC sample software.
;        This routine uses the same technic of manual-calculation.
;
; Example: 	 0000000001100100 (64h)
;	       +------------------
;    (10h) 1010| 0000001111101000 (03E8h)
;		       1010
;	       -------------------
;			1011
;			1010
;	       -------------------
;			   1010
;			   1010
;	       -------------------
;			      0
;
; Allocations (MSB <-> LSB):
;	EX[3] EX[2] EX[1] EX[0] X[3] X[2] X[1] X[0]
;	EY[3] EY[2] EY[1] EY[0] Y[3] Y[2] Y[1] Y[0]
;-------------------------------------------------------------------------

div_xy

;	*** Initialize ***

		clrf	CALCTMP1+0		; Clear temporary buffer.
		clrf	CALCTMP1+1		;
		clrf	CALCTMP1+2		;
		clrf	CALCTMP1+3		;
		clrf	CALCTMP1+4		;

		movlw	64			; Set data length.
		movwf	MDLOOPCT		;

;	*** Divide loop (64 times) ***

div_loop
		bcf	STATUS, C		; Shift CALCTMP1(Mod) + EX + X
		rlf	X+0, f			; to left.
		rlf	X+1, f			;
		rlf	X+2, f			;
		rlf	X+3, f			;
		rlf	EX+0, f			;
		rlf	EX+1, f			;
		rlf	EX+2, f			;
		rlf	EX+3, f			;
		rlf	CALCTMP1+0, f		;
		rlf	CALCTMP1+1, f		;
		rlf	CALCTMP1+2, f		;
		rlf	CALCTMP1+3, f		;
		rlf	CALCTMP1+4, f		;

		movf	CALCTMP1+0, w		; CALCTMP2(Work) <- CALCTMP1.
		movwf	CALCTMP2+0		;
		movf	CALCTMP1+1, w		;
		movwf	CALCTMP2+1		;
		movf	CALCTMP1+2, w		;
		movwf	CALCTMP2+2		;
		movf	CALCTMP1+3, w		;
		movwf	CALCTMP2+3		;
		movf	CALCTMP1+4, w		;
		movwf	CALCTMP2+4		;

		movf	Y+0, w			; CALCTMP2 <- CALCTMP2 - Y.
		subwf	CALCTMP2+0, f		; (Signed operation)
		movlw	1			;
		btfss	STATUS, C		; <- Positive if CF=1.
		subwf	CALCTMP2+1, f		;
		btfss	STATUS, C		;
		subwf	CALCTMP2+2, f		;
		btfss	STATUS, C		;
		subwf	CALCTMP2+3, f		;
		btfss	STATUS, C		;
		subwf	CALCTMP2+4, f		;
						;
		movf	Y+1, w			;
		subwf	CALCTMP2+1, f		;
		movlw	1			;
		btfss	STATUS, C		;
		subwf	CALCTMP2+2, f		;
		btfss	STATUS, C		;
		subwf	CALCTMP2+3, f		;
		btfss	STATUS, C		;
		subwf	CALCTMP2+4, f		;
						;
		movf	Y+2, w			;
		subwf	CALCTMP2+2, f		;
		movlw	1			;
		btfss	STATUS, C		;
		subwf	CALCTMP2+3, f		;
		btfss	STATUS, C		;
		subwf	CALCTMP2+4, f		;
						;
		movf	Y+3, w			;
		subwf	CALCTMP2+3, f		;
		movlw	1			;
		btfss	STATUS, C		;
		subwf	CALCTMP2+4, f		;

		btfsc	CALCTMP2+4, 7		; Skip below if CALCTMP2 < 0.
		goto	div_next		; (MSB of CALCTMP2 is '1')

		movf	CALCTMP2+0, w		; CALCTMP1(Mod) <- CALCTMP2 &
		movwf	CALCTMP1+0		; LSB of X <- '1'.
		movf	CALCTMP2+1, w		;
		movwf	CALCTMP1+1		;
		movf	CALCTMP2+2, w		;
		movwf	CALCTMP1+2		;
		movf	CALCTMP2+3, w		;
		movwf	CALCTMP1+3		;
		movf	CALCTMP2+4, w		;
		movwf	CALCTMP1+4		;
						;
		bsf	X+0, 0			;
div_next
		decfsz	MDLOOPCT, f		; Continue if loop > 0.
		goto	div_loop		;

;	*** Return from subroutine ***

		movf	CALCTMP1+0, w		; EY(Mod) <- CALCTMP1.
		movwf	EY+0			;
		movf	CALCTMP1+1, w		;
		movwf	EY+1			;
		movf	CALCTMP1+2, w		;
		movwf	EY+2			;
		movf	CALCTMP1+3, w		;
		movwf	EY+3			;

		return

;-------------------------------------------------------------------------
; General: Rotate left. (16 bits length)
;   Input : W (Pointer of entry)
;-------------------------------------------------------------------------

rl16

;	*** Rotate left ***

		movwf	FSR

		bcf	STATUS, C		; Set '0' on LSB.
		rlf	INDF, f			; Lower.
		incf	FSR, f
		rlf	INDF, f			; Upper.

		decf	FSR, f
		btfsc	STATUS, C
		bsf	INDF, 0			; Set '1' on LSB if CF = '1'.

;	*** Return from subroutine ***

		return

;-------------------------------------------------------------------------
; General: Rotate right. (16 bits length)
;   Input : W (Pointer of entry)
;-------------------------------------------------------------------------

rr16

;	*** Rotate right ***

		movwf	FSR

		incf	FSR, f

		bcf	STATUS, C		; Set '0' on MSB.
		rrf	INDF, f			; Upper.
		decf	FSR, f
		rrf	INDF, f			; Lower.

		incf	FSR, f
		btfsc	STATUS, C
		bsf	INDF, 7			; Set '1' on MSB if CF = '1'.

;	*** Return from subroutine ***

		return

;-------------------------------------------------------------------------
; General: Rotate left. (32 bits length)
;   Input : W (Pointer of entry)
;-------------------------------------------------------------------------

rl32

;	*** Rotate left ***

		movwf	FSR

		bcf	STATUS, C		; Set '0' on LSB.
		rlf	INDF, f			; Lower.
		incf	FSR, f
		rlf	INDF, f			; Middle lower.
		incf	FSR, f
		rlf	INDF, f			; Middle upper.
		incf	FSR, f
		rlf	INDF, f			; Upper.

		decf	FSR, f
		decf	FSR, f
		decf	FSR, f
		btfsc	STATUS, C
		bsf	INDF, 0			; Set '1' on LSB if CF = '1'.

;	*** Return from subroutine ***

		return

;-------------------------------------------------------------------------
; General: Rotate right. (32 bits length)
;   Input : W (Pointer of entry)
;-------------------------------------------------------------------------

rr32

;	*** Rotate right ***

		movwf	FSR

		incf	FSR, f
		incf	FSR, f
		incf	FSR, f

		bcf	STATUS, C		; Set '0' on MSB.
		rrf	INDF, f			; Upper.
		decf	FSR, f
		rrf	INDF, f			; Middle upper.
		decf	FSR, f
		rrf	INDF, f			; Middle lower.
		decf	FSR, f
		rrf	INDF, f			; Lower.

		incf	FSR, f
		incf	FSR, f
		incf	FSR, f
		btfsc	STATUS, C
		bsf	INDF, 7			; Set '1' on MSB if CF = '1'.

;	*** Return from subroutine ***

		return

;-------------------------------------------------------------------------
; General: shift left. (16 bits length)
;   Input : W (Pointer of entry)
;-------------------------------------------------------------------------

sl16

;	*** Shift left ***

		movwf	FSR

		bcf	STATUS, C		; Set '0' on LSB.
		rlf	INDF, f			; Lower.
		incf	FSR, f
		rlf	INDF, f			; Upper.

;	*** Return from subroutine ***

		return

;-------------------------------------------------------------------------
; General: Shift right. (16 bits length)
;   Input : W (Pointer of entry)
;-------------------------------------------------------------------------

sr16

;	*** Shift right ***

		movwf	FSR

		incf	FSR, f

		bcf	STATUS, C		; Set '0' on MSB.
		rrf	INDF, f			; Upper.
		decf	FSR, f
		rrf	INDF, f			; Lower.

;	*** Return from subroutine ***

		return

;-------------------------------------------------------------------------
; General: Shift left. (32 bits length)
;   Input : W (Pointer of entry)
;-------------------------------------------------------------------------

sl32

;	*** Shift left ***

		movwf	FSR

		bcf	STATUS, C		; Set '0' on LSB.
		rlf	INDF, f			; Lower.
		incf	FSR, f
		rlf	INDF, f			; Middle lower.
		incf	FSR, f
		rlf	INDF, f			; Middle upper.
		incf	FSR, f
		rlf	INDF, f			; Upper.

;	*** Return from subroutine ***

		return

;-------------------------------------------------------------------------
; General: Shift right. (32 bits length)
;   Input : W (Pointer of entry)
;-------------------------------------------------------------------------

sr32

;	*** Shift right ***

		movwf	FSR

		incf	FSR, f
		incf	FSR, f
		incf	FSR, f

		bcf	STATUS, C		; Set '0' on MSB.
		rrf	INDF, f			; Upper.
		decf	FSR, f
		rrf	INDF, f			; Middle upper.
		decf	FSR, f
		rrf	INDF, f			; Middle lower.
		decf	FSR, f
		rrf	INDF, f			; Lower.

;	*** Return from subroutine ***

		return

;-------------------------------------------------------------------------
; General: Wait procedure.
;   Input : W (Time in milliseconds)
;   Output: None.
;           W register & variable WAIT1, WAIT2 will be broken.
;-------------------------------------------------------------------------

wait_ms

;	*** Wait loop ***

		movwf	WAIT1			; Set milliseconds loop counter.
wm_loop1
		movlw	.249			; Set microseconds loop counter.
		movwf	WAIT2			; (996 microseconds / 4)
wm_loop2
		nop
		nop
		nop
		nop
		nop
		nop
		nop
		decfsz	WAIT2, f			; Continue if loop2 > 0.
		goto	wm_loop2		;

		nop
		nop
		nop
		nop
		nop
		nop
		decfsz	WAIT1, f			; Continue if loop1 > 0.
		goto	wm_loop1		;

;	*** Return from subroutine ***

		return

;-------------------------------------------------------------------------
; General: Wait procedure. (Fixed 100 microseconds)
;   Input : None.
;   Output: None.
;           W register & variable WAIT1 will be broken.
;-------------------------------------------------------------------------

wait_100us

;	*** Wait loop ***

		movlw	48			; Set 100 microseconds loop counter.
		movwf	WAIT1			; (96 microseconds)
wu_loop
		nop
		nop
		decfsz	WAIT1, f			; Continue if loop > 0.
		goto	wu_loop			;

;	*** Return from subroutine ***

		nop
		nop
		nop
		nop
		nop
		return

; End of COUNTER.ASM

	end
