;************************************************************************
; Soubry Henk wrote:													*
; "Following the example of Bob Blick's original propeler clock I made 	*
; my own version of this fancy clock and have re-written the code from	*
; scratch. 																*
; This 16f628 code is for the stationary part of the clock and will 	*
; drive the primary transformer coil, and use RC5 to switch the motor	*
; on and off (and adjust the speed)"									*
;																		*
; - Rewritten, extended to an alarm clock:								*
;	- alarm timing (hour and minute with day of week masking)			*
;	  20 on 16F628(A) and 16F648A, 40 on 16F87 and 16F88,				*
;	- Alarm sound and relais drive output,								*
;	- Push button wake up, automatic standby timer,						*
;	- Date, Time, Alarm data and settings stored in PCF8583 RTC,		*
;	- Automatic time synchronization to RTC's time,						*
;	- Error led to sign time, alarm data invalid (low battery indicator)*
;	- I2C bus for handling other devices (LCD display),					*
;	- Menu buttons,														*
;	- Extended IR command set,											*
;	- RS232 channel to connect to PC,									*
;																		*
;************************************************************************
;                                                        		       	*
;    Filename:	    base.asm                                        	*
;    Date:          20/08/2002                                        	*
;    Last Modified: 09/05/2009                                        	*
;    File Version:  2.0E                                        		*
;                                                                     	*
;  Based on:                                                           	*
;    Author:        Soubry Henk                                       	*
;    Company:       Soubry Software Service                           	*
;                                                                     	*
;  Rewritten - Extended                                                	*
;                                                                     	*
;************************************************************************
;                                                                    	*
;    Files required:                                                  	*
;           keys.asm                                                	*
;                                                                     	*
;************************************************************************
;                                                                     	*
;    Notes:                                                           	*
;    	Pin assignment                                                	*
;      		Port A            common for 16F628 and 16F87-88       	  	*
;				RA0 = SCL line of I2C bus, use 2k7 pullup to pic's Vdd	*
;				RA1 = SDA line of I2C bus, use 2k7 pullup to pic's Vdd	*
;               RA2 = Ir receiver                                     	*
;               RA3 = Menu push button 								  	*
;               RA4 = Wake up push button					open drain	*
;				RA5 = Alarm sound off push button 			input only	*
;																		*
;      		Port B						16F628(A)-648A		16F87-88	*
;				Index LED output					RB0		RB0			*
;				UART RxD							RB1		RB2			*
;				UART TxD							RB2		RB5			*
;				CCP1 PWM output - FET GATE for coil	RB3		RB3			*
;				Motor Shutdown output				RB4		RB4			*
;				Alarm sound output					RB5		RB1			*
;				Relais drive output					RB6		RB6			*
;				Error LED output					RB7		RB7			*
;                                                                     	*
;************************************************************************
;																		*
;  RTC Ram layout														*
;	00	01	02	03	04	05	06	07	08	09	0A	0B	0C	0D	0E	0F		*
;	cmd		sec	min	hou	day	mon											*
;						y10	wda											*
;																		*
;	10	11	12	13	14	15	16	17	18	19	1A	1B	1C	1D	1E	1F		*
;	Yea	~Y	Ccp	Pr2	Ccp Pr2	AST	AST	HS	HS						Mode	*
;			run	run	sl	sl	H	L	beg	end								*
;																		*
;	20	21	22	23	24	25	26	27	28	29	2A	2B	2C	2D	2E	2F		*
;	-- Alarm 0  --	-- Alarm 1  --	-- Alarm 2  --	-- Alarm 3  --		*
;	WDa	Hou	Min	Md	WDa	Hou	Min	Md	WDa	Hou	Min	Md	WDa	Hou	Min	Md		*
;																		*
;	30	31	32	33	34	35	36	37	38	39	3A	3B	3C	3D	3E	3F		*
;	-- Alarm 4  --	-- Alarm 5  --	-- Alarm 6  --	-- Alarm 7  --		*
;	WDa	Hou	Min	Md	WDa	Hou	Min	Md	WDa	Hou	Min	Md	WDa	Hou	Min	Md		*
;																		*
;	40	41	42	43	44	45	46	47	48	49	4A	4B	4C	4D	4E	4F		*
;	-- Alarm 8  --	-- Alarm 9  --	-- Alarm 10 --	-- Alarm 11 --		*
;	WDa	Hou	Min	Md	WDa	Hou	Min	Md	WDa	Hou	Min	Md	WDa	Hou	Min	Md		*
;																		*
;	50	51	52	53	54	55	56	57	58	59	5A	5B	5C	5D	5E	5F		*
;	-- Alarm 12 --	-- Alarm 13 --	-- Alarm 14 --	-- Alarm 15 --		*
;	WDa	Hou	Min	Md	WDa	Hou	Min	Md	WDa	Hou	Min	Md	WDa	Hou	Min	Md		*
;																		*
;	60	61	62	63	64	65	66	67	68	69	6A	6B	6C	6D	6E	6F		*
;	-- Alarm 16 --	-- Alarm 17 --	-- Alarm 18 --	-- Alarm 19 --		*
;	WDa	Hou	Min	Md	WDa	Hou	Min	Md	WDa	Hou	Min	Md	WDa	Hou	Min	Md		*
;																		*
;	70	71	72	73	74	75	76	77	78	79	7A	7B	7C	7D	7E	7F		*
;	-- Alarm 20 --	-- Alarm 21 --	-- Alarm 22 --	-- Alarm 23 --		*
;	WDa	Hou	Min	Md	WDa	Hou	Min	Md	WDa	Hou	Min	Md	WDa	Hou	Min	Md		*
;																		*
;	80	81	82	83	84	85	86	87	88	89	8A	8B	8C	8D	8E	8F		*
;	-- Alarm 24 --	-- Alarm 25 --	-- Alarm 26 --	-- Alarm 27 --		*
;	WDa	Hou	Min	Md	WDa	Hou	Min	Md	WDa	Hou	Min	Md	WDa	Hou	Min	Md		*
;																		*
;	90	91	92	93	94	95	96	97	98	99	9A	9B	9C	9D	9E	9F		*
;	-- Alarm 28 --	-- Alarm 29 --	-- Alarm 30 --	-- Alarm 31 --		*
;	WDa	Hou	Min	Md	WDa	Hou	Min	Md	WDa	Hou	Min	Md	WDa	Hou	Min	Md		*
;																		*
;	A0	A1	A2	A3	A4	A5	A6	A7	A8	A9	AA	AB	AC	AD	AE	AF		*
;	-- Alarm 32 --	-- Alarm 33 --	-- Alarm 34 --	-- Alarm 35 --		*
;	WDa	Hou	Min	Md	WDa	Hou	Min	Md	WDa	Hou	Min	Md	WDa	Hou	Min	Md		*
;																		*
;	B0	B1	B2	B3	B4	B5	B6	B7	B8	B9	BA	BB	BC	BD	BE	BF		*
;	-- Alarm 36 --	-- Alarm 37 --	-- Alarm 38 --	-- Alarm 39 --		*
;	WDa	Hou	Min	Md	WDa	Hou	Min	Md	WDa	Hou	Min	Md	WDa	Hou	Min	Md		*
;																		*
;************************************************************************
;-----[ Error level set ]------------------------------------------------------------------------
	Errorlevel 	0,-302		; bank kijellsre figyelmeztet zenetek kikapcsolva


	ifdef	__16F628
		list	p=16f628              ; 16F628 can be used
		#include <p16f628.inc>        ; processor specific variable definitions
EEPROM_SIZE	EQU	128
    endif
	ifdef	__16F628A
		list	p=16f628A             ; 16F628A can be used
		#include <p16f628A.inc>       ; processor specific variable definitions
EEPROM_SIZE	EQU	128
	endif
	ifdef	__16F648A
		list	p=16f648A             ; 16F648A can be used
		#include <p16f648A.inc>       ; processor specific variable definitions
EEPROM_SIZE	EQU	256
	endif
	ifdef	__16F87
		list	p=16f87				  ; 16F87 can be used
		#include <p16f87.inc>		  ; processor specific variable definitions
		#define	T0IE	TMR0IE
		#define	T0IF	TMR0IF
EEPROM_SIZE	EQU	256
	endif
	ifdef	__16F88
		list	p=16f88				  ; 16F88 can be used
		#include <p16f88.inc>		  ; processor specific variable definitions
		#define	T0IE	TMR0IE
		#define	T0IF	TMR0IF
EEPROM_SIZE	EQU	256
	endif

	ifndef	EEPROM_SIZE
		error	"Incompatible processor type selected"
	endif

	#include "keys628_URC22.asm"

	ifdef	__16F88
		__CONFIG _CONFIG1, _CP_OFF & _CCP1_RB3 & _WRT_PROTECT_OFF & _WDT_OFF & _BODEN_ON & _PWRTE_ON & _HS_OSC & _MCLR_OFF & _LVP_OFF
		__CONFIG _CONFIG2, _IESO_OFF & _FCMEN_OFF
	else
	  ifdef	__16F87
		__CONFIG _CONFIG1, _CP_OFF & _CCP1_RB3 & _WRT_PROTECT_OFF & _WDT_OFF & _BODEN_ON & _PWRTE_ON & _HS_OSC & _MCLR_OFF & _LVP_OFF
		__CONFIG _CONFIG2, _IESO_OFF & _FCMEN_OFF
	  else
		__CONFIG _CP_OFF & _WDT_OFF & _BODEN_ON & _PWRTE_ON & _HS_OSC & _MCLRE_OFF & _LVP_OFF
	  endif
	endif

#define	VersionMajor	0x02
#define	VersionMinor	0x0E

	__idlocs		0xB000 | (VersionMajor << 8) | (VersionMinor)


#define RC5AddrCheck			; Comment this line if RC5 Address checking not required
;;;#define	UseI2CLcd			; Comment this line not to use I2C LCD (not implemented yet)

; Default mode settings at first power up
#define	AutoStandby				; Comment this line to disable auto standby
;#define	WakeUpRun				; Comment this line if wake up in sleep mode
#define	DeapSleeping			; Comment this line if coil has to be powered in sleep mode
;#define	TimeReInit				; Comment this line not to reinit time from RTC at midnight
;#define	EnableHourSign			; Comment this line to disable sound at hour change
;#define	SignMode				; Comment this line to enable as many beeps as hour (1..12)

; Serial port Baud rate selection
;BaudRate	EQU	.2400			; Set the BaudRate of RS232 communication
;BaudRate	EQU	.4800			; Set the BaudRate of RS232 communication
BaudRate	EQU	.9600			; Set the BaudRate of RS232 communication
;BaudRate	EQU	.19200			; Set the BaudRate of RS232 communication
;BaudRate	EQU	.38400			; Set the BaudRate of RS232 communication

#define	RTC_ADDR		0xA0	; RTC's address with A0 pin grounded
;#define	RTC_Control	0x00	; Control register
#define	RTC_Second		0x02	; Address of Second
;#define	RTC_Minute	0x03	; Address of Minute
;#define	RTC_Hour	0x04	; Address of Hour     / AM-PM flag and mode
;#define	RTC_Day		0x05	; Address of Day      / 2 lsb of Year
;#define	RTC_Month	0x06	; Address of Week day / Month
#define RTC_Year        0x10    ; Address of Year
#define RTC_YearNeg     0x11    ; Address of negated value of Year
#define	RTC_PWMData		0x12	; Address of PWM data
#define	RTC_AST			0x16	; Address of AutoStandbyTimer
#define	RTC_HourSnd		0x18	; Address of hour sound parameters
#define	RTC_Mode		0x1F	; Address of mode
#define	RTC_AlarmData	0x20	; Address of alarm data (.20*4 or .40*4 bytes)

	ifdef	UseI2CLcd
#define	LCD_ADDR		0x70	; LCD's address
	endif


; PWM settings for Sleep mode
PR2Sleep	EQU	0x3D
CCPR1LSleep	EQU	0x0C

; PWM settings for Run mode
PR2RUN		EQU	0x3D
CCPR1LRUN	EQU	0x28


F_OSC	EQU	.20000000			; Oscillator frequency in Hz
OSC		EQU	F_OSC/4

	ifndef BaudRate
		error "No BaudRate defined"
	endif
	if ( (BaudRate!=.2400) && (BaudRate!=.4800) && (BaudRate!=.9600) && (BaudRate!=.19200) && (BaudRate!=.38400) )
		error "Nonstandard BaudRate defined"
	endif

;***** VARIABLE DEFINITIONS

;Vars in shared memory. This memory is available from any bank

  cblock	0x070
	Scratch		; memory locations for general use
	Scratch2	;
	Scratch3	; Alarm index
	BCD			; BCD routine temp var, I2C delay counter

	I2CSlaveAdr	; I2C slave address of current device
	I2CWordAdr	; I2C word address of current transfer
	I2CBitCnt	; I2C bit counter of data in/out routines
	I2CShiftReg	; I2C data shift register

	flags		;
	Mode		;

	w_temp		; variable used for context saving
	status_temp	; variable used for context saving
	fsr_temp	; variable used for context saving
  endc
	IF fsr_temp > 0x7F
		ERROR "To many variables used in Common RAM"
	ENDIF

  cblock	0x20
;Vars for time-keeping
  ;!! Keep variable order - Timecheck use indirect access to them v
	Second2			; 0..119 counting half seconds
	Minute			; 0.. 59
	Hour			; 0.. 23
	Day				; 1.. Dmon-1
	Month			; 1.. 12
	Year			; 0.. 99
  ;!! Keep variable order - Timecheck use indirect access to them ^

	DMon			; days in current Month + 1
	WDay			; Day of week 0..6

	SubSec_L		; Sub second timer
	SubSec_H

	SleepTimer		; Down counter in second, started at standby.

	StandbyTimerL	; Down counter for auto standby
	StandbyTimerH	;

	SoundCnt		; Down counter for alarm sound generation
	SwitchCnt		; Down counter for relais drive

	AutoStbyTimL	; Auto Standby time in seconds
	AutoStbyTimH	;

	PR2_Sleep		; PWM frequency for sleep mode
	PR2_Run			; PWM frequency for run   mode

	CCP1_Sleep		; PWM pulse width for sleep mode
	CCP1_Run		; PWM pulse width for run   mode

	RC5_flags		; RC5 decoding state flags
	RC5_Tmr			; RC5 time counter
	RC5_BitCnt		; RC5 decoders bit counter
	RC5_Addr		; Address field of RC5 message just received
	RC5_Cmd			; Command field of RC5 message just received
	RC5_Cmd2		; storage for previous cmd

	Alarms			; Number of alarms can be used on processor

	UartRxChNum		; Uart Rx fifo number of avaible chars
	UartRxChWp		; Uart Rx fifo write pointer
	UartRxChRp		; Uart Rx fifo read  pointer

	UartTxChNum		; Uart Tx fifo number of avaible chars
	UartTxChWp		; Uart Tx fifo write pointer
	UartTxChRp		; Uart Tx fifo read  pointer

	Rs232Cmd		; Last command code received from uart
	Rs232ParNum		; Number of paraneters of command
	Rs232Flags		; Command decoding state flags

	Rs232StrPtr		; 16 bit pointer to string to be sent
	Rs232StrPtrH
	Rs232Temp		; Temporary storage in UART routines
	Rs232Temp2		; Temporary storage in UART routines
	Rs232Fsr		; Save for FSR in UART routines
	Rs232Status		; Save for STATUS in UART routines

	HourSndBeg
	HourSndEnd

  endc
	IF HourSndEnd >= 0x70
		ERROR "To many variables used in Bank0"
	ENDIF

;Vars indirectly accessed are on Bank1
  cblock	0x0A0	; Alarm table - Indirectly accessed
	Alarm0WDay		; Bits 7: Global enable, 6..0: Enable on Day of week 7..1
	Alarm0Hour		; 0..23 or 32 for don't care
	Alarm0Minute	; 0..59
	Alarm0Mode		; Bit 7: 0 - Alarm:
							; Bit 6: turn propeller on
							; Bit 5: turn propeller off
							; Bit 4: Enable sound
					; Bit 7: 1 - Relais switched on:
							; Bit 6..0: turn on time in minutes

	Alarm1:4
	Alarm2:4
	Alarm3:4
	Alarm4:4
	Alarm5:4
	Alarm6:4
	Alarm7:4
	Alarm8:4
	Alarm9:4
	Alarm10:4
	Alarm11:4
	Alarm12:4
	Alarm13:4
	Alarm14:4
	Alarm15:4
	Alarm16:4
	Alarm17:4
	Alarm18:4
	Alarm19:4

	; Not to use memory 0x0F0..0x0FF  -  it will overwrite 0x070..0x07F
  endc
	IF  Alarm19+3 >= 0xF0
		ERROR "To many variables used in Bank1"
	ENDIF

  cblock	0x120	; Uart buffers
					; More RAM space avaible: 48 for 16F628(A) and 80 for 16F684A ,16F87, 16F88
	UartTxChar:	.32	; Uart Tx fifo

	UartRxChar:	.16	; Uart Rx fifo
 	; Not to use memory 0x170..0x17F  -  it will overwrite 0x070..0x07F
  endc

	ifdef	__16F628
	  IF  UartRxChar+.15 >= 0x150
		ERROR "To many variables used in Bank2"
	  ENDIF
    endif
	ifdef	__16F628A
	  IF  UartRxChar+.15 >= 0x150
		ERROR "To many variables used in Bank2"
	  ENDIF
    endif

	IF  UartRxChar+.15 >= 0x170
		ERROR "To many variables used in Bank2"
	ENDIF

	ifdef	TMR0IE
;Vars indirectly accessed are on Bank3 on 16F87 and 16F88
  cblock	0x1A0	; Alarm table - Indirectly accessed
	Alarm20:4
	Alarm21:4
	Alarm22:4
	Alarm23:4
	Alarm24:4
	Alarm25:4
	Alarm26:4
	Alarm27:4
	Alarm28:4
	Alarm29:4
	Alarm30:4
	Alarm31:4
	Alarm32:4
	Alarm33:4
	Alarm34:4
	Alarm35:4
	Alarm36:4
	Alarm37:4
	Alarm38:4
	Alarm39:4

	; Not to use memory 0x1F0..0x1FF  -  it will overwrite 0x070..0x07F
  endc
	IF  Alarm39+3 >= 0x1F0
		ERROR "To many variables used in Bank3"
	ENDIF
	endif

;**********************************************************************

; PORTA bits
bSCL		EQU	0				; I2C SCL line
bSDA		EQU	1				; I2C SDA line
bRC5inp		EQU	2				; IR receiver's output
bButton2	EQU	3				; Menu button
bButton		EQU	4				; Power up push button
bAlarmRes	EQU	5				; Alarm sound off push button

; PORTB bits
bLED		EQU	7				; Visible LED output
bRelais		EQU	6				; Relais drive output
bMotorOFF	EQU	4				; Motor controll output
bTrafo		EQU	3				; PWM controll of coil
bIndexLed	EQU	0				; Index LED output

; Difference in pinout 
	ifdef	__16F88
bTxD		EQU	5				; UART's TxD line
bRxD		EQU	2				; UART's RxD line
bSoundOut	EQU	1				; Buzzer output
	else
bSoundOut	EQU	5				; Buzzer output
bTxD		EQU	2				; UART's TxD line
bRxD		EQU	1				; UART's RxD line
	endif

; Flags
bStandBy	EQU	0				; Standby mode
bSleeping	EQU	1				; Sleep mode
bCoilPower	EQU	2				; High power on coil
bIndexOn	EQU	3				; Index LED on
;			EQU	4
;			EQU	5
bNewTime	EQU	6				; Check the time
bTimeInv	EQU	7				; Time invalid

; RC5_flags
bRC5_WaitStart	EQU	0
bRC5_DataReady	EQU	1
bRC5_prev_inp	EQU	bRC5inp		; Has to be on the same bit position as bRC5inp
bRC5_Idle		EQU	3
bRC5_HalfBit	EQU	4
;				EQU	5
bRC5_ReSynced	EQU	6
;				EQU	7

; RS232Flags
bCmdExec		EQU	7			; Execution of a command is in progress
bTimeSet		EQU 6			; A date/time setting command was received

; Mode bits
bAutoStby		EQU	0			; Auto standby timer enable
bWakeRun		EQU	1			; Wake Run afther power up
bDeapSleep		EQU	2			; No coil drive in standby mode
bHourSign		EQU	3			; Sound at hour change
bHourSignMode	EQU	4			; Hour sign mode (One sound / as many as hour (1..12))
;				EQU	5
;				EQU	6
bReinitTime		EQU	7			; Enable reinit time from RTC

; Alarms						; Number of alarm enrties avaible
;	Bit5		0				; Bank3 not avaible: 16F628(A), 16F648A :.20=0x14
;	Bit5		1				; Bank3     avaible: 16F87    , 16F88	:.40=0x28

;**********************************************************************
; PORTA

#define		SCL			PORTA,bSCL
#define		SDA			PORTA,bSDA

; RC5_flags
#define		RC5_WaitStart	RC5_flags,bRC5_WaitStart
#define		RC5_DataReady	RC5_flags,bRC5_DataReady
#define		RC5_prev_inp	RC5_flags,bRC5inp		; Has to be on the same bit position as bRC5inp
#define		RC5_Idle		RC5_flags,bRC5_Idle
#define		RC5_HalfBit		RC5_flags,bRC5_HalfBit
;							RC5_flags,5
#define		RC5_ReSynced	RC5_flags,bRC5_ReSynced
;							RC5_flags,7

; flags
#define 	StandBy		flags,bStandBy
#define		Sleeping	flags,bSleeping
#define		CoilPower	flags,bCoilPower
#define		IndexOn		flags,bIndexOn
#define		NewTime		flags,bNewTime
#define		TimeInv		flags,bTimeInv

; Mode
#define		AutoStby	Mode,bAutoStby
#define		WakeRun		Mode,bWakeRun
#define		DeapSleep	Mode,bDeapSleep
#define		HourSign	Mode,bHourSign
#define		HourSignMode	Mode,bHourSignMode
#define		ReinitTime	Mode,bReinitTime

;******************************************************************************
;	Define macro's for a Select - Case
;******************************************************************************
select_w	macro
sel_last	set 0                     ;setup variable only
         	endm
case     	macro   val
	        xorlw   val ^ sel_last
			btfsc	STATUS,Z
sel_last set val
         	endm
;
;**********************************************************************
		org     0x0000           	; processor reset vector
		clrf	PORTA				; Clear output for I2C SCL & SDA
		clrf	PORTB				;
		movlw	(1<<bLED)|(1<<bIndexLed)|(1<<bMotorOFF);Turn off error and index led, motor 
		goto    main            	; go to beginning of program


;**********************************************************************
		org     0x0004           	; interrupt vector location

		movwf	w_temp				; context saveing
		swapf	w_temp, f
		swapf	STATUS, w
		movwf	status_temp

		clrf 	STATUS				; select Bank0, IRP=0 for interrupt stuff

;--------
INT_TMR0

		btfss	INTCON,T0IF			; Test if a TMR0 interrupt occured

		goto	INT_RB				; nope, goto RB port change interrupt
									; do the TMR0 stuff, we get here every 200uSec
		nop
		movlw	.9					; TMR0 = 9
		movwf	TMR0				;

		btfsc	Second2,0			; Chopp alarm sound output
		goto	OffSound

		movf	SoundCnt,f			; Counter == 0 ?
		btfsc	STATUS,Z
		goto	OffSound			; Yes - Turn off sound

		bsf		PORTB,bSoundOut		; Turn on sound output
		goto	CountSubSec

OffSound
		bcf		PORTB,bSoundOut		; Turn off sound output

CountSubSec
		incfsz	SubSec_L,f			; Increment 16-bit Sub Second counter
		goto	lTime_1				; stil counting
		incfsz	SubSec_H,f			;
		goto	lTime_1				; stil counting

		movlw	0x3C				; load counter SubSecond = 0x10000 - 2500 = 0xF63C
		movwf	SubSec_L			;
		movlw	0xF6				;
		movwf	SubSec_H			;

		incf	Second2,f			; Move time
		bsf		NewTime				; Sign new time for TimeCheck

		movlw	1<<bLED				; Toggle visible led if time read from RTC invalid
		btfsc	TimeInv
		xorwf	PORTB,f
		btfss	TimeInv				; If time valid
		bsf		PORTB,bLED			; Turn off visible led

		btfsc	Second2,0			; Second passed
		goto	lTime_1

		movf	SoundCnt,f			; If SoundCnt != 0
		btfsc	STATUS,Z
		goto	lChkStby

		decf	SoundCnt,f			; Decrement sound counter

lChkStby
		btfss	StandBy				; In standby?
		goto	lTime_2				; no, count down StandbyTimer
		movf	SleepTimer,f		;
		btfsc	STATUS,Z			; SleepTimer = 0 ?
		goto	lTime_1				; yes, stop countdown
		decf	SleepTimer,f		; count down sleeptimer in seconds steps
		goto	lTime_1

lTime_2
		movf	StandbyTimerL,w		; If 16 bit StandbyTimer != 0
		iorwf	StandbyTimerH,w
		btfsc	STATUS,Z
		goto	lTime_1

		decf	StandbyTimerL,f		; Decrement it
		incf	StandbyTimerL,w
		btfss	STATUS,Z
		goto	lTime_1
		decf	StandbyTimerH,f

lTime_1
		movf	Second2,w			; Check for minute change
		xorlw	.120
		btfss	STATUS,Z
		goto	lRC5_1

		movf	SwitchCnt,f			; Check relais switch counter
		btfsc	STATUS,Z
		goto	lRC5_1				; Skip if counter == 0

		decfsz	SwitchCnt,f			; Decrement timer
		goto	lRC5_1

		bcf		PORTB,bRelais		; Turn off relais

lRC5_1								; start RC5 stuff here
		btfsc	RC5_DataReady
		goto	lRC5_Exit
		btfss	RC5_Idle
		goto	lRC5_Not_Idle		;
		decfsz	RC5_Tmr,f			;
		goto	lRC5_Exit			;
		btfsc	PORTA,bRC5inp		; test input
		bcf		RC5_Idle			; input = high, cancel Idle state
		incf	RC5_Tmr,f			; continue Idle state until input = high
		goto	lRC5_Exit			;

lRC5_Not_Idle
		btfss	RC5_WaitStart		;
		goto	lRC5_ReSync			;

lRC5WaitStart
		btfsc	PORTA,bRC5inp		; test input
		goto	lRC5_Exit			; no startbit
		bcf		RC5_WaitStart		; start received
		movlw	.13					; 13 bits to receive
		movwf	RC5_BitCnt			;
		clrf 	RC5_Addr			;
		clrf	RC5_Cmd				;
		movlw	.6					;
		goto	lRC5_Reload			;

lRC5_ReSync							;
		movf	PORTA,w				;
		xorwf	RC5_flags,w			;
		andlw	1<<bRC5inp			;
		btfsc	STATUS,Z			;
		goto	lRC5_no_sync		;
		bsf		RC5_ReSynced		;
		xorwf	RC5_flags,f			; Save new input in RC5_flags
		movlw	.6					; re-sync the timer
		btfss	RC5_HalfBit			;
		movlw	.2					;
		movwf	RC5_Tmr				;

lRC5_no_sync
		btfsc	RC5_HalfBit			;
		goto	lRC5_2nd_Half		;

lRC5_1st_Half
		decfsz	RC5_Tmr,f			;
		goto	lRC5_Exit			;
		bcf		STATUS,C			;
		btfsc	PORTA,bRC5inp		;
		bsf		STATUS,C			; C = RC5inp
		rlf		RC5_Cmd,f			;
		rlf		RC5_Addr,f			;
		bsf		RC5_HalfBit			; indicate that the first half bit is received
		bcf		RC5_ReSynced		;
		movlw	.4					;
		goto	lRC5_Reload

lRC5_2nd_Half
		btfsc	RC5_ReSynced		;
		goto	lReSyncOK			;
		decfsz	RC5_Tmr,f			;
		goto	lRC5_Exit			;

lRC5_Error
		movlw	(1<<bRC5_WaitStart) | (1<<bRC5_Idle)
		movwf	RC5_flags
		movlw	.128				;
lRC5_Reload
		movwf	RC5_Tmr				;
		goto	lRC5_Exit			;

lReSyncOK							; test second bit half
		bcf		RC5_HalfBit			;
		decfsz	RC5_BitCnt,f		;
		goto	lRC5_Exit			;
		rlf		RC5_Cmd,f			; Shift left to complete Addr
		rlf		RC5_Addr,f			;
		rlf		RC5_Cmd,w			; RC5_Cmd remains unchanged
		rlf		RC5_Addr,f			; Complete address
		bcf		RC5_Cmd,7			;
		btfss	RC5_Addr,6			;
		bsf		RC5_Cmd,7			; Bit 7 of RC5_Cmd = Inv Bit6 in RC5_Addr
									; RC5_Addr and RC5_Cmd was cleared and shifted left 13+2 times, so C = 0
		btfsc	RC5_Addr,5			;
		bsf		STATUS,C			; C = ToggleBit
		rrf		RC5_Cmd,f			; ToggleBit in bit 7 of RC5_Cmd
		movlw	0x1F
		andwf	RC5_Addr,f
		bsf		RC5_DataReady		; Only lower 5 bits of RC5_Addr are valid

lRC5_Exit
		bcf		INTCON,T0IF			; clear TMR0 interrupt flag before return

;--------
INT_RB
;		btfss	INTCON,RBIF			; Test if a port change interrupt occured
;		goto	INT_UART			; nope, Int_Uart

;		bcf		INTCON,RBIF			; clear RB port change interrupt flag

;--------
INT_UART

		movf	FSR,w				; Save FSR
		movwf	fsr_temp
		bsf		STATUS,IRP			; Buffers are on Bank2

		btfss	PIR1,RCIF			; Is it a receive interrupt
		goto	INT_UART_TX

		movf	UartRxChWp,w		; Put receive char to buffer if no error
		movwf	FSR
		movf	RCSTA,w				; Check for error
		andlw	(1<<FERR) | (1<<OERR)
		btfss	STATUS,Z
		goto	UART_ERR			; If error not to read char
		movf	RCREG,w				; Read character
		movwf	INDF				; Write it to buffer
		incf	FSR,w				; Move pointer in circular way
		andlw	0x0F
		iorlw	low(UartRxChar)
		movwf	UartRxChWp
		incf	UartRxChNum,f		; Sign one more char in buffer
		goto	INT_UART_TX

UART_ERR							; Receive error, reenable receive
		movf	RCREG,w				; Clear receive interrupt flag, drop character
		bcf		RCSTA,CREN			; Turn off receiver
		nop
		bsf		RCSTA,CREN			; Turn on  receiver

INT_UART_TX							; Is it a transmit interrupt
		btfss	PIR1,TXIF
		goto	INT_UART_EX

		movf	UartTxChNum,f		; Get next char from buffer if there is one
		btfsc	STATUS,Z
		goto	INT_TX_NOCHAR		; No more chars in buffer

		movf	UartTxChRp,w		; Get pointer to buffer
		movwf	FSR
		movf	INDF,w				; Get character from buffer
		movwf	TXREG				; Send it
		incf	FSR,w				; Move pointer
		andlw	0x1F				; in circular way
		iorlw	low(UartTxChar)
		movwf	UartTxChRp
		decfsz	UartTxChNum,f		; Sign one char read from buffer
		goto	INT_UART_EX			; Skip disabling transmit interrupt if more chars in buffer

INT_TX_NOCHAR						; No more chars to transmit
		bsf 	STATUS,RP0
		bcf		PIE1,TXIE			; Disable transmit int
		bcf 	STATUS,RP0

INT_UART_EX
		movf	fsr_temp,w			; Restore FSR
		movwf	FSR

;--------
INT_EXIT
		swapf	status_temp,w		; Restore context for main program
		movwf	STATUS				; This will also restore bank selection and IRP
		swapf	w_temp,w
		retfie                  	; Return from interrupt

;******************************************************************************
;	Check correct progress of sec, min, hour, day, day of week, month, year

TimeCheck
		movlw	Second2
		movwf	FSR
									; FSR -> Second2
		movlw	.120				; Check for second overflow
		call	CheckAndIncNext
									; FSR -> Minute
		movlw	.60					; Check for minute overflow
		call	CheckAndIncNext
									; FSR -> Hour
		movlw	.24					; Check for hour overflow
		call	CheckAndIncNext
									; FSR -> Day
		btfsc	STATUS,C        	; if Day incremented increment WDay as well
		incf	WDay,f

		movlw	.7					; if Day of week == 7
		subwf	WDay,w
		btfsc	STATUS,C        	; if Day of week < 7 then c=0 (borrow)
		movwf	WDay				; Day of week = 0

		movlw	.32					; DMon = 32
		movwf	DMon
		movf	Month,w				;
		movwf	Scratch				; For months <8 odd  months have 31 days
		btfsc	Month,3
		incf	Scratch,f			; For months >7 even months have 31 days
		btfss	Scratch,0
		decf	DMon,f
		xorlw	.2					; Month = February ?
		btfss	STATUS,Z			;
		goto	lNotFeb2			; continue
		decf	DMon,f				; February has max 29 days
		movf	Year,w
		andlw	0x03
		btfss	STATUS,Z
		decf	DMon,f				; Year is not a LeapYear, dec DMon for Feb

lNotFeb2
		movf	DMon,w
		call	CheckAndIncNext		; Check for day overflow
		btfsc	STATUS,C        	; if day cleared, correct day
		incf	Day,f
									; FSR -> Month
		movlw	.13					; Check for month overflow
		call	CheckAndIncNext
		btfsc	STATUS,C 	    	; if month cleared, correct month
		incf	Month,f
									; FSR -> Year
		movlw	.100				;
		subwf	Year,w				; Year < 100 ?
		btfsc	STATUS,C			;
		clrf	Year				;

		bcf		NewTime				; Clear new time flag
		btfsc	Rs232Flags,bTimeSet	; If time was set by uart command
		call	StoreTime			; Write new time and date to RTC
		bcf		Rs232Flags,bTimeSet	; Clear time set flag
		return

CheckAndIncNext
		subwf	INDF,w				; test INDF value to w
		btfsc	STATUS,C    	    ; if INDF < w then c=0 (borrow)
		movwf	INDF				; Reset value
		incf	FSR,f				; Move to next unit
		btfsc	STATUS,C    	    ; If reseted, increment next unit
		incf	INDF,f
		return						; C = 1, if carry was to next unit

;******************************************************************************
;	Check for Alarm conditions

AlarmCheck
		clrf	Scratch				; Get mask of daw of week
		incf	WDay,w				; Wday code is 0..6
		movwf	Scratch2
		bsf		STATUS,C			; Set C
Alarm_Wday
		rlf		Scratch,f			; Clears C
		decfsz	Scratch2,f
		goto	Alarm_Wday

Alarm_loop_init						; Scratch has now a 1 on bit number Wday
		movlw	Alarm0WDay			; Pointer to alarm table
		movwf	FSR

Alarm_loop
		btfss	INDF,7				; Check alarm enable flag
		goto	Alarm_dis			; Disabled

		movf	INDF,w				; Get alarm day of week flags
		andwf	Scratch,w			; Mask with actual day of week mask
		btfsc	STATUS,Z
		goto	Alarm_dis			; Not enabled for actual day of week

		incf	FSR,f				; Move pointer to alarm hour
		movf	Hour,w				; Get actual hour
		xorwf	INDF,w				; Compare with alarm hour
		movwf	Scratch2			; Store result

		btfsc	INDF,5				; If alarm hour >= 32, hour masked out from comparation
		clrf	Scratch2

		incf	FSR,f				; Move pointer to alarm minute
		movf	Minute,w			; Get actual minute
		xorwf	INDF,w				; Compare with alarm minute
		iorwf	Scratch2,w			; Combine with hour comp. result
		btfss	STATUS,Z
		goto	Alarm_dis			; Not Z, no alarm

		incf	FSR,f				; Move pointer to alarm minute
		btfsc	INDF,7				; Test for alarm mode
		goto	AlarmSwitch			; Goto if switch

		btfsc	INDF,6				; Test for propeller start
		call	NotStandbyPb		; Start up propeller
		btfsc	INDF,5				; Test for propeller stop
		call	ToStandby			; Stop propeller
		btfss	INDF,4				; Test for sound enabled
		goto	Alarm_dis

		movlw	.8					; Init sound counter
		movwf	SoundCnt
		goto	Alarm_dis

AlarmSwitch
		movf	INDF,w				; Get turn on time (minutes)
		andlw	0x7F
		movwf	SwitchCnt			; Set up counter
		btfss	STATUS,Z
		bsf		PORTB,bRelais		; Turn on  relais
		btfsc	STATUS,Z
		bcf		PORTB,bRelais		; Turn off relais

Alarm_dis							; Move to next alarm setting
		bsf		FSR,0
		bsf		FSR,1
		incf	FSR,f
		movlw	0xF0				; Check for end of table
		xorwf	FSR,w
		btfss	STATUS,Z
		goto	Alarm_loop			; Loop for all alarm entries

		btfss	Alarms,5			; Check for controllers with mode RAM
		goto	Alarm_loop_ex

		btfsc	STATUS,IRP			; Bank3 was used ?
		goto	Alarm_loop_ex		; End of alarm table
		bsf		STATUS,IRP			; Use Bank3
		goto	Alarm_loop_init
Alarm_loop_ex
		bcf		STATUS,IRP			; IRP = 0
		return

;******************************************************************************
;	Process RC5 commands

ProcessRC5
		movlw	'A'					; RC5 packet info will be sent
		call	UART_SEND
		movf	RC5_Addr,w			; Send address
		call	UART_SEND_BYTE
		movf	RC5_Cmd,w			; Send command
		call	UART_SEND_BYTE_CR

	ifdef	RC5AddrCheck
		movf	RC5_Addr,w			;
		xorlw	RemoteAddr			; test if RC5_Addr = RemoteAddr
		btfss	STATUS,Z			;
		goto	ProcessRC5Done		;
	endif

		movf	RC5_Cmd,w			; Get command code
		andlw	0x7F				; Mask toggle bit

		select_w
		case	WIDTH_DN			;
		  goto	DecWidth			; Decrement pulse width frequency
		case	WIDTH_UP			;
		  goto	IncWidth			; Increment pulse width frequency
		case	PER_UP				;
		  goto	DecFreq				; Decrement PWM frequency
		case	PER_DN				;
		  goto	IncFreq				; Increment PWM frequency
		case	POWUPCOIL			;
		  goto	PowerUpCoil			; Power up coil
		case	POWDNCOIL			;
		  goto	PowerDownCoil		; Power down coil

									; Toggle functions
		movf	RC5_Cmd,w			; Get command code
		xorwf	RC5_Cmd2,f			; Compare with previous one
		btfsc	STATUS,Z			;
		goto	ProcessRC5Done		; Exit if the same code

		movf	RC5_Cmd,w			; Get command code
		andlw	0x7F				; Mask toggle bit

		select_w

		case	TINDEX
		  goto	ToggleIndex	 		; Toggle index LED
		case	ALM_SND
		  clrf	SoundCnt			; Alarm sound off
		case	REL_ON	 			;
		  goto	RelaisOn			; Turn relais on
		case	REL_OFF	 			;
		  goto	RelaisOff			; Turn relais off
		case	STANDBY				;
		  goto	ToggleStndBy		; StandBy
;		default
		call	ReloadASTimer		; Reload timer if valid command sent to clock

		goto	ProcessRC5Done		;

RelaisOn
		bsf		PORTB,bRelais
		movlw	0xFF				; Turn on relais output
		goto	LoadSwitchCnt

RelaisOff							; Turn off relais output
		bcf		PORTB,bRelais
LoadSwitchCnt
		movwf	SwitchCnt
		goto	ProcessRC5Done		;

ToggleIndex							; Toggle index led
		movlw	1 << bIndexLed		;
		xorwf	PORTB,f				;
		movlw	1 << bIndexOn		; Toggle in flags too
		xorwf	flags,f
		goto	ProcessRC5Done		;

IncFreq								; Increment PWM frequency - decrement PR2
		bsf		STATUS,RP0			; Bank1
		decf	PR2,f				;
		goto	SetFreq

DecFreq								; Decrement PWM frequency - increment PR2
		bsf		STATUS,RP0			; Bank1
		incf	PR2,f				;

SetFreq								; Store frequency data in RTC
		movlw	RTC_PWMData+1		; Use run addresses
		btfsc	CoilPower			; Coil powered up?
		addlw	0x02				; No - Use sleep addresses
		call	SetWordAdr
		movlw	PR2_Run				; Update in memory	use run variable
		btfsc	CoilPower			; Coil powered up?
		movlw	PR2_Sleep			; No - Use sleep variable
		movwf	FSR
		movf	PR2,w				; Get frequency
		bcf		STATUS,RP0			; Bank0
		goto	IndfToRTC			; Write to INDF and send on I2C

IncWidth							; Increment pulse width
		decf	CCPR1L,f
		goto	SetWidth

DecWidth							; Decrement pulse width
		incf	CCPR1L,f
SetWidth
		movlw	RTC_PWMData			; Use run address
		btfsc	CoilPower			; Coil powered up?
		addlw	0x02				; No - Use sleep addresses
		call	SetWordAdr
		movlw	CCP1_Run			; Update in memory	use run variable
		btfsc	CoilPower			; Coil powered up?
		movlw	CCP1_Sleep			; No - Use sleep variable
		movwf	FSR
		movf	CCPR1L,w			; Get pulse width

IndfToRTC							; Write to INDF and send on I2C
		movwf	INDF
		call	I2CByteWrite
		goto	ProcessRC5Done		;

PowerUpCoil							; High power drive to trafo coil
		call	InitPWMHighPower	;
		goto	ProcessRC5Done		;

PowerDownCoil						; Low power drive to trafo coil
		call	InitPWMLowPower		;
		goto	ProcessRC5Done		;

ToggleStndBy						; Toggle mode
		btfsc	StandBy				;
		goto	NotStandby			;

		call	ToStandby
		goto	ProcessRC5Done		;

NotStandby							; high power drive to trafo coil
		call	NotStandbyPb
;		goto	ProcessRC5Done		;

ProcessRC5Done
		movf	RC5_Cmd,w			;
		movwf	RC5_Cmd2			; Store new command code
		bcf		RC5_DataReady		;
		return						;

;******************************************************************************
;	Init variables from RTC

InitData
		call	InitTime			; Get time from RTC
AlmInit
		movlw	Alarm0WDay			; Pointer to Alarm table
		movwf	FSR
AlmClear							; Clear all Alarm
		clrf	INDF
		incf	FSR,f
		movlw	0xF0
		xorwf	FSR,w
		btfss	STATUS,Z
		goto	AlmClear

		btfss	Alarms,5			; Check for controllers for more RAM
		goto	AlmInitEx
		btfsc	STATUS,IRP			; Bank3 was used ?
		goto	AlmInitEx			; End of alarm table
		bsf		STATUS,IRP			; Use Bank3
		goto	AlmInit
AlmInitEx
		bcf		STATUS,IRP

		btfsc	TimeInv				; If time read from RTC is invalid
		return

		movlw	RTC_AlarmData		; Address of alarm data in RTC
		call	SetWordAdr

AlmLdInit
		movlw	Alarm0WDay			; Read Alarm table
		movwf	FSR					; Point to Alarm table
AlmLd
		call	I2CByteRead			; Read alarm data from RTC
		movwf	INDF
		incf	FSR,f
		movlw	0xF0
		xorwf	FSR,w
		btfss	STATUS,Z			; Loop for all alarm records
		goto	AlmLd

		btfss	Alarms,5			; Check for controllers wiht more RAM
		goto	Alarm_loop_ex		; End of alarm table , set IRP = 0
		btfsc	STATUS,IRP			; Bank3 was used ?
		goto	Alarm_loop_ex		; End of alarm table , set IRP = 0
		bsf		STATUS,IRP			; Use Bank3
		goto	AlmLdInit

;******************************************************************************
;	Init PWM

InitPWM
		movlw	b'00000101'			;
		movwf	T2CON				; Timer2=ON, Prescaler = 1/4
		movlw	0x0C				; Set CCP in PWM mode
		movwf	CCP1CON				;

; Low power

InitPWMLowPower
		movf	PR2_Sleep,w			; set frequency for sleep mode
		bsf		STATUS,RP0			; Bank1
		movwf	PR2					;
		btfsc	DeapSleep			; If deap sleep enabled
		bsf		TRISB,bTrafo		; Disable trafo FET
		bcf		STATUS,RP0			; Bank0
		movf	CCP1_Sleep,w		; Set dutycycle for sleep mode
		movwf	CCPR1L				;
		bcf		CoilPower
		return

;--------
; Wake up from sleep / standby

NotStandbyPb
		bcf		PORTB,bIndexLed		; turn on index led
		bsf		IndexOn
		bcf		StandBy				; spin motor
		bcf		Sleeping			; stop sleeping
		bcf		PORTB,bMotorOFF		; startup MIC 2940A

; High power

InitPWMHighPower
		movf	PR2_Run,w			; High power to trafo
		bsf 	STATUS,RP0			; Bank1
		bcf		TRISB,bTrafo		; Enable trafo FET
		movwf	PR2					; PR2 set for high power
		bcf 	STATUS,RP0			; Bank0
		movf	CCP1_Run,w			;
		movwf	CCPR1L				; CCPR1L = Run
		bsf		CoilPower

; Reload Auto standby timer

ReloadASTimer
		movf	AutoStbyTimH,w		; Count is in seconds
		movwf	StandbyTimerH
		movf	AutoStbyTimL,w
		movwf	StandbyTimerL
		return

;--------
; Go to standby

ToStandby							; Go to standby
		movlw	.5					;
		movwf	SleepTimer			; SleepTimer = 5 seconds

		bsf		PORTB,bIndexLed		; turn off index led
		bcf		IndexOn
		bsf		StandBy				; motor in standby
		bsf		PORTB,bMotorOFF		; shutdown MIC 2940A

ClearASTimer
		clrf	StandbyTimerH
		clrf	StandbyTimerL
		return

;**********************************************************************
; Initialization

main
InitIO
;;;		clrf	PORTA				; Clear output for I2C SCL & SDA
;;;		clrf	PORTB				;
;;;		movlw	(1<<bLED)|(1<<bIndexLed)|(1<<bMotorOFF); Done at reset 

		movwf	PORTB
		bsf		STATUS,RP0
		movlw	0xFF				; Set PORTA 0..7 as input
		movwf	TRISA				; I2C routines turn on bSCL and bSDA output if needed
		movlw	0xFF ^ ( (1<<bLED) | (1<<bRelais) | (1<<bSoundOut) | (1<<bMotorOFF) | (1<<bIndexLed) )
 		movwf	TRISB				; Disable trafo drive

		clrf	PIE1				; dissable all possible interrupt flag
		movlw	b'10000001'			; set up timer0, prescaler(bit3)bypassed
		movwf	OPTION_REG			; send w to option.

		bsf		STATUS,RP1			; Bank3
		movlw	.20
		movwf	Rs232Temp			; Detect controller type
		addwf	Rs232Temp,w			; 16F87, 16F88 will read .20, others 0

		clrf	STATUS				; Bank0, IRP=0
		movwf	Alarms				; Alarms,5 == 1 if Bank3 usable

		movlw	0x9B				; Address of ANSEL
		movwf	FSR

		btfsc	Alarms,5			; On 16F87, 16F88
		clrf	INDF				; Use all PORTA pins as digital I/O, A/D is off afther reset

		movlw	0x07				;
		btfss	Alarms,5			; On 16F628(A), 16F648A
		movwf	0x1F				; CMCON - Turn comparators OFF, use all PORTA pins as digital I/O

InitSubSec
		movlw	0x3C				; load counter SubSecond = 0x10000 - .2500 = 0xF63C
		movwf	SubSec_L			;
		movlw	0xF6				;
		movwf	SubSec_H			;

		clrf	SoundCnt			; Clear down counter for alarm sound
		clrf	SwitchCnt			; Clear down counter for relais controll

		clrf	Rs232Flags			; Clear uart's command decode flags
		clrf	RC5_flags			; clear flags
		clrf	RC5_Tmr				;
		bsf		RC5_WaitStart		; RC5 is waiting for startbit

		movlw	(1<<bStandBy)|(1<<bSleeping)
		movwf	flags				; Init flags: Sleeping

		call 	InitData			; Init data from RTC

		call	UART_INIT			; Init uart and it's buffers

		call	InitPWM				; Init the low power PWM output to drive primary coil

		clrf	SleepTimer			; SleepTimer = 0 seconds
		call	ClearASTimer		; Clear auto standby timer

		btfsc	WakeRun				; Wake up in run mode
		call	NotStandbyPb		; Start in Active mode

		clrf	TMR0				; Restart timer

SendMessage
		movlw	(1<<GIE) | (1<<PEIE) | (1<<T0IE)
		movwf	INTCON 				; Enable global, extended peripheral and TMR0 interrupt

		call	SEND_MESSAGE		; Send message

;================================================================================
;	Main loop

MainLoop
		btfsc	RC5_DataReady		; Decode Command received from RC5 receiver
		call	ProcessRC5			;

		call	UARTCmdDec			; Decode Command received from UART

		btfss	NewTime				; test new time flag
		goto	TestButton

 		call	TimeCheck			; half second past, do TimeCheck

		movf	Second2,f
		btfss	STATUS,Z			; Minute passed
		goto	TestButton

		call	AlarmCheck			; Check for alarm condition

ChkHourSnd
		movf	Minute,w
		btfss	STATUS,Z			; At hour change
		goto	TestButton

		btfss	HourSign			; If hour sound enabled
		goto	CheckMidnight

		movf	HourSndBeg,w		; If afther begin hour
		subwf	Hour,w
		btfss	STATUS,C
		goto	CheckMidnight

		incf	Hour,w				; If before end hour
		subwf	HourSndEnd,w
		btfss	STATUS,C
		goto	CheckMidnight

		movf	SoundCnt,f			; If no alarm pending
		btfss	STATUS,Z
		goto	CheckMidnight

		movlw	1					; Set only one sound
		btfss	HourSignMode		; If hour mode
		goto	EnHourSound

		movlw	.12					; If hour >=12
		subwf	Hour,w				; W = Hour - 12
		btfss	STATUS,C
		movf	Hour,w
		btfsc	STATUS,Z			; If W ==0
		addlw	.12					; W = W+12

EnHourSound
		movwf	SoundCnt			; Set sound count

CheckMidnight
		movf	Hour,w				; At midnight
		btfss	STATUS,Z
		goto	TestButton

		btfsc	ReinitTime			; If enabled
		call	InitTime			; Reinit time from RTC

TestButton
		btfss	PORTA,bButton		; Button pressed
		call	NotStandbyPb

		btfsc	Sleeping			; yes, in standby: already sleeping?
		goto	MainLoop			; yes, sleeping, continue main loop

		btfsc	StandBy				; in standby?
		goto	CheckToSleep		; no, continue main loop

		movf	StandbyTimerH,w		; Check for StandbyTimer(16 bit)==0
		iorwf	StandbyTimerL,w
		btfss	STATUS,Z
		goto	MainLoop

		btfsc	AutoStby			; If AutoStandby enabled
		call	ToStandby			; Go to standby
		goto	MainLoop

CheckToSleep
		movf	SleepTimer,w		;
		btfss	STATUS,Z			; SleepTimer == 0
		goto	MainLoop			; no, continue main loop

		call	InitPWMLowPower		; Low power drive to trafo coil

		bsf		Sleeping			; Sleeping
		goto	MainLoop			;

;******************************************************************************
;	UART functions
;******************************************************************************

;******************************************************************************
; Command decoding

UARTCmdDec
		movlw	high(CmdParNum)
		movwf	PCLATH

		btfsc	Rs232Flags,bCmdExec	; If command execution in progress
		goto	GetParam			; Get parameters of command

GetCommand
		call	UART_RECEIVE		; Get the command character
		btfsc	STATUS,C			; Was there avaible character
		return						; Return if not
		addlw	-'a'				; Check the range 'a'...'z'
		btfss	STATUS,C
		return
		movwf	Rs232Cmd			; Save code of command
		addlw	-('z'-0x60)
		btfsc	STATUS,C
		return
		movf	Rs232Cmd,w			; Get code of command
		call	CmdParNum			; Get number of parameters
		movwf	Rs232ParNum
		bsf		Rs232Flags,bCmdExec	; Sign command execution in progress

GetParam
		movf	Rs232ParNum,w		; Are there enought parameters in buffer
		subwf	UartRxChNum,w
		btfss	STATUS,C
		return						; Return if not

CmdParOk
		movf	Rs232Cmd,w			; Get command code
		bcf		Rs232Flags,bCmdExec	; No command execution in progress

		goto	CmdExec				; Execute it and return to caller at the end of execution

;**********************************************************************
; Init uart

UART_INIT:
		clrf	UartRxChNum			; Init receive fifo
		movlw	low(UartRxChar)
		movwf	UartRxChWp
		movwf	UartRxChRp

		clrf	UartTxChNum			; Init transmit fifo
		movlw	low(UartTxChar)
		movwf	UartTxChWp
		movwf	UartTxChRp

		bsf 	STATUS,RP0			; Bank1

	if	BaudRate<.9600				; Set up Baud rate
		bcf		TXSTA,BRGH
		movlw	(F_OSC/.64/BaudRate)-1
	else
		bsf		TXSTA,BRGH
		movlw	(F_OSC/.16/BaudRate)-1
	endif

		movwf	SPBRG
		bsf		TXSTA,TXEN			; Enable transmission

		bsf		PIE1,RCIE			; Enable  receive interrupt
		bcf		PIE1,TXIE			; Disable transmit interrupt

		bcf 	STATUS,RP0			; Bank0

		bsf		RCSTA,SPEN			; Enable receive
		bsf		RCSTA,CREN

		return

;**********************************************************************
; Send a byte in BCD format with uart as two ASCII character
; Keeps IRP, RP1, RP0 bits of STATUS
; Keeps FSR

UART_SEND_BCD
		call	Conv2BCD

;**********************************************************************
; Send a byte with uart as two hex. ASCII character
; Keeps IRP, RP1, RP0 bits of STATUS
; Keeps FSR

UART_SEND_BYTE
		movwf	Scratch				; Save data to send
		swapf	Scratch,w			; Send high nibble first
		call	UART_SEND_NIBBLE
		movf	Scratch,w			; Send low  nibble

;**********************************************************************
; Send a nibble (lower 4 bit of W) with uart as a hex. ASCII character
; Keeps IRP, RP1, RP0 bits of STATUS
; Keeps FSR

UART_SEND_NIBBLE
		andlw	0x0F				; Keep only low nibble
		movwf	Rs232Temp2
		sublw	0x09
		clrw
		btfss	STATUS,C			; If nibble > 9 add 7, and use lower case character
		movlw	0x27
		addwf	Rs232Temp2,w
		addlw	0x30				; Convert to ACSII

;**********************************************************************
; Send a character (w), write it to uart send buffer
; Keeps IRP, RP1, RP0, DC, C bits of STATUS
; Keeps FSR

UART_SEND							; Sends char in w

		btfsc	UartTxChNum,5		; Test number of chars in buff
		goto	UART_SEND

		movwf	Rs232Temp			; Save character to send
		swapf	STATUS,w
		movwf	Rs232Status			; Save Status
		movf	FSR,w				; Save FSR
		movwf	Rs232Fsr

		bsf 	STATUS,RP0			; Bank1
		bcf		PIE1,TXIE			; Disable TX interrupt
		bcf 	STATUS,RP0			; Bank0

		bsf		STATUS,IRP			; Buffer in on Bank2

		movf	UartTxChWp,w		; Get write pointer
		movwf	FSR

		movf	Rs232Temp,w			; Get char to send
		movwf	INDF				; Store it in out buffer
		incf	FSR,w				; Move the pointer
		andlw	0x1F				; in circular way
		iorlw	low(UartTxChar)
		movwf	UartTxChWp
		incf	UartTxChNum,f		; Increment number of avaible characters

		bsf 	STATUS,RP0			; Bank1
		bsf		PIE1,TXIE			; Enable TX interrupt

UART_Exit
		bcf 	STATUS,RP0			; Bank0
		movf	Rs232Fsr,w			; Restore FSR
		movwf	FSR
		swapf	Rs232Status,w		; Restore STATUS
		movwf	STATUS				; Restore IRP, Bank selection
		movf	Rs232Temp,w			; Get char sent / just received
		return

;**********************************************************************
; Get a byte from receive buffer and convert it form BCD to bin
; Keeps IRP, RP1, RP0 bits of STATUS
; Keeps FSR

UART_REC_BCD
		call	UART_REC_BYTE
		goto	BcdToBin

;**********************************************************************
; Get a byte from receive buffer
; Keeps IRP, RP1, RP0 bits of STATUS
; Keeps FSR

UART_REC_BYTE
		clrf	Rs232Temp2			; Clear data to be read
		call	UART_REC_NIBBLE		; Receive high nibble
		movwf	Rs232Temp2
		swapf	Rs232Temp2,f		; Swap to high nibble

UART_REC_NIBBLE
		call	UART_RECEIVE
		btfsc	Rs232Temp,6			; Check for 'A'..'F' or 'a'..'f'
		addlw	-7
		addlw	-0x30
		andlw	0x0F				; Keep low nibble
		addwf	Rs232Temp2,w
		return

;**********************************************************************
; Get a character from receive buffer
; Keeps IRP, RP1, RP0, DC, C bits of STATUS
; Keeps FSR

UART_RECEIVE						; Gets char into w, C == 1 if no char was in fifo

		bsf		STATUS,C			; Set no character received
		swapf	STATUS,w			; Save STATUS
		movwf	Rs232Status
		movf	FSR,w				; Save FSR
		movwf	Rs232Fsr

		bsf		STATUS,IRP			; Buffer in on Bank2

		bsf 	STATUS,RP0			; Bank1
		bcf		PIE1,RCIE			; Disable RC interrupt
		bcf 	STATUS,RP0			; Bank0

		clrf	Rs232Temp
		movf	UartRxChNum,w		; Check number of avaible chars
		btfsc	STATUS,Z
		goto	NO_REC_CHAR			; Jump if no character avaible

		movf	UartRxChRp,w		; Get read pointer
		movwf	FSR

		movf	INDF,w				; Read it from buffer
		movwf	Rs232Temp
		incf	FSR,w				; Move the pointer
		andlw	0x0F				; in circular way
		iorlw	low(UartRxChar)
		movwf	UartRxChRp
		decf	UartRxChNum,f		; Decrement number of avaible characters
		bcf		Rs232Status,C+4		; Valid character received - !Status was swapped!

NO_REC_CHAR
		bsf 	STATUS,RP0			; Bank1
		bsf		PIE1,RCIE			; Enable RC interrupt

		goto	UART_Exit			; Restore STATUS and FSR

;**********************************************************************
; Send init message

SEND_MESSAGE
		movlw	high(InitMsg)		; Init string pointer high part
		movwf	Rs232StrPtrH
		movlw	low(InitMsg)		; Init string pointer low  part

;**********************************************************************
; Send a string (high part of address must be in Rs232StrPtrH)

SEND_STRING
		movwf	Rs232StrPtr			; Save string pointer low  part
lString0
		call	MsgTabl				; Get next character
		movwf	Rs232Temp2			; Save it
		andlw	0x7F				; Mask bit7 off
		call	UART_SEND			; Send character
		btfss	Rs232Temp2,7		; Check for end of string mark
		goto	lString0			; Loop for all characters
		return

;******************************************************************************
;	I2C RTC functions
;******************************************************************************

;******************************************************************************
; Read the time from RTC

InitTime							; Init time from I2C RTC

		movlw	RTC_Second			; Address of Seconds
		call	SetWordAdr

		call	I2CByteRead			; Read seconds
		call	BcdToBin
		movwf	Second2
		addwf	Second2,f
									; I2CWordAdr = 0x03
		call	I2CByteRead			; Read minute
		call	BcdToBin
		movwf	Minute
									; I2CWordAdr = 0x04
		call	I2CByteRead			; Read hour
		call	BcdToBin3F			; Mask format and AM/PM
		movwf	Hour
									; I2CWordAdr = 0x05
		call	I2CByteRead			; Read day
		call	BcdToBin3F			; Mask Year bits
		movwf	Day
		swapf	Scratch,f
		rrf		Scratch,f
		rrf		Scratch,w
		movwf	Scratch2			; Save year bit 1..0
									; I2CWordAdr = 0x06
		call	I2CByteRead			; Read Month
		andlw	0x1F
		call	BcdToBin
		movwf	Month
		swapf	Scratch,f
		rrf		Scratch,w
		andlw	0x07				; Remove swapped Month
		movwf	WDay
									; I2CWordAdr = 0x07

		movlw	RTC_Year			; Address of Year
		call	I2CByteReadStoreAddr; Read year
									; I2CWordAdr = 0x11
		movwf	Year
		call	I2CByteRead			; Read year ^ 0xff
									; I2CWordAdr = 0x12
		xorlw	0xFF				; Complement read data
		xorwf	Year,w				; Has to be same as Year
		btfss	STATUS,Z
		goto	InitTimeDef			; Year not valid, init with default values

		call	I2CByteRead			; CCP1_Run
		movwf	CCP1_Run
									; I2CWordAdr = 0x13
		call	I2CByteRead			; PR2_Run
		movwf	PR2_Run
									; I2CWordAdr = 0x14
		call	I2CByteRead			; CCP1_Sleep
		movwf	CCP1_Sleep
									; I2CWordAdr = 0x15
		call	I2CByteRead			; PR2_Sleep
		movwf	PR2_Sleep
									; I2CWordAdr = 0x16
		call	I2CByteRead			; Auto Standby Time high
		movwf	AutoStbyTimH		;
									; I2CWordAdr = 0x17
		call	I2CByteRead			; Auto Standby Time  low
		movwf	AutoStbyTimL		;
									; I2CWordAdr = 0x18

		call	I2CByteRead			; Hour sound begin
		movwf	HourSndBeg			;
									; I2CWordAdr = 0x19

		call	I2CByteRead			; Hour sound end
		movwf	HourSndEnd			;
									; I2CWordAdr = 0x1A

		movlw	RTC_Mode
		call	I2CByteReadStoreAddr; Mode
		movwf	Mode				;
									; I2CWordAdr = 0x20

		movf	Year,w				; Has the RTC incremented Year
		xorwf	Scratch2,w
		andlw	0x03
		btfsc	STATUS,Z
		return						; No, return

		incf	Year,f				; Yes, increment Year
StoreYear
		movlw	RTC_Year			; Address of Year
		movwf	I2CWordAdr
		movf	Year,w				; Write Year
		call	I2CByteWrite
									; I2CWordAdr = 0x11
		comf	Year,w
		goto	I2CByteWrite		; Write complement of Year

;******************************************************************************
; Init time with default value: 2001-01-01 12:00:00, day of week: 1 ,set time invalid
; Init CCPR1L, RP2 for Run and Sleep mode with default values
; Init Auto Standby Timer with default value
; Init mode with default value
; Write default data to RTC

InitTimeDef
;;		clrf	Hour 				; May start from 00:00:00
		movlw	.12					; why do clocks always start
		movwf	Hour				; at 12:00 ?
;;
		clrf	Minute
        clrf	Second2
		movlw	.1
		movwf	Day
		clrf	WDay
		movwf	Month
		movwf	Year
		bsf		TimeInv				; Sign time not valid

									; Write default data to RTC
		movlw	RTC_PWMData			; Set word address
		call	SetWordAdr			; Set RTC's slave address

									; PWM parameters
		movlw	CCPR1LRUN			; Set pulse width for run mode
		movwf	CCP1_Run			;
		call	I2CByteWrite		;
		movlw	PR2RUN				; Set frequency for run mode
		movwf	PR2_Run				;
		call	I2CByteWrite
		movlw	CCPR1LSleep			; Set pulse width for sleep mode
		movwf	CCP1_Sleep
		call	I2CByteWrite
		movlw	PR2Sleep			; set frequency for sleep mode
		movwf	PR2_Sleep			;
		call	I2CByteWrite

		movlw	high(.3600)			; Auto Standby time
		movwf	AutoStbyTimH		;
		call	I2CByteWrite
		movlw	low(.3600)			; Auto Standby Timer
		movwf	AutoStbyTimL		;
		call	I2CByteWrite

		movlw	.8
		movwf	HourSndBeg			; Hour sound begin hour
		call	I2CByteWrite

		movlw	.20
		movwf	HourSndEnd			; Hour sound end   hour
		call	I2CByteWrite

		movlw	RTC_Mode			; Mode
		movwf	I2CWordAdr

ModeCode	set	0x00
	ifdef	AutoStandby
ModeCode	set	ModeCode | (1<<bAutoStby)
	endif
	ifdef	WakeUpRun
ModeCode	set	ModeCode | (1<<bWakeRun)
	endif
	ifdef	DeapSleeping
ModeCode	set ModeCode | (1<<bDeapSleep)
	endif
	ifdef	EnableHourSign
ModeCode	set	ModeCode | (1<<bHourSign)
	endif
	ifdef	SignMode
ModeCode	set	ModeCode | (1<<bHourSignMode)
	endif
	ifdef	TimeReInit
ModeCode	set ModeCode | (1<<bReinitTime)
	endif

		movlw	ModeCode			; Init Mode
		movwf	Mode
		goto	I2CByteWrite

;******************************************************************************
; Convert binary number to BCD

Conv2BCD							; Convert w to BCD (0 <= w <= 99)
		clrf	BCD					; Clear high nibble
l_bcd_1
		addlw	-.10				; Sub .10
		incf	BCD,f				; Inc high nibble
		btfsc	STATUS,C			; Loop if no carry
		goto	l_bcd_1
		addlw	.10					; Re add .10, C=1
		decf	BCD,f				; Correct high nibble
		swapf	BCD,f				; Swap to high part of byte
		addwf	BCD,w				; Add to low nibble, C=0
		return

;******************************************************************************
; Convert BCD number to binary - mask bit 7..6 off

BcdToBin3F							; Convert BCD to binary for Hour and Day
		andlw	0x3F

; Convert BCD number to binary

BcdToBin							; Convert BCD to binary
		movwf	BCD
		andlw	0x0F
		btfsc	BCD,4
		addlw	.10
		btfsc	BCD,5
		addlw	.20
		btfsc	BCD,6
		addlw	.40
		btfsc	BCD,7
		addlw	.80
		return

;******************************************************************************
; Writes the time to RTC

StoreTime
		call	SetSlaveAdr			; Set Slave address of RTC

		movlw	0x80				; Disable counting command
		call	SendRtcCmd
									; I2CWordAdr = 0x01
		incf	I2CWordAdr,f		; Address of seconds is 2
									; I2CWordAdr = 0x02

		clrc						; carry = 0
		rrf		Second2,w			; Store Second = (Seconds2 >> 1)
		call	I2CBcdWrite
									; I2CWordAdr = 0x03
		movf	Minute,w			; Store Minute
		call	I2CBcdWrite
									; I2CWordAdr = 0x04
		movf	Hour,w				; Store Hour
		call	I2CBcdWrite
									; I2CWordAdr = 0x05
		swapf	Year,w				; Store Day and Year bit 1..0
		movwf	Scratch
		rlf		Scratch,f
		rlf		Scratch,f
		movlw	0xC0
		andwf	Scratch,f
		movf	Day,w
		call	BCDiorWrite
									; I2CWordAdr = 0x06
		swapf	WDay,w				; Store Month and Wday
		movwf	Scratch
		rlf		Scratch,f
		movlw	0xE0
		andwf	Scratch,f
		movf	Month,w
		call	BCDiorWrite
									; I2CWordAdr = 0x07
		call	EnableCount
		bcf		TimeInv
		goto	StoreYear

BCDiorWrite
		call	Conv2BCD
		iorwf	Scratch,w
		goto	I2CByteWrite

; Send enable count command to RTC
EnableCount
		movlw	0x00				; Enable counting command

; Send command to RTC
SendRtcCmd
		clrf	I2CWordAdr
		goto	I2CByteWrite

;******************************************************************************
;	I2C low level functions
;******************************************************************************

; Temporary variables are in common Ram - Can be used from Bank of TRISA too
; START selects BANK1 to access TRISA   - STOP select BANK0 before return

; At reset SCL and SDA bits in PORTA are cleared

;******************************************************************************
; Reads a byte from RTC to Scratch, address is in w
; store address in I2CWordAdr, I2CWordAdr incremented afther execution

I2CByteReadStoreAddr
	movwf	I2CWordAdr			; Store address

; Reads a byte from RTC to Scratch, address is in I2CWordAdr
; I2CWordAdr incremented afther execution

I2CByteRead
	CALL 	START				; Generate Start
	MOVF	I2CSlaveAdr,w		; Get slave address for write
	CALL 	OUT_BYTE			; Send slave address byte + nack
	CALL 	OUT_BYTE_ADDR		; Send word  address byte + nack

	CALL 	START				; Generate repeted start
	INCF	I2CSlaveAdr,w		; Get slave address for read
	CALL 	OUT_BYTE			; Send byte + nack
	CALL	IN_BYTE				; Get the byte to Scratch + send nack

ToStop
	CALL 	STOP				; Generate stop condition
	incf	I2CWordAdr,f		; Increment word address
	movf	Scratch,W			; Return data just read
	bcf		STATUS,RP0			; Back to Bank 0
	return

;******************************************************************************
; Writes W to RTC in BCD format, address is in I2CWordAdr
; I2CWordAdr incremented afther execution

I2CBcdWrite
	call	Conv2BCD			; Convert w to BCD

; Writes W to RTC, address is in I2CWordAdr
; I2CWordAdr incremented afther execution

I2CByteWrite
	movwf	Scratch				; Save data to be written to RTC
	CALL 	START				; Generate Start
	MOVF	I2CSlaveAdr,w		; Get slave address for write
	CALL 	OUT_BYTE			; Send slave address byte + nack
	CALL 	OUT_BYTE_ADDR		; Send word  address byte + nack
	MOVF 	Scratch, W			; Get output data
	CALL 	OUT_BYTE			; Send data byte + nack
	goto	ToStop

;******************************************************************************
; Generate stop condition on I2C bus

STOP:							; SDA 0 -> 1 while SCL == 1
								; SCL must be LOW afther (N)ACK's CLOCK_PULSE
	CALL	LOW_SDA				; SDA -> 0 - NACK has done it
	CALL	HIGH_SCL			; SCL -> 1 and make 5us stop setup time

; Make SDA high by making it input

HIGH_SDA:						; high impedance by making SDA an input
	BSF 	SDA					; make SDA pin an input
	GOTO 	DELAY_5US			; SDA -> 1 and make 5us bus free time

;******************************************************************************
; Generate start / repeated start condition on I2C bus

START:							; SDA 1 -> 0 while SCL == 1, then SCL -> 0
	BSF 	STATUS, RP0			; Bank 1 - Access TRISA
	CALL 	HIGH_SDA			; SDA 0 -> 1 ; wait 5 us - For repeated start
	CALL 	HIGH_SCL			; SCL 0 -> 1 ; make 5 us Start setup time
	CALL 	LOW_SDA				; SDA 1 -> 0 ; make 5 us Start hold  time
	GOTO 	LOW_SCL				; SCL 1 -> 0 ; wait 5 us

;******************************************************************************
; Shift in a byte from I2C bus, clock out nack, store data to Scratch

IN_BYTE							; Read byte on i2c bus
	CALL 	HIGH_SDA			; Configure SDA as input
	movlw	.8
	movwf	I2CBitCnt
;	bsf		I2CBitCnt,3			; Load 8 to I2CBitCnt, OUT_BYTE cleared I2CBitCnt
IN_BIT
	CALL 	HIGH_SCL			; SCL -> 1 ; 5us wait
	BCF		STATUS,RP0			; Bank 0 to read PORTA
	BCF		STATUS,C			; clear carry
	BTFSC 	SDA					; test SDA bit
	BSF 	STATUS,C			; set carry if SDA == 1
	BSF		STATUS,RP0			; Bank 1 to control TRISA
	RLF 	Scratch,F			; Scratch = (Scratch << 1) | input bit
	CALL 	LOW_SCL				; SCL -> 0 ; 5us wait
	DECFSZ 	I2CBitCnt,F			; decrement bit counter
	GOTO 	IN_BIT
	GOTO	NACK

;******************************************************************************
; Shift out a byte to I2C bus, clock in (n)ack, get data from I2CWordAdr

OUT_BYTE_ADDR
	MOVF	I2CWordAdr,w		; output address to be read

; Shift out a byte to I2C bus, clock in (n)ack, data is in w

OUT_BYTE:						; send o_byte on I2C bus
	movwf	I2CShiftReg			; Store data to send
	MOVLW 	.8
	MOVWF 	I2CBitCnt			; Loop for 8 bits
OUT_BIT:
	BTFSC 	I2CShiftReg,7		; if  one, send a  one
	CALL 	HIGH_SDA			; SDA at logic one
	BTFSS 	I2CShiftReg,7		; if zero, send a zero
	CALL 	LOW_SDA				; SDA at logic zero
	CALL 	CLOCK_PULSE			; SCL -> 1 ; 5 us wait, SCL -> 0 ; 5 us wait
	RLF 	I2CShiftReg,F		; left shift, move mext bit to bit7
	DECFSZ 	I2CBitCnt,F			; decrement bit counter - Leaves with I2CBitCnt = 0
	GOTO 	OUT_BIT

;******************************************************************************
; Clock in/out (n)ack

NACK:							; bring SDA high and clock, SDA must be input
	CALL 	HIGH_SDA
	CALL	CLOCK_PULSE			; SCL -> 1 ; 5 us wait, SCL -> 0 ; 5 us wait

; Make SDA low by making it output

LOW_SDA:						; SDA -> 0 ; 5 us wait
	BCF 	SDA					; make SDA pin an output
	GOTO 	DELAY_5US

;******************************************************************************
; Generate clock pulse 			; SCL -> 1 ; 5 us wait, SCL -> 0 ; 5 us wait

CLOCK_PULSE:					; SCL momentarily to logic one
	CALL 	HIGH_SCL			; SCL -> 1 ; 5 us wait

; Make SCL low by making it output

LOW_SCL:						; SCL -> 0 ; 5 us wait
	BCF 	SCL					; make SCL pin an output
	GOTO	DELAY_5US

;******************************************************************************
; Make SCL high by making it input

HIGH_SCL:						; SCL -> 1 ; wait 5 us
	BSF 	SCL					; make SCL pin an input

; Delay 5uS @ 20MHz

DELAY_5US:						; provides nominal >5 us delay @20MHz
								; 1 instuction takes 200ns
	MOVLW	 .8					; 0.2 us
	MOVWF 	BCD					; 0.2 us
DELAY_1:						; Loop of 3 inst. time
	DECFSZ 	BCD, F				; 7*0.2+0.4 us
	GOTO 	DELAY_1				; 7*0.4 us
	RETURN						; Return 0.4 us

;******************************************************************************
; Command routines
;******************************************************************************

;******************************************************************************

UART_SEND_COMMAND
		movf	Rs232Cmd,w
		addlw	'A'					; Answare code in uppercase
		goto	UART_SEND

;******************************************************************************
; Send version

SendVer
		call	UART_SEND_COMMAND
		movlw	VersionMajor		; Send major version
		call	UART_SEND_BYTE
		movlw	'.'
		call	UART_SEND
		movlw	VersionMinor		; Send minor version
UART_SEND_BYTE_CR
		call	UART_SEND_BYTE		; Send a byte and a Cr Lf
SendCrLf
		movlw	high(CrLf)			; Send a Cr Lf sequence
		movwf	Rs232StrPtrH
		movlw	low(CrLf)
		goto	SEND_STRING

;******************************************************************************
; Clear seconds

ClrSec
		clrf	Second2				; Clear seconds
		goto	NewTim

;******************************************************************************
; Set Time

SetTime
		call	UART_REC_BCD		; Get hour
		movwf	Hour
		call	UART_REC_BCD		; Get minute
		movwf	Minute
		call	UART_REC_BCD		; Get second
		movwf	Second2
		addwf	Second2,f			; Doubble it
NewTim
		bsf		Rs232Flags,bTimeSet	; Sign time has to be saved
		bsf		NewTime				; Set new time ready

;******************************************************************************
; Get Time

GetTime
		call	UART_SEND_COMMAND
		movf	Hour,w				; Send hour
		call	UART_SEND_BCD
		movf	Minute,w			; Send minute
		call	UART_SEND_BCD
		clrc						; Get second
		rrf		Second2,w
		goto	UART_SEND_BCD_CR	; Send second

;******************************************************************************
; Set Date and day of week

SetDate
		call	UART_REC_BCD		; Get year
		movwf	Year
		call	UART_REC_BCD		; Get month
		addlw	0
		btfsc	STATUS,Z			; Correct 0 to 1
		addlw	1
		movwf	Month
		call	UART_REC_BCD		; Get day
		addlw	0
		btfsc	STATUS,Z			; Correct 0 to 1
		addlw	1
		movwf	Day
		call	UART_REC_BCD		; Get day of week
		andlw	0x07				; Convert 1..7 to 0..6
		addlw	-1
		movwf	WDay
		btfsc	WDay,7
		clrf	WDay
		bsf		Rs232Flags,bTimeSet	; Sign time has to be saved
		bsf		NewTime				; Set new time ready

;******************************************************************************
; Get Date and day of week

GetDate
		call	UART_SEND_COMMAND
		movf	Year,w
		call	UART_SEND_BCD		; Send year
		movf	Month,w
		call	UART_SEND_BCD		; Send month
		movf	Day,w
		call	UART_SEND_BCD		; Send day
		incf	WDay,w				; Convert 0..6 to 1..7
UART_SEND_BCD_CR
		call	UART_SEND_BCD		; Send day of week
		goto	SendCrLf

;******************************************************************************
; Init Alarm pointers (IRP:FSR and RTC)

AlarmPtr
		call	UART_REC_BCD		; Get alarm number
		movwf	FSR
		movf	Alarms,w			; There are 40 alarm entries on 16F87, 16F88
		subwf	FSR,w				; 20 on others
		decf	Alarms,w			; If number > limit use last alarm entry
		btfsc	STATUS,C
		movwf	FSR
		movf	FSR,w
		movwf	Scratch3			; Store it
		addwf	FSR,f				; One entry has 4 locations
		rlf		FSR,f
		movf	FSR,w
		movwf	I2CWordAdr

		movlw	.4*.20				; Alarm entries >=20 are on Bank3
		subwf	FSR,w
		btfss	STATUS,C
		goto	PtrOk
		movwf	FSR					; Modify pointer
		bsf		STATUS,IRP			; Use Bank3

PtrOk
		movlw	Alarm0WDay
		addwf	FSR,f				; Init memory pointer

		movlw	RTC_AlarmData
		addwf	I2CWordAdr,w		; Init RTC address

SetWordAdr
		movwf	I2CWordAdr			; Save word address
SetSlaveAdr
		movlw	RTC_ADDR			; Init Slave address
		movwf	I2CSlaveAdr
		return

;******************************************************************************
; Set Alarm

SetAlarm
		call	AlarmPtr			; Get pointer to alarm table
		call	UART_REC_BYTE		; Get enable flags
		movwf	INDF				; Store it in memory
		call	I2CByteWrite		; Store it in RTC
		incf	FSR,f				; Move pointer
		call	UART_REC_BCD		; Get alarm hour
		movwf	INDF				; Store it in memory
		call	I2CByteWrite		; Store it in RTC
		incf	FSR,f
		call	UART_REC_BCD		; Get alarm minute
		movwf	INDF				; Store it in memory
		call	I2CByteWrite		; Store it in RTC
		incf	FSR,f
		call	UART_REC_BYTE		; Get alarm mode, duration
		movwf	INDF				; Store it in memory
		call	I2CByteWrite		; Store it in RTC

		bcf		FSR,0
		bcf		FSR,1				; Move pointer to beginning of entry
		goto	SendAlarm

;******************************************************************************
; Get Alarm

GetAlarm
		call	AlarmPtr			; Get pointer to alarm table
SendAlarm
		call	UART_SEND_COMMAND
		movf	Scratch3,w
		call	UART_SEND_BCD		; Send alarm number
		movf	INDF,w				; Get alarm enable flags
		call	UART_SEND_BYTE
		incf	FSR,f
		movf	INDF,w				; Get alarm hour
		call	UART_SEND_BCD
		incf	FSR,f
		movf	INDF,w				; Get alarm minute
		call	UART_SEND_BCD
		incf	FSR,f
		movf	INDF,w				; Get alarm mode, duration
CLR_IRP_SEND_B_CR
		bcf		STATUS,IRP			; IRP = 0
		goto	UART_SEND_BYTE_CR

;******************************************************************************
; Set PWM data

SetPWM
		movlw	RTC_PWMData			; Init RTC word address to PWM data
		call	SetWordAdr
		call	UART_REC_BYTE		; Get pulse width for run mode
		movwf	CCP1_Run
		call	I2CByteWrite		; Write to RTC
		call	UART_REC_BYTE		; Get period for run mode
		movwf	PR2_Run
		call	I2CByteWrite
		call	UART_REC_BYTE		; Get pulse width for sleep mode
		movwf	CCP1_Sleep
		call	I2CByteWrite
		call	UART_REC_BYTE		; Get period for sleep mode
		movwf	PR2_Sleep
		call	I2CByteWrite

;******************************************************************************
; Get PWM data

GetPWM
		call	UART_SEND_COMMAND
		movf	CCP1_Run,w
		call	UART_SEND_BYTE		; Send pulse width for run mode
		movf	PR2_Run,w
		call	UART_SEND_BYTE		; Send period for run mode
		movf	CCP1_Sleep,w
		call	UART_SEND_BYTE		; Send pulse width for sleep mode
		movf	PR2_Sleep,w
		call	UART_SEND_BYTE		; Send period for sleep mode
		movf	CCPR1L,w
		call	UART_SEND_BYTE		; Send current pulse width
		bsf		STATUS,RP0
		movf	PR2,w
		bcf		STATUS,RP0
		goto	UART_SEND_BYTE_CR	; Send current periode

;******************************************************************************
; Set Auto Sleep Timer

SetAST
		movlw	RTC_AST
		call	SetWordAdr
		call	UART_REC_BYTE		; Get high byte of AST
		movwf	AutoStbyTimH
		call	I2CByteWrite
		call	UART_REC_BYTE		; Get low  byte of AST
		movwf	AutoStbyTimL
		call	I2CByteWrite

;******************************************************************************
; Get Auto Sleep Timer

GetAST
		call	UART_SEND_COMMAND
		movf	AutoStbyTimH,w		; Send high byte of AST
		call	UART_SEND_BYTE
		movf	AutoStbyTimL,w
		goto	UART_SEND_BYTE_CR	; Send low  byte of AST

;******************************************************************************
; Set Mode

SetMode
		movlw	RTC_Mode
		call	SetWordAdr
		call	UART_REC_BYTE		; Get mode
		movwf	Mode
		call	I2CByteWrite

;******************************************************************************
; Get Mode

GetMode
		call	UART_SEND_COMMAND
		movf	Mode,w
		goto	UART_SEND_BYTE_CR	; Send mode

;******************************************************************************
; Set relais state

SetRel
		call	UART_REC_BYTE		; Get relais controll
		andlw	1					; Keep lsb only
		movwf	Scratch
		call	UART_REC_BYTE		; Get duration
		btfss	Scratch,0
		goto	SetRel0

		movwf	SwitchCnt
		bsf		PORTB,bRelais		; Turn on relais output
		goto	SetRel2

SetRel0
		clrf	SwitchCnt			; Clear counter
		bcf		PORTB,bRelais		; Turn off relais output
SetRel2

;******************************************************************************
; Get relais state

GetRel
		call	UART_SEND_COMMAND
		clrw
		btfsc	PORTB,bRelais		; Get relais state
		addlw	1
		call	UART_SEND_BYTE		; Send it
		movf	SwitchCnt,w
		goto	UART_SEND_BYTE_CR	; Send remaining on time

;******************************************************************************
; Get number of alarms

GetAln
		call	UART_SEND_COMMAND
		movf	Alarms,w			; There are .20 alarms for 16F628(A) and 16F648A
		call	UART_SEND_BCD		; There are .40 alarms for 1687 and 16F88
		movlw	RemoteAddr
		goto	UART_SEND_BYTE_CR	; Send RC5 address of clock

;******************************************************************************
; Set hour sound parameters

SetHSnd
		movlw	RTC_HourSnd
		call	SetWordAdr

		call	UART_REC_BCD		; Get begin hour
		movwf	HourSndBeg
		call	I2CByteWrite

		call	UART_REC_BCD		; Get end   hour
		movwf	HourSndEnd
		call	I2CByteWrite

;******************************************************************************
; Get hour sound parameters

GetHSnd
		call	UART_SEND_COMMAND
		movf	HourSndBeg,w
		call	UART_SEND_BCD		; Send begin hour
		movf	HourSndEnd,w
		goto	UART_SEND_BCD_CR	; Send end   hour

;******************************************************************************
; Get / Set propeller state

SetProp
		call	UART_REC_BYTE		; Get controll info
		movwf	Scratch
		btfsc	Scratch,7			; If bit 7 set
		goto	SendProp			;  read flags only

		btfsc	Scratch,0			; Bit 0 set
		call	NotStandbyPb		;  start propeller
		btfss	Scratch,0			; Bit 0 cleared
		call	ToStandby			;  stop  propeller

SendProp
		call	UART_SEND_COMMAND
		movf	flags,w				; Send flags
		andlw	0xBF
		bsf		STATUS,RP0			; Change Time check needed flag to deep sleep
		btfsc	TRISB,bTrafo
		iorlw	0x40
		bcf		STATUS,RP0
		call	UART_SEND_BYTE
		movf	StandbyTimerH,w		; Send high byte of remaining run time
		call	UART_SEND_BYTE
		movf	StandbyTimerL,w
		goto	UART_SEND_BYTE_CR	; Send low  byte of remaining run time


;******************************************************************************
; Execute Rc5 command

XeqRc5
		call	UART_REC_BYTE		; Get address code
		andlw	0x1F				; Only 5 bit valid
		movwf	RC5_Addr
		call	UART_REC_BYTE		; Get command code
		movwf	RC5_Cmd
		bsf		RC5_DataReady		; Set address and command valid
		return						; Answare will be send in ProcessRC5

;******************************************************************************
; Init I2C slave and word address

InitI2CAdrs
		call	UART_REC_BYTE		; Get slave address
		andlw	0xFE				; Mask R/W bit off
		movwf	I2CSlaveAdr
		call	UART_REC_BYTE		; Get word address
		movwf	I2CWordAdr
		return

;******************************************************************************
; Set reg on I2C bus

SetI2C
		call	InitI2CAdrs
		call	UART_REC_BYTE		; Get data to be written
		call	I2CByteWrite		; Write it, increments I2CWordAdr
		decf	I2CWordAdr,f		; Correct it
		goto	SendI2CReg

;******************************************************************************
; Get reg on I2C bus

GetI2C
		call	InitI2CAdrs
SendI2CReg
		call	UART_SEND_COMMAND
		movf	I2CSlaveAdr,w
		call	UART_SEND_BYTE		; Send slave address
		movf	I2CWordAdr,w
		call	UART_SEND_BYTE		; Send word address
		call	I2CByteRead			; Read the data
		goto	UART_SEND_BYTE_CR	; Send it

;******************************************************************************
; Init FSR

InitFsr
		call	UART_REC_BYTE		; Get high part of address
		andlw	1
		movwf	Scratch
		bcf		STATUS,IRP
		btfsc	Scratch,0			; If on Bank2-3
		bsf		STATUS,IRP			; IRP=1
		call	UART_REC_BYTE		; Get low  part of address
		movwf	FSR
		return

;******************************************************************************
; Set internal reg

SetReg
		call	InitFsr				; Init FSR
		call	UART_REC_BYTE		; Get data
		movwf	INDF				; Write to reg
		goto	SendReg

;******************************************************************************
; Get internal reg

GetReg
		call	InitFsr				; Init FSR
SendReg
		call	UART_SEND_COMMAND
		movf	Scratch,w
		call	UART_SEND_BYTE		; Send high part of address
		movf	FSR,w
		call	UART_SEND_BYTE		; Send low  part of address
		movf	INDF,w				; Read reg
		goto	CLR_IRP_SEND_B_CR	; Send it

;******************************************************************************

	org	0x0700

;******************************************************************************
; String tables

InitMsg								; String tables (last character of string has bit7 set)
	dt	"Propeller clock"
CrLf
	retlw	0x0a
	retlw	0x8d					; Sign end of string - bit7 set

;******************************************************************************

	org	0x7C0

;******************************************************************************
; Jump tables

;******************************************************************************
; Return parameter count of a command

CmdParNum
	addwf	PCL,f
	retlw	.4		;'a'
	retlw	.8		;'b'
	retlw	0		;'c'
	retlw	0		;'d'
	retlw	.6		;'e'
	retlw	.4		;'f'
	retlw	.2		;'g'
	retlw	0		;'h'
	retlw	.4		;'i'
	retlw	0		;'j'
	retlw	.10		;'k'
	retlw	.2		;'l'
	retlw	.8		;'m'
	retlw	0		;'n'
	retlw	0		;'o'
	retlw	0		;'p'
	retlw	.4		;'q'
	retlw	0		;'r'
	retlw	.6		;'s'
	retlw	0		;'t'
	retlw	.6		;'u'
	retlw	0		;'v'
	retlw	.2		;'w'
	retlw	.4		;'x'
	retlw	0		;'y'
	retlw	.4		;'z'

;******************************************************************************
; Go to execution routine of a command

CmdExec
	addwf	PCL,f			;	; Commands and parameters		; Answare
							;	; -- Time and date parameters transfered in BCD --
							;	; -- #YY, #MM, #DD, #hh, #mm, #ss, #nn, #hb, #he

 			goto	XeqRc5	;a  ; Execute Rc5 command #aa #cc	; 'A' #aa #cc
								; On RC5 receive					Receive: 'Aaacc'
								;		aa= RC5 address code (5 bits)
								;		cc= RC5 command code - bit7 toggle bit

								; Example: Send:'a1d0c'				Receive: 'A1d0c'

			goto	SetDate	;b	; Set date 'b' #YY #MM #DD #wd	; 'B' #YY #MM #DD #wd
								;		YY=00..99, MM=01..12, DD=01..(28,29,30,31)
								;		Day of week wd=01..07

								; Example: Send:'b09042007'			Receive: 'B09042007'

 			goto	ClrSec	;c	; Clear seconds	'c'				; 'C' #hh #mm #ss
								;		hh=00..23, mm=00..59, ss=00

 								; Example: Send:'c'					Receive: 'C073000'

			goto	GetDate	;d	; Get date 'd' 					; 'D' #YY #MM #DD #wd

								; Example: Send:'d'					Receive: 'B09042007'

;;V;; For debugging only
			goto	SetI2C	;e	; Write reg on I2C 'e' #Sa #Wa #va; 'E' #Sa #Wa #va
								;		Sa= Slave address, Wa= Word address, Va=value

								; Example: Send:'ea00f20'			Receive: 'Ea00f20'

			goto	GetI2C	;f	; Read reg on I2C 'f' #Sa #Wa	; 'F' #Sa #Wa #va

								; Example: Send:'fa00f'				Receive: 'fa00f20'
;;^;; For debugging only

			goto	SetMode	;g	; Set mode 'g' #Mo				; 'G' #Mo
								;		Mo= Mode 
								;	mode:	bit	0			; Auto standby timer enable
								;			bit	1			; Wake Run afther power up
								;			bit	2			; No coil drive in sleep mode
								;			bit	3			; Sound at hour change
								;			bit	4			; Hour sign mode (One sound / as many as hour (1..12))
								;			bit	5
								;			bit	6
								;			bit	7			; Enable reinit time from RTC at midnight

 								; Example: Send:'g01'				Receive: 'G01'

			goto	GetMode	;h	; Get mode 'h'					; 'H' #Mo
								;		Mo= Mode 
								;			bit	0			; Auto standby timer enable
								;			bit	1			; Wake Run afther power up
								;			bit	2			; No coil drive in sleep mode
								;			bit	3			; Sound at hour change
								;			bit	4			; Hour sign mode (One sound / as many as hour (1..12))
								;			bit	5
								;			bit	6
								;			bit	7			; Enable reinit time from RTC at midnight

 								; Example: Send:'h'					Receive: 'H01'

 			goto	SetAST	;i	; Set AST 'i' #Ah #Al 			; 'I' #Ah #Al
								;			Ah:Al= 16 bit AutoStandby Time in seconds

 								; Example: Send:'i0E10'				Receive: 'I0e10'

 			goto	GetAST	;j	; Get AST 'j'  					; 'J' #Ah #Al
								;			Ah:Al= 16 bit AutoStandby Time in seconds

 								; Example: Send:'j'					Receive: 'J0e10'

			goto	SetAlarm;k	; Set alarm 'k'#nn #ff #hh #mm #uu; 'K' #nn #ff #hh #mm #uu
								;	 		nn=00..19, ff: bit7-enab; bit6..0-enable on weekday 7..1
								; 			hh=00..23, if hh==32 hour masked off from comparation
								;			mm=00..59,
								;			uu: 7: mode 0= Alarm , 6: Turn propeller on , 5: Turn propeller off , 4: Enable sound
								;			uu: 7: mode 1= Relais, 6..0= switch for (6..0) minutes; 0 minutes turns off relais 

 								; Example: Send:'k01ff075000'		Receive: 'K01ff075000'
 								; Example: Send:'k01ff0750C0'		Receive: 'K01ff0750c0'

			goto	GetAlarm;l	; Get alarm 'l' #nn				; 'L' #nn #ff #hh #mm #uu
								;	 		nn=00..19, ff: bit7-enab; bit6..0-enable on weekday 7..1
								; 			hh=00..23, mm=00..59,
								;			uu: 7: mode 0= Alarm , 6: Turn propeller on , 5: Turn propeller off , 4: Enable sound
								;			uu: 7: mode 1= Relais, 6..0= switch for (6..0) minutes; 0 minutes turns off relais 

 								; Example: Send:'l01'				Receive: 'L01ff075000'
 								; Example: Send:'l01'				Receive: 'L01ff0750c0'

 			goto	SetPWM	;m	; Set PWM 'm' #Cr #Pr #Cs #Ps	; 'M' #Cr #Pr #Cs #Ps #Cc #Pc
								;			Cr= CCPR1L setting for Run   mode
								;			Pr= PR2    setting for Run   mode
								;			Cs= CCPR1L setting for Sleep mode
								;			Ps= PR2    setting for Sleep mode
								;			Cc= current CCPR1L setting
								;			Pc= current PR2    setting

 								; Example: Send:'m283d0c3d'			Receive: 'M283d0c3d0c3d'

			goto	GetPWM	;n	; Get PWM 'n'  					; 'N' #Cr #Pr #Cs #Ps #Cc #Pc
								;			Cr= CCPR1L setting for Run   mode
								;			Pr= PR2    setting for Run   mode
								;			Cs= CCPR1L setting for Sleep mode
								;			Ps= PR2    setting for Sleep mode
								;			Cc= current CCPR1L setting
								;			Pc= current PR2    setting

 								; Example: Send:'n'					Receive: 'N283d0c3d0c3d'

 			goto	GetAln	;o	; Get number of alarms 'o' 		; 'O' #An #Ad
								;			An= Number of avaible alarms
								;			Ad= RC5 Address of clock
								;
								; Get number of alarms
								; Example: Send:'o'					Receive: 'O201d' - on 16F628(A), 16F648A
								; Example: Send:'o'					Receive: 'O401d' - on 16F87, 16F88

		goto SEND_MESSAGE	;p	; Connect 'p'					; "Propeller clock"
 								; Example: Send:'p'					Receive: 'Propeller clock'

			goto	SetHSnd	;q	; Set hour sound 'q' #hb #he	; 'Q' #hb #he
								; 			hb= hour sound begin hour (bcd)
								; 			he= hour sound end   hour (bcd)

 								; Example: Send:'q0820'				Receive: 'Q0820'

			goto	GetHSnd	;r	; Get hour sound 'r'			; 'R' #hb #he
								; 			hb= hour sound begin hour (bcd)
								; 			he= hour sound end   hour (bcd)

 								; Example: Send:'r'					Receive: 'R0820'

			goto	SetTime	;s	; Set time 's' #hh #mm #ss		; 'S' #hh #mm #ss
								;		hh=00..23, mm=00..59, ss=00..59

 								; Example: Send:'s073005'			Receive: 'S073005'

			goto	GetTime	;t	; Get time 't' 					; 'T' #hh #mm #ss
								;		hh=00..23, mm=00..59, ss=00..59

 								; Example: Send:'t'					Receive: 'T073005'

;;V;; For debugging only
 			goto	SetReg	;u	; Set reg 'u' #rh #rl #vv		; 'U' #vv
								;		rh:rl=reg address, vv=value

 								; Example: Send:'u0110aa'			Receive: 'U0110aa'
;;^;; For debugging only

			goto	SendVer	;v	; Get version 'v'				; 'V' #ma '.' #mi

 								; Example: Send:'v'					Receive: 'V2.0d'

			goto	SetProp	;w	; Get/Set propeller state 'w' #wc; 'W' #ff #Rh #Rl
								;		wc=0x80 get state only
								;		wc=1 for Start
								;		wc=0 for Stop
								;		ff=flags
								;			bit	0:	in standby mode
								;			bit	1:	in sleep mode
								;			bit	2:	high power on coil
								;			bit	3:	index LED on
								;			bit	4:
								;			bit	5:
								;			bit	6:	in deep sleep mode
								;			bit	7:	time invalid
								;
								;		Rh:Rl: remaining run time in seconds

 								; Example: Send:'w01				Receive: 'W040e10'
 								; Example: Send:'w00				Receive: 'W050000'

								;  only read propeller state:
 								; Example: Send:'w80'				Receive: 'W430000'

 			goto	SetRel	;x	; Set relais 'x' #rr #vv		; 'X' #rr #vv
								;		rr=1 on, r=0 off, vv: on time in minutes

 								; Example: Send:'x0103				Receive: 'X0103'

 			goto	GetRel	;y	; Get relais 'y'				; 'Y' #rr #Rc
								;		rr=1 on, r=0 off, vv: on time in minutes

 								; Example: Send:'y0103'				Receive: 'Y0103'

;;V;; For debugging only
			goto	GetReg	;z	; Send reg 'z' #rh #rl			; 'Z' #vv
								;		rh:rl=reg address, vv=value

 								; Example: Send:'z0110'				Receive: 'Z0103aa'
;;^;; For debugging only



	org		0x7F8

;******************************************************************************
; Get next character of a string

MsgTabl
		movf	Rs232StrPtrH,w		; Get string pointer high byte
		movwf	PCLATH				; Load it to PCLATH
		movf	Rs232StrPtr,w		; Get string pointer low  byte
		incfsz	Rs232StrPtr,f		; Increment pointer as
		goto	lString1
		incf	Rs232StrPtrH,f		; 16 bit number
lString1
		movwf	PCL					; Computed goto to string table
		return						; movwf	PCL cannot be the last instruction

; Deapest Stack 					; These processors have an 8 level stack
;	1	call	UARTCmdDec
;	2	call	I2CByteWrite
;	3	CALL 	OUT_BYTE
;	4	CALL 	CLOCK_PULSE
;	5	CALL 	HIGH_SCL
;	6	Interrupt routine
;	7
;	8


	END                     		; directive 'end of program'
