// #define _DEBUG

#include "stm8s.h"


#ifdef _DEBUG
#include <string.h>
#include <stdio.h>
#endif

#define F_CPU 4000000UL
/*
 * Következő verzióban
 * - Hardver oldalon legyen reset gomb
 * - Hardverben legyen megvonható az RF modultól az áram, energiatakarákosság miatt
 *
 */


/*  STM8S003
PD4      RF433    1 +-----------+  20  PD3 / E      LCD Enable    
PD5 / TX          2 | *         |  19  PD2 / BTN
PD6 / AIN6  RX    3 |           |  18  PD1 / SWIM
NRST              4 |           |  17  PC7   PD7    LCD D7
PA1               5 |           |  16  PC6   PD6    LCD D6
PA2               6 |           |  15  PC5   PD5    LCD D5
VSS               7 |           |  14  PC4   PD4    LCD D4
VCAP              8 |           |  13  PC3   RS     LCD Register Select
VDD               9 |           |  12  PB4 / SCL
PA3        LED   10 +-----------+  11  PB5 / SDA
*/

volatile uint16_t timercounter=0;
void tim4(void) __interrupt(23)
{
    // millisec számláló
    timercounter++;
    TIM4->SR1=0;
}  


void CLKConfig_reg() // 4MHz PRE SCALER 4
{
    CLK->ICKR = 0;                              //  Belső óra reset
    CLK->ICKR |= CLK_ICKR_HSIEN;                //  HSI engedélyezés
    CLK->ECKR = 0;                              //  Külső órajel tiltása
    while ((CLK->ICKR & CLK_ICKR_HSIRDY) == 0); //  Várakozás a HSI-re
    // CLK->CKDIVR = 0b00001000;                            //  16 Mhz / 2DIV
    CLK->CKDIVR = 0b00010000;                            //  16Mhz / 4DIV
    CLK->PCKENR1 = 0xff;                        //  Minden kap órajelet
    CLK->PCKENR2 = 0xff;                        //  itt is
    CLK->CCOR = 0;                              //  CCO tiltása
    CLK->HSITRIMR = 0;                          //  
    CLK->SWIMCCR = 0;                           //  SWIM CLOCK DIV2
    CLK->SWR = 0xe1;                            //  HSI a mester órajel
    CLK->SWCR = 0;                              //  SWCR tötlése
    CLK->SWCR |= CLK_SWCR_SWEN;                 //  Swicth start
    while ((CLK->SWCR & CLK_SWCR_SWBSY) != 0);  //  várakozás az órajelre
}
 
void TIM4Config_reg()
{
    TIM4->PSCR=0x04;    // TIM4 DIV4
    TIM4->ARR=0xFF;     // auto reload
    TIM4->IER=TIM4_IER_UIE; // megszakítás engedélyezve
    TIM4->CR1=TIM4_CR1_CEN; // TIM4 engedélyezve
}

void TIM1Config_reg()
{
    // felfelé számláló, nem újratöltő
    TIM1->PSCRH=0;  
    TIM1->PSCRL=4;  // TIM1 DIV4
    TIM1->ARRH=0xFF;
    TIM1->ARRL=0xFF;
    TIM1->CR1=TIM1_CR1_OPM | TIM1_CR1_CEN; // ENABLE, egy impulzus mód
}

void uart_init_reg()
{
    UART1->CR1 = 0;
    UART1->CR2 = 0;
    UART1->CR4 = 0;
    UART1->CR3 = 0;
    UART1->CR5 = 0;
    UART1->GTR = 0;
    UART1->PSCR = 0;
    // UART1->BRR2 = 0x0B;      //  115200
    // UART1->BRR1 = 0x08;      //  0x008B
    // UART1->BRR2 = 0x05;      //  115200
    // UART1->BRR1 = 0x04;      //  8Mhz-nél
    UART1->BRR2 = 0x03;      //  115200
    UART1->BRR1 = 0x02;      //  4Mhz-nél
    
    UART1->CR3 = UART1_CR3_CPOL | UART1_CR3_CPHA | UART1_CR3_LBCL;
    UART1->CR2 = UART1_CR2_TEN;
}


void awu_init_reg()
{
    AWU->CSR&=~AWU_CSR_AWUEN;
    // 5.280 s - 30.720 s 
    AWU->TBR = 0b1111;
    // AWU->APR = 28;
    AWU->APR = 3;
    AWU->CSR|=AWU_CSR_AWUEN;
    
}

// nincs más dolga csak kiolvasni a regisztert 
void awu_int(void) __interrupt(1)
{
    uint8_t r;
    r=AWU->CSR;
}
 
void putchar (char c){UART1->DR=(u8)c;while ((UART1->SR & UART1_SR_TXE) == 0);}

void delayms(uint32_t ms) {uint16_t T=timercounter;while ((uint16_t)(timercounter - T) < ms) ;} 
void Delay(uint32_t ms) {uint32_t i;while(ms--){ i=2600; while(i--) ; } } 
void Delay_us(uint32_t us) {while(us--){ nop();nop();nop();nop(); } } 
static inline void delay_ms(uint16_t ms) {
    uint32_t i;
    for (i = 0; i < ((F_CPU / 18000UL) * ms); i++)
        __asm__("nop");
}

#define HIGH    1
#define LOW     0


// Should be ~ 1300uS (long - high)
#define LP_MIN 450
#define LP_MAX 600
// Should be ~ 500uS (short - high)
#define SP_MIN 140
#define SP_MAX 160
// Should be ~ 1900uS (short - low for Alecto)
#define ASP_MIN 1300
#define ASP_MAX 1600
// Should be ~ 3800uS (long - low for Alecto)
#define ALP_MIN 2900
#define ALP_MAX 3300
// Should be ~ 9200uS (end - low for Alecto)
#define AEP_MIN 7000
#define AEP_MAX 7700


#define STEP_DETECT     0   // 8 SP és nincs LP
#define STEP_DETECTED   1   // 100 ból egy AEP
#define STEP_SIGN       2   // ASP ALP ASP ALP
#define STEP_DATA       3   // 32 ASP ALP
#define STEP_PRINT      4   // 
#define STEP_LIGHT      5   // 
#define STEP_CLEAN      6   // 
#define STEP_DEBUG      7   // 
#define STEP_SLEEP      8   // 

#define IS_SP(x)    (x>SP_MIN) && (x<SP_MAX)
#define IS_NOTLP(x)    (x<=LP_MIN) && (x>=LP_MAX)
#define IS_ASP(x)    (x>ASP_MIN) && (x<ASP_MAX)
#define IS_ALP(x)    (x>ALP_MIN) && (x<ALP_MAX)
#define IS_AEP(x)    (x>AEP_MIN) && (x<AEP_MAX)

#define BUFFLENGTH  100
volatile uint8_t step = STEP_DETECT;
volatile uint8_t ptr1=0;
volatile uint8_t ptr2=0;
volatile uint8_t ptr=0;
volatile uint16_t buff1[BUFFLENGTH]={0,0,0,0,0,0,0,0,0,0,
                            0,0,0,0,0,0,0,0,0,0,
                            0,0,0,0,0,0,0,0,0,0,
                            0,0,0,0,0,0,0,0,0,0,
                            0,0,0,0,0,0,0,0,0,0,
                            0,0,0,0,0,0,0,0,0,0,
                            0,0,0,0,0,0,0,0,0,0,
                            0,0,0,0,0,0,0,0,0,0,
                            0,0,0,0,0,0,0,0,0,0,
                            0,0,0,0,0,0,0,0,0,0
                            };
volatile uint16_t buff2[BUFFLENGTH]={0,0,0,0,0,0,0,0,0,0,
                            0,0,0,0,0,0,0,0,0,0,
                            0,0,0,0,0,0,0,0,0,0,
                            0,0,0,0,0,0,0,0,0,0,
                            0,0,0,0,0,0,0,0,0,0,
                            0,0,0,0,0,0,0,0,0,0,
                            0,0,0,0,0,0,0,0,0,0,
                            0,0,0,0,0,0,0,0,0,0,
                            0,0,0,0,0,0,0,0,0,0,
                            0,0,0,0,0,0,0,0,0,0
                            };
volatile uint16_t _spacer = 0;
volatile uint16_t *buff=buff1;
volatile uint8_t buffflag=0;



void gpiod_int(void) __interrupt(6)
{
    uint8_t dir;
    uint8_t r;
    uint16_t width;
    // kiolvasni a D portot
    r=GPIOD->IDR;
    // ha sleep akkor csak a BTN-t kell nézni
    if (step==STEP_SLEEP)
    {
        // PD2 -őn van a BTN
        dir = r & (1 << 2);
        // mivel a láb felhúzó állapotban van, a 0 jelenti a lenyomást
        // ha le van nyomva, akkor irány a LIGHT
        if (dir==0) step=STEP_LIGHT;
    } else {
    // ez a 433Mhz RF modul jelváltozása PD4
        dir = r & (1 << 4);
        // a buffer páros elemére kerül a 0 jelszint ideje, páratlanra az 1-é
        // ha elcsúszás van ez a sor korrigál
        if (dir && ptr%2 ) {buff[ptr]=0; ptr++; ptr%=BUFFLENGTH;}
        // jelszélesség
        width=(TIM1->CNTRH << 8) | TIM1->CNTRL;
        // beírva a buffer jelenlegi helyére
        buff[ptr]=width;
        // Timer nullázás és indítás
        TIM1->CNTRH=0; TIM1->CNTRL=0; TIM1->CR1 |= 1;
        // ha a szinkronjel van és felfutó az ág és a státusz DETECT
        if (dir && IS_AEP(width) && (step==STEP_DETECT) )
        {
            // a buffert kell cserélni, hogy ha jön adat, ne a kiértékelés alattit írja felül
            if (buff==buff2) {buff=buff1; ptr2=ptr;}
            else {buff=buff2; ptr1=ptr; }
            // státusz átállítása DATA-ra
            step=STEP_DATA;
        }
        // gyűrűs buffer léptetése
        ptr++; ptr%=BUFFLENGTH;
    }
}  

/*
A bitek jelentése
    // Header 0 till 3
    // ID: 4 till 11
    // Battery: 12
    // TX mode: 13
    // Channel: 14 till 15
    // Temperature: 16 till 27
    // Humidity: 28 till 35
                
Pár minta a beérkező jelekre
                
870  1787  871  1779  1782  866  870  862  1789  867  1776  1778  1784  1783  867  861  856  863  871  1782  864  869  1782  872  866  870  870  871  853  866  1784  1788  863  1780  1776  1790  4279                
872  1788  875  1794  1784  873  873  863  1789  868  1788  1785  1786  1791  874  866  864  876  867  1784  870  867  1786  873  872  877  868  874  874  874  1789  1787  879  1793  1790  1785  4278
0101 1001011 1 1 00 000100100000 00110111

              1100010111000 000100100000 00110100
**bits: 0011 0001011 1 1 00 000100011111 00110100_
**temp: -7618, humy: 26728, ch: 0
466  
3030  468  3024  458  3034  470  1461  470  1474  457  1468  475  1473  459  1471  456  3039  463  1462  474  1476  459  1470  455  3039  
513  1483  455  3037  448  1482  454  3039  466  3035  455  1474  463  1481  449  1478  450  3041  464  1476  453  3036  455  3035  455  

3046  463  3027  458  1474  454  1486  453  1475  457  1484  455  1476  451  3048  449  1483  447  1483  467  1474  460  3028  449  3051  
460  3031  457  3031  458  3032  471  1467  453  1478  470  3029  454  3032  456  1483  462  3029  454  1473  471  1470  451  7177  

*/


#define     E_0  GPIOD->ODR &= ~(1 << 3)
#define     E_1  GPIOD->ODR |=  (1 << 3)

#define     RS_0  GPIOC->ODR &= ~(1 << 3)
#define     RS_1  GPIOC->ODR |=  (1 << 3)

#define LCD_delay   delay_ms    

void LCD_data(uint8_t data,uint8_t type)
{
    uint8_t temp;
    
    LCD_delay(10);
    temp = (data <<4) & 0xF0;
    GPIOC->ODR &= 0x0F; GPIOC->ODR |= data & 0xF0;
    if (type==0) RS_0; // parancs mód
    else RS_1;         // adat mód
    E_1; LCD_delay(10); E_0;
    GPIOC->ODR &= 0x0F; GPIOC->ODR |= temp;
    E_1; LCD_delay(10); E_0;
}

void LCD_init()
{
    LCD_delay(100);
    GPIOC->ODR &= 0x0F; GPIOC->ODR |= 0x30;
    E_1; LCD_delay(5); E_0;
    LCD_delay(5);
    E_1; LCD_delay(5); E_0;
    LCD_delay(5);
    E_1; LCD_delay(5); E_0;
    LCD_delay(5);

    LCD_data(0x02,0); // Return home
    LCD_delay(20);
    LCD_data(0x28,0); // Func Set 2 line, 5x8
    LCD_data(0x08,0); // Disp off
    LCD_data(0x01,0); // Clear display
    LCD_data(0x06,0); // Shift Display to Right
    LCD_data(0x0C,0); // Disp on
}
void LCD_goto(u8 c, u8 r) {
    if (r == 0) LCD_data(0x80 | c,0);
    else LCD_data(0xC0 | c,0);
}

void LCD_str(const char *text)
{
    while (*text) LCD_data(*text++,1);
}

void LCD_int(uint32_t dt)
{
    uint32_t i;
    for (i=100000;i>=1;i/=10) {
        if (dt>=i) LCD_data( '0' + ((dt/i) % 10) , 1);
    }
    if (dt==0) LCD_data('0',1);
}


#define     LED1_1  GPIOA->ODR |=  (1 << 3)
#define     LED1_0  GPIOA->ODR &= ~(1 << 3)

void main()
{
    uint16_t T_SER = 0; 
    uint16_t T_LED = 0; 
    uint16_t T_WRK = 0; 
    uint16_t T_BTN = 0; 
    uint16_t T_DLY = 0; 
    int16_t i=0;
    uint16_t a1=0;
    uint16_t a2=0;
    uint8_t v=0;
    uint8_t l=0;
    
    
    uint8_t tdelay=1;
    uint8_t ch=0;
    uint8_t battery=0;
    uint8_t tx_mode=0;
    int16_t temp=0;
    uint16_t humy=0;
    uint8_t keystate=0;
    
    uint16_t *_buff=buff1;
    uint8_t _ptr=0;

    uint8_t bits[40]={0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0}; 
    CLKConfig_reg();
    
    GPIOD->DDR &= ~(1<<4); GPIOD->CR2 |= (1<<4); // Input with interrupt
    GPIOD->DDR &= ~(1<<2); GPIOD->CR1 |= (1<<2); // Input pull up no interrupt
    
    LED1_0; GPIOA->DDR |= (1<<3); GPIOA->CR1 |= (1<<3); // Output push pull
    E_0;  GPIOD->DDR |= (1<<3); GPIOD->CR1 |= (1<<3); // Output push pull
    RS_0; GPIOC->DDR |= (1<<3); GPIOC->CR1 |= (1<<3); // Output push pull
    /* GPIO C7 - C4 */ GPIOC->ODR &= 0x0F; GPIOC->DDR |= 0xF0; GPIOC->CR1 |= 0xF0; // Output push pull


#ifdef _DEBUG
    uart_init_reg();
    printf("debug enabled\r\n"); 
#endif    
    // TIM2Config_reg();
    awu_init_reg();
    TIM1Config_reg();
    TIM4Config_reg();
    EXTI->CR1|=0b11000000; // PORTA mindkét ágban
    rim(); // enable interrupts
    LED1_1;
    LCD_init();
    LCD_str("   WS   ");
    LCD_goto(0,1);
    LCD_str("  Sync  ");
    T_DLY=timercounter;
    while (1) 
    {
        if((uint16_t)(timercounter - T_BTN) > 100){
            T_BTN = timercounter;
            v=GPIOD->IDR & (1<<2); 
            if (v!=keystate)
            {
                keystate=v;
                if ((keystate & (1<<2))==0) // 0 mivel felhúzó
                    step=STEP_LIGHT;
            }
        }
        if ( ((uint16_t)(timercounter - T_DLY) > 3000) && tdelay) {
            tdelay=0;
            LED1_0;
            if (temp==0) step=STEP_CLEAN;
        }
        
        if((uint16_t)(timercounter - T_WRK) > 100){
            T_WRK = timercounter;
            switch (step)
            {
                case STEP_DATA:
                    if (buff==buff2) {_buff=buff1; _ptr=ptr1; }
                    else {_buff=buff2; _ptr=ptr2; }
                    if (_ptr%2!=0) _ptr--;
                    
                    for (i=36;i>=0;i--)
                    {
                        a1=_buff[_ptr];
                        if (IS_ALP(a1)) bits[i]='1';
                        else if (IS_ASP(a1)) bits[i]='0';
                        else bits[i]='_';
                        _ptr-=2; if (_ptr>=BUFFLENGTH) _ptr=BUFFLENGTH-2; 
                    }
                    // 010000010001010110001011110000010001
                    // 00000010001010110001011110000010001
                    if (bits[0]=='0' && bits[1]=='1' && bits[2]=='0' && bits[3]=='1')
                    {
                        temp=0;
                        humy=0;                
                        ch = ((bits[14]=='1' ? 1 : 0)<<1) + (bits[15]=='1' ? 1 : 0);
                        for (i=16;i<=27;i++) {temp<<=1; temp |= bits[i]=='1' ? 1 : 0; }
                        for (i=28;i<=35;i++) {humy<<=1; humy |= bits[i]=='1' ? 1 : 0; }
                        if (temp > 3840) temp -= 4096;
                        step=STEP_PRINT;
                    } else {
                        step=STEP_DETECT;
        #ifdef _DEBUG                    
                        step=STEP_DEBUG;
        #endif           
                    }
                    break;
                case STEP_LIGHT:
                    LED1_1;
                    GPIOD->CR2 &= ~(1<<4); // D4 RF433Mhz no WakeUp
                    halt();
                    GPIOD->CR2 |= (1<<4);
                    LED1_0;
                    step=STEP_DETECT;
                    break;
                case STEP_PRINT:
                    
                    LCD_goto(0,0);
                    if (temp<0) {
                        v=1; a2=-temp;
                    } else {
                        v=0; a2 = temp;
                    }
                    a1=a2/1000;
                    if (v && (temp<=-100)) LCD_data( '-', 1);
                    else if (a1==0) LCD_data( ' ', 1);
                    else LCD_data( '0' + (a1%10) , 1);
                    
                    a1=a2/100;
                    if (v && (temp>-100)) LCD_data( '-', 1);
                    else if (v) LCD_data( '0' + (a1%10) , 1);
                    else if (a1==0) LCD_data( ' ', 1);
                    else LCD_data( '0' + (a1%10) , 1);

                    a1=a2/10;
                    LCD_data( '0' + (a1%10) , 1);
                    LCD_data( '.', 1);
                    a1=a2/1;
                    LCD_data( '0' + (a1%10) , 1);
                    LCD_data( '\xDF', 1);
                    LCD_data( 'C', 1);
                    LCD_goto(2,1);
                    a1=humy/10;
                    if (a1==0) LCD_data( ' ', 1);
                    else LCD_data( '0' + (a1%10) , 1);
                    a1=humy/1;
                    LCD_data( '0' + (a1%10) , 1);
                    LCD_data('%',1);
                    LCD_data(' ',1);
                    
                    LCD_goto(0,1);
                    LCD_data( '*', 1);
                    step=STEP_SLEEP;
                    break;
                case STEP_SLEEP:
                    GPIOD->CR2 &= ~(1<<4); // D4 RF433Mhz no WakeUp
                    GPIOD->CR2 |= (1<<2);  // D2 BTN WakeUP
                    for (i=0;i<13;i++)
                    {
                        halt();
                        nop();
                        nop();
                        if (step==STEP_LIGHT) break;
                    }
                    
                    GPIOD->CR2 &= ~(1<<2);
                    GPIOD->CR2 |= (1<<4);
                    LCD_goto(0,1);
                    LCD_data( ' ', 1);
                    if (step==STEP_LIGHT) break;
                    step=STEP_DETECT;
        #ifdef _DEBUG
                    step=STEP_DEBUG;
        #endif           
                    break;
                case STEP_CLEAN:
                    LCD_goto(0,0);
                    LCD_str(" --.-\xDF""C ");
                    LCD_goto(0,1);
                    LCD_str("  --%  ");
                    step=STEP_DETECT;
                    break;
            }
        }
            
        if((uint16_t)(timercounter - T_SER) > 1000){
            T_SER = timercounter;
        #ifdef _DEBUG
            if (step==STEP_DETECT)
                printf("\n**");
            else
                printf("\n***");
            
            if (step==STEP_DEBUG)
            {
                printf("\n**bits: %s\n" , bits);
                printf("**temp: %d, humy: %d, ch: %d\n" , temp, humy, ch);
                if (buff==buff2) {_buff=buff1; _ptr=ptr1; }
                else {_buff=buff2; _ptr=ptr2; }
                for (i=0;i<25;i++) printf("%u  ",_buff[0 + i]); printf("\n");
                for (i=0;i<25;i++) printf("%u  ",_buff[25 + i]); printf("\n");
                for (i=0;i<25;i++) printf("%u  ",_buff[50 + i]); printf("\n");
                for (i=0;i<25;i++) printf("%u  ",_buff[75 + i]); printf("\n");
                step=STEP_DETECT;
            }
        #endif           
         
        }

    }
    
}    
// 123456789012345678
// 11 11011000 10110011
