/* 
 * File:   lcd.h
 * Author: Jan Kubovy <jan@kubovy.eu>
 */

#ifndef LCD_H
#define	LCD_H

#ifdef	__cplusplus
extern "C" {
#endif
    
#include <stdbool.h>
#include <stdint.h>

typedef enum {
    DISPLAY_DATETIME_NONE,
    DISPLAY_DATETIME_I2C_LCD,
    DISPLAY_DATETIME_USART,
    DISPLAY_DATETIME_LED_DISPLAY        
} DISPLAY_DATETIME_TARGET;    

#define LCD_ADDRESS        0x27 // change this according to ur setup
#define LCD_COLS           20
#define LCD_ROWS           4

#define LCD_ROM_CODE_A00    // A00 - asia, A02 - europe

// Commands
#define LCD_CLEARDISPLAY    0x01        // 0b00000001
#define LCD_RETURNHOME      0x02        // 0b00000010
#define LCD_ENTRYMODESET    0x04        // 0b00000100
#define LCD_DISPLAYCONTROL  0x08        // 0b00001000
#define LCD_CURSORSHIFT     0x10        // 0b00010000
#define LCD_FUNCTIONSET     0x20        // 0b00100000
#define LCD_SETCGRAMADDR    0x40        // 0b01000000
#define LCD_SETDDRAMADDR    0x80        // 0b10000000

// Flags for display entry mode
#define LCD_ENTRYRIGHT      0x00
#define LCD_ENTRYLEFT       0x02        // 0b00000010
#define LCD_ENTRYSHIFTINCREMENT 0x01
#define LCD_ENTRYSHIFTDECREMENT 0x00

// Flags for display on/off control
#define LCD_DISPLAYON       0x04        // 0b00000100
#define LCD_DISPLAYOFF      0x00        // 0b00000000
#define LCD_CURSORON        0x02        // 0b00000010
#define LCD_CURSOROFF       0x00        // 0b00000000
#define LCD_BLINKON         0x01        // 0b00000001
#define LCD_BLINKOFF        0x00        // 0b00000000

// Flags for display/cursor shift
#define LCD_DISPLAYMOVE     0x08        // 0b00001000
#define LCD_CURSORMOVE      0x00        // 0b00000000
#define LCD_MOVERIGHT       0x04        // 0b00000100
#define LCD_MOVELEFT        0x00        // 0b00000000

// Flags for function set
#define LCD_8BITMODE        0x10        // 0b00010000
#define LCD_4BITMODE        0x00        // 0b00000000
#define LCD_2LINE           0x08        // 0b00001000
#define LCD_1LINE           0x00        // 0b00000000
#define LCD_5x10DOTS        0x04        // 0b00000100
#define LCD_5x8DOTS         0x00        // 0b00000000

// Flags for backlight
#define LCD_BACKLIGHT       0x08        // 0b00001000
#define LCD_NOBACKLIGHT     0x00        // 0b00000000

// Lines
#define LCD_LINE1           0x80        // 0b10000000
#define LCD_LINE2           0xC0        // 0b11000000
#define LCD_LINE3           0x94        // 0b10010100
#define LCD_LINE4           0xD4        // 0b11010100
    
#define En            0b00000100        // 0x04 Enable bit
#define Rw            0b00000010        // 0x02 Read/Write bit
#define Rs            0b00000001        // 0x01 Register select bit

/**
 * LCD module intialization.
 * 
 * @param address I2C 7bit address of the LCD.
 * @param cols Number of columns.
 * @param rows Number of rows.
 */
void LCD_initialize(uint8_t address, uint8_t cols, uint8_t rows);

/**
 * Clear the content of the LCD.
 */
void LCD_clear(void);

/**
 * Turn LCD's backlight on or off.
 * 
 * @param on Whether to turn the backlight on or off.
 */
void LCD_backlight(bool on);

/**
 * Send a command to the LCD.
 * 
 * @param command The command.
 */
void LCD_send_cmd(uint8_t command);

/**
 * Send data to the LCD.
 * 
 * @param data The data.
 */
 
void LCD_send_data(uint8_t data);

/*
 * <p><b>Function prototype:</b>LCD_load_customchars( uint8_t*, uint8_t, uint8_t );</p>
 * <p><b>Summary:</b>Sajt kaaktereket tlt fel az LCD-re</p>
 * <p><b>Description:</b></p>
 * <p><b>Precondition:</b></p>
 * mint a feltltend karakterek szma</p>
 * <p><b>Parameters:</b>
 * @param a pattern tmb pointere</p>
 * @param a feltltend karakter sorszma (mint az ASCII)</p>
 * @param a feltltend karakterer szma</p>
 * <p><b>Returns:</b> -</p>
 * <p><b>Example:</b></p>
 * <code>
 * </code>
 * <p><b>Remarks:</b> A pattern tmbnek legalbb 8-szor akkornak kell lenni mint a feltltend karakterek szma</p>
 */

void LCD_load_customchars( uint8_t *charMap, uint8_t location, uint8_t nChars );


/**
 * Send string to the LCD.
 * 
 * The string length will be trimmed to the number of the LCD's columns. Also
 * multiple lines may be send at once by separating the line with a new line 
 * (\n). All lines bigger than the number of the LCD's rows will be ignore.
 * 
 * There are some modifiers which can be added on the beginning of each line:
 * <li><code>|c|</code>: will center the text</li>
 * <li><code>|l|</code>: will left-align the text (the default)</li>
 * <li><code>|r|</code>: will right-align the text</li>
 * <li><code>|d|MS|</code>: type animation with <code>MS</code> millisecond delay between each character</li>
 * 
 * @param str String to be sent.
 * @param line Line the string should be display on or begin at in case of multi-line string.
 */
void LCD_send_string(char *str, uint8_t line);


void LCD_strReplace( char* );


void LCD_clearLine( uint8_t line );


#ifdef LCD_ROM_CODE_A00
//char replaceTable[] = { '',  '', '',  '',  '',  '',  '',  '',  '',  '' };
char replaceTable[] = { 0xFC, 0xf6, 0xB0, 0xE1, 0xE9, 0xEd, 0xF3, 0xF5, 0xFA, 0xFC };  
char lcdTable[] =     { 0x5F, 0xEF, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06 };

uint8_t customChar[64] = { 
    0x02, 0x04, 0x0E, 0x01, 0x0F, 0x11, 0x0F, 0x00,     // 0 
    0x02, 0x04, 0x0E, 0x11, 0x1F, 0x10, 0x0E, 0x00,     // 1 
    0x04, 0x08, 0x0C, 0x04, 0x04, 0x04, 0x0E, 0x00,     // 2 
    0x02, 0x04, 0x0E, 0x11, 0x11, 0x11, 0x0E, 0x00,     // 3 
    0x05, 0x0A, 0x0E, 0x11, 0x11, 0x11, 0x0E, 0x00,     // 4 
    0x02, 0x04, 0x11, 0x11, 0x11, 0x13, 0x0D, 0x00,     // 5 
    0x05, 0x0A, 0x11, 0x11, 0x11, 0x13, 0x0D, 0x00,     // 6 ,
    0x0C, 0x12, 0x12, 0x0C, 0x00, 0x00, 0x00, 0x00      // 7  0xB0 (176)
};

#endif


#ifdef LCD_ROM_CODE_A02 
char replaceTable = { '',   '',  '',  '',  '',   '',  '',  '',  '',  '', '',   '',   '',  '',  '',  '', '\0'  };
char lcdTable =     { 0xE0, 0xC0, 0xE9, 0xC9, 0xED, 0xED', 0xF3, 0xD3, 0xF6, 0xD6, 0x00, 0x01', 0xFC, 0xDC, 0x02, 0x03, '\0' } };

char customChar[8][8] = { 
    { 0x05, 0x0A, 0x00, 0x0E, 0x11, 0x11, 0x11, 0x0E },     // 
    { 0x05, 0x0A, 0x0E, 0x11, 0x11, 0x11, 0x11, 0x0E },     // 

    { 0x02, 0x04, 0x0E, 0x11, 0x1F, 0x10, 0x0E, 0x00 },     // 
    { 0x04, 0x08, 0x0C, 0x04, 0x04, 0x04, 0x0E, 0x00 },     // 
    { 0x02, 0x04, 0x0E, 0x11, 0x11, 0x11, 0x0E, 0x00 },     // 
    { 0x05, 0x0A, 0x0E, 0x11, 0x11, 0x11, 0x0E, 0x00 },     // 
    { 0x02, 0x04, 0x11, 0x11, 0x11, 0x13, 0x0D, 0x00 },     // 
    { 0x05, 0x0A, 0x11, 0x11, 0x11, 0x13, 0x0D, 0x00 },     // ,
    { 0x0E, 0x11, 0x11, 0x11, 0x0E, 0x00, 0x00, 0x00 } };   // 
#endif


#endif	/* LCD_H */