/*
   DS1307
   The DS1307 operates in the standard mode (100kHz) only. 

   the default RTC mode 24 hours
 */
#include <stdint.h>
#include <time.h>
#include "../globals.h"
#include "rtc.h"
#include "lcd.h"

#ifdef USE_DS1307_VCARD
uint8_t vcardStr[ RTC_VCARD_SIZE ] = "ZSoft - 2019";
#endif
RTC_MODE_T rtc_mode = RTCMODE_24HOURS;
char DateStr[ LCD_COLS + 1 ];
char TimeStr[ LCD_COLS + 1 ];



void RTC_reset(void)
{
    I2C_start();
    I2C_select( RTC_ADDRESS, I2C_MODE_WRITE );
    I2C_write( RTC_ADDRESS_RAM );
    for( uint8_t i = RTC_ADDRESS_RAM; i <= RTC_ADDRESS_RAMEND; i++ ) 
        I2C_write( 0xFF );
    I2C_stop();
    __delay_ms( 10 );
}


void RTC_initialize( bool clearRam )
{
    I2C_start();
    I2C_select( RTC_ADDRESS, I2C_MODE_WRITE );
    I2C_write( RTC_ADDRESS_SECONDS );
    for( uint8_t i = 0; i < 8; i++ ) I2C_write( 0x00 );
    I2C_stop();
    
    if( clearRam ) RTC_reset();
    
#ifdef USE_DS1307_VCARD
    RTC_writeVcard( vcardStr );
#endif
}


void RTC_Start( bool start )
{
    uint8_t sec = I2C_receive( RTC_ADDRESS, RTC_ADDRESS_SECONDS );
    
    if( start ) {
        BitClear( sec, 7 );
    }
    else {
        BitSet( sec, 7 );
    }
    I2C_start();
    I2C_select( RTC_ADDRESS, I2C_MODE_WRITE );
    I2C_write( RTC_ADDRESS_SECONDS );
    I2C_write( sec );
    I2C_stop();
}


void RTC_setMode( RTC_MODE_T mode )
{
    uint8_t rb = I2C_receive( RTC_ADDRESS, RTC_ADDRESS_HOURS );
    uint8_t wb = 0;
    
    if( ( BitTest( rb, 6 ) ) != mode ) {
        if( mode == RTCMODE_24HOURS ) {  // 12 -> 24
            wb = Bcd2Dec( rb & 0x1F );
            if( BitTest( rb, 5 ) ) wb += 12;
            wb = Dec2Bcd( wb ); 
        } else {                         // 24 -> 12
            wb = Bcd2Dec( rb & 0x3F );
            if( wb > 12 ) {
                wb = Dec2Bcd( wb - 12 );
                BitSet( wb, 5 );
            }
            BitSet( wb, 6 );    
        }
        I2C_select( RTC_ADDRESS, I2C_MODE_WRITE );
        I2C_write( RTC_ADDRESS_HOURS );
        I2C_stop();
        I2C_start();
        I2C_write( wb );
        I2C_stop();
    }
}


void RTC_setControl( uint8_t ctrlByte )
{
    I2C_start();
    I2C_select( RTC_ADDRESS, I2C_MODE_WRITE ); 
    I2C_write( RTC_ADDRESS_CONTROL );
    I2C_write( ctrlByte );
    I2C_stop();
}


void RTC_setDateTime(struct tm dateTime)
{
    uint8_t writeBuffer[7] = { Dec2Bcd( dateTime.tm_sec ),
                               Dec2Bcd( dateTime.tm_min ),
                               Dec2Bcd( dateTime.tm_hour ),
                               Dec2Bcd( weekDay( dateTime.tm_year, dateTime.tm_mon, dateTime.tm_mday ) + 1 ),
                               Dec2Bcd( dateTime.tm_mday ),
                               Dec2Bcd( dateTime.tm_mon ),
                               Dec2Bcd( dateTime.tm_year % 100 ) };

    if( rtc_mode == RTCMODE_12HOURS )
    {
        if( dateTime.tm_hour > 12 )
        {
            writeBuffer[ RTCRB_HOURS ] = Dec2Bcd( dateTime.tm_hour - 12 );
            BitSet( writeBuffer[ RTCRB_HOURS ], 5 );
        }
        BitSet( writeBuffer[ RTCRB_HOURS ], 6 );
    }
    I2C_start();
    I2C_select( RTC_ADDRESS, I2C_MODE_WRITE );
    I2C_write( RTC_ADDRESS_SECONDS );
    for( uint8_t i = 0; i < 7 ; i++ ) I2C_write( writeBuffer[ i ] );
    I2C_stop();
}
 


void RTC_getDateTime(void)
{
    uint8_t b = 0;
    uint8_t a[7];
    
    I2C_start();
    I2C_select( RTC_ADDRESS, I2C_MODE_WRITE );
    I2C_write( RTC_ADDRESS_SECONDS );
    I2C_repeated_start();
    I2C_select( RTC_ADDRESS, I2C_MODE_READ );

    for( uint8_t i = 0; i < 7; i++ )
    {
        b = I2C_read( i == 6 ? I2C_ACK : I2C_NACK );
        switch ( i ) {
            case 0: rtcTime.tm_sec = Bcd2Dec( b ) & 0x7F ; break;
            case 1: rtcTime.tm_min = Bcd2Dec( b ) & 0x7F; break;
            case 2: if( BitTest( b, 6 ) ) {  // 12 rs md
                        rtcTime.tm_hour = Bcd2Dec( b ) & 0x1F;
                        rtc_mode = RTCMODE_12HOURS;
                        // if( BitTest b, 5 ) du
                    } else {
                        rtcTime.tm_hour = Bcd2Dec( b ) & 0x3F;
                        rtc_mode = RTCMODE_24HOURS;
                    }
                    break;
            case 3: rtcTime.tm_wday = Bcd2Dec( b ) & 0x07; break;
            case 4: rtcTime.tm_mday = Bcd2Dec( b ) & 0x3F; break;
            case 5: rtcTime.tm_mon = Bcd2Dec( b ) & 0x1F; break;
            case 6: rtcTime.tm_year = Bcd2Dec( b ) + RTC_EPOCH; break;
            default: ;
        }
    }
    I2C_stop();
}


#ifdef USE_DS1307_VCARD
void RTC_writeVcard(uint8_t *str ) 
{
    uint8_t count = 0;
    
    I2C_start();
    I2C_select( RTC_ADDRESS, I2C_MODE_WRITE );
    I2C_write( RTC_ADDRESS_VCARD );
    while( *str != '\0' ) 
    {
        I2C_write( *str++ );
        count++;
    }
    while( count++ < RTC_VCARD_SIZE ) I2C_write( '\0' );
    I2C_stop();
}


void RTC_getVcard(uint8_t* str)
{
    uint8_t count = 0;
    memset( str, 0, RTC_VCARD_SIZE + 1 );
    I2C_start();
    I2C_select( RTC_ADDRESS, I2C_MODE_WRITE);
    I2C_write( RTC_ADDRESS_VCARD );
    I2C_repeated_start();
    while( count++ <= RTC_VCARD_SIZE - 1 ) {
        *str++ = I2C_read( I2C_NACK );
        if( ( *str == '\0' ) || ( *str == 0xFF ) ) break;
    }
    I2C_stop();
    *str = '\0';        // fuse
}    
#endif // USE_DS1307_VCARD
    

void RTC_shiftRtc()
{
    if( ++( rtcTime.tm_sec ) == 60 ){
        rtcTime.tm_sec = 0;;       
        if( ++( rtcTime.tm_min ) == 60 ) {
            rtcTime.tm_min = 0;
            if( ++( rtcTime.tm_hour ) > 23 ) {
                rtcTime.tm_hour = 0;
                if( ++( rtcTime.tm_wday ) > 7 ) rtcTime.tm_wday = 1;
                if( ++( rtcTime.tm_mday ) > DayofMonth[ rtcTime.tm_mon ] +
                    ( rtcTime.tm_mon == 2 && isLeapYear( rtcTime.tm_year ) ? 1 : 0 ) ) {
                    rtcTime.tm_mday = 1;
                    if( ++( rtcTime.tm_mon ) == 13 ) {
                        rtcTime.tm_mon = 1;
                        ++rtcTime.tm_year;
                    }  
                }
            }
        }
    }
}


void RTC_display_to_LCD(RTC_DISPLAY mode) 
{
    memset( DateStr, 0, LCD_COLS + 1 );
    memset( TimeStr, 0, LCD_COLS + 1 );
    sprintf( DateStr, "|c|%4d.%02d.%02d\n", rtcTime.tm_year, rtcTime.tm_mon, rtcTime.tm_mday ); //, &weekDayName[ rtcTime.tm_wday + 1 ] );
    sprintf( TimeStr, "|c|%2d:%02d:%02d\n", rtcTime.tm_hour, rtcTime.tm_min, rtcTime.tm_sec );

    switch( mode ) {
        case RTC_DISPLAY_DATE: LCD_send_string( DateStr, 3 ); break;
        case RTC_DISPLAY_TIME: LCD_send_string( TimeStr, 2 ); break;
        case RTC_DISPLAY_DATETIME:
            LCD_send_string( DateStr, 1 );
            LCD_send_string( TimeStr, 2 );
            break;
        default: ;    
    }
}


void RTC_display_to_UART(RTC_DISPLAY mode) 
{
    memset( DateStr, 0, LCD_COLS + 1 );
    memset( TimeStr, 0, LCD_COLS + 1 );
    sprintf( DateStr, "|c|%4d.%02d.%02d %s", rtcTime.tm_year, rtcTime.tm_mon, rtcTime.tm_mday, weekDayName[ rtcTime.tm_wday + 1 ] );
    sprintf( TimeStr, "|c|%2d:%02d:%02d", rtcTime.tm_hour, rtcTime.tm_min, rtcTime.tm_sec );

    if( mode == RTC_DISPLAY_DATE ) {
        printf( "%s\n\r\r", DateStr );
    }
    else if( mode == RTC_DISPLAY_TIME ) {
        printf( "%s\n\r\r", TimeStr );
    }
    else if( mode == RTC_DISPLAY_DATETIME ) {
        printf( "%s\n\r", DateStr );
        printf( "%s\n\r\r", TimeStr );
    }
}
