/*
 * drv_mcp2515.c
 */

#include <p18cxxx.h>
#include <stddef.h>	/* NULL */
#include <delays.h>

#include "types.h"
#include "drv_spi.h"
#include "drv_mcp2515.h"

#define DISABLE_CAN_INT()	(INTCON3bits.INT1IE = 0)
#define ENABLE_CAN_INT()	(INTCON3bits.INT1IE = 1)

/* Chip Select */
#define MCP2515_CS_PORT_ADDRESS (&PORTB)
#define MCP2515_CS_BIT (4)

/* Reset */
//#define MCP2515_RESET (PORTBbits.RB5)

static void CAN_Reset(void);
static void CAN_Mode(BYTE mode);
static BYTE CAN_ReadRegister(BYTE addr);
static void CAN_ReadSeveralRegisters(BYTE addr, BYTE len, BYTE *ptr);
static void CAN_WriteRegister(BYTE addr, BYTE val);
static void CAN_WriteSeveralRegisters(BYTE addr, BYTE len, BYTE *ptr);
static void CAN_ModifyRegister(BYTE addr, BYTE mask, BYTE data);
static void CAN_Write_Can_ID(BYTE addr, BYTE type, DWORD cid);
static void CAN_Read_Can_ID(BYTE addr, BYTE *type, DWORD *cid);
static void CAN_Tx(BYTE type, DWORD cid, BYTE rtr, BYTE *data, BYTE dlc);
static void CAN_Rx(BYTE RxBuffer, CAN_message_t *cf);

static unsigned char wr_p;
static unsigned char rd_p;
static unsigned char dataCnt;

#define CAN_BUF_SIZE (5)

static CAN_message_t can_buf[CAN_BUF_SIZE];
struct CANStatus CANStatus;

static SPI_message_t SPI_msg;
static BYTE mcp2515_cmd[4];

/*
 * CAN_Reset
 */
static void CAN_Reset(void)
{
#ifdef MCP2515_RESET
	MCP2515_RESET = 0;
	DelayMs(250);
	MCP2515_RESET = 1;
#else
	mcp2515_cmd[0] = MCP_RESET;
	SPI_msg.SDO[0].addr = &mcp2515_cmd[0];
	SPI_msg.SDO[0].len = 1;
	SPI_msg.SDO[1].len = SPI_msg.SDI.len = 0;
	SPI_Task(&SPI_msg);
#endif
}

/*
 * CAN_Mode, a CAN modok kozti valtasokat vegzi el
 */
static void CAN_Mode(BYTE mode)
{
	CAN_WriteRegister(CANCTRL, mode);
	while ((CAN_ReadRegister(CANSTAT) & 0xE0) != mode);
}

/*
 * CAN_ReadRegister, adott cimu regiszter olvasasa
 */
static BYTE CAN_ReadRegister(BYTE addr)
{
	BYTE ret;

	mcp2515_cmd[0] = MCP_READ;
	mcp2515_cmd[1] = addr;

	SPI_msg.SDO[0].addr = &mcp2515_cmd[0];
	SPI_msg.SDO[0].len = 2;
	SPI_msg.SDO[1].len = 0;
	SPI_msg.SDI.addr = &ret;
	SPI_msg.SDI.len = 1;
	SPI_Task(&SPI_msg);

	return ret;
}

/*
 * CAN_ReadSeveralRegisters, adott cimu regisztertol kezdve 'len' db regiszter olvasasa
 */
static void CAN_ReadSeveralRegisters(BYTE addr, BYTE len, BYTE *ptr)
{
	if (len && ptr) {		/* len != 0 && ptr != NULL */
		mcp2515_cmd[0] = MCP_READ;
		mcp2515_cmd[1] = addr;

		SPI_msg.SDO[0].addr = &mcp2515_cmd[0];
		SPI_msg.SDO[0].len = 2;
		SPI_msg.SDO[1].len = 0;
		SPI_msg.SDI.addr = ptr;
		SPI_msg.SDI.len = len;
		SPI_Task(&SPI_msg);
	}
}

/*
 * CAN_WriteRegister, adott cimu regiszter irasa
 */
static void CAN_WriteRegister(BYTE addr, BYTE val)
{
	mcp2515_cmd[0] = MCP_WRITE;
	mcp2515_cmd[1] = addr;
	mcp2515_cmd[2] = val;

	SPI_msg.SDO[0].addr = &mcp2515_cmd[0];
	SPI_msg.SDO[0].len = 3;
	SPI_msg.SDO[1].len = SPI_msg.SDI.len = 0;
	SPI_Task(&SPI_msg);
}

/*
 * CAN_WriteSeveralRegisters, adott cimu regisztertol kezdve 'len' db regiszter irasa
 */
static void CAN_WriteSeveralRegisters(BYTE addr, BYTE len, BYTE *ptr)
{
	if (len && ptr) {		/* len != 0 && ptr != NULL */
		mcp2515_cmd[0] = MCP_WRITE;
		mcp2515_cmd[1] = addr;

		SPI_msg.SDO[0].addr = &mcp2515_cmd[0];
		SPI_msg.SDO[0].len = 2;
		SPI_msg.SDO[1].addr = ptr;
		SPI_msg.SDO[1].len = len;
		SPI_msg.SDI.len = 0;
		SPI_Task(&SPI_msg);
	}
}

/*
 * CAN_ModifyRegister
 * ahol a mask bitjei 1-ben vannak, ott a data bitjei ervenyesulnek
 * ahol a mask bitjei 0-ban vannak, ott maradnak az eredeti bitek
 */
static void CAN_ModifyRegister(BYTE addr, BYTE mask, BYTE data)
{
	mcp2515_cmd[0] = MCP_BITMOD;
	mcp2515_cmd[1] = addr;
	mcp2515_cmd[2] = mask;
	mcp2515_cmd[3] = data;

	SPI_msg.SDO[0].addr = &mcp2515_cmd[0];
	SPI_msg.SDO[0].len = 4;
	SPI_msg.SDO[1].len = SPI_msg.SDI.len = 0;
	SPI_Task(&SPI_msg);
}

/* CAN_Tx
 */
static void CAN_Tx(BYTE type, DWORD cid, BYTE rtr, BYTE *data, BYTE dlc)
{
	BYTE addr, canstat;
	int tries = 5;				/* a probalkozasok szama */
	bool foundOK = false;

	while (tries-- > 0) {
		DISABLE_CAN_INT();
		addr = TXB0CTRL;
		while (addr <= TXB2CTRL) {
			canstat = CAN_ReadRegister(addr);
			if (canstat & TXREQ)
				addr += 0x10;		/* tart meg a kuldes, nezzuk a kovetkezot */
			else
				break;
		}
		ENABLE_CAN_INT();

		if (addr > TXB2CTRL) {
			Delay10KTCYx(5);		/* 10 milisec */
		} else {
			foundOK = true;
			tries = 0;
		}
	}

	if (!foundOK) {	/* nem volt szabad puffer, nincs mit tenni */
		CANStatus.CAN_SEND_TIMEOUT = 1;
		return;
	}

	if (dlc > 8)
		dlc = 8;

	DISABLE_CAN_INT();
	CAN_Write_Can_ID(addr + 1, type, cid);
	CAN_WriteRegister(addr + 5, (rtr ? (RTR_MASK | dlc) : dlc));
	CAN_WriteSeveralRegisters(addr + 6, dlc, data);
	CAN_ModifyRegister(addr, TXREQ | HIGH_PRIO, TXREQ | HIGH_PRIO);		/* RTS */
	ENABLE_CAN_INT();
}

/*
 * CAN_Rx, RxBuffer = 0 vagy 1 (vagy 1-nel nagyobb)
 * ha 1 (vagy 1-nel nagyobb), akkor az 1-es Buffer-t olvassa
 */
static void CAN_Rx(BYTE RxBuffer, CAN_message_t *cm)
{
	BYTE addr;

	addr = (RxBuffer ? RXB1CTRL : RXB0CTRL);
	CAN_Read_Can_ID(addr + 1, &cm->type, &cm->id);
	cm->rtr = ((CAN_ReadRegister(addr) & RXRTR) ? 1 : 0);
	cm->dlc = CAN_ReadRegister(addr + 5) & 0x0F;
	CAN_ReadSeveralRegisters(addr + 6, cm->dlc, cm->data);
}

/*
 * CAN_Init
 */
void CAN_Init(BYTE CNF1_data, BYTE CNF2_data, BYTE CNF3_data)
{
	DISABLE_CAN_INT();

//	LATBbits.LATB4 = 1;
//	TRISBbits.TRISB4 = 0;	/* kimenet */
//	TRISBbits.TRISB1 = 1;	/* bemenet */

	INTCON3bits.INT1IP = 0;		/* alacsony prioritasu */
	INTCON2bits.INTEDG1 = 0;	/* lefuto elre */
	INTCON3bits.INT1IF = 0;
	INTCON3bits.INT1IE = 1;

	SPI_msg.SpeedBits = 0x01;	/* 1.25MHz = 0x01 */
	SPI_msg.expected_char_mode = false;
	SPI_msg.SDIChar = 0xFF;
	SPI_msg.CS_address = (ram char *)MCP2515_CS_PORT_ADDRESS;
	SPI_msg.CS_bit = MCP2515_CS_BIT;
	SPI_msg.SDO[2].len = 0;		/* mindig 0 */

	CAN_Reset();
	CAN_Mode(MODE_CONFIG);

	/* baudrate beallitas */
	CAN_WriteRegister(CNF1, CNF1_data);
	CAN_WriteRegister(CNF2, CNF2_data);
	CAN_WriteRegister(CNF3, CNF3_data | SOF_DISABLE | WAKFIL_DISABLE);

	/* TXnRTS labak digitalis bemenetek lesznek */
	CAN_WriteRegister(TXRTSCTRL, 0x00);

	/* maszkok/szurok kikapcsolasa es ROLLOVER engedelyezese */
	CAN_WriteRegister(RXB0CTRL, RXB0CTRL_RXM_OFF | RXB0CTRL_BUKT);

	/* maszkok/szurok kikapcsolasa */
	CAN_WriteRegister(RXB1CTRL, RXB1CTRL_RXM_OFF);

	/* maszkok tiltasa */
	CAN_Write_Can_ID(RXM0SIDH, CAN_EXT_ID, 0);
	CAN_Write_Can_ID(RXM1SIDH, CAN_EXT_ID, 0);

	/* szurok tiltasa */
	CAN_Write_Can_ID(RXF0SIDH, CAN_EXT_ID, 0);
	CAN_Write_Can_ID(RXF1SIDH, CAN_EXT_ID, 0);
	CAN_Write_Can_ID(RXF2SIDH, CAN_EXT_ID, 0);
	CAN_Write_Can_ID(RXF3SIDH, CAN_EXT_ID, 0);
	CAN_Write_Can_ID(RXF4SIDH, CAN_EXT_ID, 0);
	CAN_Write_Can_ID(RXF5SIDH, CAN_EXT_ID, 0);

	/* interrupt */
	CAN_WriteRegister(CANINTE, ERRIE | RX1IE | RX0IE);

	/* RXnBF lbak */
	CAN_WriteRegister(BFPCTRL, 0x00);

	//CAN_Mode(MODE_LOOPBACK | ONESHOT_DISABLE);
	CAN_Mode(MODE_NORMAL | ONESHOT_DISABLE);

	wr_p = 0;
	rd_p = 0;
	dataCnt = 0;
	CANStatus.RxBufferEmpty = 1;	// alapban ures
	CANStatus.RxBufferFull = 0;	// alapban biztosan nincs tele
	CANStatus.RxOverFlow = 0;		// alapban nincs rafutas
	CANStatus.CAN_ERROR = 0;
	CANStatus.CAN_SEND_TIMEOUT = 0;

	ENABLE_CAN_INT();
}

/*
 * CAN_Write_Can_ID
 */
static void CAN_Write_Can_ID(BYTE addr, BYTE type, DWORD cid)
{
	BYTE buf[4];

	if (type == CAN_STD_ID) {
		buf[SIDL] = cid << 5;
		buf[SIDH] = cid >> 3;
	} else {
		WORD cid2;

		buf[EID0] = cid;
		buf[EID8] = cid >> 8;
		cid2 = cid >> 16;
		buf[SIDL] = cid2 & 0x03;
		buf[SIDL] |= (cid2 & 0x1C) << 3;
		buf[SIDL] |= EXIDE;
		buf[SIDH] = cid2 >> 5;
	}
	CAN_WriteSeveralRegisters(addr, 4, buf);
}

/*
 * CAN_Read_Can_ID
 */
static void CAN_Read_Can_ID(BYTE addr, BYTE *type, DWORD *cid)
{
	BYTE buf[4];
	DWORD id;

	*type = CAN_STD_ID;
	CAN_ReadSeveralRegisters(addr, 4, buf);
	id = ((DWORD)buf[SIDL] >> 5) + ((DWORD)buf[SIDH] << 3);
	if (buf[SIDL] & EXIDE) {
		*type = CAN_EXT_ID;
		id = (id << 2) + ((DWORD)buf[SIDL] & 0x03);
		id <<= 16;
		id += (DWORD)buf[EID8] << 8;
		id += (DWORD)buf[EID0];
	}
	*cid = id;
}

/*
 * CANRTRPacket
 */
void CANRTRPacket(DWORD id)
{
	CAN_Tx(CAN_EXT_ID, id, RTR_PACKET, NULL, 0);
}

/*
 * CANDATAPacket
 */
void CANDATAPacket(DWORD id, BYTE *ptr, BYTE len)
{
	CAN_Tx(CAN_EXT_ID, id, DATA_PACKET, ptr, len);
}

/*
 * MCP2515_INT_handler, IT-bol !!!
 */
void MCP2515_INT_handler(void)
{
	CAN_message_t temp;
	BYTE intf = CAN_ReadRegister(CANINTF);

	if (intf & RX0IF) {
		if (CANStatus.RxBufferFull) {
			CAN_Rx(0, &temp);
			CANStatus.RxOverFlow = 1;	// tulcsordulas
		} else {
			CANStatus.RxOverFlow = 0;
			CANStatus.RxBufferEmpty = 0;
			CAN_Rx(0, &can_buf[wr_p]);
			dataCnt++;
			if (dataCnt == CAN_BUF_SIZE)
				CANStatus.RxBufferFull = 1;
			wr_p++;
			if (wr_p == CAN_BUF_SIZE)
				wr_p = 0;
		}
		CAN_ModifyRegister(CANINTF, RX0IF, 0);
	}

	if (intf & RX1IF) {
		if (CANStatus.RxBufferFull) {
			CAN_Rx(1, &temp);
			CANStatus.RxOverFlow = 1;	// tulcsordulas
		} else {
			CANStatus.RxOverFlow = 0;
			CANStatus.RxBufferEmpty = 0;
			CAN_Rx(1, &can_buf[wr_p]);
			dataCnt++;
			if (dataCnt == CAN_BUF_SIZE)
				CANStatus.RxBufferFull = 1;
			wr_p++;
			if (wr_p == CAN_BUF_SIZE)
				wr_p = 0;
		}
		CAN_ModifyRegister(CANINTF, RX1IF, 0);
	}

	if (intf & ERRIF) {
		BYTE err;

		err = CAN_ReadRegister(EFLG);
		if (err & RX0OVR)
			CAN_ModifyRegister(EFLG, RX0OVR, 0);
		if (err & RX1OVR)
			CAN_ModifyRegister(EFLG, RX1OVR, 0);
		CAN_ModifyRegister(CANINTF, ERRIF, 0);
		CANStatus.CAN_ERROR = 1;
	}
}

unsigned char CANIntGetPacket(CAN_message_t *ret)
{
	if (CANStatus.RxBufferEmpty)
		return 0;	/* nincs a puffer-ben csomag */

	DISABLE_CAN_INT();
	CANStatus.RxBufferFull = 0;	/* egyet kiveszunk a puffer-bol, tehat mar nincs tele */
	*ret = can_buf[rd_p];
	rd_p++;
	if (rd_p == CAN_BUF_SIZE)
		rd_p = 0;
	dataCnt--;
	if (dataCnt == 0)
		CANStatus.RxBufferEmpty = 1;
	ENABLE_CAN_INT();

	return 1;
}
