/* PICCOLO project
 * Copyright (c) 2009-2010 Istvan Cserny (cserny@atomki.hu)
 *
 */

#include "piccolo_all.h"

/** \file
LED vezrlse nyomgombbal s kapcsolval. 

Ez a program a vges llapotgp megkzeltssel valstja meg a LED1 kijelz 
az SW1 nyomgombbal s az SW3 kapcsolval trtn vezrlst az albbiak szerint:
   -# LED1-et kikapcsoljuk
   -# Az SW1 nyomgomb megnyomst s elengedst kveten LED1-et bekapcsoljuk.
   -# Az SW1 nyomgomb jabb megnyomst s elengedst kveten a programot az SW3 llstl 
fggen gy folytatdik: 
          - Ha SW3 =0 (a kapcsol zr), akkor az 1. pontnl folytatjuk
          - Ha SW3 = 1 (a kapcsol nyitott), akkor a 4. pontnl folytatjuk
   -# Villogtatjuk LED1-et (LED1 llapotnak tbillentse utn 100 ms vrakozs kvetkezzen)
   -# Az SW1 nyomgomb megnyomst kveten bekapcsoljuk LED1-et (stabilan vilgt)
   -# Az SW1 nyomgomb felengedst kveten a program az 1. pontnl folytatdik

Hardver igny:
   - PICCOLO projekt alapkapcsols PIC18F14K50 vagy PIC18F4550 mikrovezrlvel,
     esetleg a Microchip Low Pin Count USB vagy a PICDEM Full Speed USB demkrtya
   - USB kapcsolat
   - SW3 kapcsol vagy jumper az RB5 kivezets s VSS (fld) kz.
*/

/*** INTERRUPT KISZOLGL ELJRSOK **********************************/
#if defined(USE_INTERRUPT)
	///Magas priorits programmegszakts kiszolglsa
	#pragma interrupt hi_isr
	void hi_isr() {
	#if defined(USB_INTERRUPT)
	    USBDeviceTasks(); //--- USB programmegszakts kiszolglsa
	#endif  //USB_INTERRUPT
	}
	///Alacsony priorits programmegszakts kiszolglsa
	#pragma interruptlow lo_isr
	void lo_isr() {
	}
#endif //USE_INTERRUPT


//-- A lehetsges llapotok felsorolsa 
typedef enum  { 
  STATE_RESET = 0,              //Reset
  STATE_WAIT_FOR_PRESS1,        //Els lenyomsra vr
  STATE_WAIT_FOR_RELEASE1,      //Els felengedsre vr
  STATE_WAIT_FOR_PRESS2,        //Msodik lenyomsra vr
  STATE_WAIT_FOR_RELEASE2,      //Msodik felengedsre vr
  STATE_BLINK,                  //Villogs
  STATE_WAIT_FOR_RELEASE3       //Harmadik felengedsre vr
} STATE;

STATE e_LastState = STATE_RESET;

/** Nyomkvet zenet kirsa minden llapotvltskor. 
 * Az elz llapotot a globlis e_LastState vltoz rzi.
 * Kirs akkor trnik, ha az aktulis llapot klnbzik az elz llapottl.
 * \param STATE e_currentState - az aktulis llapot
 */
void printNewState (STATE e_currentState) {
  if (e_LastState != e_currentState) {
    switch (e_currentState) {
      case STATE_WAIT_FOR_PRESS1:
        outString("STATE_WAIT_FOR_PRESS1 - LED is off\n");
        break;
      case STATE_WAIT_FOR_RELEASE1:
        outString("STATE_WAIT_FOR_RELEASE1\n");
        break;
      case STATE_WAIT_FOR_PRESS2:
        outString("STATE_WAIT_FOR_PRESS2 - LED is on\n");
        break;
      case STATE_WAIT_FOR_RELEASE2:
        outString("STATE_WAIT_FOR_RELEASE2 - on SW3 go to blink else go to PRESS1\n");
        break;
      case STATE_BLINK:
        outString("STATE_BLINK - LED blinks, waiting for SW1 press\n");
        break;
      case STATE_WAIT_FOR_RELEASE3:
        outString("STATE_WAIT_FOR_RELEASE3 - LED is on\n");
        break;
      default:
        break;
    }
  }
  e_LastState = e_currentState;  //megjegyezzk az utols llapotot
}

#define SW1_PRESSED()   SW1==0  //lenyomott llapot felttele
#define SW1_RELEASED()  SW1==1  //felengedett llapot felttele

//-- SW3 kapcsol definilsa
#define SW3 PORTBbits.RB5		//SW3 kapcsol RB5-re csatlakozik

//-- A fprogram -------------------------------------------
void main(void) {
	unsigned char c;			//LED villogtatshoz szmll
	STATE e_mystate;			//aktulis llapot trolja
	InitializeSystem();			//USB, LED-ek s SW1 inicializlsa

//-- SW3 kapcsol inicializlsa
	TRISBbits.TRISB5=1;			//RB5 legyen bemenet
#if defined(__18F14K50)
	//-- PIC18F14K50 esetben RB5 analg funkcijt le kell tiltani
	//-- A bels felhzs bitenknt s a portra nzve is belltand
	ANSELHbits.ANS11=0;			//RB5 analg funkci letiltsa
	WPUBbits.WPUB5=1;			//RB5 felhzs bekapcsolsa
	INTCON2bits.RABPU=0;		//Felhzs ltalnos engedlyezse
#else
	//-- PIC18F4550 esetben RB5 mindig digitlis ki/bemenet
	//-- PORTB bels felhzs bitenknt nem llthat
	INTCON2bits.RBPU=0;			//PORTB felhzsainak engedlyezse
#endif
	
//-- kivrjuk az els karakter berkezst
    while (!usb_cdc_kbhit()) {
        ProcessIO();
    }
    outString("LED kapcsolgatsa nyomgombbal\n");
	BlinkUSBStatus_enabled=0;	//az USB sttusz kijelzst letiltjuk
	LEDport = 0;				//minden LED kikapcsolsa
	
  e_mystate = STATE_WAIT_FOR_PRESS1; //kezdllapot megadsa
  while (1) {
    printNewState(e_mystate);		//llapotvltskor nyomkvet zenet kirsa
    switch (e_mystate) {
      case STATE_WAIT_FOR_PRESS1:
        mLED_1_Off();				//kikapcsolja a LED-et
        if (SW1_PRESSED()) e_mystate = STATE_WAIT_FOR_RELEASE1;
        break;
     case STATE_WAIT_FOR_RELEASE1:
        if (SW1_RELEASED()) e_mystate = STATE_WAIT_FOR_PRESS2;
        break;
      case STATE_WAIT_FOR_PRESS2:
        mLED_1_On();				//bekapcsolja a LED-et
        if (SW1_PRESSED()) e_mystate = STATE_WAIT_FOR_RELEASE2;
        break;
     case STATE_WAIT_FOR_RELEASE2:
        if (SW1_RELEASED()) {
        //SW3 llapottl fgg, hogy merre folytassuk
           if(SW3) { 
		     e_mystate = STATE_BLINK;				//LED1 villogtats, ha SW3 nyitva
		     c=1;									//LED villogtats szmllja
		   }
           else e_mystate = STATE_WAIT_FOR_PRESS1;	//LED1 kikapcsols, ha SW3 zrt
        }
        break;
      case STATE_BLINK:
		if(--c==0) {
           mLED_1_Toggle();			//villogtatja a LED-et
		   c=10;					//ksleltets = 10 * 20 ms
		}
        if (SW1_PRESSED()) e_mystate = STATE_WAIT_FOR_RELEASE3;
        break;
     case STATE_WAIT_FOR_RELEASE3:
        mLED_1_On();				//a LED llapott befagyasztjuk '1'-re
        if (SW1_RELEASED()) e_mystate = STATE_WAIT_FOR_PRESS1;
        break;
      default:
        e_mystate = STATE_WAIT_FOR_PRESS1;
    } 							//switch(e_mystate)  vge
    delay_ms(20);				//pergsmentest ksleltets
   }							//a while (1) ciklustrzs vge
}
