/* PICCOLO project
 * Copyright (c) 2009-2010 Istvan Cserny (cserny@atomki.hu)
 *
 */

#include "piccolo_all.h"
#define USE_OR_MASKS			//timers.h-ben "VAGY" mvelettel egyesthet makrk legyenek
#include "timers.h"				//makrk s fggvnydek az idztkhz 


/** \file
* Annak bemutatsa, hogyan hasznlhatjuk a periodikus programmegszaktst
* egy nyomgomb pergsmentestsre, feleslegess tve a pergsmentest ksleltetsek 
* alkalmazst. Az llapotgp a fprogramban van megvalstva, de a nyilvntartott 
* llapotok szmt cskkentettk azltal, hogy a nyomgomb megnyoms s flengeds 
* figyelst egy msik llapotgp vgzi, ami az interrupt kiszolgl eljrsban van 
* megvalstva, s egy szemafor lltsval jelzi a fprogramnak a nyomgomb lenyoms 
* s felengs ciklusnak bekvetkeztt. A szemafort az ISR lltja '1'-be s a 
* fprogram trli.
*
* A program vges llapotgp megkzeltssel valstja meg a LED1 kijelznek 
* az SW1 nyomgombbal s az SW3 kapcsolval trtn vezrlst az albbiak szerint:
*   -# LED1-et kikapcsoljuk
*   -# Az SW1 nyomgomb megnyomst s elengedst kveten LED1-et bekapcsoljuk.
*   -# Az SW1 nyomgomb jabb megnyomst s elengedst kveten a programot az SW3 llstl 
*    fggen gy folytatdik: 
*          - Ha SW3 = 0 (a kapcsol zr), akkor az 1. pontnl folytatjuk
*          - Ha SW3 = 1 (a kapcsol nyitott), akkor a 4. pontnl folytatjuk
*   -# Villogtatjuk LED1-et (LED1 llapotnak tbillentse utn 100 ms vrakozs kvetkezzen)
*   -# Az SW1 nyomgomb megnyomst kveten bekapcsoljuk LED1-et (stabilan vilgt)
*   -# Az SW1 nyomgomb felengedst kveten a program az 1. pontnl folytatdik
*
* Hardver igny:
*   - PICCOLO projekt alapkapcsols PIC18F14K50 vagy PIC18F4550 mikrovezrlvel,
*     esetleg a Microchip Low Pin Count USB vagy a PICDEM Full Speed USB demkrtya
*   - USB kapcsolat
*   - SW3 kapcsol vagy jumper az RB5 kivezets s VSS (fld) kz.
*
*  Projekt konfigurci: A program mkdshez a HID_BOOTLOADER szimblumon kvl  
*  az USE_INTERRUPT s az USE_USB szimblumokat is definilni kell.

*/

#define SW1_PRESSED()  SW1_minta==0 //lenyomott llapot felttele
#define SW1_RELEASED() SW1_minta==1 //felengedett llapot felttele

//-- SW3 kapcsol definilsa
#define SW3 PORTBbits.RB5			///<SW3 kapcsol RB5-re csatlakozik

volatile unsigned char delay, 		///<ksleltets szmllja
					SW1_minta, 		///<a nyomgomb mintavtelezett llapota
					pnrSW1;			///<Press & Release szemafor

typedef enum  {
  WAIT_FOR_PRESS = 0, 
  WAIT_FOR_RELEASE
} ISRSTATE;

ISRSTATE isrState=WAIT_FOR_PRESS;	///<ISR llapota

//-- A fprogram lehetsges llapotainak felsorolsa 
typedef enum  { 
  STATE_RESET = 0,              	//Kezdllapot
  STATE_WAIT_FOR_PNR1,        		//Els lenyomsra vr
  STATE_WAIT_FOR_PNR2,        		//Msodik lenyomsra vr
  STATE_BLINK,                  	//Villogs
  STATE_WAIT_FOR_RELEASE3       	//Harmadik felengedsre vr
} MAINSTATE;


MAINSTATE LastState = STATE_RESET;	///<Az utoljra kirt llapot	

#pragma interrupt hi_isr
void hi_isr(void) {
//-- TIMER0 megszakts kiszolglsa --------
	if(INTCONbits.TMR0IF && INTCONbits.TMR0IE) {
	  SW1_minta = SW1;     			//mintavtelezi a nyomgomb llapott
	  switch (isrState) {
	    case WAIT_FOR_PRESS:
	      if (SW1_PRESSED() && pnrSW1==0)	//nem dolgozunk fel jabb gombnyomst, 
	        isrState = WAIT_FOR_RELEASE;	//mg a szemafor nincs felhasznlva!
	      break;
	    case WAIT_FOR_RELEASE:
	      if (SW1_RELEASED()) {
	        isrState = WAIT_FOR_PRESS;
	        pnrSW1 = 1; 			//belltja a szemafort (pnr = Pressed aNd Released)
	        break;
	      }
	    default:
	      isrState = WAIT_FOR_RELEASE;
	  }
	  INTCONbits.TMR0IF = 0;		//trli az interrupt jelzbitet
	  if(delay) delay--;			//szmll cskkents, amg delay > 0
	}
//-- Az USB megszaktsok kiszolglsa ----------
#if defined(USB_INTERRUPT)
    USBDeviceTasks(); 				//--- USB programmegszakts kiszolglsa
#endif  //USB_INTERRUPT
}

#pragma interruptlow lo_isr
	void lo_isr() {
	}

/** Nyomkvet zenet krsa minden llapotvltskor
 * \param currentState a fprogram aktulis llapota
 * \param LastState (globlis vltoz) a fprogram utoljra kirt llapota
 */
void printNewState (MAINSTATE currentState) { 
  if (LastState != currentState) {
    switch (currentState) {
      case STATE_WAIT_FOR_PNR1:
        outString("STATE_WAIT_FOR_PNR1 - LED1 is off\n");
        break;
      case STATE_WAIT_FOR_PNR2:
        outString("STATE_WAIT_FOR_PNR2 - LED1 is on\n");
        break;
      case STATE_BLINK:
        outString("STATE_BLINK - LED1 blinks, waiting for SW1 press\n");
        break;
      case STATE_WAIT_FOR_RELEASE3:
        outString("STATE_WAIT_FOR_RELEASE3 - LED is on\n");
        break;
    }
  }
  LastState = currentState;    //megjegyzi a legutbbi llapotot
}

//-- A fprogram -------------------------------------------
void main(void) {
	MAINSTATE mystate;			//a fprogram llapotjelzje
    mystate = STATE_RESET;		//a fprogram kezdllapota
	isrState=WAIT_FOR_PRESS;	//ISR kezdllapota
	pnrSW1 = 0;					//a szemafor trlse
//-- A megszaktsi rendszer inicializlsa -----------------
	RCONbits.IPEN=1;			//Ktszint megszaktsi md belltsa
	INTCONbits.GIEH=1;			//A magas priorits interrupt engedlyezse
	INTCONbits.GIEL=0;			//Az alacsony priorits interrupt tiltsa
    InitializeSystem();			//USB, LEDport s SW1 inicializlsa
//-- Timer0 konfigurlsa periodikus programmegszaktshoz (~20 ms peridussal)
	INTCON2bits.TMR0IP=1;		//Timer0 megszakts magas priorits legyen
	OpenTimer0(TIMER_INT_ON 
				| T0_16BIT 
				| T0_SOURCE_INT 
				| T0_PS_1_4 );
//-- SW3 kapcsol inicializlsa
	TRISBbits.TRISB5=1;			//RB5 legyen bemenet
#if defined(__18F14K50)
	//-- PIC18F14K50 esetben RB5 analg funkcijt le kell tiltani
	//-- A bels felhzs bitenknt s a portra nzve is belltand
	ANSELHbits.ANS11=0;			//RB5 analg funkci letiltsa
	WPUBbits.WPUB5=1;			//RB5 felhzs bekapcsolsa
	INTCON2bits.RABPU=0;		//Felhzs ltalnos engedlyezse
#else
	//-- PIC18F4550 esetben RB5 mindig digitlis ki/bemenet
	//-- PORTB bels felhzs bitenknt nem llthat
	INTCON2bits.RBPU=0;			//PORTB felhzsainak engedlyezse
#endif
    //-- kivrjuk az els karakter berkezst
    while (!usb_cdc_kbhit()) {
        ProcessIO();
    }
	#if defined(__18F4550)
		outString("PICCOLO-4550");
	#elif defined(__18F14K50)
		outString("PICCOLO-14K50");
	#endif
    outString(" - ledswitch_tmr0.c program\n");
	BlinkUSBStatus_enabled=0;	//az USB sttusz kijelzst letiltjuk
	LEDport = 0;				//minden LED kikapcsolsa

/* Figyeljk meg, hogy a perodikus programmegszaktskor trtn mintavtelezs
 * flslegess teszi a pergsmentest ksleltetst a while ciklus vgn!
 */
  while (1) {
    printNewState(mystate);  		//nyomkvet kirs minden llapotvltskor
    switch (mystate) {
//-- Els gomblenyomsra vrunk ---------------
    case STATE_WAIT_FOR_PNR1:
      mLED_1_Off(); 				//kikapcsolja LED1-et
      if (pnrSW1) {
         pnrSW1 = 0; 				//trli a szemafort
         mystate = STATE_WAIT_FOR_PNR2;	//jabb gomlenyomsra vr
      }
      break;
//-- Msodik gomblenyomsra vrunk ------------
    case STATE_WAIT_FOR_PNR2:
      mLED_1_On(); 					//bekapcsolja LED1-et
      if (pnrSW1) {
        pnrSW1 = 0; 				//trli a szemafort
//-- SW3 llapota dnti el, hogy merre kell folytatni a programot
        if (SW3) mystate = STATE_BLINK;	//LED1 villogtats
        else mystate = STATE_WAIT_FOR_PNR1;
      }
      break;
//-- Villog zemmd ---------------------------
    case STATE_BLINK:
	  if(delay==0) {
        mLED_1_Toggle();     		//villogtatja LED1-et lenyomsig
        delay=12;		    		//villogtatsi ksleltets ~250 ms
      }
      //-- Villogs kzben a harmadik lenyomsra vrunk
      if (SW1_PRESSED()) mystate = STATE_WAIT_FOR_RELEASE3;
      break;
//-- Harmadik felengedsre vrunk ------
    case STATE_WAIT_FOR_RELEASE3:
      mLED_1_On();   				//Bekapcsolva hagyja a LED-et felengedsig
      //Figyelem! Itt pnrSW1-re kell vrni SW1_RELEASED helyett, mert
      //pnrSW1 felengedskor belltdik, amit TRLNI KELL!
      if (pnrSW1) {
        pnrSW1 = 0;
        mystate = STATE_WAIT_FOR_PNR1;
      }
      break;
//-- csak STATE_RESET esetn tved ide a program ----
    default:
      mystate=STATE_WAIT_FOR_PNR1;	//Kezdskor az els gombnyomsra vrjon
    } //switch(mystate) vge
//-- A fprogram egyb tevkenysge: letben tartjuk az USB kommunikcit
	ProcessIO();
  } // while(1) ciklus vge
}

