/* PICCOLO project
 * Copyright (c) 2009-2010 Istvan Cserny (cserny@atomki.hu)
 *
 */

#include "piccolo_all.h"

/** \file
* Ksrletezs az ADC s a feszltsgreferencik (FVR, 5 bites DAC) konfigurlshoz.
* Az egyszer parancsrtelmez az albbi utastsokat fogadja el:
*  - #Akkmmnn - ADC konfigurlsa (ADCON0=kk, ADCON1=MM, ADCON2=nn)
*  - #Dnn - LEDport belltsa nn rtkre
*  - #H - segt szveg kirsa
*  - #L - LED villogtats tiltsa/engedlyezse
*  - #Mnn - az ADC nn. csatornjnak kiolvassa ("mrs")
*  - #Vkkmmnn - a VREF modul konfigurlsa (REFCON0=kk, REFCON1=MM, REFCON2=nn)
*
* Hardver igny:
*   - PICCOLO projekt alapkapcsols PIC18F14K50 vagy PIC18F4550 mikrovezrlvel,
*     esetleg a Microchip Low Pin Count USB vagy a PICDEM Full Speed USB demkrtya
*   - USB kapcsolat
*
*Projekt konfigurci: A program mkdshez a HID_BOOTLOADER szimblumon kvl  
*  az USE_INTERRUPT s az USE_USB szimblumokat is definilni kell!
*/

//-- Projekt konfigurci ellenrzse 
#if !defined(USE_INTERRUPT) 
	#error "Nincs definilva az USE_INTERRUPT szimblum!"
#endif //USE_INTERRUPT
#if !defined(USE_USB)
	#error "Nincs definilva az USE_USB szimblum!"
#endif //USE_USB
#if !defined(__18F14K50) 
	#error "Ez a program PIC18F14K50 mikrovezrlre kszlt!"
#endif //__18F14K50

//-- Magas priorits programmegszakts kiszolglsa
#pragma interrupt hi_isr
void hi_isr() {
    USBDeviceTasks(); //--- USB programmegszakts kiszolglsa
}
//-- Alacsony priorits programmegszakts kiszolglsa
#pragma interruptlow lo_isr
void lo_isr() {
}

/** A PIC18F14K50 mikrovezrl AN10 s AN11 csatornjnak engedlyezses, s az
 *  ADC inicializlsa (AN10 kivlasztsa, rajel FOSC/64, TACQ = 20TAD,
 *  konverzis eredmny jobbra igaztsa). 
 */ 
void InitADC(void)  {
	CONFIG_AN10_AS_ANALOG();		//AN10(RB4) analg bemenet belltsa 
	CONFIG_AN11_AS_ANALOG();		//AN11(RB5) analg bemenet belltsa 
	ADCON0=0x28;					//AN10 kivlaszts, ADC mg letiltva
	ADCON1=0;						//VREF+ = VDD, VREF- = VSS
    ADCON2=0xBE;					//20TAD, FOSC/64, jobbra igazts	
	ADCON0bits.ADON=1;				//Az ADC engedlyezse
}

/** Egy analg csatorna kivlasztsa, s megmrse.
 *  A fggvny blokkol tpus: kivrja a mrs vgt (~46 us).
 * \param chan a kivlasztott csatorna AN0, AN1, ... stb. jells szerinti sorszma
 */ 
unsigned int ReadADC(unsigned char chan) { 
	ChangeBits(ADCON0, chan<<2, 0b00111100); 
	ADCON0bits.GO = 1;              // Start AD conversion
	while (ADCON0bits.NOT_DONE);    // Wait for conversion
	return (((unsigned int)ADRESH)<<8)|(ADRESL);
}

void main(void) {
unsigned int adat;
char c;
    InitializeSystem();
	InitADC();						//ADC konfigurlsa
	REFCON0=FVR_ENABLE|FVR1S_2048;	//FVR inicializlsa 2.048 V-ra
	while(!FVR_STATUS);				//kivrjuk, amg FVR stabil lesz
	REFCON1=DAC_ENABLE | 
			D1LPS_LO |				//Energiatakarkos mdban a kimenet = VSS 
			DACOUT_DISABLE |		//DAC csak bels hasznlatra
			D1PSS_FVR |				//VREF+ = FVR
			D1NSS_VSS;				//VREF- = VSS				
	REFCON2=0x10;					//DAC = a maximlis rtk fele
    while (!usb_cdc_kbhit()) {
        ProcessIO();                //-- Megvrjuk az els karakter rkezst
    }
    outString("PICCOLO-14K50 adc.c program\n");
    while (1) {
//--- j parancsra vrunk. A parancs #-jellel kezddik
        do { 
			c=usb_cdc_getc();
        } while (c!='#');
		c=usb_cdc_getc(); 			//Parancskarakter beolvassa
		usb_cdc_putc(c);			//Parancskarakter tkrzse
		switch (c) {
		case 'A':    				//ADC modul konfigurlsa
			ADCON0=get2hex();
			ADCON1=get2hex();
			ADCON2=get2hex();
		break;
        case 'D':    				//LEDport belltsa
            LEDport=get2hex();
            break;
        case 'H':    				//Help kirsa
            outString("elp:\n");
            outString("#Akkmmnn - ADC konfigurlsa\n");
            outString("#Dnn - LEDport belltsa\n");
            outString("#H - help kirsa\n");
            outString("#L - LED villogtats tilts/engedlyezs\n");
            outString("#Mnn - ADC(nn) csatorna kiolvassa\n");
            outString("#Vkkmmnn - VREF modul konfigurlsa\n");
            break;
        case 'L':    				//LED villogs tilts/engedlyezs
            BlinkUSBStatus_enabled=!BlinkUSBStatus_enabled;
            if (BlinkUSBStatus_enabled) {
                usb_cdc_putc('1');
            } else {
                usb_cdc_putc('0');
            }
            break;
		case 'M':					//Kivlasztott csatorna mrse
			c=get2hex();			//csatorna sorszma
			usb_cdc_putc(' ');
			adat=ReadADC(c);		//csatorna mrse
			out4hex(adat);			//eredmny kiratsa
		break;			
		case 'V':    				//VREF modul konfigurlsa
			REFCON0=get2hex();
			REFCON1=get2hex();
			REFCON2=get2hex();
		break;
		} //switch
		outString("\n");
    } //while
}     //main
