/* PICCOLO project
 * Copyright (c) 2009-2010 Istvan Cserny (cserny@atomki.hu)
 *
 */

#include "piccolo_all.h"
#include "delays.h"

/** \file
 *  breszts alvs zemmdbl az INT2 interrupt segtsgvel.
 *  Ebben a programban az elektronikusan ki-s bekacsolhat kszlkek 
 *  (pl. TV, video, mobiltelefon) mkdst modellezzk. A kszlk 
 *  mkdst LED4 villogsa fogja jelkpezni, a ki- bekapcsolst pedig az
 *  INT2 bemenetre kttt nyomgombbal vgezhetjk. Amikor a "kszlket" kikapcsoljuk,
 *  a mikrovezrlt is energiatakarkos, "kszenlti" llapotba kapcsoljuk a SLEEP
 *  utasts segtsgvel. Az bresztshez most az INT2 megszaktst hasznljuk.  
 * 
 * Mivel a megszaktst nemcsak az bresztshez, hanem a kikapcsolshoz is hasznljuk, 
 * definilnunk kell a megszaktst kiszolgl eljrst. A magas priorits megszaktst vlasztottuk, 
 * teht a hi_isr() eljrsban szolgljuk ki a megszaktsokat. A pergsmentests rdekben minden 
 * interrupt utn a kontaktuspergs vrhat idejre ideiglenesen letiltjuk a tovbbi INT2 megszaktsokat.
 * 
 *  Hardver kvetelmnyek: az "A ksrleti ramkr" cm fejezetben bemutatott referencia 
 *  ramkrt ki kell egszteni: az INT2 bemenet s VSS kz egy nyomgombot (SW3), 
 *  az INT2 bemenet s VDD kz pedig egy 10 kohmos felhzellenllst kell ktni.
 *  PIC18F4550 esetn a bels felhzst bekapcsoltuk, gy a kls felhzs elhagyhat.
 *
 *  Projekt konfigurci: a helyes mkdshez a HID_BOOTLOADER mellett a USE_INTERRUPT
 *  szimblumot is definilni kell! 
 */

volatile enum {KI,BE} allapot;	///<llapotjelz

///Magas priorits programmegszakts kiszolglsa
	#pragma interrupt hi_isr
	void hi_isr() {				//INT2 megszakts kiszolglsa
		if(allapot==KI) {allapot=BE; } else {allapot=KI;}
		INTCON3bits.INT2IE=0;	//INT2 interrupt ideiglenes letiltsa!!!
	}

///Alacsony priorits programmegszakts kiszolglsa
	#pragma interruptlow lo_isr
	void lo_isr() {
	}

///Az INT2 bemenet inicializlsa
void initINT2(void) {
	#if defined(__18F4550)
		ADCON1 = 7;				//B port minden lba legyen digitlis
		TRISB=0xFF;				//B port minden lba legyen bemenet
		INTCON2bits.RBPU=0;		//legyen B port minden lba felhzva (ne lebegjen)
	#elif defined(__18F14K50)
		ANSELbits.ANS6=0;		//RC2 legyen digitlis	
	#endif
		INTCON2bits.INTEDG2=0;	//INT2 negatv lre legyen rzkeny
		INTCON3bits.INT2IP=1;	//INT2 magas priorits interrupt legyen
		INTCON3bits.INT2IF=0;	//az INT2 interrupt jelzbitjnek trlse
		INTCON3bits.INT2IE=1;	//az INT2 interrupt engedlyezse	
}

/** Ksleltet eljrs, amely a Microchip PIC18 knyvtri eljrsait hasznlja.
 * A bels ciklus kb. 1 ms ksleltetst vgez. 
 * \param d az elrt vrakozsi id, 1 ms egysgekben
 */ 
void delay_ms(unsigned int d) {
unsigned int j;
	for(j=0; j<d; j++) {
		Delay1KTCYx(12);		//12 000 TCY ksleltets (~ 1ms)
	}
}

//-- A fprogram -------------------------------------------
void main(void) {
unsigned int i;
	LEDport=0;					//LEDport inicializlsa
	LEDtris=0xF7;				//Csak LED4 legyen kimenet, a tbbi bemenet
	RCONbits.IPEN=1;			//Ktszint megszaktsi md belltsa
	INTCONbits.GIEH=1;			//A magas priorits interrupt engedlyezse
	INTCONbits.GIEL=0;			//Az alacsony priorits interrupt tiltsa
	initINT2();					//INT0 inicializlsa
	allapot=KI;					//Kikapcsolt kezdllapotbl indulunk
	while (1) {
	  if(allapot==KI) {
		LEDport=0;				//beszntetjk a mkdst
		delay_ms(25);			//pergsmentest ksleltets
		INTCON3bits.INT2IF=0;	//INT2 interrupt jelzbit trlse
		INTCON3bits.INT2IE=1;	//az INT2 interrupt engedlyezse
		Sleep();				//energiatakarkos zemmdba vltunk
		Nop();					//kszenlti llapot, interruptra vrunk
		mLED_4_On();			//jrakezdjk a mkdst
		delay_ms(25);			//Pergsmentest ksleltets)
		INTCON3bits.INT2IF=0;	//INT2 interrupt jelzbit trlse
		INTCON3bits.INT2IE=1;	//az INT2 interrupt engedlyezse
	  }
	  if(allapot==BE) {
		delay_ms(250);			//Villogtats flperidus ideje
		mLED_4_Toggle();		//Villogunk...
	  }
   }
}

