/* PICCOLO project
 * Copyright (c) 2009-2011 Istvan Cserny (cserny@atomki.hu)
 */

/** \file
 *  I2C tmogati fggvnyek. \see piccolo_i2c.h a rszletekrt!
 */

#include "piccolo_config.h"
#include "piccolo_i2c.h"

/** Az I2C modul konfigurlsa s engedlyezse a bitrate_kHz 
 * paramter ltal megadott adatsebessggel.
 * \param bitrate_kHz a kHz egysgekben megadott adattviteli sebessg
 */
void i2c_init(unsigned int bitrate_kHz) {
  #define clk_tmp CLOCK_FREQ/4000L
  SSPADD = clk_tmp/bitrate_kHz -1;
  SSPSTAT = SLEW_OFF | SMBUS_DISABLE;
  SSPCON1 = MASTER;
  SSPCON2 = 0x00;
  I2C_SCL = 1;
  I2C_SDA = 1;
  SSPCON1 |= SSPENB; 			// az MSSP egysg engedlyezse
}

/** Mvelet: Beolvas egy bjtot az I2C buszrl, maj kikldi az \em ack2Send 
 * paramterrel megadott nyugtz bitet a slave eszkznek.
 * \param ack2Send a nyugtz bit (0: ACK, 1: NAK), amelyet a beolvass vgn kikldnk a slave-nek
 * \return a beolvasott bjt
 */

uint8 i2c_getc(uint8 ack2send) {
  i2c_idle();
  SSPCON2bits.RCEN = 1;                 // a master engedlyezse egy bjt vtelre
  while ( !SSPSTATbits.BF );            // megvrjuk, amg a bjt berkezik 
  i2c_ack(ack2send);					// kikldjk a nyugtz bitet
  return ( SSPBUF );                    // visszatrnk a bjttal
}

/** Mvelet: Kir egy bjtot az I2C buszra, maj megvrja nyugtzst
 * \param data ki1rand adatbjt
 * \return a nyugtzs eredmnye ((0: ACK, 1: NAK)
 */
uint8 i2c_putc(uint8 data) {
  SSPBUF = data;                        // kirakunk egy bjtot SSPBUF-ba
  while( SSPSTATbits.BF );              // megvrjuk,amg az rslezajlik  
  i2c_idle();                           // megvrjuk a busz aktivits vgt
  return ( SSPCON2bits.ACKSTAT );       // a nyugtz bittel trnk vissza
}

/** Tranzakci: Egy bjt rsa 
 * \param addr  a slave I2C cme
 * \param data  a kikldeni kvnt bjt
 */
void i2c_write1(uint8 addr,uint8 data) {
  i2c_start();
  i2c_putc(addr & 0xFE);
  i2c_putc(data);
  i2c_stop();
}

/** Tranzakci: Kt bjt rsa
 * \param addr a slave I2C cme
 * \param d1  az els kikldeni kvnt bjt
 * \param d2  a msodik kikldeni kvnt bjt
 */
void i2c_write2(uint8 addr,uint8 d1, uint8 d2) {
  i2c_start();
  i2c_putc(addr & 0xFE);
  i2c_putc(d1);
  i2c_putc(d2);
  i2c_stop();
}

/** Tranzakci: Adott szm adatbjt rsa 
 * \param addr a slave I2C cme
 * \param *data a krand adatok mutatja
 * \param cnt a kirand adatbjtok szma
 */
void i2c_writeN(uint8 addr, uint8* data, uint8 cnt) {
  uint8 i;
  i2c_start();
  i2c_putc(addr & 0xFE);
  for (i=0; i < cnt; i++) {
    i2c_putc(*data);
    data++;
  }
  i2c_stop();
}

/** Tranzakci: Egy bjtot olvas az \em addr cm eszkzrl. 
 *  A beolvasott adat a \em p1 mutatval megcmzett helyre kerl.
 * \param addr a slave I2C cme
 * \param *p1 a beolvasott adat eltrolsi helynek mutatja
 */
void i2c_read1(uint8 addr,uint8* p1) {
  i2c_start();
  i2c_putc(addr | 0x01);
  *p1 = i2c_getc(I2C_NAK);
  i2c_stop();
}

/** Tranzakci: Kt bjtot olvas az \em addr cm eszkzrl. 
 *  A beolvasott adatok a \em p1 s \em p2 mutatval megcmzett 
 *  bjtokba kerlnek elhelyezsre.
 * \param addr a slave I2C cme
 * \param *p1 az els beolvasott adat eltrolsi helynek mutatja
 * \param *p2 a msodik beolvasott adat eltrolsi helynek mutatja
 */
void i2c_read2(uint8 addr,uint8* p1, uint8* p2) {
  i2c_start();
  i2c_putc(addr | 0x01);
  *p1 = i2c_getc(I2C_ACK);
  *p2 = i2c_getc(I2C_NAK);
  i2c_stop();
}

/** Tranzakci: A \em cnt paramterrel megadott darabszm bjtot olvas 
 *  az \em addr cm eszkzrl. A beolvasott adat a \em p1 mutatval megcmzett 
 *  helytl kezdden kerlnek elhelyezsre, egyms utn.
 * \param addr a slave I2C cme
 * \param *p1 a beolvasott adatok eltrolsi helynek mutatja
 * \param cnt a beolvasand adatbjtok szma
 */ 
void i2c_readN(uint8 addr,uint8* p1, uint8 cnt) {
  uint8 i;
  i2c_start();
  i2c_putc(addr | 0x01);
  for (i=0; i < cnt; i++) {
    if(i != cnt-1) *p1 = i2c_getc(I2C_ACK);
    else  *p1 = i2c_getc(I2C_NAK);
    p1++;
  }
  i2c_stop();
}








