/* PICCOLO project
 * Copyright (c) 2010-2011 Istvan Cserny (cserny@atomki.hu)
 *
 */
#include "piccolo_all.h"
#include "piccolo_i2c.h"

/** \file
 * I2C plda: Kt hmrt kezelnk az I2C tmogati fggvnyekkel.
 * NXP gyrtmny, LM75A tpus digitlis hmrket vagy vele
 * kompatibilis (LM75, TCN75A, TCN75) tpus hmrket hasznlunk. 
 * A fokokban kirt hmrsklet mellett a kiolvasott
 * nyers adatot is megjelentjk (hexadecimlisan).
 * 
 * Hardver igny:
 * - PICCOLO projekt ksrleti ramkr, melyet kt LM75A (vagy ms, 
 *   kompatibilis tpus) I2C hmrvel kell kiegszteni, s az I2C busz vonalait 
 *   egy-egy 2-10 kohmos ellenllssal fel kell hzni. 
 * - Az eredmnyek megjelentshez az USB kapcsolatot is hasznljuk.
 *
 * Projekt konfigurci: A HID_BOOTOADER szimblumon kvl csak az USE_USB
 * szimblumot definiljuk.
 */ 

/** Az LM75A digitlis hmr CONFIG regiszternek belltsa.
 * \param addr a hmr egyedi I2C cme.
 */
void LM75_init(uint8 addr) { 	
  i2c_start();						//j tranzakcit kezdnk
  i2c_putc(addr &0xFE);				//Slave cm s write md
  i2c_putc(0x01);					//A vezrlregiszter cme
  i2c_putc(0x18);					//Konfigurci belltsa (FT=6,normal mode)
  i2c_stop();						//A tranzakci vge		
}

/** Az LM75A digitlis hmr TEMP regiszternek kiolvassa.
 *  A visszatrsi rtk 16 bites, eljeles szm, melyben a 
 *  11 bites adat balra igaztott. Kiolvass eltt belltjuk a regisztermutatt.
 */
int LM75_readTemp(uint8 addr) {
union16 temp;
  i2c_start();
  i2c_putc(addr & 0xFE);			//Slave cm s rs md	
  i2c_putc(0x00);					//A TEMP regiszter cme
  i2c_rstart();						//RESTART a mdvltshoz
  i2c_putc(addr | 0x01);			//Slave cm s olvass md
  temp.hi_byte=i2c_getc(I2C_ACK);	//egy bjt fogadsa nyugtzssal (fokok)
  temp.lo_byte=i2c_getc(I2C_NAK);	//trtsz olvassa, negatv nyugtzssal
  i2c_stop();						//tranzakci vge	
  return ((int16)temp.word);
}

/** A kiolvasott hmrsklet tszmtsa tizedfokokra, kerektssel
 * s kiratsa egy tizedesjegyre.
 * \param data 16 bites eljeles adat, melyben a fels bjt az egszeket, az
 * als bjta a trtrszeket tartalmazza
 */
void LM75_kiiras(int16 data) {
int32 tmp32; 
	tmp32=(data*10L + 128L)/256;	//tizedfokokra szmtjuk t, kerektssel
	outdec(tmp32,1);			//Kirats, egy tizedesre
	outString(" C (0x");
	out4hex((uint16)data);		//Nyers adat hexadcimlis kirsa
	outString(") ");
}

void main(void) {
int16 temp;
  InitializeSystem();
  while (!usb_cdc_kbhit()) {	//Vrunk a CDC Terminlra
    ProcessIO();
  }
  i2c_init(100);                //100 kHz-es I2C bitrta 
  LM75_init(0x90);              //a T1 hmr belltsa norml mdba
  LM75_init(0x92);				//a T2 hmr belltsa norml mdba
  outString("\nIsten hozott a PICCOLO projekthez!\n");
  outString("2xLM75_i2c.c program, MCU: ");
#if defined(__18F4550)
  outString("PIC18F4550\n");
#elif defined(__18F14K50)
  outString("PIC18F14K50\n");
#endif
  while (1) {
    delay_ms(2000);
	temp=LM75_readTemp(0x90);	//els hmr kiolvassa
	outString("T1 = ");
	LM75_kiiras(temp);			//a hmrsklet kiratsa
	temp=LM75_readTemp(0x92);	//msodik hmr kiolvassa
	outString("T2 = ");
	LM75_kiiras(temp);			//a hmrsklet kiratsa
	outString("\n");
  }
}