/* PICCOLO project
 * Copyright (c) 2009-2010 Istvan Cserny (cserny@atomki.hu)
 *
 */

/** \file
 *  Ez a header llomny gondoskodik a RESET s az interupt vektorok
 *  thelyezsrl, s a PIC18F14K50 s a PIC18F4550 ksrleti ramkrk kztti
 *  trivilis eltrsek feloldsrl. A \ref piccolo_config.h tartalmtl, vagy
 *  a fordtnak kzvetlenl tadott makrodefinciktl fggen tovbbi llomnyok
 *  becsatolsra is sor kerlhet, illetve feltteles fordts segtsgvel 
 *  alternatv kdrszletek kerlnek bele a lefordtott projektbe.
 *
 *  Figyelem! ez a header llomny csak a fprogramba csatolhat be!
 *
 *  A legfontosabb konfigurcis paramterek:
 *  - <b>__18F14K50:</b> Azt jelzi, hogy a Configure/Select Device menben PIC18F14K50 mikrovezrlt 
 *    vlasztottuk. Ezt a szimblumot a fordt automatikusan definilja. 
 *
 *  - <b>__18F4550:</b>  Azt jelzi, hogy a Configure/Select Device menben PIC18F4550 mikrovezrlt 
 *    vlasztottuk. Ezt a szimblumot a fordt automatikusan definilja. 
 *
 *  - <b>USE_USB:</b> Akkor definiljuk, ha hasznlni kvnjuk az USB kapcsolatot (\ref piccolo_config.h 
 *    llomnyban vagy az MPLAB IDE Project/Build Options/Project menjben definilhatjuk)
 *
 *  - <b>USE_INTERRUPT:</b> Akkor definiljuk, ha hasznlni kvnjuk a programmegszaktst 
 *    (vagy a \ref piccolo_config.h llomnyban, vagy az MPLAB IDE Project/Build Options/Project 
 *    menben definilhatjuk). <b>Figyelem! Programmegszakts esetn gondoskodnunk kell a magas s az 
 *    alacsony priorits interruptok kiszolglst ellt hi_isr() s lo_isr() eljrsokrl!</b>  
 *    Ha az USE_USB s az USE_INTERRUPT szimblumok definilva vannak, akkor automatikusan az
 *    USB_INTERRUPT szimblum is definilsra kerl, ellenkez esetben pedig az USB_POLLING 
 *    kerl definilsra. A \ref hello-int.c mintaprogramban lthat, hogy az USB interruptos 
 *    kiszolglsa hogyan vgezhet.
 *
 *  - <b>Blinking_at_start:</b> azt engedlyezhetjk vele, hogy LED1 s LED2 kijelezze az USB kapcsolat 
 *    llapott. A lehetsges rtkek: 0 = tilts, 1 = engedlyezs. 
 * 
 *  - <b>HID_BOOTLOADER:</b> Akkor definiljuk, ha HID bootloadert hasznlunk (\ref piccolo_config.h  
 *    llomnyban vagy az MPLAB IDE Project/Build Options/Project menpontjban definilhatjuk). 
 *    Ezzel szinkronban a linker programnak is t kell adni a HID_BOOTLOADER szimbolumot 
 *    (parancssori /uHID_BOOTLOADER opci megadsa a Project/Build Options/Project menben, MPLINK belltsai lapjn).
 *   
 *  - <b>MCHPUSB_BOOTLOADER:</b> Akkor definiljuk, ha az MCHPUSB bootloadert hasznlunk (\ref piccolo_config.h  
 *    llomnyban vagy a Project/Build Options/Project menben definilhatjuk). Ezzel szinkronban a
 *    Linker programnak is t kell adni a MCHPUSB_BOOTLOADER szimbolumot (parancssori /u opci megadsa
 *    a Project/Build Options/Project menben, MPLINK belltsai lapjn). 
 * 
 *    \see piccolo_config.h projekt szint konfigurcis belltsok
 *    \see piccolo_usb.h az USB kapcsolat alkalmazsszint kezelshez szksges fggvnyek s vltozk
 *    \see piccolo-14k50.h a PIC18F14K50 mintaramkr hardverspecifikus makrdefincii
 *    \see piccolo-4550.h a PIC18F4550 mintaramkr hardverspecifikus makrdefincii
 */

#ifndef PICCOLO_ALL
#define PICCOLO_ALL 

/*  INCLUDE llomnyok becsatolsa ***********************************/
#include "piccolo_config.h" 	//Implicit mdon becsatolja p18cxxx.h-t is!

    #warning *** P R O J E K T   K O N F I G U R  C I  ***
#if defined(USE_USB)
    #include "piccolo_usb.h" 
    #warning * USE__USB        : DEFINILT                 *
#else
    #warning * USE__USB        : NEM DEFINILT             * 
#endif
#if defined(USE_INTERRUPT)
    #warning * USE__INTERRUPT  : DEFINILT                 *
#else
    #warning * USE__INTERRUPT  : NEM DEFINILT             * 
#endif
#if defined(HID_BOOTLOADER)
    #warning * HID__BOOTLOADER : DEFINILT                 *
#else
    #warning * HID__BOOTLOADER : NEM DEFINILT             * 
#endif
#if defined(MCHPUSB_BOOTLOADER)
    #warning * MCHPUSB__BOOTLOADER : DEFINILT             *
#endif

    #warning *********************************************** 

/** Inline fggvny, amely egy regiszter valamelyik bitcsoportjt
 * atomi mvelettel mdostja. 
 * \param reg a mdostand regiszter neve
 * \param val a berand rtk (helyirtken helyesen!)
 * \param mask a mdostand bitcsoportot kijell maszk
 */
#define ChangeBits(reg,val,mask) reg ^= ((reg^val) & mask)	///<Bitcsoport trsa atomi mvelettel


/*  Vektorok thelyezse  ********************************************/
    void hi_isr(void);     ///< Magas priorits programmegszakts kiszolglsa
    void lo_isr(void);     ///< Alacsony priorits programmegszakts kiszolglsa

#if defined(HID_BOOTLOADER)
	#define RESET_VECTOR			0x1000
	#define HIGH_INTERRUPT_VECTOR	0x1008
	#define LOW_INTERRUPT_VECTOR	0x1018
#elif defined(MCHPUSB_BOOTLOADER)	
	#define RESET_VECTOR			0x800
	#define HIGH_INTERRUPT_VECTOR	0x808
	#define LOW_INTERRUPT_VECTOR	0x818
#else	
	#define RESET_VECTOR			0x00
	#define HIGH_INTERRUPT_VECTOR	0x08
	#define LOW_INTERRUPT_VECTOR	0x18
#endif

/*  RESET VEKTOR ****************************************************/
// Ha nem hasznlunk bootloadert, akkor a C18 startup kdja automatikuan ltrehozza!
// Ezrt az albbi sorok csak bootloader hasznlata esetn kellenek.
#if defined(HID_BOOTLOADER)||defined(MCHPUSB_BOOTLOADER)		
	extern void _startup (void);
/** Az thelyzett RESET vektor.  
 *  A bootloader hasznlattl fgg abszolt cmre kerl.
 */
	#pragma code REMAPPED_RESET_VECTOR = RESET_VECTOR
	void _reset (void) 
	{
	    _asm goto _startup _endasm
	}
#endif
	
/** Az thelyzett magas priorits interrupt vektor.  
 *  A bootloader hasznlattl fgg abszolt cmre kerl.
 */
#pragma code REMAPPED_HIGH_INTERRUPT_VECTOR = HIGH_INTERRUPT_VECTOR
void Remapped_High_ISR (void)
{
	 _asm goto hi_isr _endasm
}

/** Az thelyzett alacsony priorits interrupt vektor.  
 *  A bootloader hasznlattl fgg abszolt cmre kerl.
 */
#pragma code REMAPPED_LOW_INTERRUPT_VECTOR = LOW_INTERRUPT_VECTOR
void Remapped_Low_ISR (void)
{
	 _asm goto lo_isr _endasm 
}	

#pragma code
//res eljrsokat gyrtunk az elszabadult interruptok megfkezsre,
//ha a felhasznl nem akar programmegszaktst hasznlni!
#ifndef USE_INTERRUPT
	void hi_isr()
	{
	}	
	
	void lo_isr()
	{	
	}
#endif //USE_INTERRUPT

#endif //PICCOLO_ALL