/* ***************************************************************************
**  File Name    : uart2spi.c
**  Version      : 1.0
**  Description  : UART to SPI Gateway
**  Author       : RWB
**  Target       : PICJazz 20PIN Board: PIC18F14K22
**  Compiler     : Microchip C18 v3.36 C Compiler
**  IDE          : Microchip MPLAB IDE v8.46
**  Programmer   : PICKit3, Firmware Suite Version 01.25.20
**  Last Updated : 25 Aug 2010
** ***************************************************************************/
#include <p18cxxx.h>
#include <delays.h>
#include <string.h>

/*
** PIC18F14K22 Configuration Bit:
**
** FOSC = IRC        - Internal RC Oscillator
** PLLEN = OFF       - PLL is under software control
** PCLKEN = ON       - Primary Clock Enable
** FCMEN = OFF       - Fail-Safe Clock Monitor disabled
** PWRTEN = OFF      - Power Up Timer disabled
** BOREN = OFF       - Brown-out Reset disabled in hardware and software
** WDTEN = OFF       - WDT is controlled by SWDTEN bit of the WDTCON register
** MCLRE = ON        - MCLR pin enabled, RE3 input pin disabled
** LVP = OFF         - Single-Supply ICSP disabled
*/
#pragma config FOSC = IRC, PLLEN = OFF, PCLKEN = ON
#pragma config FCMEN = OFF, BOREN = OFF, PWRTEN = OFF
#pragma config WDTEN = OFF, MCLRE = ON, LVP = OFF

// Using Internal Clock of 16 Mhz
#define FOSC 16000000L
#define BAUD_RATE 19200
#define MAX_CMD 20

rom char prompt[] = "\nSPI>";
rom char errcmd[] = "\nUnknown Command!\n";
char sdigit[3]={'0','0','\0'};
unsigned char hexval[5];
unsigned char spi_cs, spi_freq, spi_bus;

// SPI Gateway Help Screen
#define MAX_HELP 60
// .........|.........|.........|.........|.........|.........|
// 123456789012345678901234567890123456789012345678901234567890
// ........10........20........30........40........50........60
rom char helpscr[][MAX_HELP]=
 {"PICJazz Board: PIC18F14K22 UART to SPI Gateway\n",
  "http://www.ermicro.com/blog\n",
  "\nSPI Gateway Commands:\n\n",
  "[A] - Auto SPI Data Transmission (Hex Format)\n",
  "      Example: A0x40,0x12,0xFF\n",
  "[C] - Clear Screen\n",
  "[D] - Display SPI Configuration\n",
  "[F] - SPI Clocks F0-Fosc/4, F1-Fosc/16, and F2-Fosc/64\n",
  "[M] - SPI Bus Modes M0, M1, M2, and M3\n",
  "      Mode 0 - CKP=0,CKE=1 : Mode 1 - CKP=0,CKE=0\n",
  "      Mode 2 - CKP=1,CKE=1 : Mode 3 - CKP=1,CKE=0\n",
  "[R] - Received SPI Slave Data\n",
  "[S] - SPI Chip Select S0-Low and S1-High\n",
  "[T] - Transmit SPI Master Data:\n",
  "      0x - Hex Data Format: T0x8A\n",
  "      0b - Binary Data Format: T0b10001010\n",
  "[U] - Use default SPI Configuration\n",
  "[?] - Show Help\n"};

// Delay in 1 ms (approximately) for 16 MHz Internal Clock
void delay_ms(unsigned int ms)
{
  do {
    Delay1KTCYx(4);
  } while(--ms);
}

void uart_init(void)
{
  TRISBbits.TRISB5 = 1; // Set Port B5 for UART RX
  TRISBbits.TRISB7 = 0; // Set Port B7 for UART RX

  // Baud Rate formula for SYNC=0 (Async), BRG16=0 (8-bit), BRGH=0 (low speed)
  // 0.16% Error for 16 MHz Oscillator Clock. Actual baud rate will be 19230.
  // BAUD_RATE = FOSC / (64 x (SPBRG + 1)) 

  SPBRG = (int)(FOSC/(64UL * BAUD_RATE) - 1);
  TXSTA = 0b00100000;  // Async, 8 bit and Enable Transmit (TXEN=1)
  RCSTA = 0b10010000;  // Serial Port Enable, Async,8-bit and Enable Receipt (CREN=1)
  BAUDCON=0x00;
}

int _user_putc(char ch)
{
  if (ch == '\n')
    _user_putc('\r');

  // Send Data when TXIF bit is clear
  while(!PIR1bits.TXIF) continue;
  TXREG = ch;
}

char _user_getc(void)
{
  // Get Data when RCIF bit is clear
  while(!PIR1bits.RCIF) continue;
  return RCREG;
}

void uart_gets(char *buf, unsigned char len)
{
  unsigned char cnt;

  cnt=0;
  while(cnt < len) {
    *buf = _user_getc();  // Get a character from the USART
    _user_putc(*buf);     // Echo Back
    cnt++;
    if (*buf == '\r') {   // Check for CR
      *buf='\0';          // Replace it with Null Terminate String
      break;
    }
    if (*buf == '\b') {   // Check for Backspace
      _user_putc(32);
      _user_putc('\b');
      buf--;buf--;
      cnt--;
    }
    buf++;
  }
  *buf='\0';              // Null Terminate String

  _user_putc('\n');
}

void uart_puts_P(rom char *buf)
{
  while(*buf) {
    _user_putc(*buf);
    buf++;
  }
}

void uart_puts(char *buf)
{
  while(*buf) {
    _user_putc(*buf);
    buf++;
  }
}

void ansi_cl(void)
{
  // ANSI clear screen: cl=\E[H\E[J
  _user_putc(27);
  _user_putc('[');
  _user_putc('H');
  _user_putc(27);
  _user_putc('[');
  _user_putc('J');
}

void ansi_me(void)
{
  // ANSI turn off all attribute: me=\E[0m
  _user_putc(27);
  _user_putc('[');
  _user_putc('0');
  _user_putc('m');
}

void disp_help(void)
{
  unsigned char i;

  // Display the Help Screen
  for(i=0;i < (sizeof(helpscr)/MAX_HELP);i++) {
    uart_puts_P(helpscr[i]);
  }
}

void spi_init(void)
{
  /* Initial the PIC18F14K22 SPI Peripheral */
  TRISCbits.TRISC6 = 0;  // RC6/SS - Output (Chip Select)
  TRISCbits.TRISC7= 0;   // RC7/SDO - Output (Serial Data Out)
  TRISBbits.TRISB4= 1;   // RB4/SDI - Input (Serial Data In)
  TRISBbits.TRISB6= 0;   // RB6/SCK - Output (Clock)

  SSPSTAT = 0x40;        // Set SMP=0 and CKE=1. Notes: The lower 6 bit is read only
  SSPCON1 = 0x20;        // Enable SPI Master, CKP=0 and Fosc/4
  PORTCbits.RC6 = 1;     // Disable Chip Select

  // Initial SPI Setup Status Variables
  spi_cs=1;
  spi_freq=0;
  spi_bus=0;
}

void spi_mode(unsigned char cmd)
{
  switch(cmd) {
    case '0':  // Mode 0
      spi_bus=0;
      SSPSTATbits.CKE=1;
      SSPCON1bits.CKP=0;
      uart_puts_P((rom char *)"SPI Mode 0\n");
      break;
    case '1':  // Mode 1
      spi_bus=1;
      SSPSTATbits.CKE=0;
      SSPCON1bits.CKP=0;
      uart_puts_P((rom char *)"SPI Mode 1\n");
      break;
    case '2':  // Mode 2
      spi_bus=2;
      SSPSTATbits.CKE=1;
      SSPCON1bits.CKP=1;
      uart_puts_P((rom char *)"SPI Mode 2\n");
      break;
    case '3':  // Mode 3
      spi_bus=3;
      SSPSTATbits.CKE=0;
      SSPCON1bits.CKP=1;
      uart_puts_P((rom char *)"SPI Mode 3\n");
      break;
    default:
      uart_puts_P(errcmd);
  }
}

void spi_slavecs(unsigned char cmd)
{
  switch(cmd) {
    case '0':  // SPI Slave Select Low
      spi_cs=0;
      PORTCbits.RC6 = 0;    // Low
      uart_puts_P((rom char *)"SPI Chip Select Low\n");
      break;
    case '1':  // SPI Slave Select High
      spi_cs=1;
      PORTCbits.RC6 = 1;    // High
      uart_puts_P((rom char *)"SPI Chip Select High\n");
      break;
    default:
      uart_puts_P(errcmd);
  }
}

void spi_clock(unsigned char cmd)
{
  switch(cmd) {
    case '0':  // SPI Clock Fosc/4
      spi_freq=0;
      SSPCON1 &= 0xF0;
      uart_puts_P((rom char *)"SPI Clock Fosc/4\n");
      break;
    case '1':  // SPI Clock Fosc/16
      spi_freq=1;
      SSPCON1 &= 0xF1;
      uart_puts_P((rom char *)"SPI Clock Fosc/16\n");
      break;
    case '2':  // SPI Clock Fosc/64
      spi_freq=2;
      SSPCON1 &= 0xF2;
      uart_puts_P((rom char *)"SPI Clock Fosc/64\n");
      break;
    default:
      uart_puts_P(errcmd);
  }
}

// Implementing integer value from 0 to 255
char *num2hex(unsigned char num)
{
   unsigned char idx;
   char hexval[]={"0123456789ABCDEF"};     

   idx = 0;                     // Start with index 0
   while(num >= 16){            // Keep Looping for larger than 16
     idx++;                     // Increase index
     num -= 16;                 // Subtract number with 16
   }      

   sdigit[0]='0';               // Default first Digit to '0'
   if (idx > 0)
     sdigit[0]=hexval[idx];     // Put the First digit

   sdigit[1]=hexval[num];       // Put the Second Digit
   return sdigit;
}

int hex2num(char *cmd)
{
   unsigned char num;

   if (strlen(cmd) != 2)  // 8-bit Data
     return -1;

   // Start from MSB to LSB
   num=0;
   while(*cmd) {
     num = num << 4;
     if (*cmd >= 'A' && *cmd <= 'F') {
       num += *cmd - 55;
	 } else if (*cmd >= 'a' && *cmd <= 'f') {
       num += *cmd - 87;
     } else if (*cmd >= '0' && *cmd <= '9') {
       num += *cmd - 48;
     } else {
       return -1;
     }
	 cmd++;
   }
   return num;
}

int bin2num(char *cmd)
{
  unsigned char num,cnt;
  unsigned char binval[]={128,64,32,16,8,4,2,1};    

  if (strlen(cmd) != 8 ) // 8-bit Data
    return -1;

  // Start from MSB to LSB
  num=0;
  for(cnt=0;cnt<8;cnt++) {
    if (cmd[cnt] >= '0' && cmd[cnt] <= '1') {
      num += binval[cnt] * (cmd[cnt] - 48);
	} else
	  return -1;
  }
  return num;
}

unsigned char SPI_WriteRead(unsigned char data)
{
  unsigned char slave_data;

  // Send the SPI Master data
  uart_puts_P((rom char *)"SPI Master Send: 0x");
  uart_puts(num2hex(data));
  SSPBUF = data;

  // Wait for Data Transmit/Receipt complete
  while(!SSPSTATbits.BF);  

  // Return SPI Slave Data
  slave_data=SSPBUF;
  uart_puts_P((rom char *)", SPI Slave Return: 0x");
  uart_puts(num2hex(slave_data)); _user_putc('\n');

  return(slave_data);
}

void spi_transmit(char *cmd)
{
  unsigned char data;

  if (*cmd != '0') {
    uart_puts_P(errcmd);
    return;
  }
  cmd++;         // Skip to the next token  

  switch(*cmd) {
    case 'x':    // Hex Data
    case 'X':    // Hex Data
      data=hex2num(cmd + 1);
      if (data >= 0)
        data=SPI_WriteRead((unsigned char) data);
      else
       uart_puts_P(errcmd);
      break;
    case 'b':    // Binary Data
    case 'B':    // Binary Data
      data=bin2num(cmd + 1);
      if (data >= 0)
        data=SPI_WriteRead((unsigned char) data);
      else
        uart_puts_P(errcmd);
      break;
    default:
      uart_puts_P(errcmd);
  }
}

void auto_transmit(char *cmd)
{
  unsigned char cnt,hexcnt;

  // Set Slave CS Pin Low
  spi_slavecs('0');

  // Transmit SPI Data
  cnt=0;
  hexcnt=0;
  while(cmd[cnt]) {
    if (cmd[cnt] != ',') {       // Check for Command Separator
      hexval[hexcnt++]=cmd[cnt];
      if (hexcnt > 4) {          // Max Hex Format 0xFF (4)
	 uart_puts_P(errcmd);
	 return;
      }
    } else {
      hexval[hexcnt]='\0';	     // Null Terminated
      spi_transmit(hexval);
      hexcnt=0;
    }
    cnt++;
  }  

  // Transmit the last SPI Data
  hexval[hexcnt]='\0';           // Null Terminated
  spi_transmit(hexval);

  // Set Slave CS Pin High
  spi_slavecs('1');
}

void disp_config()
{
  // Display SPI Chip Select Status
  uart_puts_P((rom char *)"SPI CS Status: ");
  switch (spi_cs) {
    case 0:
      uart_puts_P((rom char *)"Low");
      break;
    case 1:
      uart_puts_P((rom char *)"High");
      break;
    default:
      uart_puts_P(errcmd);
  }

  // Display SPI Frequency Setup
  uart_puts_P((rom char *)", SPI Clock: ");
  switch (spi_freq) {
    case 0:
      uart_puts_P((rom char *)"Fosc/4");
      break;
    case 1:
      uart_puts_P((rom char *)"Fosc/16");
      break;
    case 2:
      uart_puts_P((rom char *)"Fosc/64");
      break;
    default:
      uart_puts_P(errcmd);
  }

  // Display SPI Mode Setup
  uart_puts_P((rom char *)", SPI Bus Mode: ");
  switch (spi_bus) {
    case 0:
      _user_putc('0');
      break;
    case 1:
      _user_putc('1');
      break;
    case 2:
      _user_putc('2');
      break;
    case 3:
      _user_putc('3');
      break;
    default:
      uart_puts_P(errcmd);
  }
  _user_putc('\n');
}

void disp_header()
{
   uart_puts_P(helpscr[0]);
   uart_puts_P(helpscr[1]);
   uart_puts_P((rom char *)"\nEnter SPI Command (? for help):\n");
}

void main(void)
{
  char cmd[MAX_CMD + 1];
  unsigned char data;

  OSCCON=0x70;         // Select 16 MHz internal clock

  TRISC = 0x00;        // Set All on PORTC as Output
  TRISA = 0x30;        // Input for RA4 and RA5
  TRISB = 0x00;
  ANSEL = 0x00;        // Set PORT AN to analog input
  ANSELH = 0x00;       // Set PORT AN8 to AN11 as Digital I/O 

  // Initial the UART
  uart_init();
  delay_ms(1);         // Make 1 ms delay here for the UART initialization

  // Initial default SPI
  spi_init();

  // Clear and Initial ANSI Terminal
  ansi_cl();
  ansi_me();
  ansi_cl();
  disp_header();

  for(;;) {
    uart_puts_P(prompt); uart_gets(cmd,MAX_CMD);
    switch (cmd[0]) {
      case 'a':       // Automatic SPI Transmit
      case 'A':       // Automatic SPI Transmit
        auto_transmit(cmd + 1);
        break;
      case 'c':       // Clear Screen
      case 'C':       // Clear Screen
        if (cmd[1] != '\0') {
          uart_puts_P(errcmd);
        } else {
          ansi_cl();
          disp_header();
        }
        break;
      case 'd':       // Display SPI Configuration
      case 'D':       // Display SPI Configuration
        if (cmd[1] != '\0') {
          uart_puts_P(errcmd);
        } else {
          disp_config();
        }
        break;
      case 'f':       // Set SPI Master Clock
      case 'F':       // Set SPI Master Clock
        spi_clock(cmd[1]);
        break;
      case 'm':       // Set SPI Bus Mode
      case 'M':       // Set SPI Bus Mode
        spi_mode(cmd[1]);
        break;
      case 'r':       // Receive SPI Slave Data
      case 'R':       // Receive SPI Slave Data
        if (cmd[1] != '\0') {
          uart_puts_P(errcmd);
        } else {
          data=SPI_WriteRead(0x00);
          uart_puts_P((rom char *)"SPI Read: 0x");
          uart_puts(num2hex(data)); _user_putc('\n');
        }
        break;
      case 's':       // SPI Slave Chip Select
      case 'S':       // SPI Slave Chip Select
        spi_slavecs(cmd[1]);
        break;
      case 't':       // Transmit SPI Master Data
      case 'T':       // Transmit SPI Master Data
        spi_transmit(cmd + 1);
        break;
      case 'u':       // Used Default Configuration
      case 'U':       // Used Default Configuration
        if (cmd[1] != '\0') {
          uart_puts_P(errcmd);
        } else {
          spi_init();
          uart_puts_P((rom char *)"Default SPI Config:\n");
          disp_config();
        }
        break;
      case '?':       // Display Help
        if (cmd[1] != '\0')
          uart_puts_P(errcmd);
        else
          disp_help();
        break;
      default:
        uart_puts_P(errcmd);
    }
  }
}

/* EOF: uart2spi.c */