/* PICCOLO project
 * Copyright (c) 2009-2011 Istvan Cserny (cserny@atomki.hu)
 *
 */
#include "piccolo_all.h"
#include "piccolo_i2c.h"


/** \file
 * I2C mintaprogram: Egy Microchip gyrtmny, 24LC515 tpus EEPROM 
 * rsa s olvassa az I2C tmogati fggvnyek felhasznlsval.
 *
 * Hardver kvetelmnyek: a ksrleti ramkrt egy 24LC515 I2C EEPROM-mal kell
 * kiegszteni, melynek A0 s A1 cmvonalait GND-re, A2 kivlaszt s a WP
 * bemenetet pedig Vdd-re ktjk. Az SDA s az SCL kivezetseket a mikrovezrl
 * SDA s SCL kivezetseire ktjk. Mindkt buszvonalat egy-egy 2,2 kohm-mos 
 * ellenllssal a tpfeszltsgre hzzuk.
 */

#define EEPROM 0xA0			//az EEPROM I2C cme, fldre kttt cmvonalakkal
#define BLKSIZE 64			//blokkmret: egy memrialap 16 bjtos
//-- Projekt konfigurci ellenrzse 
#if defined(USE_INTERRUPT) 
	#error "Ne legyen definilva az USE__INTERRUPT szimblum!"
#endif //USE_INTERRUPT
#if !defined(USE_USB)
	#error "Nincs definilva az USE__USB szimblum!"
#endif //USE_USB

/** Vrakozs arra, hogy az EEPROM befejezze az rst. Amg az rs tart,
 *  addig az EEPROM nem kld nyugtz jelet a cmnek kikldsekor.
 *  A fggvny blokkol tpus, addig nem tr vissza, amg az EEPROM foglalt.
 * \param addr a vizsglni kvnt slave eszkz I2C cme.
 */
void MCP24LC515_wait(uint8 addr) {
  uint8 ack;
  do {
    i2c_start();
    ack = i2c_putc(addr);					//ACK bit figyelse
    i2c_stop();
  } while (ack == I2C_NAK);
}

/** Egy memrialap (64 bjt) rsa a bemen adatbufferbl, az EEPROM egy
 *  megadott lapcmtl kezdden.
 *  \param addr az EEPROM I2C cme
 *  \param memaddr a memrialap kezdcme, ahov runk
 *  \param *pbuf mutat az adatbuffer kezdethez
 */
void MCP24LC515_write(uint8 addr,  union16 memaddr, uint8* pbuf) {
  pbuf[1] = memaddr.lo_byte;				//a lapcmet is a bufferbe tesszk
  pbuf[0] = memaddr.hi_byte;
  if (memaddr.hi_byte & 0x80) {				// ha MSB='1', belltjuk a blokkvlaszt bitet
    addr = addr | 0x08;
  }
  MCP24LC515_wait(addr);					//Vrunk, ha az EEPROM elfoglalt
  i2c_writeN(addr,pbuf,BLKSIZE+2);			//Az adatbuffer kirsa (cm+adatok)
}

/** Egy memrialap (64 bjt) olvassa az EEPROM egy megadott lapcmtl kezdden,
 *  s eltrolsa az adatbufferbe.
 *  \param addr az EEPROM I2C cme
 *  \param memaddr a beolvasni kvnt memrialap kezdcme
 *  \param *pbuf mutat az adatbuffer kezdethez
 */
void MCP24LC515_read(uint8 addr,union16 memaddr,uint8* pbuf) {
  if (memaddr.hi_byte & 0x80) {				// ha MSB='1', belltjuk a blokkvlaszt bitet
    addr = addr | 0x08;
  }
  MCP24LC515_wait(addr);
  i2c_write2(addr,memaddr.hi_byte,memaddr.lo_byte);		//lapcm belltsa
  i2c_readN(addr,pbuf,BLKSIZE);				//memrialap olvassa
}


void main (void) {
  uint8  buf[BLKSIZE+2];					//2 extra bjt kell a cmnek
  union16 MemAddr,ReadAddr;
  uint8  i,cmd;
  InitializeSystem();
  while (!usb_cdc_kbhit()) {				//Vrunk a CDC Terminlra
    ProcessIO();
  }
  cmd=usb_cdc_getc();						//az els karakter a parancs neve
  i2c_init(400);                    		//400 kHz-es I2C bitrta 
  MemAddr.word = 0;							//A memria 0 cmtl kezdnk
  ReadAddr.word = 0;
  while (1) {
	if(cmd=='w' || cmd=='W') {
//-- rssal kezdnk
    outString("Gpeljen be 64 karaktert!\n");
//-- a buffer els bjtja a cmnek van fenntartva
    for (i = 2; i< BLKSIZE+2; i++) {
      buf[i] = usb_cdc_getc();
      usb_cdc_putc(buf[i]);
    }
    outString("\nEltroljuk az adatot\n");
//-- ktszer rjuk ki az adatokat egyms utn, hogy a foglaltsg ellenrzst is kiprbljuk
    out4hex(MemAddr.word);
    outString(": rs indul...");
    MCP24LC515_write(EEPROM,MemAddr,buf);	//Memria lap rsa
    MemAddr.word += BLKSIZE;		 		//Memria cm lptetse
    outString(" ...rs vge!\n");
    out4hex(MemAddr.word);
    outString(": rs mgegyszer...");
    MCP24LC515_write(EEPROM,MemAddr,buf);	//Memria lap rsa mgegyszer
    MemAddr.word += BLKSIZE;		 		//Memria cm lptetse
    outString(" ...rs vge!\n");
 	}  else { 
//-- Olvasssal folytatjuk	
    out4hex(ReadAddr.word);
	outString(": olvass indul...");
    MCP24LC515_read(EEPROM,ReadAddr,buf);	// olvass
	outString(" ...olvass vge!\n");
    for (i = 0;i<BLKSIZE; i++) {
      usb_cdc_putc(buf[i]);
    }
	outString("\n");
    ReadAddr.word += BLKSIZE;		 		//Memria cm lptetse
    out4hex(ReadAddr.word);
	outString(": olvass mgegyszer...");
    MCP24LC515_read(EEPROM,ReadAddr,buf);	// olvass
	outString(" ...olvass vge!\n");
    for (i = 0;i<BLKSIZE; i++) {
      usb_cdc_putc(buf[i]);
    }
    ReadAddr.word += BLKSIZE;		 		//Memria cm lptetse
	}
	outString("\nNyomjon egy gombot a folytatshoz!\n");
	cmd=usb_cdc_getc();
  }
}
