/* PICCOLO project
 * Copyright (c) 2009-2011 Istvan Cserny (cserny@atomki.hu)
 *
 */
#include "piccolo_all.h"
#include "spi.h"


/** \file
 * SPI mintaprogram: Egy Microchip gyrtmny, 25LC256 tpus EEPROM
 * rsa s olvassa az SPI tmogati fggvnyek felhasznlsval.
 *
 * Hardver kvetelmnyek: a ksrleti ramkrt egy 25LC256 SPI EEPROM-mal kell
 * kiegszteni, az sszektst az albbi tblzat szerint alaktsuk ki:
 *  \code
 *  PIC18                       25LC256
 *  ===========================================
 *  SDI  <--------------------- SO  (2)
 *  SDO  ---------------------> SI  (5)
 *  SCK  ---------------------> SCK (6)
 *  LED4 -- Slave Select -----> CS  (1)
 *  VDD  ---------------------- VCC (8), WP(3), HOLD(7)
 *  GND  ---------------------- GND (4)
 * \endcode
 */


#define BLKSIZE   64                        //blokkmret: egy memrialap 64 bjtos
//-- A 25LC256 EEPROM ltal elfogadott parancsok
#define CMD_WRSR  0x01                      //Sttuszregiszter rsa
#define CMD_WRITE 0x02                      //rs a megadott cmtl kezdden
#define CMD_READ  0x03                      //Olvass a megadott cmtl kezdden
#define CMD_WRDI  0x04                      //Letiltja az rst
#define CMD_RDSR  0x05                      //Sttuszregiszter olvassa
#define CMD_WREN  0x06                      //Engedlyezi az rst

//-- Slave Select kimenet definilsa
#define SLAVE_ENABLE()     mLED_4 = 0       //Slave kivlasztsa
#define SLAVE_DISABLE()    mLED_4 = 1       //Slave eszkz letiltsa


/** Vrakozs arra, hogy az EEPROM befejezze az rst. Amg az rs tart,
 *  addig az EEPROM STATUS regiszternek legals bitje (WIP) '1'-ben ll.
 *  A fggvny blokkol tpus, addig nem tr vissza, amg az EEPROM foglalt.
 */
void waitFor25LC256(void) {
  uint8 flag;
  do {
    SLAVE_ENABLE();                         //kiadjuk a Chip Enable jelet
    WriteSPI(CMD_RDSR);                     //Sttuszregiszter olvassa parancs
    flag = ReadSPI();                       //Olvasunk, de valamit akkor is kldeni kell!
    SLAVE_DISABLE();                        //megszntetjk a Chip Enable jelet
  } while (flag & 0x01); 
}

/** Egy memrialap (64 bjt) rsa a bemen adatbufferbl, az EEPROM egy
 *  megadott cmtl kezdden.
 *  \param MemAddr a memrialap kezdcme, ahov runk
 *  \param buf mutat az adatbuffer kezdethez
 */
void memWrite25LC256(union16 MemAddr, uint8 *pbuf) {
  uint8 i;
  waitFor25LC256();                         //Vrunk, ha az EEPROM elfoglalt
  SLAVE_ENABLE();
  WriteSPI(CMD_WREN);                       //rs jraengedlyezse
  SLAVE_DISABLE();
  Nop();
  SLAVE_ENABLE();
  WriteSPI(CMD_WRITE);                      //Adatblokk rsa
  WriteSPI(MemAddr.hi_byte );
  WriteSPI(MemAddr.lo_byte);
  for (i=0; i < BLKSIZE; i++) {
    WriteSPI(*pbuf++);                      //Az adatbuffer kirsa
  }
  SLAVE_DISABLE();
}

/** Egy memrialap (64 bjt) olvassa az EEPROM egy megadott cmtl kezdden,
 *  s eltrolsa az adatbufferbe.
 *  \param u16_MemAddr a beolvasni kvnt memrialap kezdcme
 *  \param *pu8_buf mutat az adatbuffer kezdethez
 */
void memRead25LC256(union16 MemAddr, uint8 *pbuf) {
  uint8 i;
  waitFor25LC256();                         //Vrunk, ha az EEPROM elfoglalt
  SLAVE_ENABLE();
  WriteSPI(CMD_READ);                       //Adatblokk olvassa
  WriteSPI(MemAddr.hi_byte );
  WriteSPI(MemAddr.lo_byte);
  for (i=0; i < BLKSIZE; i++) {
    *pbuf++=ReadSPI();
  }
  SLAVE_DISABLE();
}

/** Egy eljel nlkli bjtot ktjegy hexadecimlis formban kir
 * a kimeneti bufferbe. Ez a fggvny meghvja a blokkol tpus
 *  _user_putc() fggvnyt! 
 */ 
void out2hex(unsigned char t) {
	unsigned char c;
		c=(char)((t>>4) & 0x0F);
		if (c>9) c+=7;
		_user_putc(c+'0');
		c=(char)(t & 0x0F);
		if (c>9) c+=7;
		_user_putc(c+'0');
}


void main (void) {
  uint8 buf[BLKSIZE];
  union16 MemAddr,ReadAddr;
  uint8 cmd,i,dummy;
  InitializeSystem();
  DISABLE_ALL_ANALOG();
  mInitAllLEDs();
  LEDport = 0;
  SLAVE_DISABLE();
  OpenSPI(SPI_FOSC_16,MODE_00,SMPEND);
  while (!usb_cdc_kbhit()) {                //Vrunk a CDC Terminlra
    ProcessIO();
  }
  outString("\nIsten hozott a PICCOLO projekthez!\n");
  outString("25LC256_spi.c program, MCU: ");
#if defined(__18F4550)
  outString("PIC18F4550\n");
#elif defined(__18F14K50)
  outString("PIC18F14K50\n");
#endif
  MemAddr.word  = 0;                        //A memria 0 cmtl kezdnk
  ReadAddr.word = 0;
  while (1) {
    outString("\nParancs ('w' rs, 'r' olvass): ");
    cmd=usb_cdc_getc(); usb_cdc_putc(cmd);
    outString("\n");
    if (cmd=='w' || cmd=='W') {
//-- rs ---------------------------------
      outString("Gpeljen be 64 karaktert!\n");
      for (i = 0; i< BLKSIZE; i++) {
        buf[i] = usb_cdc_getc();
        usb_cdc_putc(buf[i]);
      }
      outString("\n");
//-- ktszer rjuk ki az adatokat egyms utn, hogy a foglaltsg ellenrzst is kiprbljuk
      out4hex(MemAddr.word); outString(": rs indul...\n");
      memWrite25LC256(MemAddr,buf);         //Memria lap rsa
      MemAddr.word += BLKSIZE;              //Memria cm lptetse
      delay_ms(10);							//vrunk, amg az rs lezajlik		
      memWrite25LC256(MemAddr,buf);         //Memria lap rsa mgegyszer
      MemAddr.word += BLKSIZE;              //Memria cm lptetse
      out4hex(MemAddr.word); outString(": rs vge!\n");
    }
//-- Olvass -------------------------------
    if (cmd=='r' || cmd=='R') {
      out4hex(ReadAddr.word); outString(": ");
      memRead25LC256(ReadAddr,buf);         // olvass
      for (i = 0;i<BLKSIZE; i++) {
        out2hex(buf[i]); usb_cdc_putc(' ');
      }
      ReadAddr.word += BLKSIZE;             //Memria cm lptetse
      outString("\n");
      for (i = 0;i<BLKSIZE; i++) {
        usb_cdc_putc(buf[i]);
      }
      outString("\n");
      out4hex(ReadAddr.word); outString(": ");
      memRead25LC256(ReadAddr,buf);         // olvass
      for (i = 0;i<BLKSIZE; i++) {
        out2hex(buf[i]); usb_cdc_putc(' ');
      }
      ReadAddr.word += BLKSIZE;             //Memria cm lptetse
      outString("\n");
      for (i = 0;i<BLKSIZE; i++) {
        usb_cdc_putc(buf[i]);
      }
      outString("\n");
    }
  }
}
