//----------------------------------------------------------------------------
#if TXBUFX_SIZE < 2
#error TXBUFx_SIZE is too small.  It must be larger than 1.
#elif ((TXBUFX_SIZE & (TXBUFX_SIZE-1)) != 0)
#error TBUF_SIZE must be a power of 2.
#endif

#if RXBUFX_SIZE < 2
#error RXBUFX_SIZE is too small.  It must be larger than 1.
#elif ((RXBUFX_SIZE & (RXBUFX_SIZE-1)) != 0)
#error RXBUFX_SIZE must be a power of 2.
#endif

//----------------------------------------------------------------------------
struct bufx_r {
  unsigned int in;                      // Next In Index
  unsigned int out;                     // Next Out Index
  char buf [RXBUFX_SIZE];               // Buffer
};

struct bufx_t {
  unsigned int in;                      // Next In Index
  unsigned int out;                     // Next Out Index
  char buf [TXBUFX_SIZE];               // Buffer
};

volatile static struct bufx_r rbufx = { 0, 0, };
#define SIO_RBUFLEN ((unsigned short)(rbufx.in - rbufx.out))

volatile static struct bufx_t tbufx = { 0, 0, };
#define SIO_TBUFLEN ((unsigned short)(tbufx.in - tbufx.out))

static unsigned int txx_restart = 1;    // NZ if TX restart is required

__weak void usartx_cbrx(char rxch)
{
}


/*----------------------------------------------------------------------------
  USARTX_IRQHandler
  Handles USARTX global interrupt request.
 *----------------------------------------------------------------------------*/
void USARTX_IRQHandler(void)
{
  unsigned int usr;

  usr = USARTX->SR;

  #if GPIOX_PORTNUM(USARTX_RX) >= GPIOX_PORTNUM_A
  unsigned int udr;
  if (usr & USART_SR_RXNE)
  {                                     // RX ?
    udr = USARTX->DR;
    USARTX->SR &= ~USART_SR_RXNE;       // clear interrupt
    if(!(usr & (USART_SR_ORE | USART_SR_NE | USART_SR_FE)))
      if (((rbufx.in - rbufx.out) & ~(RXBUFX_SIZE - 1)) == 0)
      {
        rbufx.buf [rbufx.in & (RXBUFX_SIZE - 1)] = (char)udr;
        rbufx.in++;
      }
    usartx_cbrx((unsigned char)udr);
  }
  #endif

  #if GPIOX_PORTNUM(USARTX_TX) >= GPIOX_PORTNUM_A
  if (usr & USART_SR_TXE)
  {                                     // TX ?
    USARTX->SR &= ~USART_SR_TXE;        // clear interrupt
    if (tbufx.in != tbufx.out)
    {
      USARTX->DR = tbufx.buf [tbufx.out & (TXBUFX_SIZE - 1)] & 0x00FF;
      tbufx.out++;
      txx_restart = 0;
    }
    else
    {
      txx_restart = 1;
      USARTX->CR1 &= ~USART_CR1_TXEIE;  // disable TX interrupt if nothing to send
    }
  }
  #endif
}

/*------------------------------------------------------------------------------
  transmit a character
 *------------------------------------------------------------------------------*/
char usartx_sendchar(char c)
{
  while(SIO_TBUFLEN >= TXBUFX_SIZE);

  tbufx.buf[tbufx.in & (TXBUFX_SIZE - 1)] = c; /* Add data to the transmit buffer */
  tbufx.in++;

  if (txx_restart)
  {                                     /* If transmit interrupt is disabled, enable it */
    txx_restart = 0;
    USARTX->CR1 |= USART_CR1_TXEIE;     /* enable TX interrupt */
  }

  return (0);
}

/*------------------------------------------------------------------------------
  receive a character (if buffer is empty: return -1)
 *------------------------------------------------------------------------------*/
char usartx_getchar(char * c)
{
  if (rbufx.in == rbufx.out)
    return 0;                           /* empty */

  *c = rbufx.buf[rbufx.out];
  rbufx.out = (rbufx.out + 1) & (RXBUFX_SIZE - 1);
  return 1;
}

/*------------------------------------------------------------------------------
  initialize the buffers
 *------------------------------------------------------------------------------*/
void usartx_init(void)
{
  tbufx.in = 0;                                    // Clear com buffer indexes
  tbufx.out = 0;
  txx_restart = 1;

  rbufx.in = 0;
  rbufx.out = 0;

  USARTX_CLOCLK_ON;

  /* Enable the USARTx Interrupt */
  NVIC->ISER[(((uint32_t)(int32_t)USARTX_IRQn) >> 5UL)] = (uint32_t)(1UL << (((uint32_t)(int32_t)USARTX_IRQn) & 0x1FUL));
  NVIC->IP[((uint32_t)(int32_t)USARTX_IRQn)] = (uint8_t)((UART_PRIORITY << (8U - __NVIC_PRIO_BITS)) & (uint32_t)0xFFUL);

  #if GPIOX_PORTNUM(USARTX_RX) >= GPIOX_PORTNUM_A
  RCC->APB2ENR |= GPIOX_CLOCK(USARTX_RX);
  GPIOX_MODE(MODE_FF_DIGITAL_INPUT, USARTX_RX);
  #define USARTX_CR1_RXNEIE  USART_CR1_RXNEIE
  #define USARTX_CR1_RE      USART_CR1_RE
  #else
  #define USARTX_CR1_RXNEIE  0
  #define USARTX_CR1_RE      0
  #endif

  #if GPIOX_PORTNUM(USARTX_TX) >= GPIOX_PORTNUM_A
  RCC->APB2ENR |= GPIOX_CLOCK(USARTX_TX);
  GPIOX_MODE(MODE_PP_ALTER_50MHZ, USARTX_TX);
  #define USARTX_CR1_TE      USART_CR1_TE
  #else
  #define USARTX_CR1_TE      0
  #endif

  USARTX->CR1 = USARTX_CR1_RXNEIE | USARTX_CR1_TE | USARTX_CR1_RE | USART_CR1_PEIE;
  USARTX->BRR = USARTX_BRR_CALC;
  USARTX->CR1 |= USART_CR1_UE;
  #undef USARTX_CR1_RXNEIE
  #undef USARTX_CR1_RE
  #undef USARTX_CR1_TE
}

/*------------------------------------------------------------------------------
  printf redirect
 *------------------------------------------------------------------------------*/
#if USARTX_PRINTF == 1
int _write (int fd, char *ptr, int len)
{
  static unsigned int usartx_inited = 0;       /* 0: not intit (call the uart_init), 1: after init */
  int i = 0;
  if(!usartx_inited)
  {
    usartx_init();
    usartx_inited = 1;
  }
  while (*ptr && (i < len)) {
    usartx_sendchar(*ptr);
    i++;
    ptr++;
  }
  return i;
}
#endif

#undef  USARTX
#undef  USARTX_IRQHandler
#undef  USARTX_IRQn
#undef  USARTX_CLOCLK_ON
#undef  USARTX_BRR_CALC
#undef  USARTX_RX
#undef  USARTX_TX
#undef  TXBUFX_SIZE
#undef  RXBUFX_SIZE
#undef  USARTX_PRINTF
#undef  USART_IO_SETMODE
#undef  usartx_inited
#undef  txx_restart
#undef  bufx_r
#undef  bufx_t
#undef  rbufx
#undef  tbufx
#undef  usartx_init
#undef  usartx_sendchar
#undef  usartx_getchar
