; Subliminal clock by Graham Richter
; (c) March 2001
;
; A 12-hour digital clock that works on the memory effect of the human retina. Only one column
; of the current time is shown at once, cycling quickly through all the columns, leaving the complete
; image on a moving eye.
;
;	PORTB are the LEDs
;	PORTA:0 is piezo buzzer
;	      2 is HR button
;	      3 is MIN button

; Time keeping:
; Prescaler=256,TMR0=256,TMR=15
; therefore 256*256*15=983040 instructions for 1 real second (@4MHz 1 instruction=1 microsecond)
; *60=58982400 instrs for 1 minute
; +1 second compensation every minute (but our 1 second is actually 983040 instructions)
; = 59965440 isntructions per minute
; *60=3597926400
; +2 seconds compensation every hour
; =3599892480 instr per hour
; *12=43198709760
; +1 sec every 12 hours
; =43199692800 instr per 12 hours
; but this should be 43200000000 at 4MHz
; difference is 307200 instructions
; so next second should be 307200/983040*15(ie TMR)
; so TMR=4.6875 for next second.
; difference =5-4.6875=0.3125
; but 256*256*0.3125=20480 instructions
; so clock is inaccurate by 20.48ms/12 hours
; which is 1.22 seconds per month or 14 seconds per year.

	include "p16f84a.inc"
	list p=16f84a, R=Dec
        errorlevel      -302,-305,-216    ; No Bank or dest select warnings
        __config 0x3ff1
        
        #define Hour 2
        #define Min  3
        #define Buzz 0
        

	cblock	0xc
		STATBUF,WBUF,SEC,MINL,MINM,HRL,HRM,TMP,COL
		BUF,SHBUF
		del1,del2
	endc


	org 0
	goto	start
	org 4
	goto	int
	
int	bcf	INTCON,T0IF		; Interrupt when TMR0 overflows
	movwf	WBUF			; Save w and STATUS regs
	movfw	STATUS
	movwf	STATBUF
	
	decfsz	TMP			;  If TMP=0, make it 15: One second has passed
	goto	cont
	movlw	15
	movwf	TMP
	
	incf	SEC
	movlw	60
	xorwf	SEC,w			; Compare secs to 60, if true, make 0 and inc minutes.
	btfss	STATUS,Z
	goto	cont
	clrf	SEC
	
	movlw	-1		; Compensation (every minute add one second to count)
	movwf	SEC
	incf	MINL
	movlw	10
	xorwf	MINL,w
	btfss	STATUS,Z
	goto	cont
	clrf	MINL
	
	incf	MINM
	movlw	6
	xorwf	MINM,w
	btfss	STATUS,Z
	goto	cont
	clrf	MINM
	
	movlw	-2		; Compensation (add 2 secs every hour)	
	movwf	SEC	
	incf	HRL
	movlw	3
	xorwf	HRL,w
	btfsc	STATUS,Z
	goto	hrflip
hrcont	movlw	10
	xorwf	HRL,w
	btfss	STATUS,Z
	goto	cont
	clrf	HRL
	incf	HRM
	goto	cont
	
hrflip	movlw	1
	xorwf	HRM,w
	btfss	STATUS,Z
	goto	hrcont
	movlw	10
	movwf	TMP		; Compensation every 12 hours (TMP 10 not 16 for 0.672s catch up)
	clrf	HRM
	clrf	HRL
	incf	HRL
	goto	hrcont


	
cont	bcf	STATUS,RP0
	movlw	0
	movwf	TMR0		; Restore w and STATUS
	movfw	STATBUF
	movwf	STATUS
	movfw	WBUF
	retfie
	
	
start	movlw	11111110b
	tris 	PORTA		; PORTA:0 is output for buzzer
	clrw
	tris	PORTB		; PORTB all output for LEDS
	movwf	PORTB

	movlw	0xff		; Turn all LEDs on
	movwf	PORTB
	bsf	STATUS,RP0	; Bank 1
	bsf	INTCON,T0IE
	bcf	OPTION_REG,T0CS ; TMR0 clock on instr cycle
	bcf	OPTION_REG,PSA	; TMR0 prescaler

	bsf	OPTION_REG,PS0	; 1:256 prescaler ratio
	bsf	OPTION_REG,PS1  ; ie TMR0 increments every 256 clock cycles.
	bsf	OPTION_REG,PS2

	bcf	STATUS,RP0	; Bank 0

	movlw	0		; Start at 01:00
	movwf	SEC
	movlw	0
	movwf	MINL
	movlw	0
	movwf	MINM
	movlw	0
	movwf	HRM
	movlw	1
	movwf	HRL
	
	movlw	1	
	movwf	TMP
	movlw	255
	movwf	TMR0
	bsf	INTCON,GIE	; Enable global interrupts
	
inf	btfsc	PORTA,Hour	; Test for hour set button
	call	SetH
	btfsc	PORTA,Min	; Test for min set button
	call	SetM

	call	del
	movfw	HRM		; First hour digit
	xorlw	0
	btfsc	STATUS,Z
	goto	nohm		; Only one hour digit, skip the initial 0
	movwf	HRM
	call	show
	call	del
nohm	movfw	HRL		; Second hour digit
	call	show	
	call	del
	call	isc		; Colon
	call	del
	movfw	MINM		; First min digit
	call	show
	call	del
	movfw	MINL		; Second Min Digit
	call	show
	call	isb		; Display a blank digit between.
	
	goto	inf		; Repeat forever

SetM	call	del		; Wait for bouncing of switch
	call	del
	btfss	PORTA,Min	; Confirm press
	return
	bcf	INTCON,GIE	; No interrupts, timer stops
	incf	MINL	
	movlw	10
	xorwf	MINL,w
	btfss	STATUS,Z
	goto	cont3
	clrf	MINL
	incf	MINM
	movlw	6
	xorwf	MINM,w
	btfss	STATUS,Z
	goto	cont3
	clrf	MINM
cont3	btfsc	PORTA,Min	; Wait for switch to release
	goto	cont3		
	call	del		; Debounce
	call	del
	bsf	INTCON,GIE	; Start timer again.
	return	


SetH	call	del		; See comments above for minute set
	call	del
	btfss	PORTA,Hour
	return
	bcf	INTCON,GIE
	incf	HRL
	movlw	3
	xorwf	HRL,w
	btfsc	STATUS,Z
	goto	hrflip2
hrcont2	movlw	10
	xorwf	HRL,w
	btfss	STATUS,Z
	goto	cont2
	clrf	HRL
	incf	HRM
	goto	cont2
	
hrflip2	movlw	1
	xorwf	HRM,w
	btfss	STATUS,Z
	goto	hrcont2
	clrf	HRM
	clrf	HRL
	incf	HRL
	goto	hrcont2
	
cont2	btfsc	PORTA,Hour
	goto	cont2
	call	del
	call	del
	bsf	INTCON,GIE
	return

	
	
show	movwf	SHBUF		; Show char in w
	xorlw	1
	btfsc	STATUS,Z	; Is char 1?
	goto	is1
	movfw	SHBUF
	xorlw	2
	btfsc	STATUS,Z	
	goto	is2
	movfw	SHBUF
	xorlw	3
	btfsc	STATUS,Z
	goto	is3
	movfw	SHBUF
	xorlw	4
	btfsc	STATUS,Z
	goto	is4
	movfw	SHBUF
	xorlw	5
	btfsc	STATUS,Z
	goto	is5
	movfw	SHBUF
	xorlw	6
	btfsc	STATUS,Z
	goto	is6
	movfw	SHBUF
	xorlw	7
	btfsc	STATUS,Z
	goto	is7
	movfw	SHBUF
	xorlw	8
	btfsc	STATUS,Z
	goto	is8
	movfw	SHBUF
	xorlw	9
	btfsc	STATUS,Z
	goto	is9
	movfw	SHBUF
	xorlw	0
	btfsc	STATUS,Z
	goto	is0


is0	movlw	8			; 8 columns to cycle through per character
	movwf	COL
shlp0	movfw	COL
	call	lookup0			; this holds bitmap for a 0.
	movwf	PORTB
	call	del			; Delay to keep LED on.
	decfsz	COL
	goto	shlp0
	return

is1	movlw	8
	movwf	COL
shlp1	movfw	COL
	call	lookup1
	movwf	PORTB
	call	del
	decfsz	COL
	goto	shlp1
	return

is2	movlw	8
	movwf	COL
shlp2	movfw	COL
	call	lookup2
	movwf	PORTB
	call	del
	decfsz	COL
	goto	shlp2
	return

is3	movlw	8
	movwf	COL
shlp3	movfw	COL
	call	lookup3
	movwf	PORTB
	call	del
	decfsz	COL
	goto	shlp3
	return

is4	movlw	8
	movwf	COL
shlp4	movfw	COL
	call	lookup4
	movwf	PORTB
	call	del
	decfsz	COL
	goto	shlp4
	return

is5	movlw	8
	movwf	COL
shlp5	movfw	COL
	call	lookup5
	movwf	PORTB
	call	del
	decfsz	COL
	goto	shlp5
	return

is6	movlw	8
	movwf	COL
shlp6	movfw	COL
	call	lookup6
	movwf	PORTB
	call	del
	decfsz	COL
	goto	shlp6
	return

is7	movlw	8
	movwf	COL
shlp7	movfw	COL
	call	lookup7
	movwf	PORTB
	call	del
	decfsz	COL
	goto	shlp7
	return

is8	movlw	8
	movwf	COL
shlp8	movfw	COL
	call	lookup8
	movwf	PORTB
	call	del
	decfsz	COL
	goto	shlp8
	return

is9	movlw	8
	movwf	COL
shlp9	movfw	COL
	call	lookup9
	movwf	PORTB
	call	del
	decfsz	COL
	goto	shlp9
	return
	
isc	movlw	8
	movwf	COL
shlpc	movfw	COL
	call	lookupc
	movwf	PORTB
	call	del
	decfsz	COL
	goto	shlpc
	return	
	
isb	movlw	8		; blank
	movwf	COL
shlpb	movfw	COL
	call	lookupb
	movwf	PORTB
	call	del
	decfsz	COL
	goto	shlpb
	return		
	
del		
	movlw	15
	movwf	del1
lp1z	movlw	25
	movwf	del2
lp2z	decfsz	del2
	goto	lp2z
	decfsz	del1
	goto	lp1z	
	
	
	return
	
	

	org	0x200		; Third program memory page to avoid PCL spillover

lookup0	addlw	255
	clrf	PCLATH
	incf	PCLATH
	incf	PCLATH		; Third page
	addwf	PCL	
	dt 	00000000b
	dt	01111110b
	dt 	11000001b
	dt 	10110001b
	dt 	10001101b
	dt 	10000011b
	dt 	01111110b
	dt 	00000000b ;<-bottom left (LSB under)
lookup1	addlw	255
	clrf	PCLATH
	incf	PCLATH
	incf	PCLATH
	addwf	PCL
	dt 	00000000b
	dt	00000001b
	dt 	11111111b
	dt 	11111111b
	dt 	11000001b
	dt 	01000001b
	dt 	01000001b
	dt 	00000000b	
lookup2	addlw	255
	clrf	PCLATH
	incf	PCLATH
	incf	PCLATH
	addwf	PCL
	dt 	00000000b
	dt	01100001b
	dt 	11110001b
	dt 	10011001b
	dt 	10001101b
	dt 	11000111b
	dt 	01000011b
	dt 	00000000b		
lookup3	addlw	255
	clrf	PCLATH
	incf	PCLATH
	incf	PCLATH
	addwf	PCL
	dt 	00000000b
	dt	01100110b
	dt 	11111111b
	dt 	10011001b
	dt 	10011001b
	dt 	11000011b
	dt 	01000010b
	dt 	00000000b
lookup4	addlw	255
	clrf	PCLATH
	incf	PCLATH
	incf	PCLATH
	addwf	PCL
	dt 	00000000b
	dt	00000100b
	dt 	11111111b
	dt 	01000100b
	dt 	00100100b
	dt 	00010100b
	dt 	00001100b
	dt 	00000000b			
lookup5	addlw	255
	clrf	PCLATH
	incf	PCLATH
	incf	PCLATH
	addwf	PCL
	dt 	00000000b
	dt	10000110b
	dt 	10001001b
	dt 	10001001b
	dt 	10001001b
	dt 	10001011b
	dt 	11111010b
	dt 	00000000b			
lookup6	addlw	255
	clrf	PCLATH
	incf	PCLATH
	incf	PCLATH	
	addwf	PCL
	dt 	00000000b
	dt	00000110b
	dt 	10001001b
	dt 	10001001b
	dt 	01001001b
	dt 	00101110b
	dt 	00011100b
	dt 	00000000b			
lookup7	addlw	255
	clrf	PCLATH
	incf	PCLATH
	incf	PCLATH	
	addwf	PCL
	dt 	00000000b
	dt	11000000b
	dt 	10100000b
	dt 	10010000b
	dt 	10001000b
	dt 	10000111b
	dt 	11000011b
	dt 	00000000b			
lookup8	addlw	255
	clrf	PCLATH
	incf	PCLATH
	incf	PCLATH	
	addwf	PCL
	dt 	00000000b
	dt	01100110b
	dt 	10011001b
	dt 	10011001b
	dt 	10011001b
	dt 	10011001b
	dt 	01100110b
	dt 	00000000b			
lookup9	addlw	255
	clrf	PCLATH
	incf	PCLATH
	incf	PCLATH
	addwf	PCL
	dt 	00000000b
	dt	01111000b
	dt 	10011100b
	dt 	10010010b
	dt 	10010001b
	dt 	10010001b
	dt 	01100000b
	dt 	00000000b			
	
lookupc	addlw	255
	clrf	PCLATH
	incf	PCLATH
	incf	PCLATH
	addwf	PCL
	dt 	00000000b
	dt	00000000b
	dt 	00000000b
	dt 	00110110b
	dt 	00110110b
	dt 	00000000b
	dt 	00000000b
	dt 	00000000b
	
lookupb	addlw	255		; blank
	clrf	PCLATH
	incf	PCLATH
	incf	PCLATH
	addwf	PCL
	dt 	00000000b
	dt	00000000b
	dt 	00000000b
	dt 	00000000b
	dt 	00000000b
	dt 	00000000b
	dt 	00000000b
	dt 	00000000b					
	
	
	end
	

