/****************************************************************************/
/*                                                                          */
/*                ACS-2602 - Kis Szkop                                      */
/*                                                                          */
/****************************************************************************/
/*  Target Chip: Xilinx XC2S200-PQ208 (Spartan-II)                          */
/*--------------------------------------------------------------------------*/
/* Gorbe megjelenites                                                       */
/*--------------------------------------------------------------------------*/
/* History:                                                                 */
/*  2007.01.05: elso verzio                                                 */
/*  2007.03.11: atirkalas, FIFO bemenet                                     */
/*  2007.03.12: pontrajzolo atirasa gyorsabbra (vonalhuzas: 6 CLK66/pont)   */
/*  2007.03.13: aprobb hibak javitgatasa                                    */
/*  2007.03.22: X kezdeti ertek a 'mode' regiszteren keresztul megadhato    */
/*  2008.04.16: min/max is tud zoom-olni (tomboket rajzol)                  */
/*  2009.04.04: fifo teljes atirasa                                         */
/****************************************************************************/

/*
 Ez a modul rajzolja az LCD-re a gorbeket. A bemeneten van egy 256x16 FIFO.
 Ket uzemmod van. Az egyik a sima rajzolas, akkor csak az also 8 bit szamit
 a bemeneten. Egyszeruen itt jonnek az ertekek, amiket ez a modul Y-nak tekint
 es osszekoti oket. Az X iranyu elmozdulast (1..63) a mode regiszterben kell
 megadni.
 A masik uzemmod, a min-max rendszer, amikor a bejovo adat min-max parokbol all.
 Ilyenkor minden min-max par alapjan ket dolgot csinal. Eloszor rajzol egy fuggoleges
 tombot vagy csak 1 pixel szeles vonalat (zoomin szeleset) min-tol max-ig magassagig,
 majd huz egy osszekoto vonalat a kovetkezo min-max vonalhoz.
 Az osszekoto vonal kiindulasi es vegzodesi Y koordinataja a ket
 min-max tol fugg.
 
 A modul a bemenetere erkezo ertekeket limitalja es eltolja, hogy a tovabbi
 feldolgozas mar csak 0-199 kozotti ertekeket kapjon.

 A mode regiszterben megadhato egy Y felezes is, ami egy fuggoleges kompresszio.
 
*/
 

`define VMIN    (128-100)
`define VMAX    (128+100)

module  curve(
    CLK66,
    regrd, regwr, rega, regdi, regdo,
    newdata, datain, curverun, busy,
    rama, ramrd, ramwr, ramack, ramdon, ramdi, ramdo
);
input           CLK66;

input           regrd;  
input           regwr;
input   [7:0]   rega;
input   [7:0]   regdi;
output  [7:0]   regdo;

input           newdata;
input   [15:0]  datain;
input           curverun;
output          busy;

output  [16:0]  rama;
output          ramrd, ramwr;
input           ramack, ramdon;
input   [7:0]   ramdi;
output  [7:0]   ramdo;

/* ---- Internal signals -------------------------------------------------- */

reg     [3:0]   state = 0;
reg             busy = 0;
reg     [7:0]   mode = 0;
reg     [7:0]   zoomin = 1;
wire    [7:0]   min, max;
reg     [7:0]   lmin, lmax;
reg     [7:0]   linecnt;

reg             datareq;            /* Fifo olvasas igeny                   */

reg     [7:0]   fifobc;             /* Fifo byte szamlalo                   */
reg             fifobn;             /* fifobc >= 1                          */
reg     [7:0]   fifowp;             /* Iro pointer                          */
reg     [7:0]   fiforp;             /* Olvaso pointer                       */
wire    [15:0]  fifoout;            /* A fifo kimenete                      */
wire    [15:0]  fbout;              /* FIFO ram kimenete                    */
reg     [2:0]   drdy;
wire            datardy;
reg             supp;

wire    [15:0]  fifoin;

reg     [7:0]   Y;
reg     [7:0]   Y2;
reg     [8:0]   DY;
reg     [8:0]   DX;
reg             LI = 0;             /* Vonalat kell huzni (legyszi)         */
reg             ST = 0;             /* Nem kell vonal, csak X lepjen        */

reg     [3:0]   lstat = 0;
reg     [8:0]   mY, mY2, mX;
reg     [2:0]   sx;
reg             LIEND = 0;
reg     [8:0]   count;
reg     [8:0]   change;
reg     [8:0]   mm;
reg             swap, mxe, mye, sigy;
reg     [8:0]   dy;
reg             PNT = 0;

reg     [1:0]   pstat = 0;
reg             PNTACK;
reg     [15:0]  urama;
reg     [15:0]  vaddr;
reg     [16:0]  rama = 17'h10000;
reg             ramrd = 0;
reg             ramwr = 0;
reg     [7:0]   ramdo;

//  assign regdo = (regrd && rega == `CUR_ZOOM) ? 
//      {fifodr, fifobn, fifob1, fifob0, fifonew, state} : 8'hzz;

    assign regdo = 8'hzz;
    
    assign fifoin[15:8] = (datain[15:8] < `VMIN) ? (`VMIN - `VMIN) :
                          (datain[15:8] > `VMAX) ? (`VMAX - `VMIN) : 
                                                   (datain[15:8] - `VMIN);
    
    assign fifoin[7:0]  = (datain[7:0] < `VMIN) ? (`VMIN - `VMIN) :
                          (datain[7:0] > `VMAX) ? (`VMAX - `VMIN) : 
                                                  (datain[7:0] - `VMIN);

    assign datardy = drdy[2] && !supp;
    assign fifoout = fbout;
    
    always @ (posedge CLK66) begin
            
        drdy <= {drdy[1:0], fifobn};
    
        if(! busy) begin
            fiforp <= 0;
            fifowp <= 0;
            fifobc <= 0;
            fifobn <= 0;
            drdy   <= 0;
            supp   <= 0;
        end
        else begin
            if(newdata)
                fifowp <= fifowp + 1;
            
            if(newdata && !(datareq && datardy)) begin
                fifobc <= fifobc + 1;
                fifobn <= 1;
            end
                
            if(datareq && datardy && !newdata) begin
                fifobc <= fifobc - 1;
                if(fifobc == 1) begin
                    fifobn <= 0;
                    drdy <= 0;
                end
            end
            
            if(datardy && datareq) begin
                fiforp <= fiforp + 1;
                supp <= 1;
            end
            else
                supp <= 0;
        end
    end

    RAMB4_S16_S16 fiforam(
        .CLKA(CLK66), .RSTA(1'b0), .ENA(1'b1), .ADDRA(fifowp), .WEA(newdata), .DIA(fifoin),
        .CLKB(CLK66), .RSTB(1'b0), .ENB(1'b1), .ADDRB(fiforp), .WEB(1'b0), .DOB(fbout)
    );
    
    assign min = fifoout[7:0];
    assign max = fifoout[15:8];
    
    always @ (posedge CLK66) begin
        
        if(regwr && rega == `CUR_LCDAL)
            urama[7:0] <= regdi;
            
        if(regwr && rega == `CUR_LCDAH)
            urama[15:8] <= regdi;
            
        if(regwr && rega == `CUR_ZOOM)
            zoomin <= regdi;
            
        if(regwr && rega == `CUR_MODE)
            mode  <= regdi;
        
        if(state == 0) begin
            if(curverun) begin
                busy <= 1;
                state <= 1;
                datareq <= 1;
            end
            else
                busy <= 0;
        end
        else if(state == 1 && datardy) begin
            datareq <= 0;
            lmin <= min;
            lmax <= max;
            linecnt <= zoomin;
            if(mode[`CUR_MODE_MINMAX])
                state <= 2;
            else begin
                datareq <= 1;
                state <= 4;
            end
        end
        else if(state == 2) begin       // huzza a fuggoleges vonalat
            Y  <= lmin;
            Y2 <= lmax;
            DX <= 0;
            LI <= 1;
            linecnt <= linecnt - 1;
            if(linecnt == 1)            // ha ez az utolso vonal, akkor 3-ra megy
                state <= 3;     
            else
                state <= 6;             // egyebkent 6-ra
        end
        else if(state == 6 && LIEND) begin
            DX <= 1;                    // meghuzta a vonalat, akkor lep egyet
            ST <= 1;                    // jobbra
            state <= 7;
        end
        else if(state == 7 && LIEND) begin
            LI <= 0;
            ST <= 0;
            state <= 2;
        end
        else if(state == 3 && LIEND) begin  // az utolso fugg. vonal utan jon az 
            LI <= 0;                        // osszekoto
            datareq <= 1;
            state <= 8;
        end
        else if(state == 8 && datardy) begin
            datareq <= 0;
            if(lmin > max) begin
                Y  <= lmin;
                Y2 <= max;
                DX <= 1;
            end
            else if(lmax < min) begin
                Y  <= lmax;
                Y2 <= min;
                DX <= 1;
            end
            else begin
                DX <= 1;
                ST <= 1;
            end
            
            LI     <= 1;
            lmin   <= min;
            lmax   <= max;
            state  <= 5;
        end
        else if(state == 4 && datardy) begin
            datareq <= 0;
            Y  <= lmin;
            Y2 <= min;
            DX <= zoomin;
            LI <= 1;
            
            lmin   <= min;
            lmax   <= max;
            state  <= 5;
        end
        else if(state == 5 && LIEND) begin
            LI <= 0;
            ST <= 0;
            linecnt <= zoomin;
            if(!curverun && !datardy)
                state <= 0;
            
            else if(mode[`CUR_MODE_MINMAX])
                state <= 2;
            else begin
                state <= 4;
                datareq <= 1;
            end
        end
    end
    
    /************************************************************/
    /*  Vonalhuzo                                               */
    /************************************************************/
    /* Bejon:   busy - ha nulla, akkor az mX-et reseteli        */
    /*             Y - vonal kezdopontja                        */
    /*            Y2 - vonal vegpontja                          */
    /*            DX - X iranyu hossz                           */
    /* mode[`..HALFY]- Y-t es Y2-t elosztja kettovel az elejen  */
    /*            LI - mehet a huzas                            */
    /*            ST - nem rajzol, csak X-et allitja            */
    /*                                                          */
    /* Kimegy: LIEND - vege                                     */
    /*            mX - az utolso X pozicio                      */
    /************************************************************/
        
    always @ (posedge CLK66) begin
        
        LIEND <= 0;
        if(! busy)
            mX <= mode[`CUR_MODE_XLOW];
        
        if(lstat == 0 && LI && !LIEND) begin
            /*
             *  Felezest es sigy szamitast megcsinalja mindenkeppen
             */
            mY    <= (mode[`CUR_MODE_HALFY]) ?  (Y >> 1)  :  Y;
            mY2   <= (mode[`CUR_MODE_HALFY]) ? (Y2 >> 1)  : Y2;
            sigy  <= (Y > Y2) ? 1 : 0;  // Y csokkenni fog
            
            
            if(ST) begin
                /*
                 *  Ha csak STep, akkor X-et lepteti, es kesz is
                 */
                mX <= mX + DX;
                LIEND <= 1;
            end
            else
                lstat <= 1;
        end
        else if(lstat == 1) begin
            if(sigy)
                DY <= mY - mY2;
            else
                DY <= mY2 - mY;
            lstat <= 2;
        end
        else if(lstat == 2) begin
            swap <= (DX > DY) ? 0 : 1;
            lstat <= 3;
        end
        else if(lstat == 3) begin
            if(! swap) begin
                dy <= DY;
                count  <= DX;
                change <= {2'b0, DX[7:1]};
                mm  <= DY - DX;
                mxe <= 1;
                mye <= 0;
            end
            else begin
                dy <= DX;
                count  <= DY;
                change <= {2'b0, DY[7:1]};
                mm  <= DX - DY;
                mxe <= 0;
                mye <= 1;
            end
            PNT   <= 1;
            lstat <= 4;
        end
        else if(lstat == 4 && PNTACK) begin
            if(count == 0) begin
                PNT   <= 0;
                LIEND <= 1;
                lstat <= 0;
            end
            else begin
                count <= count - 1;
                if((change + mm) & 9'b1_00000000) begin // if(change+mm < 0)
                    change <= change + dy;
                    mX <= mX + mxe;
                    mY <= mY + ((mye == 0) ? 0: (sigy) ? -1 : 1);
                end
                else begin
                    change <= change + mm;
                    mX <= mX + 1;
                    mY <= mY + ((sigy) ? -1 : 1);
                end
            end
        end
    end 
    
    /************************************************************/
    /*  Pontrajzolo                                             */
    /*                                                          */
    /* Az Y koordinatat fejre allitja, azaz a 0 lesz alul, 199  */
    /* lesz felul                                               */
    /************************************************************/
    /* Bejon:   mX - X koordinata                               */
    /*          mY - Y koordinata                               */
    /*       urama - terulet bal also sarka az LCD ramban       */
    /*         PNT - rajzolj! (legyszi)                         */
    /* Kimegy:                                                  */
    /*      PNTACK - kesz                                       */
    /*                                                          */
    /************************************************************/

    always @ (posedge CLK66) begin
        
        vaddr <= mY * 40;
        PNTACK <= 0;
        
        if(ramack) begin
            ramrd <= 0;
            ramwr <= 0;
        end
        
        /*
         * mX es mY egyutt jon PNT-tal 
         */
        if(pstat == 0 && PNT) begin
            pstat <= 1;
        end
        else if(pstat == 1 || (pstat == 3 && ramdon)) begin
            
            rama[15:0] <= urama + mX[7:3] - vaddr;
            sx <= mX[2:0];
            
            if(PNT) begin
                ramrd <= 1;
                pstat <= 2;
                PNTACK <= 1;
            end
            else
                pstat <= 0;
        end
        else if(pstat == 2 && ramdon) begin
            ramdo <= ramdi | (128 >> sx);
            ramwr <= 1;
            pstat <= 3;
        end
    end
endmodule

