/****************************************************************************/
/*                                                                          */
/*                ACS-2602 - Kis Szkop                                      */
/*                                                                          */
/****************************************************************************/
/*  Target Chip: Xilinx XC2S200-5PQ208 (Spartan-II)                         */
/*--------------------------------------------------------------------------*/
/* Gorbe megjelenites                                                       */
/*--------------------------------------------------------------------------*/
/* History:                                                                 */
/*  2007.01.05: xc2s30, belso ramba tarolos elso verzio                     */
/*  2007.03.14: SDRAM-os valtozat                                           */
/*  2007.03.22: 'mode' regiszter bitek valtoztatasa                         */
/*  2007.03.23: buffer feltoltesi hiba, javitva (masodszorra is)            */
/*  2007.03.24: a fenti hiba harmadik javitasa                              */
/*  2007.07.23: CH1 + CH2 kiszedese                                         */
/*  2007.07.25: CH1 - CH2 elojel javitasa                                   */
/*  2007.07.28: irq regiszter inicializalasa hianyzott                      */
/*  2008.11.27: offset hozzaadasa                                           */
/*  2008.11.28: az egesz modul ujrairasa es ezzel a duplajara gyorsitasa    */
/****************************************************************************/

module  graph(
    CLK66,
    regrd, regwr, rega, regdi, regdo, irq,
    sdrd, sddon, sdaddr, sdbufa, sdwe, sdbufdata,
    cdata, cnew, crun, cbusy
);
input           CLK66;

input           regrd;  
input           regwr;
input   [7:0]   rega;
input   [7:0]   regdi;
output  [7:0]   regdo;
output          irq;

output          sdrd;
input           sddon;
output  [12:0]  sdaddr;
input   [6:0]   sdbufa;
input           sdwe;
input   [31:0]  sdbufdata;  

output  [15:0]  cdata;
output          cnew;
output          crun;
input           cbusy;

/* ---- Internal signals -------------------------------------------------- */

/*
 *  User writeable registers
 */
reg     [7:0]   mode;           /* Mode bitek                               */
reg     [12:0]  usdaddr1;       /* Adat cime (SDRAM block)                  */
reg     [12:0]  usdaddr2;       /* Adat cime (SDRAM block)                  */ 
reg     [8:0]   offset;         /* Adat cim  (SDRAM offset)                 */
reg     [15:0]  zoomout;        /* min/max blokk hossza                     */
reg     [7:0]   uwcnt;          /* Kiirando adatok szama                    */
reg     [8:0]   b1offs;         /* Megjelenitett gorbe Y offset (+- 200)    */
reg     [8:0]   b2offs;

reg             start = 0;
reg             run = 0;
reg             stop = 0;
reg             irq = 0;

wire            datareq;        /* A buffer fill es SDRAM readert inditja   */
reg     [9:0]   bufar;          /* Buffer read address                      */
reg     [3:0]   prdy = 0;       /* pipe ready bitek                         */
wire    [7:0]   buf1out, buf2out;/* A ket buffer kimenete                   */
reg     [8:0]   tmp1, tmp2;     /* Atmeneti valtozok                        */
reg     [7:0]   dat1, dat2;     /* Pipe elso stage eredmenyek               */
reg     [7:0]   new;            /* Pipe masodik stage eredmeny              */
reg     [7:0]   min, max;       /* Pipe harmadik stage adatok (kimenet)     */
reg     [7:0]   wcnt;           /* Kiirt adatok szamlaloja (csokkeno)       */
reg     [15:0]  zcnt;           /* min/max szamitas blokk szamlalo          */
reg             newmmb = 0;     /* Uj min/max block szamitasa kezdodik      */
reg             newdw;          /* Lesz-e cnew                              */
reg             cnew = 0;       /* Uj adat van a kimeneten                  */


/*
 *  Buffer fill
 */
reg     [1:0]   drdy;           /* Megtoltott bufferek szama (0-1-2)        */
reg     [1:0]   pbrd;           /* brdy elozo allapota                      */
reg     [1:0]   breq;           /* block read request (nem trivialis)       */
reg             strt;           /* az elso block read-et jelzi (segedreg.)  */

/*
 *  SDRAM block reader
 */
reg     [1:0]   brdy = 0;       /* Minden beolvasott blokk utan eggyel no   */
reg             bufab = 0;      /* buffer A/B select (a brdy also bitje)    */ 
reg             buf12;          /* buffer 1/2 select                        */
reg     [1:0]   brstat = 0;
reg             sdrd;           /* sdram read request                       */
reg     [12:0]  sdaddr1;
reg     [12:0]  sdaddr2;
reg     [12:0]  sdaddr;
wire    [7:0]   bufaw;          /* buffer irasi cim                         */
wire            bufwe1, bufwe2; /* buffer write enable                      */


    /*
     *  A kimeno adat 16 bites, igy all ossze
     */
    assign cdata = {max, min};
    
    /*
     *  CPU read, status regiszter
     */
    assign regdo = (regrd && rega == `CUR_MODE) ? 
        {irq, run, cbusy, 5'b0} : 8'hzz;
    
    /*
     *  A vezerles 'run' jele megy a gorberajzolo modul engedelyezo bemenetere
     *  es a helyi buffertolto es sdram olvaso logikara is.
     */
    assign crun = run;
    assign datareq = run;

    /********************************************************/
    /*  User regiszterek irasa                              */
    /********************************************************/
    
    always @ (posedge CLK66) begin
        if(regwr && rega == `CUR_MODE) begin
            mode  <= regdi;
            start <= 1;
        end
        else
            start <= 0;
            
        if(regwr && rega == `CUR_ADDR1L)
            usdaddr1[7:0] <= regdi;
            
        if(regwr && rega == `CUR_ADDR1H)
            usdaddr1[12:8] <= regdi;
        
        if(regwr && rega == `CUR_ADDR2L)
            usdaddr2[7:0] <= regdi;
            
        if(regwr && rega == `CUR_ADDR2H)
            usdaddr2[12:8] <= regdi;
            
        if(regwr && rega == `CUR_OFFS1L)
            b1offs[7:0] <= regdi;
        
        if(regwr && rega == `CUR_OFFS1H)
            b1offs[8] <= regdi;
            
        if(regwr && rega == `CUR_OFFS2L)
            b2offs[7:0] <= regdi;
        
        if(regwr && rega == `CUR_OFFS2H)
            b2offs[8] <= regdi;
    
        if(regwr && rega == `CUR_CMPRSL)
            zoomout[7:0] <= regdi;
            
        if(regwr && rega == `CUR_CMPRSH)
            zoomout[15:8] <= regdi;
        
        if(regwr && rega == `CUR_OFFSETL)
            offset[7:0] <= regdi;
            
        if(regwr && rega == `CUR_OFFSETH)
            offset[8] <= regdi[0];
        
        if(regwr && rega == `CUR_DATACNT)
            uwcnt <= regdi;
    end
    
    /********************************************************/
    /*  A vezerles resz                                     */
    /********************************************************/
    
    always @ (posedge CLK66) begin
        if(regrd && rega == `CUR_MODE)
            irq <= 0;

        if(start) begin
            run    <= 1;
            bufar  <= {1'b0, offset};
            zcnt   <= zoomout;
            wcnt   <= uwcnt;
            newmmb <= 1;
        end
        else if(run) begin
            if(cnew && wcnt == 1) begin
                run  <= 0;
                stop <= 1;
            end
        end
        else if(stop && !cbusy) begin
            stop <= 0;
            irq <= 1;
        end
        
            
        /*
         *  buffer read address mindig novekszik, ha a bufferben van adat
         */
        if(drdy)
            bufar <= bufar + 1;
        /*
         *  prdy (pipe ready) bitek jelzik az egyes stage-eknek, hogy 
         *  van ervenyes adat
         *
         *  prdy[0] jelenti, hogy a buffer kimenete ervenyes
         */
        prdy <= (prdy << 1) | (|drdy);
        
        /*
         *  pipe elso stage: a mert adathoz hozzaadja az ofszetet es hatarolja
         *  0 es 255 koze
         */
        if(prdy[0]) begin
            tmp1 = {1'b0, buf1out} + b1offs;
            tmp2 = {1'b0, buf2out} + b2offs;
            dat1 <= (tmp1[8] == 0) ? tmp1[7:0] : (b1offs[8] ? 0 : 255);
            dat2 <= (tmp2[8] == 0) ? tmp2[7:0] : (b2offs[8] ? 0 : 255);
        end
        
        /*
         *  pipe masodik stage: sima, negalt, vagy differencial kimenet
         */
        if(prdy[1]) begin
            new <= (mode[`CUR_MODE_ARIT] == `CUR_MODE_SIMPLE) ?  dat1 :
                   (mode[`CUR_MODE_ARIT] == `CUR_MODE_INV) ?    -dat1 :
                            (((dat1 ^ 8'h80) - (dat2 ^ 8'h80)) ^ 8'h80);
        end
        
        /*
         *  pipe harmadik stage: min/max kereses, ha kell
         *  kimeno adat generalasa
         */
        cnew <= 0;

        if(prdy[2]) begin
            /*
             *  uj adat a kimenetre csak akkor megy, ha megy az egesz koceraj,
             *  es a wcnt nem most fog nullara lepni.
             */
            newdw = run && !((wcnt == 1) && cnew);
            
            newmmb <= 0;
            if(mode[`CUR_MODE_MINMAX] == 0) begin
                min   <= new;
                cnew  <= newdw;
            end
            else begin
                if(newmmb) begin
                    min <= new;
                    max <= new;
                end
                else begin
                    if(new < min)
                        min <= new;
                    if(new > max)
                        max <= new;
                end
                
                if(zcnt == 0) begin
                    cnew <= newdw;
                    zcnt <= zoomout;
                    newmmb <= 1;
                end
                else begin
                    zcnt <= zcnt - 1;
                end
            end
        end
        
        if(cnew)
            wcnt <= wcnt - 1;
    end
    
    /********************************************************/
    /*  Amig kell adat, szerez                              */
    /********************************************************/
    
    always @ (posedge CLK66) begin
        if(! datareq) begin
            breq <= 0;
            strt <= 1;
            drdy <= 0;
            pbrd <= 0;
        end
        else begin
            if(strt) begin
                /*
                 * indulaskor general ket req-t
                 */
                breq <= 2;
                strt <= 0;
            end
            else begin
                if(bufar[8:0] == 511 && drdy) begin
                    /*
                     *  Ha kiirult egy buffer, akkor general egy uj req-t
                     *
                     *  Az if-ben szereplo feltetel csak 1 orajelig allhat fent,
                     *  mivel bufar mindel orajelnel lep egyet, ha drdy nem 0.
                     */
                    drdy <= drdy - 1;
                    breq <= breq + 1;
                end
                else if(pbrd != brdy) begin
                    /*
                     *  SDRAM olvaso megvaltoztatta brdy-t, azaz kesz egy blokk
                     */
                    drdy <= drdy + 1;
                    pbrd <= brdy;
                end
            end
        end
    end
        
    /********************************************************/
    /*  Amig van breq, generalja az SDRAM olvasasokat       */
    /********************************************************/
    /*
     *  Van 2db 1024 byte-os buffer. Mindketto A es B reszre van osztva.
     *  Igy keletkezik logikailag 4db 512 byteos buffer, B1A, B1B, B2A, B2B.
     *     _______________
     *  1 |___A___|___B__|
     *     _______________
     *  2 |___A___|___B__|
     *
     *  Az 1-es bufferhez usdaddr1, a 2-eshez usdaddr2 regisztereket tudja
     *  a sw beallitani. Ha egy bufferbe beolvasott egy blokkot (512 byte),
     *  akkor a buffer sdaddr regiszteret eggyel megnoveli.
     *
     *  A buffer kiiolvasasakor cim a bufar[] cimregiszterben jon. Legfelso bit
     *  A/B buffer kozott valt. A cim 1-es es 2-es bufferre azonos, a ket buffer
     *  kimenete a buf1out es buf2out 8 bites vektorok.
     */
    always @ (posedge CLK66) begin
            
        sdrd <= 0;
        
        if(! datareq) begin
            brdy  <= 0;
            bufab <= 0;
            brstat <= 0;
            sdaddr1 <= usdaddr1;
            sdaddr2 <= usdaddr2;
        end
        else if(brstat == 0 && (breq != brdy)) begin
            sdaddr <= sdaddr1;
            sdaddr1[10:0] <= sdaddr1[10:0] + 1;
            buf12 <= 0;     // buffer #1
            sdrd  <= 1;
            brstat <= mode[`CUR_MODE_2OP] ? 1 : 2;
        end
        else if(sddon && brstat == 1) begin
            sdaddr <= sdaddr2;
            sdaddr2[10:0] <= sdaddr2[10:0] + 1;
            buf12  <= 1;    // buffer #2
            sdrd   <= 1;
            brstat <= 2;
        end
        else if(sddon && brstat == 2) begin
            brdy   <= brdy + 1;
            bufab  <= ~bufab;
            brstat <= 0;
        end
    end
    
    assign bufwe1 = sdwe & (buf12 == 0);
    assign bufwe2 = sdwe & (buf12 == 1);
    assign bufaw  = {bufab, sdbufa};
    
    buffer8_32 buf1(
        .clk8(CLK66),  .addr8(bufar), .dout8(buf1out),
        .clk32(CLK66), .addr32(bufaw), .wr32(bufwe1), .din32(sdbufdata)
    );
        
    buffer8_32 buf2(
        .clk8(CLK66),  .addr8(bufar), .dout8(buf2out),
        .clk32(CLK66), .addr32(bufaw), .wr32(bufwe2), .din32(sdbufdata)
    );
endmodule           
