/****************************************************************************/
/*                                                                          */
/*                ACS-2602 - Kis Szkop                                      */
/*                                                                          */
/****************************************************************************/
/*  Target Chip: Xilinx XC2S200-PQ208 (Spartan-II)                          */
/*--------------------------------------------------------------------------*/
/* RAM interface, CPU interface                                             */
/*--------------------------------------------------------------------------*/
/* History:                                                                 */
/*  2004.01.23 - 2004.03.10: ACS-2004 ram.v                                 */
/*                                                                          */
/*  2007.01.03: kiindulas az acs-2004-bol, nincs cpu a8-a15                 */
/*  2007.03.08: atszegezes xc2s150-pq208-ra, cpu a8-a15 vissza              */
/*  2007.03.13: atiras harom csatornasra kettorol                           */
/*  2007.03.19: ACTIVEADDRESS 6-rol 7 bitre szelesitve                      */
/*  2008.02.19: Csak egy komment javitasa                                   */
/*  2008.11.27: ACTIVEADDRESS 7-rol 8 bitre szelesitve                      */
/****************************************************************************/

`define REG     0
`define RAM     1
`define CPU     0
`define INT     1

/*
 *  A CPU cim regiszterek cimzesehez hasznalt aktiv resze
 */
`define ACTIVEADDRESS   7:0

/****************************************************************************

A RAM-ot belso logika es a CPU egyarant tudja irni/olvasni.
A ciklusok 3 orajelig tartanak. A cim vegig kint van. Olvasaskor az OE a cimmel
egyutt jelenik meg, de csak 2 orajelig van kint. Az OE felfutasakor veszi el az
adatot a RAM-tol. Irasnal a WR csak a masodik orajel alatt aktiv, ahogy az adat
is. 
****************************************************************************/


module ramctrl(CLK66,
           ALE, nRD, nWR, CPUAD, CPUAH,
           nRAMOE, nRAMWR, nRAMCE, RAMDATA, RAMADDR,
           intaddr0, intrdreq0, intwrreq0, intack0, intdon0, intwdata0,
           intaddr1, intrdreq1, intwrreq1, intack1, intdon1, intwdata1,
           intaddr2, intrdreq2, intwrreq2, intack2, intdon2, intwdata2,
//         intaddr3, intrdreq3, intwrreq3, intack3, intdon3, intwdata3,
           intrdata,
           regread, regwrite, regdo, regdi, rega, sdbread, sdbwrite, sdba
);

input           CLK66;          /* 66 MHz clock input                       */
input           ALE;            /* CPU ALE                                  */
input           nRD;            /* CPU RD                                   */
input           nWR;            /* CPU WR                                   */
inout   [7:0]   CPUAD;          /* CPU Address low in / data in/out         */
input   [7:0]   CPUAH;          /* CPU Address high                         */

output          nRAMOE;         /* RAM OE                                   */
 reg            nRAMOE = 1;
output          nRAMWR;         /* RAM WR                                   */
 reg            nRAMWR = 1;
output          nRAMCE;
inout   [7:0]   RAMDATA;        /* RAM data                                 */
output  [16:0]  RAMADDR;        /* 128k RAM address                         */
 reg    [16:0]  RAMADDR;

input   [16:0]  intaddr0;       /* Belso RAM hozzaferes cim                 */
input           intrdreq0;      /* Belso RAM olvasas kerelem                */
input           intwrreq0;      /* Belso RAM iras kerelem                   */
output          intack0;        /* Belso RAM hozzaferes elfogadva           */
output          intdon0;        /* Belso RAM hozzaferes vege                */
input   [7:0]   intwdata0;      /* Belso RAM iras adat                      */

input   [16:0]  intaddr1;       /* Belso RAM hozzaferes cim                 */
input           intrdreq1;      /* Belso RAM olvasas kerelem                */
input           intwrreq1;      /* Belso RAM iras kerelem                   */
output          intack1;        /* Belso RAM hozzaferes elfogadva           */
output          intdon1;        /* Belso RAM hozzaferes vege (adat elviheto)*/
input   [7:0]   intwdata1;      /* Belso RAM iras adat                      */

input   [16:0]  intaddr2;       /* Belso RAM hozzaferes cim                 */
input           intrdreq2;      /* Belso RAM olvasas kerelem                */
input           intwrreq2;      /* Belso RAM iras kerelem                   */
output          intack2;        /* Belso RAM hozzaferes elfogadva           */
output          intdon2;        /* Belso RAM hozzaferes vege (adat elviheto)*/
input   [7:0]   intwdata2;      /* Belso RAM iras adat                      */
 
output  [7:0]   intrdata;       /* Belso RAM olvasas adat                   */
 reg    [7:0]   intrdata;

output          regread;        /* CPU regiszter olvasas                    */
 reg            regread = 0;
output          regwrite;       /* CPU regiszter iras                       */
 reg            regwrite = 0;
input   [7:0]   regdo;          /* CPU regiszter olvasas adat               */
output  [7:0]   regdi;          /* CPU regiszter iras adat                  */
output  [7:0]   rega;           /* CPU regiszter iras cim                   */

output          sdbread;
 reg            sdbread = 0;
output          sdbwrite;
 reg            sdbwrite = 0;
output  [15:0]  sdba;

/****************************************************************************/
/*  Belso regiszterek - jelek                                               */
/****************************************************************************/

reg     [7:0]   alow;           /* CPU a0..a7 LATCH                         */
reg     [1:0]   rdd = 0;        /* */
reg     [1:0]   wrd = 0;        /* */
reg             ramsp;          /* RamSpace  (1000 - EFFF)                  */
reg             sdrsp;          /* SDRAM acc (F000 - F3FF)                  */
reg             regsp;          /* RegSpace  (FF00 - FFFF)                  */
wire            rdfall, wrfall; /* RD/WR lefuto el                          */

reg     [2:0]   seq = 0;        /* RAM iras/olvasas idozitese               */
reg             delrd = 0;      /* CPU rd ciklus kesleltetesere             */
reg             delwr = 0;      /* CPU wr ciklus kesleltetesere             */
reg             crdmux = 0;     /* CPU read data mux select                 */
reg             rwdmux = 0;     /* RAM write data mux select                */
reg             intrd = 0;      /* Belso RAM olvasas van folyamatban        */
reg             intwr = 0;      /* Belso RAM iras van folyamatban           */
reg     [7:0]   rdlatch;        /* RAM-bol felolvasott adat CPU-nak         */
reg     [7:0]   rtocreg;        /* Belso reg-bol felolvasott adat CPU-nak   */
reg             rambde = 0;     /* RAM BusDriverEnable                      */
reg             crdle = 0;

reg     [3:0]   intwsel = 0;    /* belso ram iras/olvasas nyertese          */
reg     [3:0]   intdon = 0;
reg     [3:0]   intack = 0;


    assign nRAMCE = 0;
    
    assign intdon0 = intdon[0];
    assign intdon1 = intdon[1];
    assign intdon2 = intdon[2];
//  assign intdon3 = intdon[3];

    assign intack0 = intack[0];
    assign intack1 = intack[1];
    assign intack2 = intack[2];
//  assign intack3 = intack[3];
    
    /* 
     * CPU address latch es meg azt is taroljuk itt, 
     * hogy a cimtartomany kihez tartozik
     */
    always @ (ALE or CPUAD or CPUAH) begin
        if(ALE) begin
            alow  = CPUAD;
            ramsp = (CPUAH >= 8'h10 && CPUAH < 8'hf0);
            sdrsp = (CPUAH >= 8'hf0 && CPUAH < 8'hf4);
            regsp = (CPUAH == 8'hff);
        end
    end
    
    /*
     *  Az orajel felfuto elevel mintavetelezzuk a CPU RD es WR jeleket
     *  es ezek alapjan detektaljuk a lefuto eleket
     */
    always @ (posedge CLK66) begin
        rdd  <= {rdd[0], nRD};
        wrd  <= {wrd[0], nWR};
    end
    
    assign rdfall = !rdd[0] && rdd[1];
    assign wrfall = !wrd[0] && wrd[1];

    /*
     *  Kell ide egy latch, ami a RAM-bol olvasott adatot tartja, amikor
     *  mar a RAM olvasasi ciklus befejezodott
     */
    always @ (crdle or RAMDATA)
        if(crdle)
            rdlatch = RAMDATA;
    
    /*
     *  CPU data bus mux. es driver
     */
    assign CPUAD = (nRD == 1) ? 8'hzz : (crdmux == `RAM) ? rdlatch : rtocreg;

    /*
     *  Belso regiszterekbe irando adat, maga a CPU adat, 
     *  a cim a CPU cim also 8 bitje
     */
    assign regdi = CPUAD;
    assign rega = alow[`ACTIVEADDRESS];
    assign sdba = {CPUAH, alow};
    
    /*
     *  RAM data bus multiplexer es driver
     */
    assign RAMDATA = (rambde == 0) ? 8'hzz : (rwdmux == `CPU) ? CPUAD : 
                                             (intwsel[0]) ? intwdata0 :
                                             (intwsel[1]) ? intwdata1 :
                                                            intwdata2;
    
    /*
     *  Ez itt minden, ami meg hatra van...
     */
    always @ (posedge CLK66) begin

        intdon <= 0;
        intack <= 0;
        
        if(seq[1]) begin
            /*
             *  RAM ciklus kozepe
             */
            if(intwr || intrd)
                intdon <= intwsel;
            
            nRAMWR <= 1;
            nRAMOE <= 1;
            rambde <= 0;
            crdle  <= 0;
            if(intrd)
                intrdata <= RAMDATA;
        end
        if(seq[2]) begin
            /*
             *  RAM ciklus utolso orajele
             */
            intrd <= 0;
            intwr <= 0;
            seq   <= 0;
        end
        seq <= {seq[1:0], 1'b0};
        
        /*
         *
         */
        delrd <= 0;         // ezteket itt toroljuk
        delwr <= 0;
        
        
        if(((rdfall || wrfall) && ramsp) || delrd || delwr) begin   // cpu access req
            if(seq[1:0]) begin
                delrd <= rdfall | delrd; 
                delwr <= wrfall | delwr;
            end
            else begin
                /*
                 *  CPU RAM ciklus kezdodik
                 */
                crdmux  <= `RAM;                // CPU Read Data MUX select RAM
                RAMADDR <= {1'b0, CPUAH, alow}; // Cim beall
                seq <= 1;
                if(rdfall || delrd) begin       //  cpu olvasas kezdodik
                    nRAMOE <= 0;
                    crdle  <= 1;
                end
                else begin                      // cpu iras kezdodik
                    rwdmux <= `CPU;             // RAM Write Data MUX select CPU
                    nRAMWR <= 0;
                    rambde <= 1;
                end
            end
        end
        else if((seq[1:0] == 0) && (intrdreq0 || intrdreq1 || intrdreq2 || 
                                    intwrreq0 || intwrreq1 || intwrreq2)) begin
            
            if(intrdreq0 || intwrreq0) begin
                intack  <= 4'b0001;
                intwsel <= 4'b0001;
                RAMADDR <= intaddr0;
            end
            else if(intrdreq1 || intwrreq1) begin
                intack  <= 4'b0010;
                intwsel <= 4'b0010;
                RAMADDR <= intaddr1;
            end
            else begin
                intack  <= 4'b0100;
                intwsel <= 4'b0100;
                RAMADDR <= intaddr2;
            end
                
            rwdmux <= `INT;
            seq <= 1;
            
            if(intrdreq0 || 
               (intrdreq1 && !intwrreq0) ||
               (intrdreq2 && !intwrreq0 && !intwrreq1)) begin
                intrd   <= 1;
                nRAMOE  <= 0;
            end
            else begin
                intwr   <= 1;
                nRAMWR  <= 0;
                rambde  <= 1;
            end
        end
    
        /*
         *  Chip belso regisztereinek elerese:
         *  
         *  Ha a CPU olvas, akkor az adatnak a 'regdi'-en kell lennie
         *  mindaddig amig regread aktiv.
         *  
         *  Ha a CPU ir, akkor az adat a regdo-rol elveheto a regwrite
         *  utan egy orajellel valahogy igy:
         *
         *  always @ (posedge clk)  
         *     if(regwrite == 1)
         *          regiszterem <= regdo;
         */
        regread  <= 0;
        regwrite <= 0;
        sdbread <= 0;
        sdbwrite <= 0;
        
        if(rdfall && regsp) begin
            regread <= 1;
            crdmux  <= `REG;
        end
        
        if(rdfall && sdrsp) begin
            sdbread <= 1;
            crdmux  <= `REG;
        end
        
        if(wrfall && regsp)
            regwrite <= 1;
            
        if(wrfall && sdrsp)
            sdbwrite <= 1;

        if(regread || sdbread)
            rtocreg <= regdo;
    end
endmodule

