/****************************************************************************/
/*                ACS-2602 - Kis Szkop                                      */
/****************************************************************************/
/* SRAM interface                                                           */
/*--------------------------------------------------------------------------*/
/*  Target: Xilinx XC2S200-5PQ208                                           */
/*--------------------------------------------------------------------------*/
/* History:                                                                 */
/*  2005.09.08: kezdet                                                      */
/*  2005.10.18: RAM mukodes attekintese                                     */
/*  2006.01.27: Uj koncepcio kidolgozasa                                    */
/*  2006.02.13: Arbiter, complete jel, par komment                          */
/*  ----------------------------------------------------------------------- */
/*  2006.03.08: Elso HW teszt.                                              */
/*    - bre, bwe ertelmezesi zavarok, fixalva                               */
/*    - inicializalas atirva automatikusrol a 'mehet' nevu jel megjottekorra*/
/*    - UCF: minden jel LVTTL, drive=24, slew=fast                          */
/*                                                                          */
/*  2006.03.18:                                                             */
/*    - 'mehet' jel megszuntetese, mert elozoleg is jo volt az, csak nem 15,*/
/*      hanem 150 us-ot kellett volna varni. Most 327.68us                  */
/*    - prog. file generation-ban is be kell allitani a DLL startup wait-et */
/*  2006.04.20: valami bizonytalan, RCKFB bufg beteve. arbiter atirva       */
/*              kesobb egesz DLL kiszedve, azota stabil (kulso 59 MHz orajel)*/
/*  2006.05.31: apro hiba az arbiterben                                     */
/*  2007.02.06: video/kamera prioritaslogika kiszedve, mindig a video a magasabb */
/*  2007.02.28: state == 1 CM<=`RCMD_NOP berakasa                           */
/*  ======================================================================= */
/*  2007.03.08: kiindulas a DigiCam8 sdram.v-bol                            */
/*  2007.03.10: rb, rset regiszterek hozzaadasa gyorsitasi szandekkal       */
/*  2007.03.11: toggle FF-ok kiszedese, mivel azonos CLK-n jar a tobbiekkel */
/****************************************************************************/


module  sdramifc(
    CLK66,
    RD, RA, RCE, RAS, CAS, RWR, DQML, DQMH, CKE,
    srd0, srd1, srd2, srd3, 
    swr0, swr1, swr2, swr3,
    don0, don1, don2, don3,
    rad0, rad1, rad2, rad3, 
    bufa, bdw,  
    bwe0, bwe1, bwe2, bwe3,
    brd0, brd1, brd2, brd3
);

input           CLK66;                  /* 66 MHz Master clock              */

inout   [31:0]  RD;                     /* 32 bit SDRAM data bus            */
output  [11:0]  RA;                     /* 12 bit SDRAM address             */
output          RCE;                    /* SDRAM chip enable                */
output          RAS;                    /* SDRAM Row address select         */
output          CAS;                    /* SDRAM Column address select      */
output          RWR;                    /* SDRAM Write                      */
output          DQML;                   /* SDRAM data buffer enable         */
output          DQMH;                   /* SDRAM data buffer enable         */
output          CKE;                    /* SDRAM clock enable               */

input           srd0, srd1, srd2, srd3; /* Start read pulses SD -> buff     */
input           swr0, swr1, swr2, swr3; /* Start write pulses buff -> SD    */
output          don0, don1, don2, don3; /* Done outputs                     */
input   [12:0]  rad0, rad1, rad2, rad3; /* SDRAM addresses (8k x 512 blocks)*/
output  [31:0]  bdw;                    /* Buffer data to write (Common)    */
output  [6:0]   bufa;                   /* Buffer addresses 128 * 4byte     */

output          bwe0, bwe1, bwe2, bwe3; /* Buffer write enable signals      */
input   [31:0]  brd0, brd1, brd2, brd3; /* Data from buffers                */

wire    [31:0]  brd;

wire    [4:0]   dones;
wire    [3:0]   bufwe;

wire            treq0, read0;
wire            treq1, read1;
wire            treq2, read2;
wire            treq3, read3;

reg     [3:0]   r = 0;
reg     [1:0]   rb = 0;
reg             progress = 0;
reg             rset = 0;
reg             read;
reg     [12:0]  addr;

    assign CKE = 1'b1;  // always enabled
                    
    assign bwe0 = bufwe[0];
    assign bwe1 = bufwe[1];
    assign bwe2 = bufwe[2];
    assign bwe3 = bufwe[3];
    
    assign don0 = dones[0];
    assign don1 = dones[1];
    assign don2 = dones[2];
    assign don3 = dones[3];
    
    assign brd  = (rb == 0) ? brd0 :
                  (rb == 1) ? brd1 :
                  (rb == 2) ? brd2 : brd3;

    always @ (posedge CLK66) begin
        if(progress) begin              // valaki akiv
            if(dones[4]) begin
                progress <= 0;
                r        <= 0;
                rset     <= 0;
            end
        end
        else if(!rset) begin            // nincs aktiv, nincs valasztott
            if(treq0) begin
                r  <= 4'b0001;
                rb <= 0;
                rset <= 1;
            end
            else if(treq1) begin
                r  <= 4'b0010;
                rb <= 1;
                rset <= 1;
            end
            else if(treq2) begin
                r  <= 4'b0100;
                rb <= 2;
                rset <= 1;
            end
            else if(treq3) begin
                r  <= 4'b1000;
                rb <= 3;
                rset <= 1;
            end
        end
        else begin                      // valasztott valakit, aktivalja
            progress <= 1;
            read <= (rb == 0) ? read0 : (rb == 1) ? read1 : (rb == 2) ? read2 : 
                                                                        read3;
            
            addr <= (rb == 0) ? rad0 :  (rb == 1) ? rad1 :  (rb == 2) ? rad2 : 
                                                                        rad3;
        end
    end

    sdphyifc    u1(
        .CLK66(CLK66), .RD(RD), .RA(RA), .RCE(RCE), .RAS(RAS),
        .CAS(CAS), .RWR(RWR), .DQML(DQML), .DQMH(DQMH),
        .treq(progress), .read(read), .addr(addr), .chan(r), .complete(dones),
        .bufalow(bufa), .bufwe(bufwe), .bufdatao(bdw), .bufdatain(brd)
    );
    
    
    inpifc  u1a(
        .CLK66(CLK66), 
        .srd(srd0), .swr(swr0), .treq(treq0), .read(read0), .done(dones[0])
    );
    
    inpifc  u1b(
        .CLK66(CLK66),
        .srd(srd1), .swr(swr1), .treq(treq1), .read(read1), .done(dones[1])
    );
    
    inpifc  u1c(
        .CLK66(CLK66),
        .srd(srd2), .swr(swr2), .treq(treq2), .read(read2), .done(dones[2])
    );
    
    inpifc  u1d(
        .CLK66(CLK66),
        .srd(srd3), .swr(swr3), .treq(treq3), .read(read3), .done(dones[3])
    );
endmodule


module  inpifc(
    CLK66, srd, swr, treq, read, done
);

input           CLK66;
input           srd;            /* Start read block pulse (1 CLK66 wide)    */
input           swr;            /* Start write block pulse (1 CLK66 wide)   */
output          treq;           /* Transfer request to sdphyifc (level)     */
 reg            treq = 0;
output          read;           /* Read=1 or write=0 select                 */
 reg            read;
input           done;           /* 1 CLK66 wide Transfer complete pulse     */

    always @ (posedge CLK66) begin
        if(srd || swr) begin
            treq <= 1;
            read <= srd;
        end
        else if(done) begin
            treq <= 0;
        end
    end
endmodule


`define INIT_MODE   12'b00_0_00_010_0_111

`define _RCE    4'h8
`define _RAS    4'h4
`define _CAS    4'h2
`define _RWR    4'h1

`define RCMD_NOP    (`_RCE|`_RAS|`_CAS|`_RWR)
`define RCMD_ACT    (            `_CAS|`_RWR)
`define RCMD_READ   (      `_RAS      |`_RWR)
`define RCMD_WRITE  (      `_RAS            )
`define RCMD_PREC   (            `_CAS      )
`define RCMD_REF    (                  `_RWR)
`define RCMD_LOADMODE   4'b0

/* // xsynthesis attribute register_duplication sdphyifc "no"; */

module  sdphyifc(
    CLK66,
    RD, RA, RCE, RAS, CAS, RWR, DQML, DQMH,
    treq, read, chan, complete, addr, bufalow, bufwe, bufdatao, bufdatain
);

input           CLK66;
inout   [31:0]  RD;
output  [11:0]  RA;
 reg    [11:0]  RA;
output          RCE;
output          RAS;
output          CAS;
output          RWR;
output          DQML;
output          DQMH;

input           treq;               /* Transfer request                     */
input           read;               /* Read (1) or write (0) select         */
input   [3:0]   chan;               /* bufwe, complete select               */
output  [4:0]   complete;           /* 1 CLK66 wide complete output         */
 reg    [4:0]   complete = 0;
input   [12:0]  addr;               /* Block address input                  */
output  [6:0]   bufalow;            /* Buffer address low bits (128 words)  */
 reg    [6:0]   bufalow;
output  [3:0]   bufwe;              /* Buffer WEN signals                   */
 reg    [3:0]   bufwe;

output  [31:0]  bufdatao;           /* Data to buffer (read from RAM)       */
 reg    [31:0]  bufdatao = 0;
input   [31:0]  bufdatain;          /* Data to RAM (output of the buffer)   */

/* ---- Internal registers  ----------------------------------------------- */
 
reg     [3:0]   CM = `RCMD_NOP;
reg     [1:0]   DQ = ~0;
reg             rdwr;
reg     [7:0]   count;              /* Word counter                         */
reg     [8:0]   state = (1 << 8);
reg     [8:0]   after_ref = (1 << 8);
reg     [1:0]   initstat = 0;
reg     [9:0]   refcnt = 0;         /* Refresh timer                        */
reg     [15:0]  initcnt = 0;        /* Init delay                           */
reg     [2:0]   reftime;

reg             bwe = 0;            /* Buffer write enable ...              */
reg     [2:0]   bwed = 0;
reg             bre = 0;            /* Buffer read enable ...               */
reg             bred = 0;
reg     [31:0]  rdout;
wire            incbufa;
reg             lastinc = 0;        /* A complete jel eloallitasahoz kell   */


    assign RCE = CM[3];
    assign RAS = CM[2];
    assign CAS = CM[1];
    assign RWR = CM[0];
    assign DQML = DQ[1];
    assign DQMH = DQ[0];
    
    assign RD = (bred) ? rdout : 32'hzzzzzzzz;

    /* 
     * Buffer cim inkrementalasa
     */
    assign incbufa = (bwed[2] || bre);
    
    always @ (posedge CLK66) begin
    
        refcnt   <= refcnt + 1;
        initcnt  <= initcnt + 1;
        rdout    <= bufdatain;
        bufdatao <= RD;

        bwed <= {bwed[1:0], bwe};
        bufwe <= (bwed[1]) ? chan : 4'b0;
        
        bred <= bre;
        
                
        /*
         *  A complete jel 1 orajelig tart, megpedig az utols buffer
         *  iras vagy olvasas utan
         */
        lastinc  <= incbufa;
        complete <= (lastinc && !incbufa) ? (5'b10000 | chan) : 0;
        
        if(incbufa)
            bufalow <= bufalow + 1;
        else
            bufalow <= 0;
            
        if(state[8]) begin      // INIT
            if(initstat == 0 && initcnt >= 32768) begin
                initstat <= 1;
                state <= (1 << 4);  // PRECHARGE, NOP, AUTO_REFRESH, 7*NOP
            end
            else if(initstat == 1) begin
                initstat <= 2;
                state <= (1 << 6);          // AUTO_REFRESH, 7*NOP
            end
            else if(initstat == 2) begin
                RA <= `INIT_MODE;
                CM <= `RCMD_LOADMODE;
                initstat <= 3;
            end
            else if(initstat == 3) begin
                CM <= `RCMD_NOP;
                refcnt    <= 0;
                after_ref <= (1 << 0);
                state <= (1 << 0);
            end
        end
        else if(state[0]) begin
            if(refcnt >= (512+256+128)) // 896 * 15ns = 13.4us (15.625 needed)
                state <= (1 << 6);
            else if(treq) begin
                RA <= addr[12:1];
                DQ <= ~0;
                CM <= `RCMD_ACT;
                rdwr <= read;
                bre <= (read) ? 0 : 1; // buffer read == RAM write!!!
                state <= (1 << 1);
            end
        end
        else if(state[1]) begin
            CM <= `RCMD_NOP;            // ez eddig nem volt itt (2007.02.28)
            state <= (1 << 2);
        end
        else if(state[2]) begin
            DQ <= 0;//bank, auto prec, 0, 0, 8bit column
            RA <= {addr[12], 3'b0, addr[0], 7'b0};
            CM  <= (rdwr) ? `RCMD_READ : `RCMD_WRITE;
            bwe <= (rdwr) ?  1         :  0;
            count <= 127;   // 128 szo/block = 512byte
            state <= (1 << 3);
        end
        else if(state[3]) begin
            count <= count - 1;
            CM <= `RCMD_NOP;
            if(count == 0) begin    // ez volt az utolso NOP
                bwe   <= 0;
                bre   <= 0;
                DQ    <= ~0;        // nem ir, nem olvas tobbet
                state <= (1 << 4);
            end
        end
        else if(state[4]) begin
            CM <= `RCMD_PREC;
            RA[10] <= 1;            // Precharge all banks
            state <= (1 << 5);
        end
        else if(state[5]) begin
            CM <= `RCMD_NOP;
            state <= (1 << 6);
        end
        else if(state[6]) begin
            reftime <= 0;
            CM <= `RCMD_REF;
            state <= (1 << 7);
        end
        else if(state[7]) begin
            CM <= `RCMD_NOP;
            reftime <= reftime + 1;
            if(reftime == 7) begin
                state  <= after_ref;
                refcnt <= 0;
            end
        end
    end
endmodule

