/****************************************************************************/
/*                                                                          */
/*                ACS-2602 - Kis Szkop                                      */
/*                                                                          */
/****************************************************************************/
/*  Target Chip: Xilinx XC2S200-PQ208 (Spartan-II)                          */
/*--------------------------------------------------------------------------*/
/* - TOP modul                                                              */
/*--------------------------------------------------------------------------*/
/* History:                                                                 */
/*  2004.01.23 - 2004.03.10: ACS-2004 ram.v                                 */
/*                                                                          */
/*  2007.03.08: indulunk, Apuka!                                            */
/*  2007.07.08: kalibrator hozzaadasa                                       */
/*  2007.07.12: LCD IRQ hozzaadaa                                           */
/*  2007.07.23: kalibrator atirasa pos, neg regiszteresre                   */
/*  2007.07.28: IRQ kimenet open drain                                      */
/*  2007.09.18: calibrator komment hozzaadasa                               */
/*  2007.10.27: calibrator atirasa 16 bitesre                               */
/*  2007.10.31: trigger valtoztatas, AC csatolas ki, qualifier sajat komp.  */
/*  2008.04.16: calibratoron trigger is kimehet                             */
/*  2008.05.20: calibrator prescaler 10,100,1000 hozzaadasa                 */
/*  2008.05.25: calibrator DC out hozzaadasa                                */
/*  2008.11.27: regisztercimek teljes atirasa                               */
/*  2008.11.28: verzioszam bevezetese                                       */
/*  2009.03.02: Cal out DC 0 hozzaadva                                      */
/*  2009.03.18: trigger hiba javitasa, Verzio: 2.2                          */
/*  2009.03.20: Chip V3.0, pretrigger es tarolas hossza mintakban szamolando*/
/*              block helyett, ezert store es trigger jelentos atirasa      */
/*  2009.04.04: curve.v modositas                                           */
/*  2009.06.05: Chip V4.0 Ujabb trigger funkciok hozzaadasa                 */
/****************************************************************************/

`define CHIP_VER    8'h40   // Ver: 4.0

`define RAM_ALO     1
`define RAM_AHI     2
`define RAM_CTRL    3

`define CUR_MODE    4   // ff04
`define CUR_ZOOM    5   // ff05
`define CUR_LCDAL   6   // ff06
`define CUR_LCDAH   7
`define CUR_CMPRSL  8   // ff08
`define CUR_CMPRSH  9
`define CUR_ADDR1L  10  // ff0a
`define CUR_ADDR1H  11
`define CUR_ADDR2L  12  // ff0c
`define CUR_ADDR2H  13
`define CUR_OFFSETL 14  // ff0e
`define CUR_OFFSETH 15
`define CUR_OFFS1L  16  // ff10
`define CUR_OFFS1H  17
`define CUR_OFFS2L  18
`define CUR_OFFS2H  19
`define CUR_DATACNT 20


`define  CUR_MODE_MINMAX    6
`define  CUR_MODE_HALFY     7
`define  CUR_MODE_ARIT      5:3
`define  CUR_MODE_2OP       5   // bitpos
`define  CUR_MODE_XLOW      2:0

`define  CUR_MODE_SIMPLE    0
`define  CUR_MODE_INV       1
//`define    CUR_MODE_ADD       4
`define  CUR_MODE_SUB       5

`define LCD_MASK            24
`define LCD_MODE            25

`define ACQ_DROPL           28
`define ACQ_DROPH           29
`define ACQ_EXT             30

`define  ACQ_EXTMEMSEL      0
`define  ACQ_MINMAX         6

`define CAL_MODE            36
`define CAL_PLOW            37
`define CAL_PHIGH           38
`define CAL_NLOW            39
`define CAL_NHIGH           40

`define TRG_TPNT1           48
`define TRG_TPNT2           49
`define TRG_TPNT3           50
`define TRG_TPNT4           51
`define TRG_PRE1            52
`define TRG_PRE2            53
`define TRG_PRE3            54
`define TRG_PRE4            55
`define TRG_STORE1          56
`define TRG_STORE2          57
`define TRG_STORE3          58
`define TRG_STORE4          59

`define TRG_MODE1           60  // ff
`define TRG_MODE2           61  // ff
`define TRG_MODE3           62  // ff
`define TRG_MODE4           63  // ff
`define TRG_TIMEA1          64  // ff
`define TRG_TIMEA2          65
`define TRG_TIMEA3          66
`define TRG_TIMEA4          67
`define TRG_TIMEB1          68
`define TRG_TIMEB2          69
`define TRG_TIMEB3          70
`define TRG_TIMEB4          71
`define TRG_MODE5           72

`define TRG_CH1BASE         8'd80
`define TRG_CH2BASE         8'd88
`define   TRG_LLEVEL        0
`define   TRG_ULEVEL        1
`define   TRG_QLLEVEL       2
`define   TRG_QULEVEL       3

`define TRG_HOLDOFF1        96
`define TRG_HOLDOFF2        97
`define TRG_HOLDOFF3        98
`define TRG_HOLDOFF4        99
`define TRG_VIDEOL1         100
`define TRG_VIDEOL2         101
`define TRG_VIDEOP1         102
`define TRG_VIDEOP2         103
`define TRG_VIDEOM          104

//
// VIDEOL bitek
//
`define VIDEO_FILT          15
`define VIDEO_INV           14
`define VIDEO_LINE          10:0

//
// MODE1 regiszter bitek
//
`define TRG_MODE1_EN        7       // trigger engedelyezes
`define TRG_MODE1_HOLDEVS   6       // 
`define TRG_MODE1_ENSEL     5:4     // enable select: main, Q-level, Q-edge
`define TRG_MODE1_INSEL2    3:2     // trig secondary source sel (RESP es NORESP modoknal)
`define TRG_MODE1_INSEL     1:0     // trig source sel: ch1, ch2, ext, patt


//
// Triggerkutyu es qualifier bemenete: TRG_MODE1_INSEL, TRG_MODE1_INSEL2
//                                     es TRG_MODE5_QSEL bitmezo:
//
`define TRG_MODE1_INCH1     2'd0
`define TRG_MODE1_INCH2     2'd1
`define TRG_MODE1_INEXT     2'd2
`define TRG_MODE1_INPATT    2'd3


//
// Triggerkutyu engedelyezes bemenet: TRG_MODE1_ENSEL bitmezo
//
`define TRG_MODE1_ENMAIN    2'd0    // sima trigger
`define TRG_MODE1_ENQLEV    2'd1    // level qualified
`define TRG_MODE1_ENQEDGE   2'd2    // edge qualified

//
//          /===============================================================\
//          |   7   |   6   |   5   |   4   |   3   |   2   |   1   |   0   |
//          |=======|=======|=======|=======|=======|=======|=======|=======|
// MODE1    | EN    |HOLDEVS|     ENSEL     |    INSEL2     |     INSEL     |
//          |-------|-------|-------|-------|-------|-------|-------|-------|
// MODE2    | DO    | IV    | PW    | EDGE  | NEG2  | POS2  | NEG   | POS   |
//          |-------|-------|-------|-------|-------|-------|-------|-------|
// MODE3    |NORESP | RESP  | -     | -     |RESTART| OR    | B     | A     |
//          |-------|-------|-------|-------|-------|-------|-------|-------|
// MODE4    | -     | -     |   EXT-PATT    |   CH2-PATT    |   CH1-PATT    |
//          |-------|-------|-------|-------|-------|-------|-------|-------|
// MODE5    | -     | -     | -     | -     | QPOL  | -     |     QSEL      |
//          \===============================================================/

//
// Alapveto triggermodok: MODE2 regiszter bitek
//
`define TRG_MODE2_POS       0       // pos
`define TRG_MODE2_NEG       1       // neg
`define TRG_MODE2_POS2      2       // resp es no resp mereseknel a masodik 
`define TRG_MODE2_NEG2      3       // jel polaritasa
`define TRG_MODE2_EDGE      4       // edge
`define TRG_MODE2_PW        5       // pulse width
`define TRG_MODE2_IV        6       // pulse interval
`define TRG_MODE2_DO        7       // dropout

//
// Idomeresek: MODE3 regiszter bitek
//
`define TRG_MODE3_A         0       // t >= timea
`define TRG_MODE3_B         1       // t  < timeb
`define TRG_MODE3_OR        2       // barmelyik
`define TRG_MODE3_RESTART   3       // resp modnal idomeres ujraindul minden elnel
`define TRG_MODE3_RESP      6       // response edge to other ch. edge
`define TRG_MODE3_NORESP    7       // no-response edge to other ch. edge

//
// Pattern select: MODE4 regiszter bitjei
//
`define TRG_MODE4_CH1       1:0     // ch1: H, L, X
`define TRG_MODE4_CH2       3:2     // ch2: H, L, X
`define TRG_MODE4_EXT       5:4     // ext: H, L, X

`define TRG_MODE4_X         2'd0    // don't care (X)
`define TRG_MODE4_A         2'd1    // above      (H)
`define TRG_MODE4_B         2'd2    // below      (L)

//
// Qualifier select: MODE5
//
`define TRG_MODE5_QPOL      3       // 1=above/0=below ill. 1=posedge/0=negedge
`define TRG_MODE5_QSEL      1:0     // qualifier sel: ch1, ch2, ext, patt

`define BLT_SAL             128
`define BLT_SAH             129
`define BLT_DAL             130
`define BLT_DAH             131
`define BLT_SWID            132     // BYTE
`define BLT_WIDL            133     // PIXEL
`define BLT_WIDH            134
`define BLT_HEIG            135     // PIXEL
`define BLT_XBITS           136     // 
`define BLT_MODE            137     //

module top(
    ICLKOSC100,
    ALE, nRD, nWR, CPUAD, CPUAH, nIRQ,
    nRAMOE, nRAMWR, nRAMCE, RAMDATA, RAMADDR,
    RD, RA, RCE, RAS, CAS, RWR, DQML, DQMH, CKE,
    lcd_cp, lcd_frame, lcd_load, lcd_dispon, lcd_data,
    CH1IN, CH2IN, EXTTRIG,
    RCKOUT, ICLKRAM, CAL
);

input           ICLKOSC100;     /* 100 MHz clock input                      */

input           ALE;            /* CPU ALE                                  */
input           nRD;            /* CPU RD                                   */
input           nWR;            /* CPU WR                                   */
inout   [7:0]   CPUAD;          /* CPU Address low in / data in/out         */
input   [7:0]   CPUAH;          /* CPU address bus high                     */
output          nIRQ;           /* Open drain Interrupt Request output      */

output          nRAMOE;         /* RAM OE                                   */
output          nRAMWR;         /* RAM WR                                   */
output          nRAMCE;         /* RAM CE                                   */
inout   [7:0]   RAMDATA;        /* RAM data                                 */
output  [16:0]  RAMADDR;        /* 128k RAM address                         */

inout   [31:0]  RD;             /* SDRAM 32 bit bidirectional data bus      */          
output  [11:0]  RA;             /* SDRAM address bus                        */
output          RCE;
output          RAS;
output          CAS;
output          RWR;
output          DQML;
output          DQMH;
output          CKE;

output          lcd_cp;         /* LCD                                      */
output          lcd_frame;
output          lcd_load;
output          lcd_dispon;     /* Active high display on signal            */
output  [3:0]   lcd_data;

input   [7:0]   CH1IN;          /* CH1 A/D data                             */
input   [7:0]   CH2IN;          /* CH2 A/D data                             */
input           EXTTRIG;        /* Ext Trigger Input                        */

output          RCKOUT;         /* Clock out for SDRAMs                     */
input           ICLKRAM;

output          CAL;            /* Calibration signal output                */

/* ---- Internal signals -------------------------------------------------- */

wire            T, F;
wire    [7:0]   ramrddata;      /* Belso logika RAM olvasaskor ez az adat   */

wire    [16:0]  lcdaddr;
wire            lcdrd, lcdack, lcddon;

wire    [16:0]  graddr;
wire    [7:0]   grwdata;
wire            grrd, grwr, grack, grdon;

wire    [16:0]  bladdr;
wire    [7:0]   blwdata;
wire            blrd, blwr, black, bldon;


wire            regrd, regwr;
wire            sdbrd, sdbwr;
wire    [7:0]   regdi;
wire    [7:0]   regdo;
wire    [7:0]   rega;
wire    [15:0]  sdbufa;

wire    [31:0]  sd_readdata;    /* Amit az SDRAM-bol felolvasott            */
wire    [6:0]   sd_bufa;        /* SDRAM buffer address                     */

/*
 *  CPU - SDRAM jelei
 */
wire            cpu_sddon, cpu_sdrd, cpu_sdwr;
wire    [12:0]  cpu_sdbase;         // base address
wire            cpu_sdbwe;          // buffer write enable
wire    [31:0]  cpu_sdwdata;        // buffer -> SDRAM

/*
 *  Acquision - SDRAM
 */
wire            acq_sddon, acq_sdwr;
wire    [12:0]  acq_sdbase;
wire    [31:0]  acq_sdwdata;        // buffer -> SDRAM

/*
 *  Graph - SDRAM
 */
wire            gr_sdrd, gr_sddon, gr_sdbwe;
wire    [12:0]  gr_sdbase;
wire    [15:0]  cur_data;
wire            cur_new, cur_run, cur_busy;
wire            blt_irq, gr_irq, tr_irq, lcd_irq;

wire            ickosc, ickram, ck100, ck66, ckram, CLK66, CLK100, dlllock;
wire            daclock;
wire            shot;   // trigger kimenet a CAL BNC-n


    assign nIRQ = (blt_irq | gr_irq | tr_irq | lcd_irq) ? 1'b0 : 1'bz;
    
    assign T = 1'b1;
    assign F = 1'b0;

    /**************************************************************************
     * Bejon az oszcillatorbol 100MHz. Ez 'ibosc' bufferen keresztul bemegy a
     * DLL-be. A DLL-bol kijon egy 100MHz es egy 66MHz.
     *
     * A 100MHz 'ob100' bufferen keresztul hajtja meg a chip-et CLK100 jellel.
     * Ez a CLK100 a DLL visszacsatolasa is.
     *
     * A DLL 66MHz kimenete az 'ob3' bufferen keresztul kimegy a chip-bol az
     * RCKOUT labon. RCKOUT egyreszt ramegy az SDRAM-okra, masreszt visszajon a
     * chip-be az ICLKRAM labon. Abbol lesz a CLK66 belso orajel.
     **************************************************************************/
    
    IBUFG   ibosc(.I(ICLKOSC100), .O(ickosc));
    CLKDLLHF #(
        .STARTUP_WAIT("TRUE"),
        .CLKDV_DIVIDE(1.5),
        .DUTY_CYCLE_CORRECTION("TRUE")
    ) dll_int (
        .CLKIN(ickosc),
        .CLKFB(CLK100),
        .CLK0(ck100),
        .CLKDV(ck66),
        .LOCKED(dlllock)
    );
    BUFG ob100(.I(ck100), .O(CLK100));
    
    OBUF   ob3(.I(ck66), .O(RCKOUT));
    IBUFG  ib2(.I(ICLKRAM), .O(ckram));
    BUFG  ob66(.I(ckram),   .O(CLK66));
    
    version u_vers(
        .regrd(regrd), .rega(rega), .regdo(regdo)
    );
    
    calgen  u_cal(
        .CLK100(CLK100), .out(CAL), 
        .CLK66(CLK66), .regwr(regwr), .rega(rega), .regdi(regdi), .shot(shot)
    );
    
    ramctrl u_ramctrl(
        .CLK66(CLK66), .ALE(ALE), .nRD(nRD), .nWR(nWR), .CPUAD(CPUAD), .CPUAH(CPUAH),
        .nRAMOE(nRAMOE), .nRAMWR(nRAMWR), .nRAMCE(nRAMCE), .RAMDATA(RAMDATA), .RAMADDR(RAMADDR),
        
        .intaddr0(lcdaddr), .intrdreq0(lcdrd), .intwrreq0(1'b0), .intack0(lcdack),
        .intdon0(lcddon), .intwdata0(8'h00),
        
        .intaddr1(graddr), .intrdreq1(grrd), .intwrreq1(grwr), .intack1(grack),
        .intdon1(grdon), .intwdata1(grwdata),

        .intaddr2(bladdr), .intrdreq2(blrd), .intwrreq2(blwr), .intack2(black),
        .intdon2(bldon), .intwdata2(blwdata),
        
        .intrdata(ramrddata),
        
        .regread(regrd), .regwrite(regwr), .regdo(regdo), .regdi(regdi), .rega(rega),
        .sdbread(sdbrd), .sdbwrite(sdbwr), .sdba(sdbufa)
    );      
    
    lcdrefresh  u_lcd(
        .CLK66(CLK66),
        .regrd(regrd), .regwr(regwr), .rega(rega), .regdi(regdi), .regdo(regdo), .irq(lcd_irq),
        .ramaddr(lcdaddr), .ramrd(lcdrd), .ramack(lcdack), .ramdon(lcddon),
        .ramdata(ramrddata),
        .cp(lcd_cp), .frame(lcd_frame), .load(lcd_load), .data(lcd_data),
        .dispon(lcd_dispon)
    );

    curve u_curve(
        .CLK66(CLK66), .regrd(regrd), .regwr(regwr), .rega(rega),
        .regdi(regdi), .regdo(regdo),
        .ramrd(grrd), .ramwr(grwr), .ramack(grack), .ramdon(grdon),
        .rama(graddr), .ramdi(ramrddata), .ramdo(grwdata),
        .newdata(cur_new), .datain(cur_data), .curverun(cur_run), .busy(cur_busy)
    );
    
    graph u_graph(
        .CLK66(CLK66), .regrd(regrd), .regwr(regwr), .rega(rega),
        .regdi(regdi), .regdo(regdo), .irq(gr_irq),
        .sdrd(gr_sdrd),  .sddon(gr_sddon), .sdaddr(gr_sdbase), .sdbufa(sd_bufa),
        .sdwe(gr_sdbwe), .sdbufdata(sd_readdata),
        .cdata(cur_data), .cnew(cur_new), .crun(cur_run), .cbusy(cur_busy)
    );
    
    blitter u_blitter(
        .memclk(CLK66), .regrd(regrd),  .regwr(regwr), .rega(rega),
        .regdin(regdi), .regdout(regdo), .irq(blt_irq),
        .ramrdreq(blrd), .ramwrreq(blwr), .ramack(black), .ramdon(bldon),
        .ramaddr(bladdr), .ramdin(ramrddata), .ramdtow(blwdata)
    );
    
    sdramifc    u_sdram(
        .CLK66(CLK66), .CKE(CKE),
        .RD(RD), .RA(RA), .RCE(RCE), .RAS(RAS), .CAS(CAS), .RWR(RWR),
        .DQML(DQML), .DQMH(DQMH), 
        
        .srd0(cpu_sdrd),    .srd1(F),           .srd2(gr_sdrd),
        .swr0(cpu_sdwr),    .swr1(acq_sdwr),    .swr2(F),
        .don0(cpu_sddon),   .don1(acq_sddon),   .don2(gr_sddon),
        .rad0(cpu_sdbase),  .rad1(acq_sdbase),  .rad2(gr_sdbase),
        .bwe0(cpu_sdbwe),                       .bwe2(gr_sdbwe),
        .brd0(cpu_sdwdata), .brd1(acq_sdwdata),
        
        .bufa(sd_bufa), .bdw(sd_readdata)
    );
    
    cpuram  u_cpusdram(
        .CLK66(CLK66),
        .regr(regrd), .regw(regwr), .ramrd(sdbrd), .ramwr(sdbwr),
        .rega(rega),  .datai(regdi), .datao(regdo), .rama(sdbufa),
        
        .srd(cpu_sdrd), .swr(cpu_sdwr), .done(cpu_sddon),
        .basea(cpu_sdbase),
        .bdw(sd_readdata), .bufa(sd_bufa), 
        .bwe(cpu_sdbwe), .brd(cpu_sdwdata)
    );
    
    acquire  u_acq(
        .CLK100(CLK100), .CH1IN(CH1IN), .CH2IN(CH2IN), .daclock(daclock),
        .extin(EXTTRIG),
        .CLK66(CLK66), .regrd(regrd), .regwr(regwr), .rega(rega),
        .regdi(regdi), .regdo(regdo), .irq(tr_irq),
        .sdwr(acq_sdwr), .sddone(acq_sddon), .sdbase(acq_sdbase),
        .sdbufa(sd_bufa), .brd(acq_sdwdata), .shot(shot)
    );
endmodule

/*
 *   Kalibrator kimenet eloallitasa
 *  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 *
 *  'mode' regiszter also ket bitje a jel valasztas:
 *    00 - kalibralo jel
 *    01 - trigger (shot)
 *    10 - DC
 *    11 - GND
 *
 *  'mode' regiszter legfelso ket bitje a prescalert allitja:
 *    00 -   10  (out =  10MHz)
 *    01 -  100  (out =   1MHz)
 *    10 - 1000  (out = 100kHz)
 *
 * Van egy pos es egy neg regiszter. Pos+1 a jel 1-ben toltott idejet, mig 
 * neg+1 a jell nullaban toltott idejet adja meg 100ns felbontasban.
 * Tehat a regiszterekbe a szamitott ertek minusz egyet kell irni.
 *
 * 4999, 4999 adja az 1kHz, 1:1 negyszoget (prescaler = 10 esetben).
 *
 */
module  calgen(
        CLK100, out,
        CLK66, regwr, rega, regdi,
        shot
);

input           CLK100;
output          out;
input           CLK66;
input           regwr;
input   [7:0]   rega;
input   [7:0]   regdi;
input           shot;

reg             sout = 0;
reg     [9:0]   pdiv = 0;
reg     [9:0]   preset = 9;
reg             clk2;

reg     [15:0]  userp = 4999;
reg     [15:0]  usern = 4999;
reg     [5:0]   calmode = 0;
reg     [15:0]  div = 4999;


    assign out = (calmode[1:0] == 0) ? sout :
                 (calmode[1:0] == 1) ? shot : 
                 (calmode[1:0] == 2) ? 1 : 0;

    always @ (posedge CLK100) begin
        if(pdiv == 0) begin
            pdiv <= preset;
            clk2 <= 1;
        end
        else begin
            pdiv <= pdiv - 1;
            clk2 <= 0;
        end
        
        if(clk2) begin
            if(div == 0) begin
                sout <= ~sout;
                div <= (sout) ? usern : userp;
            end
            else
                div <= div - 1;
        end
    end
    
    always @ (posedge CLK66) begin
        if(regwr && rega == `CAL_PLOW)
            userp[7:0] <= regdi;
        if(regwr && rega == `CAL_PHIGH)
            userp[15:8] <= regdi;
        if(regwr && rega == `CAL_NLOW)
            usern[7:0] <= regdi;
        if(regwr && rega == `CAL_NHIGH)
            usern[15:8] <= regdi;
        if(regwr && rega == `CAL_MODE) begin
            calmode <= regdi[5:0];
            preset <= (regdi[7:6] == 2'b00) ?  9 :
                      (regdi[7:6] == 2'b01) ? 99 : 999;
        end
    end
endmodule


/*
 *  SDRAM - CPU interface
 */
module  cpuram(
    CLK66, 
    regr, regw, ramrd, ramwr, rega, datai, datao, rama,
    srd, swr, done, basea, bdw, bufa, bwe, brd
);

input           CLK66;
input           regr, regw, ramrd, ramwr;
input   [7:0]   rega;
input   [7:0]   datai;
output  [7:0]   datao;
input   [15:0]  rama;

output          srd;
 reg            srd = 0;
output          swr;
 reg            swr = 0;
input           done;
output  [12:0]  basea;
 reg    [12:0]  basea;
input   [31:0]  bdw;
input   [6:0]   bufa;
input           bwe;
output  [31:0]  brd;

reg     [7:0]   status = 0;

wire    [7:0]   ramdata;
reg             bufah;
wire    [7:0]   a32;

    assign datao = (ramrd) ? ramdata :
                   (regr && (rega == `RAM_CTRL)) ? status : 8'hzz;
                   

    always @ (posedge CLK66) begin
        if(done)
            status[2] <= 0;
        
        srd <= 0;
        swr <= 0;
        
        if(regw && (rega == `RAM_ALO))
            basea[7:0] <= datai;
            
        if(regw && (rega == `RAM_AHI))
            basea[12:8] <= datai;
        
        if(regw && (rega == `RAM_CTRL) && datai[0]) begin
            bufah  <= datai[7];
            swr    <= 1;
            status <= 8'b0000_0101;
        end
        else if(regw && (rega == `RAM_CTRL) && datai[1]) begin
            bufah  <= datai[7];
            srd    <= 1;
            status <= 8'b0000_0110;
        end
    end
    
    assign  a32 = {bufah, bufa};
    
    buffer8_32  bu1(
        .clk8(CLK66), .addr8(rama[9:0]),  .wr8(ramwr), .din8(datai), .dout8(ramdata),
        .clk32(CLK66), .addr32(a32), .wr32(bwe),  .din32(bdw),  .dout32(brd)
    );
endmodule

/*
 *  It returns the chip version number.
 */
module  version(
    regrd, rega, regdo
);
input           regrd;
input   [7:0]   rega;
output  [7:0]   regdo;

    assign regdo = (regrd && rega == 255) ? `CHIP_VER : 8'hzz;
    
endmodule


/*
 *  1024 byte-os DUAL-PORT buffer
 *  
 *  Az egyik port 1024 x 8 bit
 *  A masik        256 x 32 bit
 */
module  buffer8_32(
    clk8,  addr8,  wr8,  din8,  dout8,
    clk32, addr32, wr32, din32, dout32 
);

input           clk8;
input   [9:0]   addr8;
input           wr8;
input   [7:0]   din8;
output  [7:0]   dout8;

input           clk32;
input   [7:0]   addr32;
input           wr32;
input   [31:0]  din32;
output  [31:0]  dout32;

wire    [15:0]  bufi1, bufi2, bufo1, bufo2;


    assign  dout32[ 3: 0] = bufo1[3:0];
    assign  dout32[ 7: 4] = bufo2[3:0];
    assign  dout32[11: 8] = bufo1[7:4];
    assign  dout32[15:12] = bufo2[7:4];
    assign  dout32[19:16] = bufo1[11:8];
    assign  dout32[23:20] = bufo2[11:8];
    assign  dout32[27:24] = bufo1[15:12];
    assign  dout32[31:28] = bufo2[15:12];
    
    assign  bufi1[3:0]   = din32[ 3: 0];
    assign  bufi2[3:0]   = din32[ 7: 4];
    assign  bufi1[7:4]   = din32[11: 8];
    assign  bufi2[7:4]   = din32[15:12];
    assign  bufi1[11:8]  = din32[19:16];
    assign  bufi2[11:8]  = din32[23:20];
    assign  bufi1[15:12] = din32[27:24];
    assign  bufi2[15:12] = din32[31:28];
    
    
    RAMB4_S4_S16 ramlo(
        .CLKA(clk8), .ADDRA(addr8), .WEA(wr8), .ENA(1'b1), .RSTA(1'b0),
        .DIA(din8[3:0]), .DOA(dout8[3:0]),
        
        .CLKB(clk32), .ADDRB(addr32), .WEB(wr32), .ENB(1'b1), .RSTB(1'b0),
        .DIB(bufi1), .DOB(bufo1)
    );

    RAMB4_S4_S16 ramhi(
        .CLKA(clk8), .ADDRA(addr8), .WEA(wr8), .ENA(1'b1), .RSTA(1'b0),
        .DIA(din8[7:4]), .DOA(dout8[7:4]),
        
        .CLKB(clk32), .ADDRB(addr32), .WEB(wr32), .ENB(1'b1), .RSTB(1'b0),
        .DIB(bufi2), .DOB(bufo2)
    );
endmodule

/*
 *  1024 byte-os DUAL-PORT buffer
 *  
 *  Az egyik port 512 x 16 bit
 *  A masik port  256 x 32 bit
 */
module  buffer16_32(
    clk16, addr16, wr16, din16, dout16,
    clk32, addr32, wr32, din32, dout32 
);

input           clk16;
input   [8:0]   addr16;
input           wr16;
input   [15:0]  din16;
output  [15:0]  dout16;

input           clk32;
input   [7:0]   addr32;
input           wr32;
input   [31:0]  din32;
output  [31:0]  dout32;

wire    [15:0]  bufi1, bufi2, bufo1, bufo2;


    assign  dout32[ 7: 0] = bufo1[7:0];
    assign  dout32[15: 8] = bufo2[7:0];
    assign  dout32[23:16] = bufo1[15:8];
    assign  dout32[31:24] = bufo2[15:8];
    
    assign  bufi1[ 7:0]  = din32[ 7: 0];
    assign  bufi2[ 7:0]  = din32[15: 8];
    assign  bufi1[15:8]  = din32[23:16];
    assign  bufi2[15:8]  = din32[31:24];
    
    
    RAMB4_S8_S16 ramlo(
        .CLKA(clk16), .ADDRA(addr16), .WEA(wr16), .ENA(1'b1), .RSTA(1'b0),
        .DIA(din16[7:0]), .DOA(dout16[7:0]),
        
        .CLKB(clk32), .ADDRB(addr32), .WEB(wr32), .ENB(1'b1), .RSTB(1'b0),
        .DIB(bufi1), .DOB(bufo1)
    );

    RAMB4_S8_S16 ramhi(
        .CLKA(clk16), .ADDRA(addr16), .WEA(wr16), .ENA(1'b1), .RSTA(1'b0),
        .DIA(din16[15:8]), .DOA(dout16[15:8]),
        
        .CLKB(clk32), .ADDRB(addr32), .WEB(wr32), .ENB(1'b1), .RSTB(1'b0),
        .DIB(bufi2), .DOB(bufo2)
    );
endmodule


`include "ram.v"
`include "lcd.v"
`include "sdram.v"
`include "acq.v"
`include "curve.v"
`include "blitter.v"
`include "graph.v"
`include "trigger.v"

