/****************************************************************************/
/*                                                                          */
/*                ACS-2602 - Kis Szkop                                      */
/*                                                                          */
/****************************************************************************/
/*  Target Chip: Xilinx XC2S200-PQ208 (Spartan-II)                          */
/*--------------------------------------------------------------------------*/
/* Trigger                                                                  */
/*--------------------------------------------------------------------------*/
/* History:                                                                 */
/*  2007.03.26: acq.v ketteszedese                                          */
/*  2007.04.14: komparator hiszterezis hozzaadasa                           */
/*  2007.07.06: trigger modul mode1 regiszter irasa atrakva a blokk vegere  */
/*  2007.07.08: auto trigger inditas atirasa MODE2-read -re                 */
/*  2007.07.12: stop-ra is IRQ-t ker                                        */
/*  2007.07.23: Dropout csak az elso el utan triggerel                      */
/*  2007.07.24: irq torlese rossz helyen volt                               */
/*  2007.10.27: AC csatolas kiszedese                                       */
/*  2007.10.31: Kulon komparator a qualifiernek                             */
/*  2008.02.06: hiszterezis komparator elvi hiba javitasa                   */
/*  2008.04.16: 'shot' kimenet hozzairasa                                   */
/*  2008.04.22: trigger status 'ran' bit hozzairasa                         */
/*  2008.05.21: qualifier() level kimenet maszkolas (ena) hiba javitasa     */
/*  2008.05.21: trigtime() edge trigger maszkolas (ena) javitasa            */
/*  2008.05.23: holdoff hozzairasa                                          */
/*  2008.05.24: video trigger hozairasa                                     */
/*  2008.05.25: video trigger lp filter hozzaadasa                          */
/*  2009.03.18: debug ss regiszter beleirasa                                */
/*  2009.03.18: irq beallitas hiba (neha torolte), javitva                  */
/*  2009.03.20: Chip V3.0, pretrigger es tarolas hossza mintakban szamolando*/
/*              block helyett, ezert store es trigger jelentos atirasa      */
/*  2009.05.06: Chip V4.0, Rsponse es NoResponse triggerek hozzaadasa       */
/****************************************************************************/

/**
 * MODE1 irasa inditja vagy megallitja az egeszet, fuggoen a legfelso bittol
 *  Leallitas eseten IRQ-t ket, ha elozoleg ment.
 *
 *              7   6  5 4  3   2   1   0
 *  A statusz: IRQ TEN 0 0 RAN ST2 ST1 ST0
 *  
 *  ST[2:0] a status (TS_XXX)
 *
 *  RAN akkor all be egybe, amikor status TS_STORE-ra valt,
 *  MODE3 olvasasa torli
 *
 * MODE1 olvasasa visszaadja a statuszt es torli az IRQ-t (ISR)
 * MODE2 olvasasa visszaadja a statuszt es beallitja 'atrig'-et (AUTO trigger)
 * MODE3 olvasasa visszaadja a statuszt, es torli 'ran'-t (display.c hasznalja)
 * MODE4 olvasasa visszaadja a statuszt es nem torol semmit (debug-hoz)
 *
 * 'atrig'-et MODE1 irasa torli
 **/

`define TS_STOP     0
`define TS_START    1
`define TS_PREFILL  2
`define TS_WAIT     3
`define TS_STORE    4
`define TS_FLUSH    5

module  trigger(
    CLK66, regrd, regwr, rega, regdi, regdo, irq,
    ext, newdata, dch1, dch2, st_en, st_run, st_busy, wword, wraddr, shot
);

input           CLK66;
input           regrd, regwr;
input   [7:0]   rega;
input   [7:0]   regdi;
output  [7:0]   regdo;
output          irq;

input           ext;                /* ext trigger komparator out           */
input   [7:0]   dch1, dch2;         /* a ket csatornarol jovo adat (50MS/s) */
input           newdata;            /* ext, dch1, dch2 uj adatot tartalmaz  */
output          st_en;              /* tarolo reszt engedelyezi             */
output          st_run;             /* A tarolo taroljon                    */
input           st_busy;            /* A tarolo resz meg dolgozik           */
input           wword;              /* tarolo resz lerakott 2 mintat        */
input   [19:0]  wraddr;             /* eppen aktualis tarolasi cim          */
output          shot;               /* Trigger utan 1 a tarolas vegen 0     */

reg     [19:0]  trigpnt;

reg     [23:0]  prewords;
reg     [23:0]  storewords;
reg     [7:0]   mode1 = 0;
reg     [7:0]   mode4 = 0;

wire            ch1level, ch2level, pattern;
wire            comp, sec, qlev, qedge, t_trig, v_trig, ten;
wire            ch1qlev, ch2qlev;

reg     [2:0]   state = `TS_STOP;
reg             ran;
reg     [19:0]  wcount;
reg             irq = 0;
reg             st_en = 0;
reg             st_run = 0;
reg             mainen = 0;
reg             a_trig = 0;

reg     [7:0]   ss = 0;

wire    [7:0]   status;

    
    assign shot = (state == `TS_STORE);
    
    assign status = {irq, ten, t_trig, a_trig, ran, state};

    assign regdo = (regrd && rega == `TRG_TPNT1) ? trigpnt[7:0] :
                   (regrd && rega == `TRG_TPNT2) ? trigpnt[15:8] :
                   (regrd && rega == `TRG_TPNT3) ? {4'b0, trigpnt[19:16]} :
                   (regrd && rega == `TRG_TPNT4) ? {8'b0} :
                   (regrd && rega == `TRG_MODE1 ||
                    regrd && rega == `TRG_MODE2 ||
                    regrd && rega == `TRG_MODE3 ||
                    regrd && rega == `TRG_MODE4) ? status : 
                   (regrd && rega == `TRG_TIMEA1) ? ss : 8'hzz;
    
    trigfe u_trch1(
        .CLK66(CLK66), .basea(`TRG_CH1BASE),
        .regrd(regrd), .regwr(regwr), .rega(rega), .regdi(regdi), .regdo(regdo),
        .newdata(newdata), .chdata(dch1), .out(ch1level), .qout(ch1qlev)
    );
    
    trigfe u_trch2(
        .CLK66(CLK66), .basea(`TRG_CH2BASE),
        .regrd(regrd), .regwr(regwr), .rega(rega), .regdi(regdi), .regdo(regdo),
        .newdata(newdata), .chdata(dch2), .out(ch2level), .qout(ch2qlev)
    );

    qualifier u_qual(
        .CLK66(CLK66),
        .regrd(regrd), .regwr(regwr), .rega(rega), .regdi(regdi), .regdo(regdo),
        .newdata(newdata), .in1(ch1qlev), .in2(ch2qlev), .inext(ext), .inpatt(pattern),
        .ena(mainen), .levelout(qlev), .edgeout(qedge)
    );
    
    trigtime u_trig(
        .CLK66(CLK66),
        .regrd(regrd), .regwr(regwr), .rega(rega), .regdi(regdi), .regdo(regdo),
        .newdata(newdata), .comparator(comp), .secondary(sec), .ena(ten), .out(t_trig)
    );
    
    trigvideo u_video(
        .CLK66(CLK66),
        .regrd(regrd), .regwr(regwr), .rega(rega), .regdi(regdi), .regdo(regdo),
        .newdata(newdata), .dt1(dch1), .dt2(dch2), .ena(ten), .out(v_trig)
    );
    
    /***********************************************************/
    
    assign pattern = ((mode4[`TRG_MODE4_CH1] == `TRG_MODE4_X) ? 1 :
                      (mode4[`TRG_MODE4_CH1] == `TRG_MODE4_A) ? ch1level : !ch1level) &
                     ((mode4[`TRG_MODE4_CH2] == `TRG_MODE4_X) ? 1 :
                      (mode4[`TRG_MODE4_CH2] == `TRG_MODE4_A) ? ch2level : !ch2level) &
                     ((mode4[`TRG_MODE4_EXT] == `TRG_MODE4_X) ? 1 :
                      (mode4[`TRG_MODE4_EXT] == `TRG_MODE4_A) ? ext : !ext);

    assign comp = (mode1[`TRG_MODE1_INSEL] == `TRG_MODE1_INCH1) ? ch1level :
                  (mode1[`TRG_MODE1_INSEL] == `TRG_MODE1_INCH2) ? ch2level :
                  (mode1[`TRG_MODE1_INSEL] == `TRG_MODE1_INEXT) ? ext : pattern;
    
    assign sec  = (mode1[`TRG_MODE1_INSEL2] == `TRG_MODE1_INCH1) ? ch1level :
                  (mode1[`TRG_MODE1_INSEL2] == `TRG_MODE1_INCH2) ? ch2level :
                  (mode1[`TRG_MODE1_INSEL2] == `TRG_MODE1_INEXT) ? ext : pattern;
                  
    assign ten =  (mode1[`TRG_MODE1_ENSEL] == `TRG_MODE1_ENMAIN) ? mainen :
                  (mode1[`TRG_MODE1_ENSEL] == `TRG_MODE1_ENQLEV) ? qlev : qedge;

    always @ (posedge CLK66) begin
        
        if(regrd && rega == `TRG_MODE1)
            irq <= 0;
            
        if(state == `TS_STORE)
            ran <= 1;
        else if(regrd && rega == `TRG_MODE3)
            ran <= 0;
            
        if(regwr && rega == `TRG_PRE1)
            prewords[7:0] <= regdi;
        if(regwr && rega == `TRG_PRE2)
            prewords[15:8] <= regdi;
        if(regwr && rega == `TRG_PRE3)
            prewords[23:16] <= regdi;
            
        if(regwr && rega == `TRG_STORE1)
            storewords[7:0] <= regdi;
        if(regwr && rega == `TRG_STORE2)
            storewords[15:8] <= regdi;
        if(regwr && rega == `TRG_STORE3)
            storewords[23:16] <= regdi;
            
        if(regwr && rega == `TRG_MODE4)
            mode4 <= regdi;
        
        if(state == `TS_START) begin
            st_en  <= 1;
            st_run <= 1;
            wcount <= prewords;
            state  <= `TS_PREFILL;
        end
        else if(state == `TS_PREFILL && wword) begin
            if(wcount == 1) begin
                wcount <= storewords;
                mainen <= 1;
                state  <= `TS_WAIT;
            end
            else begin
                wcount <= wcount - 1;
            end
        end
        else if(state == `TS_WAIT && (t_trig || v_trig || a_trig)) begin
            trigpnt <= wraddr;
            state   <= `TS_STORE;
        end
        else if(state == `TS_STORE && wword) begin
            if(wcount == 1) begin
                mainen <= 0;
                st_run <= 0;
                state  <= `TS_FLUSH;
            end
            else begin
                wcount <= wcount - 1;
            end
        end
        else if(state == `TS_FLUSH && !st_busy) begin
            st_en  <= 0;
            irq    <= 1;
            state  <= `TS_STOP;
        end
        
        if(regrd && rega == `TRG_MODE2)
            a_trig <= 1;
        
        if(regwr && rega == `TRG_MODE1) begin
            mode1   <= regdi;
            st_en   <= 0;
            st_run  <= 0;
            mainen  <= 0;
            a_trig  <= 0;
            if(regdi[`TRG_MODE1_EN])
                state <= `TS_START;
            else begin
                if(st_en)
                    irq <= 1;           // ha meg ment, akkor IRQ-t ker
                ss[0] <= irq;
                ss[1] <= st_run;
                state <= `TS_STOP;
            end
        end
    end
endmodule


/*
 *  Trigger front-end: komparatorok
 */
module  trigfe(
    CLK66, regrd, regwr, rega, regdi, regdo, basea,
    newdata, chdata, out, qout
);

input           CLK66;
input           regrd, regwr;
input   [7:0]   rega;
input   [7:0]   regdi;
output  [7:0]   regdo;
input   [7:0]   basea;

input           newdata;
input   [7:0]   chdata;
output          out;
reg             out;
output          qout;
reg             qout;

reg     [7:0]   ulevel;
reg     [7:0]   llevel;
reg     [7:0]   level;

reg     [7:0]   qulev, qllev, qlev;


    assign regdo = 8'hzz;

    always @ (posedge CLK66) begin
        if(regwr && rega == (basea + `TRG_ULEVEL))
            ulevel <= regdi;
        if(regwr && rega == (basea + `TRG_LLEVEL))
            llevel <= regdi;
        if(regwr && rega == (basea + `TRG_QULEVEL))
            qulev <= regdi;
        if(regwr && rega == (basea + `TRG_QLLEVEL))
            qllev <= regdi;
    end
    
    
    always @ (posedge CLK66) begin
        if(newdata) begin
            if(chdata >= level) begin
                out <= 1;
                level <= llevel;
            end
            else begin
                out <= 0;
                level <= ulevel;
            end
            
            if(chdata >= qlev) begin
                qout <= 1;
                qlev <= qllev;
            end
            else begin
                qout <= 0;
                qlev <= qulev;
            end
        end
    end
endmodule

module  qualifier(
    CLK66, regrd, regwr, rega, regdi, regdo,
    newdata, in1, in2, inext, inpatt, ena, levelout, edgeout
);
input           CLK66;
input           regrd, regwr;
input   [7:0]   rega;
input   [7:0]   regdi;
output  [7:0]   regdo;

input           newdata;
input           in1, in2, inext, inpatt;
input           ena;
output          levelout;
output          edgeout;

reg     [7:0]   mode5 = 0;
reg     [1:0]   edet;
reg             edgeout = 0;
wire            ql1, qlev;

    assign regdo = (regrd && rega == `TRG_MODE5) ? {inext, edgeout, levelout, mode5[4:0]} :
                    8'hzz;
    
    assign levelout = qlev & ena;
    
    assign ql1  = (mode5[`TRG_MODE5_QSEL] == `TRG_MODE1_INCH1) ? in1 :
                  (mode5[`TRG_MODE5_QSEL] == `TRG_MODE1_INCH2) ? in2 :
                  (mode5[`TRG_MODE5_QSEL] == `TRG_MODE1_INEXT) ? inext : inpatt;
    
    assign qlev = mode5[`TRG_MODE5_QPOL] ? ql1 : !ql1;
    
    always @ (posedge CLK66) begin
        if(regwr && rega == `TRG_MODE5)
            mode5 <= regdi;
    end
    
    always @ (posedge CLK66) begin
        
        if(newdata) begin
            edet[1] <= edet[0];
            edet[0] <= qlev;
        end
        
        if(!ena)
            edgeout <= 0;
        else if((edet[0] == 1) && (edet[1] == 0))
            edgeout <= 1;
    end
endmodule

module  trigtime(
    CLK66, regrd, regwr, rega, regdi, regdo,
    newdata, comparator, secondary, ena, out
);
input           CLK66;
input           regrd, regwr;
input   [7:0]   rega;
input   [7:0]   regdi;
output  [7:0]   regdo;

input           newdata;
input           comparator;
input           secondary;
input           ena;
output          out;

reg     [7:0]   mode1, mode2, mode3;
reg     [31:0]  timea;
reg     [31:0]  timeb;
reg     [31:0]  holdoff;
reg     [31:0]  holdcnt;
reg             newhold;

reg     [1:0]   edet;
reg     [1:0]   edet2;
reg     [31:0]  timer;
reg     [31:0]  timex;
reg             timechk, tmeasure, ta, tb, nth, tstrt, xnrp;
reg             out = 0;
wire            edge1, edge2;

    assign regdo = 8'hzz;

    always @ (posedge CLK66) begin
        if(regwr && rega == `TRG_MODE1)
            mode1  <= regdi;
        if(regwr && rega == `TRG_MODE2)
            mode2 <= regdi;
        if(regwr && rega == `TRG_MODE3)
            mode3 <= regdi;
            
        if(regwr && rega == `TRG_TIMEA1)
            timea[7:0] <= regdi;
        if(regwr && rega == `TRG_TIMEA2)
            timea[15:8] <= regdi;
        if(regwr && rega == `TRG_TIMEA3)
            timea[23:16] <= regdi;
        if(regwr && rega == `TRG_TIMEA4)
            timea[31:24] <= regdi;
            
        if(regwr && rega == `TRG_TIMEB1)
            timeb[7:0] <= regdi;
        if(regwr && rega == `TRG_TIMEB2)
            timeb[15:8] <= regdi;
        if(regwr && rega == `TRG_TIMEB3)
            timeb[23:16] <= regdi;
        if(regwr && rega == `TRG_TIMEB4)
            timeb[31:24] <= regdi;
    end

    
    assign edge1 = (mode2[`TRG_MODE2_POS] &&  edet[0] && !edet[1]) ||
                   (mode2[`TRG_MODE2_NEG] && !edet[0] &&  edet[1]);
    
    assign edge2 = (mode2[`TRG_MODE2_POS2] &&  edet2[0] && !edet2[1]) ||
                   (mode2[`TRG_MODE2_NEG2] && !edet2[0] &&  edet2[1]);
    
    always @ (posedge CLK66) begin
        
        timechk <= 0;
        
        if(newdata) begin
            edet[1] <= edet[0];
            edet[0] <= comparator;
            
            edet2[1] <= edet2[0];
            edet2[0] <= secondary;
        end
        
        if(!ena) begin
            out <= 0;
            timer  <= 0;
            timechk <= 0;
            ta <= 0;
            tb <= 0;
            tmeasure <= 0;
            nth <= 0;
            tstrt <= 0;
            xnrp <= 0;
        end
        else if(newdata) begin              // 50 MHz clock
            
            timer <= timer + 1;
            
            if(mode2[`TRG_MODE2_PW]) begin
                if(comparator == mode2[`TRG_MODE2_POS]) begin
                    timer <= timer + 1;
                    tmeasure <= nth;
                end
                else begin
                    if(tmeasure) begin
                        timechk <= 1;
                        timex <= timer;
                    end
                    nth <= 1;
                    timer <= 0;
                    tmeasure <= 0;
                end
            end
            else if(mode2[`TRG_MODE2_IV]) begin
                if(edge1) begin
                    timechk <= nth;
                    timex <= timer;
                    timer <= 0;
                    nth <= 1;
                end
            end
            else if(mode2[`TRG_MODE2_DO]) begin
                if(edge1) begin
                    timer <= 0;
                    nth   <= 1;
                end
                else begin
                    timex <= timer;
                    timechk <= nth;
                end
            end
            else if(mode2[`TRG_MODE3_RESP]) begin
                if(edge1) begin
                    if(!tstrt || mode3[`TRG_MODE3_RESTART])
                        timer <= 0;
                    
                    tstrt <= 1;
                end
                else if(edge2) begin
                    timex   <= timer;
                    timechk <= tstrt;
                    tstrt   <= 0;
                end
            end
            else if(mode2[`TRG_MODE3_NORESP]) begin
                if(edge1) begin
                    timer <= 0; // ido 0
                    xnrp  <= 1; // mostantol varjuk a masikon a pulzust 
                    if(xnrp)    // ha ujbol ezen volt pulzus, akkor trig
                        out <= ena;
                end
                else begin
                    timex   <= timer;
                    timechk <= xnrp;
                    
                    if(edge2)
                        xnrp <= 0;
                end
            end
        end
        
        if(mode2[`TRG_MODE2_EDGE]) begin
            if(newdata) begin
                if(!mode1[`TRG_MODE1_HOLDEVS] && (holdcnt != 0)) begin
                    holdcnt <= holdcnt - 1;
                end
                
                if(edge1) begin
                    if(holdcnt == 0) begin
                        out <= ena;
                        holdcnt <= holdoff;
                    end
                    else if(mode1[`TRG_MODE1_HOLDEVS]) begin
                        holdcnt <= holdcnt - 1;
                    end
                end
            end
            
            if(newhold) begin
                holdcnt <= 0;
                newhold <= 0;
            end
        end
        else if(timechk) begin
            /*
             * timechk csak akkor allhat be egybe, ha ena is 1, ezert
             * ta es tb is csak akkor allhat be, ha ena 1
             * ezert az "out <= 1" a kovetkezo if-ben rendben van
             */
            if(!mode3[`TRG_MODE3_A] || (timex > timea))
                ta <= 1;
            else
                ta <= 0;
            
            if(!mode3[`TRG_MODE3_B] || (timex < timeb))
                tb <= 1;
            else
                tb <= 0;
        end
        if(( mode3[`TRG_MODE3_OR] && (ta || tb)) ||
           (!mode3[`TRG_MODE3_OR] && ta && tb)) begin
            out <= 1;
        end
        
        if(regwr & rega == `TRG_HOLDOFF1) begin
            holdoff[7:0] <= regdi;
            newhold <= 1;
        end
        if(regwr & rega == `TRG_HOLDOFF2) begin
            holdoff[15:8] <= regdi;
            newhold <= 1;
        end
        if(regwr & rega == `TRG_HOLDOFF3) begin
            holdoff[23:16] <= regdi;
            newhold <= 1;
        end
        if(regwr & rega == `TRG_HOLDOFF4) begin
            holdoff[31:24] <= regdi;
            newhold <= 1;
        end
    end
endmodule





`define P_L0    2'b11
`define P_S0    2'b01
`define P_L1    2'b10
`define P_S1    2'b00

// 1. felkep kezdete:   L1 L1 S1 S1 S1 S1 S1 S1 
`define ODDBEGIN    16'b10_10_00_00_00_00_00_00

// 2. felkep kezdete:   L1 L1 S1 S1 S1 S1 S1 S0
`define EVENBEGIN   16'b10_10_00_00_00_00_00_01

//
//     L1        L1   S1   S1   S1   S1   S1   S1   S0   S0   S0   S0   S0   S1   S1   S1   S1   S1         L1 
//  ___  ________  ___  ___  ___  ___  ___  ___                           ___  ___  ___  ___  ___  ________
//     \/        \/   \/   \/   \/   \/   \/   \___/\___/\___/\___/\___/\/   \/   \/   \/   \/   \/        \/
//                                                                                                              
//                                             ^ Line    1 (1 /1)                                               
//                                                                                                              
//                    5    4    3    2    1    0    IN
//          L1        L1   S1   S1   S1   S1   S1   S0   S0   S0   S0   S0   S1   S1   S1   S1        L1
//  ________  ________  ___  ___  ___  ___  ___                           ___  ___  ___  ___  ________  ______
//\/        \/        \/   \/   \/   \/   \/   \___/\___/\___/\___/\___/\/   \/   \/   \/   \/        \/
//                                                   
//                                                  ^ Line 314 (1 /2)
//
module  trigvideo(
    CLK66, regrd, regwr, rega, regdi, regdo,
    newdata, dt1, dt2, ena, out
);  

input           CLK66;
input           regrd, regwr;
input   [7:0]   rega;
input   [7:0]   regdi;
output  [7:0]   regdo;

input           newdata;            // 50MHz ref clock
input   [7:0]   dt1, dt2;           // Channel A/D
input           ena;                // output enable
output          out;                // trigger output

reg     [7:0]   mode1 = 0;          // source select
reg     [7:0]   level = 0;          // vandorol
reg     [7:0]   muxout = 0;
reg     [12:0]  kondi = 0;          // low pass filter kondija
wire    [8:0]   uc;
wire    [8:0]   ube;
wire    [8:0]   dif9;
wire    [12:0]  diff;
reg     [7:0]   data;

reg     [9:0]   ctime = 0;          // vandorlas timer
reg             comp = 0;
reg     [1:0]   edet;

reg     [11:0]  timer = 0;
reg             time_8us = 0;       // lefuto el ota eltelt 8 us
reg             time_48us = 0;      // lefuto el ota eltelt 48 us
reg             time_l0 = 0;        // nullaban volt 8us-nal tovabb
reg     [15:0]  events = 0;         // az utolso 8 esemeny
reg             newe = 0;           // uj esemeny van (lefuto elnel)

reg     [15:0]  tmr6266 = 0;        // 62-66 us ablakhoz
reg             window = 0;
reg     [15:0]  uline = 0;          // user sorszam es mode bitek
reg     [15:0]  uptrn = 0;          // user event pattern
reg     [7:0]   umask = 0;          // user event mask
reg             newline = 0;        // uj ertek a 'line'-ban
reg     [10:0]  line = 1;           // a jelenlegi sor (1-n) 1024-nel megall

reg             out = 0;

    assign regdo = 8'hzz;

    assign uc   = {1'b0, kondi[12:5]};
    assign ube  = {1'b0, muxout};
    assign dif9 = ube - uc;
    assign sig = dif9[8];
    assign diff = {sig, sig, sig, sig, dif9};
    
    /*
     *  A bejovo jelet MUX-olja (ch1/ch2), opcionalisan atnyomja egy LP filteren,
     *  ha kell negalja, koveti a DC vandorlast,
     *  majd egy hiszterezis-komparatorral eloallitja a digitalis szinkronjelet
     */
    always @ (posedge CLK66) begin
        if(newdata) begin
            muxout <= (mode1[`TRG_MODE1_INSEL] == `TRG_MODE1_INCH1) ? dt1 : dt2;
            
            kondi <= kondi + diff;
            
            if(uline[`VIDEO_FILT])
                data <= uline[`VIDEO_INV] ? ~uc[7:0] : uc[7:0];
            else
                data <= uline[`VIDEO_INV] ? ~muxout : muxout;
            
            if((comp == 1) && !(data > level+12))
                comp <= 0;
            else if((comp == 0) && (data > level+16))
                comp <= 1;
            
            if(data < level)
                level <= data;              // legalacsonyabb szint
            
            if(ctime == 0) begin
                level <= level + 1;
                ctime <= 600;               // 600*20 = 12us
            end
            else begin
                ctime <= ctime - 1; 
            end
        end
    end
    
    /*
     *  Meri a lefuto elek kozotti idot (>48us vagy nem), valamint
     *  a lefutotol felfutoig idot is meri (>8us vagy nem)
     *  Ezt a ket informaciot minden lefuto elnel shifteli egy 16 bites
     *  regiszterbe, igy abban 8 lefuto el idozitese lathato egyszerre
     */
    always @ (posedge CLK66) begin
        
        newe <= 0;
        
        edet[1] <= edet[0];
        edet[0] <= comp;
        
        if((edet[1] == 1) && (edet[0] == 0)) begin  // fall
            timer <= 0;
            time_8us  <= 0;
            time_48us <= 0;
            time_l0   <= 0;
            events[15:0] <= {events[13:0], time_48us, time_l0};
            newe <= 1;
        end
        else begin
            timer <= timer + 1;
            if(timer >= 528)                        // @ 66MHz
                time_8us <= 1;
            if(timer >= 3168)
                time_48us <= 1;
            
            if((edet[1] == 0) && (edet[0] == 1))    // rise
                time_l0 <= time_8us;
        end
    end

    /*
     *  Az utolso 8 lefuto el idozitesebol detektalja a kepszinkronokat
     *  a user beallitasok alapjan es ahhoz szinkronizalva eloallit egy 62-66us
     *  ablakot, ami a sorszinkron preciz detektalasahoz kell. Mindezek
     *  alapjan line regiszterben szamolja a sorokat. 1024-nel megall a szamlalo
     */
    always @ (posedge CLK66) begin
        
        tmr6266 <= tmr6266 + 1;
        
        if(tmr6266 == 0)
            window <= 0;
        if(tmr6266 == 4092)
            window <= 1;
        if(tmr6266 == 4356)
            window <= 0;
        
        newline <= 0;
        
        if(newe) begin
            if((!umask[7] || (events[15:14] == uptrn[15:14])) &&
               (!umask[6] || (events[13:12] == uptrn[13:12])) &&
               (!umask[5] || (events[11:10] == uptrn[11:10])) &&
               (!umask[4] || (events[9:8]   == uptrn[9:8])) &&
               (!umask[3] || (events[7:6]   == uptrn[7:6])) &&
               (!umask[2] || (events[5:4]   == uptrn[5:4])) &&
               (!umask[1] || (events[3:2]   == uptrn[3:2])) &&
               (!umask[0] || (events[1:0]   == uptrn[1:0]))) begin
                line    <= 1;
                newline <= 1;
                tmr6266 <= 0;
            end
            else if(window) begin
                if(line[10] == 0)
                    line <= line + 1;
                newline <= 1;
                tmr6266 <= 0;
            end
        end
    end
    
    always @ (posedge CLK66) begin
        if(regwr && rega == `TRG_MODE1)
            mode1 <= regdi;
            
        if(regwr && rega == `TRG_VIDEOM)
            umask <= regdi;

        if(regwr && rega == `TRG_VIDEOP1)
            uptrn[7:0] <= regdi;
            
        if(regwr && rega == `TRG_VIDEOP2)
            uptrn[15:8] <= regdi;
            
        if(regwr && rega == `TRG_VIDEOL1)
            uline[7:0] <= regdi;
        
        if(regwr && rega == `TRG_VIDEOL2)
            uline[15:8] <= regdi;
        
        if(!ena)
            out <= 0;
        else if(newline && line == uline[`VIDEO_LINE])
            out <= 1;
    end
endmodule

