/*  
  Author: Simon "Simonexc" Trochimiak
  23 January 2017
  Version: 1.0
*/

#include <Wire.h>
#include <LiquidCrystal_I2C.h>
#include <DS3231.h>
#include <OneWire.h>
#include <DallasTemperature.h>
#include <EEPROM.h>

// I'm using int only when a variable is < 0 or > 255. Otherwise I use byte to save memory.

#define TIME_DISPLAY 10 // How long the time is displayed || in seconds
#define DATE_DISPLAY 4 // How long the date is displayed || in seconds
#define BUZZER_PIN 8
#define MAX_POSSIBLE_MENU 2 // How many main screens (one for temperature and one for light)
#define LEDS 9 // to which pin the MOSFET is connected

#define MAIN_MENU_LENGTH 4 // how many options are in main menu

// TEMPERATURE SENSORS //
#define ONE_WIRE_BUS 7 // temperature read pin
#define TEMPERATURE_PRECISION 10

// BLINK //
#define BLINK_ON 1000 // It applies to some settings. It determines how long you will see the number || in milliseconds
#define BLINK_OFF 400 // It applies to some settings. It determines how long you won't see the number || in milliseconds

// TO FUNCTION //
#define X_DIFF 3
#define SLOPE 10

DS3231  rtc(SDA, SCL);
LiquidCrystal_I2C lcd(0x3F, 2, 1, 0, 4, 5, 6, 7, 3, POSITIVE);
OneWire oneWire(ONE_WIRE_BUS);

DallasTemperature sensors(&oneWire);
DeviceAddress thermometer1, thermometer2;

Time t;

int read_write_int(int addr, int value);
float read_write_float(int addr, float value);
long read_write_long(int addr, long value);
bool read_write_bool(int addr, bool value);
//void write_float(int address, float variable);
//void write_bool(int address, bool variable);

String mainMenu[MAIN_MENU_LENGTH] = {"Light", "Temperature", "Clock", "Display"};
String subMenu[MAIN_MENU_LENGTH][5] = {
  {"Mode", "Dawn Start", "Dawn Duration", "Dusk Start", "Dusk Duration"},
  {"Set Temp.", "Max difference", "Alarm", "Temp. units"},
  {"Set Time", "Set Date", "Summer time", "Set day of week"},
  {"Auto off delay", "Auto off"}
};

byte subMenuLengths[MAIN_MENU_LENGTH] = {5, 4, 4, 2};
byte menu = 0;
byte menuPos1 = 0;
byte menuPos2 = 0;

int seconds = -1;
int dtChangeTimer = -2; // dt stands for date/time
int buzzerTimer = seconds;
int backlightTimer = -2;
unsigned long long blinkTimer = millis();
bool time = true;
byte displayPos = 0;

float waterTemp;

// Temperature //
float requiredTemp = read_write_float(0, 26.00); // in degrees Centigrade || EEPROM address - 0, 1, 2
float tempDiff = read_write_float(3, 5.00); // in degrees Centigrade || EEPROM address - 3, 4, 5
bool temperatureAlert = read_write_bool(6, true); // EEPROM address - 6
byte temperatureUnit = read_write_int(19, 0); // 0 - Celsius, 1 - Fahrenheit, 2 - Kelvin || EEPROM address - 19
// Change versions
float newRequiredTemp = requiredTemp;
float newTempDiff = tempDiff;
bool newTemperatureAlert = temperatureAlert;
byte newTemperatureUnit = temperatureUnit;

// Light Control //
byte hourDawn = read_write_int(7, 8); // EEPROM address - 7
byte minDawn = read_write_int(8, 30); // EEPROM address - 8
byte hourDusk = read_write_int(9, 21); // EEPROM address - 9
byte minDusk = read_write_int(10, 30); // EEPROM address - 10
long dawnDuration = read_write_long(11, 120); // in seconds || EEPROM address - 11, 12, 13
long duskDuration = read_write_long(14, 120); // in seconds || EEPROM address - 14, 15, 16
bool autoMode = read_write_bool(17, true); // EEPROM address - 17
int fastOff = 120; // in seconds
// Change versions
byte newHourDawn = hourDawn;
byte newMinDawn = minDawn;
byte newHourDusk = hourDusk;
byte newMinDusk = minDusk;
long newDawnDuration = dawnDuration; // in seconds
long newDuskDuration = duskDuration; // in seconds
bool newAutoMode = autoMode;

// Clock //
bool summerTime = read_write_bool(18, false); // EEPROM address - 18
// Change versions
byte newSecond = 0;
byte newMinute = 0;
byte newHour = 0;
byte newDay = 0;
byte newMonth = 0;
int newYear = 0;
bool newSummerTime = summerTime;
byte newDow = 1; // dow - day of the week; 1 -> Monday, 7 -> Sunday
// How many days in each month
byte monthsLengths[12] = {31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};
// Days of week
String dows[7] = {"Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday", "Sunday"};

// Display //
byte autoOffDelay = read_write_int(20, 10); // in seconds || EEPROM address - 20
bool autoOff = read_write_bool(21, true); // EEPROM address - 21
// Change versions
byte newAutoOffDelay = autoOffDelay;
bool newAutoOff = autoOff;


bool background = HIGH;

// Blinking menu //
bool blink = false;
bool visible = false;
byte blinkPos = 0;
int showMode = -1;

// TO LIGHTS AND FUNCTION //
byte fading = 0;
double x = 0;
byte phase = 255;
long timeLeft = 0;

// SPECIAL SIGNS //
byte degree[8] = {
  B00000100,
  B00001010,
  B00000100,
  B00000000,
  B00000000,
  B00000000,
  B00000000,
  B00000000
};

byte arrow[8] = {
  B00000000,
  B00010000,
  B00011000,
  B00011100,
  B00011100,
  B00011000,
  B00010000,
  B00000000
};

// BUTTONS //
byte buttons[] = {3, 4, 5, 6};
bool buttonsPos[] = {LOW, LOW, LOW, LOW};

void setup()
{
  //Serial.begin(9600);
  
  // Initialize the rtc object
  rtc.begin();
  // Initialize the lcd object
  lcd.begin(16,2);
  lcd.backlight();
  
  lcd.createChar(0, degree);
  lcd.createChar(1, arrow);
  
  for (int i = 0; i<sizeof(buttons); i++)
    pinMode(INPUT, buttons[i]);
    
    
  sensors.begin();
  sensors.getAddress(thermometer1, 0);
  sensors.getAddress(thermometer2, 1);
  
  // set the resolution
  sensors.setResolution(thermometer1, TEMPERATURE_PRECISION);
  sensors.setResolution(thermometer2, TEMPERATURE_PRECISION);
  
  // get water temperature
  waterTemp = getTemperature();
  
}

void loop()
{
  
  // MENU/OK/ACCEPT BUTTON //
  // ON
  if (digitalRead(buttons[0]) && !buttonsPos[0]) {
    bool off = backlightTimer != -1;
    button_click(0, HIGH);
    
    if (off) {
      if (menu == 0) {
        menu = 1;
        main_menu();
      }
      else if (menu == 1) {
        menu = 2;
        sub_menu();
      }
      else if (menu == 2) {
        menu = 3;
        which_setting();
      }
      else if (menu == 3)
        ok();
    }
    
  }
  // OFF
  else if (digitalRead(buttons[0]) == LOW && buttonsPos[0])
    button_click(0, LOW);
  
  // CLOSE/BACK/CANCEL BUTTON //
  // ON
  if (digitalRead(buttons[1]) && !buttonsPos[1]) {
    bool off = backlightTimer != -1;
    button_click(1, HIGH);
    
    if (off) {
      if (menu == 1) {
        menu = 0;
        menuPos1 = 0;
        display();
      }
      else if (menu == 2) {
        menuPos2 = 0;
        menu = 1;
        main_menu();
      }
      else if (menu == 3) {
        menu = 2;
        cancel();
      }
    }
    
  }
  // OFF
  else if (digitalRead(buttons[1]) == LOW && buttonsPos[1])
    button_click(1, LOW);
  
  // DOWN/LEFT BUTTON //
  // ON
  if (digitalRead(buttons[2]) == HIGH && !buttonsPos[2]) {
    bool off = backlightTimer != -1;
    button_click(2, HIGH);
    
    if (off) {
      if (menu == 0) {
        set_variable(&displayPos, MAX_POSSIBLE_MENU-1);
        display();
      }
      else if (menu == 1) {
        set_variable(&menuPos1, MAIN_MENU_LENGTH-1);
        main_menu();
      }
      else if (menu == 2) {
        set_variable(&menuPos2, subMenuLengths[menuPos1]-1);
        sub_menu();
      }
      else if (menu == 3)
        subtract();
    }
    
  }
  // OFF
  else if (digitalRead(buttons[2]) == LOW && buttonsPos[2])
    button_click(2, LOW);
  
  // UP/RIGHT BUTTON //
  // ON
  if (digitalRead(buttons[3]) && !buttonsPos[3]) {
    bool off = backlightTimer != -1;
    button_click(3, HIGH);
    
    if (off) {
      if (menu == 0) {
        displayPos = (displayPos + 1)%MAX_POSSIBLE_MENU;
        display();
      }
      else if (menu == 1) {
        menuPos1 = (menuPos1 + 1)%MAIN_MENU_LENGTH;
        main_menu();
      }
      else if (menu == 2) {
        menuPos2 = (menuPos2 + 1)%subMenuLengths[menuPos1];
        sub_menu();
      }
      else if (menu == 3)
        add();
    }
    
  }
  // OFF
  else if (digitalRead(buttons[3]) == LOW && buttonsPos[3])
    button_click(3, LOW);
  
  t = rtc.getTime();
  if (t.sec != seconds) {
    seconds = t.sec;
    
    if (menu != 3)
      update_date_variables();
      
    light_fading();
    
    if (menu == 0)
      display();
      
    if (dtChangeTimer == -2)
      dtChangeTimer = (seconds + (int)(!time)*DATE_DISPLAY + (int)time*TIME_DISPLAY)%60;
    
    if (backlightTimer == -2)
      backlightTimer = (seconds + autoOffDelay)%60;
    
  }
  
  // Blink timer
  if (blink && blinkTimer < millis())
    which_setting();
  
  // Time/date change
  if (dtChangeTimer == seconds) {
    time = !time;
    dtChangeTimer = (seconds + (int)(!time)*DATE_DISPLAY + (int)time*TIME_DISPLAY)%60;
  }
  
  if (waterTemp < requiredTemp - tempDiff && buzzerTimer != seconds && temperatureAlert) {
    tone(BUZZER_PIN, 440, 3000);
    buzzerTimer = (seconds + 4)%60;
  }
  else if (waterTemp > requiredTemp + tempDiff && buzzerTimer != seconds && temperatureAlert) {
    tone(BUZZER_PIN, 440, 1000);
    buzzerTimer = (seconds + 2)%60;
  }
  
  if (backlightTimer == seconds && menu == 0 && autoOff) {
    backlightTimer = -1;
    lcd.noBacklight();
  }
    
}
