/****************************************************************************
 Title	:   HD44780U LCD library
 Author:    Peter Fleury <pfleury@gmx.ch>  http://jump.to/fleury
 Date:	    5-DEC-2002
 Software:  AVR-GCC 3.3 
 Target:    any AVR device, memory mapped mode only for AT90S4414/8515/Mega

 DESCRIPTION
       Basic routines for interfacing a HD44780U-based text lcd display

       Originally based on Volker Oth's lcd library,
       changed lcd_init(), added additional constants for lcd_command(),
       added 4-bit I/O mode, improved and optimized code.

       Library can be operated in memory mapped mode (LCD_IO_MODE=0) or in 
       4-bit IO port mode (LCD_IO_MODE=1). 8-bit IO port mode not supported.
       
       Memory mapped mode compatible with Kanda STK200, but supports also
       generation of R/W signal through A8 address line.

 USAGE
       See the C include lcd.h file for a description of each function
       
*****************************************************************************/
#include <inttypes.h>
#include <avr/io.h>
#include <avr/pgmspace.h>
#include "lcd.h"


/* 
** constants/macros 
*/
#define PIN(x) (*(&x - 2))  /* address of data direction register of port x */
#define DDR(x) (*(&x - 1))  /* address of input register of port x          */


#if LCD_IO_MODE
#define lcd_e_delay() __asm__ __volatile__( "rjmp 1f\n 1:" );
#define lcd_e_high()    sbi(LCD_E_PORT, LCD_E_PIN)
#define lcd_e_low()     cbi(LCD_E_PORT, LCD_E_PIN)
#define lcd_e_toggle()  toggle_e()
#endif

#if LCD_IO_MODE
#if LCD_LINES==1
#define LCD_FUNCTION_DEFAULT    LCD_FUNCTION_4BIT_1LINE 
#else
#define LCD_FUNCTION_DEFAULT    LCD_FUNCTION_4BIT_2LINES 
#endif
#else
#if LCD_LINES==1
#define LCD_FUNCTION_DEFAULT    LCD_FUNCTION_8BIT_1LINE
#else
#define LCD_FUNCTION_DEFAULT    LCD_FUNCTION_8BIT_2LINES
#endif
#endif


/* 
** function prototypes 
*/
static void delay(uint16_t us);
#if LCD_IO_MODE
static void toggle_e(void);
#endif

/*
** local functions
*/

static void delay(uint16_t us) 
/* delay for a minimum of <us> microseconds    */
/* with a 4Mhz crystal, the resolution is 1 us */
{
    while ( us ) us--;
} 


#if LCD_IO_MODE
static void toggle_e(void)
/* toggle Enable Pin */
{
    lcd_e_high();
    lcd_e_delay();
    lcd_e_low();
}
#endif


#if LCD_IO_MODE
static void inline lcd_write(uint8_t data,uint8_t rs) 
{
    /* configure data pins as output */
    outp(0xFF, DDR(LCD_DATA_PORT) );

    if (rs) {   /* write data        (RS=1, RW=0) */
    
       /* output high nibble first */
       outp( ((data>>4)&0x0F)|(1<<LCD_RS_PIN), LCD_DATA_PORT );
       lcd_e_toggle();

       /* output low nibble */
       outp( (data&0x0F)|(1<<LCD_RS_PIN), LCD_DATA_PORT );
       lcd_e_toggle();
       
    } else {    /* write instruction (RS=0, RW=0) */
           
       /* output high nibble first */
       outp( (data>>4)&0x0F, LCD_DATA_PORT );
       lcd_e_toggle();

       /* output low nibble */
       outp( data&0x0F, LCD_DATA_PORT );
       lcd_e_toggle();
    }

    /* all data pins high (inactive) */
    outp(0x0F, LCD_DATA_PORT);
}
#else
#define lcd_write(d,rs) if (rs) *(volatile uint8_t*)(LCD_IO_DATA) = d; else *(volatile uint8_t*)(LCD_IO_FUNCTION) = d;
/* rs==0 -> write instruction to LCD_IO_FUNCTION */
/* rs==1 -> write data to LCD_IO_DATA */
#endif


#if LCD_IO_MODE
static uint8_t lcd_read(uint8_t rs) 
{
    register uint8_t dataH, dataL;

    if (rs) sbi(LCD_RS_PORT, LCD_RS_PIN);    /* RS=1: read data      */
    else    cbi(LCD_RS_PORT, LCD_RS_PIN);    /* RS=0: read busy flag */
    sbi(LCD_RW_PORT, LCD_RW_PIN);            /* RW=1  read mode      */

    /* configure data pins as input */
    outp(0xF0, DDR(LCD_DATA_PORT));
    
    lcd_e_high();
    lcd_e_delay();
    dataH = inp(PIN(LCD_DATA_PORT));         /* read high nibble first */
    lcd_e_low();
    
    lcd_e_delay();                           /* Enable 500ns low       */
    
    lcd_e_high();
    lcd_e_delay();
    dataL = inp(PIN(LCD_DATA_PORT));         /* read low nibble        */
    lcd_e_low();
    
    return ( (dataH<<4) | (dataL&0x0F) );
}
#else
#define lcd_read(rs) (rs) ? *(volatile uint8_t*)(LCD_IO_DATA+LCD_IO_READ) : *(volatile uint8_t*)(LCD_IO_FUNCTION+LCD_IO_READ)
/* rs==0 -> read instruction from LCD_IO_FUNCTION */
/* rs==1 -> read data from LCD_IO_DATA */
#endif


static uint8_t lcd_waitbusy(void)
/* loops while lcd is busy, reads address counter */
{
    register uint8_t c;
    
    /* wait until busy flag is cleared */
    while ( (c=lcd_read(0)) & (1<<LCD_BUSY)) {}
    
    /* the address counter is updated 4us after the busy flag is cleared */
    delay(2);

    /* now read the address counter */
    return (lcd_read(0));  // return address counter
}


static inline void lcd_newline(uint8_t pos)
/* goto start of next line */
{
    register uint8_t addressCounter;


#if LCD_LINES==1
    addressCounter = 0;
#endif
#if LCD_LINES==2
    if ( pos < (LCD_START_LINE2) )
        addressCounter = LCD_START_LINE2;
    else
        addressCounter = LCD_START_LINE1;
#endif
#if LCD_LINES==4
    if ( pos < LCD_START_LINE3 )
        addressCounter = LCD_START_LINE2;
    else if ( (pos >= LCD_START_LINE2) && (pos < LCD_START_LINE4) )
        addressCounter = LCD_START_LINE3;
    else if ( (pos >= LCD_START_LINE3) && (pos < LCD_START_LINE2) )
        addressCounter = LCD_START_LINE4;
    else 
        addressCounter = LCD_START_LINE1;
#endif
    lcd_command((1<<LCD_DDRAM)+addressCounter);

}/* lcd_newline */


/*
** PUBLIC FUNCTIONS 
*/

void lcd_command(uint8_t cmd)
/* send commando <cmd> to LCD */
{
    lcd_waitbusy();
    lcd_write(cmd,0);
}


void lcd_gotoxy(uint8_t x, uint8_t y)
/* goto position (x,y) */
{
#if LCD_LINES==1
    lcd_command((1<<LCD_DDRAM)+LCD_START_LINE1+x);
#endif
#if LCD_LINES==2
    if ( y==0 ) 
        lcd_command((1<<LCD_DDRAM)+LCD_START_LINE1+x);
    else
        lcd_command((1<<LCD_DDRAM)+LCD_START_LINE2+x);
#endif
#if LCD_LINES==4
    if ( y==0 )
        lcd_command((1<<LCD_DDRAM)+LCD_START_LINE1+x);
    else if ( y==1)
        lcd_command((1<<LCD_DDRAM)+LCD_START_LINE2+x);
    else if ( y==2)
        lcd_command((1<<LCD_DDRAM)+LCD_START_LINE3+x);
    else /* y==3 */
        lcd_command((1<<LCD_DDRAM)+LCD_START_LINE4+x);
#endif

}/* lcd_gotoxy */


int lcd_getxy(void)
{
    return lcd_waitbusy();
}


void lcd_clrscr(void)
/* clear lcd and set cursor to home position */
{
    lcd_command(1<<LCD_CLR);
}


void lcd_home(void)
/* set cursor to home position */
{
    lcd_command(1<<LCD_HOME);
}


void lcd_putc(char c)
/* print character at current cursor position */
{
    register uint8_t pos;
    
    pos = lcd_waitbusy();   // read busy-flag and address counter
    if (c=='\n')
        lcd_newline(pos);
    else
        lcd_write(c, 1);
}


void lcd_puts(const char *s)
/* print string on lcd (no auto linefeed) */
{
    register char c;

    while ( (c = *s++) ) {
        lcd_putc(c);
    }
}


void lcd_puts_p(const char *progmem_s)
/* print string from program memory on lcd (no auto linefeed) */
{
    register char c;

    while ( (c = PRG_RDB(progmem_s++)) ) {
        lcd_putc(c);
    }
}


void lcd_init(uint8_t dispAttr)
/* initialize display and select type of cursor */
/* dispAttr: LCD_DISP_OFF, LCD_DISP_ON, LCD_DISP_ON_CURSOR, LCD_DISP_CURSOR_BLINK */
{
#if LCD_IO_MODE   
    /*------ Initialize lcd to 4 bit i/o mode -------*/
    
    outp( 0xFF, DDR(LCD_DATA_PORT) );         /* all port bits as output */
    
    delay(16000);        /* wait 16ms or more after power-on       */
    
    /* initial write to lcd is 8bit */
    outp(LCD_FUNCTION_8BIT_1LINE>>4,LCD_DATA_PORT);
    lcd_e_toggle();
    delay(4992);         /* delay, busy flag can't be checked here */
    
    outp(LCD_FUNCTION_8BIT_1LINE>>4,LCD_DATA_PORT);
    lcd_e_toggle();
    delay(64);           /* delay, busy flag can't be checked here */
    
    outp(LCD_FUNCTION_8BIT_1LINE>>4,LCD_DATA_PORT);
    lcd_e_toggle();
    delay(64);           /* delay, busy flag can't be checked here */
    
    outp(LCD_FUNCTION_4BIT_1LINE>>4,LCD_DATA_PORT); /* set IO mode to 4bit */
    lcd_e_toggle();
    
    /* from now the lcd only accepts 4 bit I/O, we can use lcd_command() */    
#else
    /*----- Initialize lcd to 8 bit memory mapped mode ------*/
    
    /* enable external SRAM (memory mapped lcd) and one wait state */    
    outp((1<<SRE)|(1<<SRW), MCUCR);

    /* reset lcd */
    delay(16000);                           /* wait 16ms after power-on     */
    lcd_write(LCD_FUNCTION_8BIT_1LINE,0);   /* function set: 8bit interface */                   
    delay(4992);                            /* wait 5ms                     */
    lcd_write(LCD_FUNCTION_8BIT_1LINE,0);   /* function set: 8bit interface */                 
    delay(64);                              /* wait 64us                    */
    lcd_write(LCD_FUNCTION_8BIT_1LINE,0);   /* function set: 8bit interface */                
    delay(64);                              /* wait 64us                    */
#endif
    lcd_command(LCD_FUNCTION_DEFAULT);      /* function set: display lines  */
    lcd_command(LCD_DISP_OFF);              /* display off                  */
    lcd_clrscr();                           /* display clear                */ 
    lcd_command(LCD_MODE_DEFAULT);          /* set entry mode               */
    lcd_command(dispAttr);                  /* display/cursor control       */

}/* lcd_init */
