/********************************************************************
* iButton.c
* Library of Dallas one wire bus functions
********************************************************************/

#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/delay.h>
#include "iButton.h"

void IBlow()
{
	//PB0 - iButton Pin
	DDRB 	|= 0x01;		//Configure PB0 output 1 - output, 0 - Input
	PORTB	&= 0xFE;
}

/***********************************************************************
 *  FUNCTION     :  OWReset()
 *  DESCRIPTION  :  Generate a 1-Wire RESET
 *  PARAMETERS   :  void
 *  RETURNS      :  1: if presence detect, 0: otherwise
 ***********************************************************************/
unsigned char OWReset(void)
{
	unsigned char presence;
	cli();
	IBlow();
	delay_us(500);
	DDRB &= 0xFE;	  	//PB0 - Input
	delay_us(45);
	delay_us(20);
	
	if(PINB0 & 0x01)
		presence = 0;
	else
		presence = 1;

	delay_us(500);
	sei();
	return presence;
}

/***********************************************************************
 *  FUNCTION     :  OWWriteBit()
 *  DESCRIPTION  :  Send a 1-Wire write bit. Provide 10uS recovery time
 *  PARAMETERS   :  1 or 0 to be written,
 *  RETURNS      :  void
 ***********************************************************************/

void OWWriteBit(unsigned char bit)
{
	cli();
	switch(bit)
	{
		case 1:
			IBlow();
			delay_us(6);
			DDRB &= 0xFE;	  	//PB0 - Input
			delay_us(64);
			break;
		case 0:
			IBlow();
			delay_us(72);
			DDRB &= 0xFE;	  	//PB0 - Input
			delay_us(10);
			break;
		default:
			break;
	}
	delay_us(10);
	sei();
}

/***********************************************************************
 *  FUNCTION     :  OWReadBit()
 *  DESCRIPTION  :  Read a 1-Wire bit.
 *  PARAMETERS   :  void,
 *  RETURNS      :  Bit read
***********************************************************************/
unsigned char OWReadBit(void)
{
	unsigned char readbit;
	cli();
	IBlow();
	DDRB &= 0xFE;	  	//PB0 - Input
	delay_us(6);
	readbit = PINB0 & 0x01;
	delay_us(60);
	sei();
	return readbit;
}

/***********************************************************************
 *  FUNCTION     :  OWReadByte()
 *  DESCRIPTION  :  Read a byte.
 *  PARAMETERS   :  void,
 *  RETURNS      :  Byte read
 ***********************************************************************/
unsigned char OWReadByte(void)
{
	unsigned char rebyte, loop;
	rebyte = 0x00;
	for(loop=0;loop<8;loop++)
	{
		rebyte >>=1;
		if(OWReadBit())
		rebyte |= 0x80;
	}
	return rebyte;
}

/***********************************************************************
 *  FUNCTION     :  OWWriteByte()
 *  DESCRIPTION  :  Write a byte.
 *  PARAMETERS   :  Byte to be written,
 *  RETURNS      :  void
 ***********************************************************************/
void OWWriteByte(unsigned char data)
{
	unsigned char loop;
	for(loop=0;loop<8;loop++)
	{
		OWWriteBit((data&0x01));
		data >>=1;
	}
}

/***********************************************************************
 *  FUNCTION     :  OWReadROM()
 *  DESCRIPTION  :  Read ROM ID
 *  PARAMETERS   :  void,
 *  RETURNS      :  ROM_ID
 ***********************************************************************/
unsigned char OWReadROM(unsigned char *ROM_ID)
{
    unsigned char   count;
	if(!OWReset()) return 0;
    OWWriteByte(READ_ROM);
    
    for(count=0;count<8;count++)
    {
        *ROM_ID = OWReadByte();
        ROM_ID++;
    }
    if(!OWReset()) return 0;
    return 1;
}

void delay_us(unsigned int us) 
/* delay for a minimum of <us> microseconds    */
/* with a 8Mhz crystal, the resolution is 1 us */
{
	unsigned int us1;
	us1=us;
	while ( us ) us--;
	while ( us1 ) us1--;
}	