;********************************************************************************************************
;				I2C Bus Initialization
;	
;
;       Program:          I2C.ASM 
;       Revision Date:   
;                         09-18-95      Compatibility with MPASMWIN 1.20
;
;********************************************************************************************************
InitI2CBus_Master:
	
	bcf	_rp0
	movf	_portb,w
	andlw	0xFC		; do not use BSF & BCF on Port Pins
	movwf	_portb		; set SDA & SCL to zero. From Now on, simply play with tris
	ReleaseBus
        clrf	Bus_Status	; reset status reg	
	return
;
;**********************************************************************************************************
;				Check If Bus Is Free
;
; Check if both SCL & SDA lines are high
;  returns status in Carry Bit (in STATUS Reg)
;		1	If bus is free
;		0	If bus is not free
;*********************************************************************************************************

IsBusFree:
		bsf	_c
		btfsc	_SDA
		btfss	_SCL
                bcf	_c
		return

;*********************************************************************************************************
;					Send Start Bit
; Returns in WREG TRUE if start bit successful, else FALSE 
;*********************************************************************************************************

TxmtStartBit:	
		bsf	_rp0		; select page 1
		bsf	_SDA		; set SDA high
		bsf	_SCL		; clock is high
;
; Setup time for a REPEATED START condition (4.7 uS)
;
                call	Delay40uSec	; only necesry for setup time
;
		bcf	_SDA		; give a falling edge on SDA while clock is high
;
		call	Delay47uSec	; only necessary for START HOLD time
;
		return


;*********************************************************************************************************
;					Send Stop Bit
;
;*********************************************************************************************************

TxmtStopBit:
		bsf	_rp0		; select page 1
		bcf	_SCL		
		bcf	_SDA		; set SDA low
		bsf	_SCL		; Clock is pulled up
		call	Delay40uSec	; Setup Time For STOP Condition 
		bsf	_SDA		; give a rising edge on SDA while CLOCK is high
;
#if _ENABLE_BUS_FREE_TIME
; delay to make sure a START bit is not sent immediately after a STOP, ensure BUS Free Time tBUF
;
		call	Delay47uSec	
#endif
;
		return	


;*********************************************************************************************************
;					Abort Transmission
;
;   Send STOP Bit & set Abort Flag
;*********************************************************************************************************

AbortTransmission:

		call	TxmtStopBit
		bsf	_Abort
		return	


;*********************************************************************************************************
;				       Get Acknowledge (From Slave Transmitter)
;
;  If SDA is LOW during Ack Clock, then slave has acknowledged
;  If SDA is HIGH during Ack Clock, then slave is busy or error has occured, so ABORT
;
;*********************************************************************************************************

Rcv_Acknowledge:
		
		return	

;*********************************************************************************************************
;					Transmit Acknowledge (To Slave Receiver)
;
;*********************************************************************************************************

Txmt_Acknowledge:

		return


;*********************************************************************************************************
;				Transmit Address (1st Byte)& Put in Read/Write Operation
;
;  Transmits Slave Addr On the 1st byte and set LSB to R/W operation
;  Slave Address must be loaded into SlaveAddr reg
;  The R/W operation must be set in Bus_Status Reg (bit _SLAVE_RW): 0 for Write & 1 for Read
;
;  On Success, return TRUE in WREG, else FALSE in WREG
;
;   If desired, the failure may tested by the bits in Bus Status Reg
;
;*********************************************************************************************************

Txmt_Slave_Addr:
	bcf	_ACK_Error		; reset Acknowledge error bit
	btfss	_10BitAddr
	goto	SevenBitAddr

TenBitAddr:
	movf	SlaveAddr+1,W
	movwf	DataByte
	bcf	DataByte,LSB
	btfsc	_Slave_RW		; if skip then write operation
	bsf	DataByte,LSB		; Read Operation
;
; Ready to transmit data : If Interrupt Driven (i.e if Clock Stretched LOW Enabled)
; then save RETURN Address Pointer
;
	call	SendData		; send high byte of 10 bit addr slave
;
; if successfully transmitted, expect an ACK bit
;
	btfss	_Txmt_Success		; if not successful, generate STOP & abort transfer
        goto	_AddrSendFail
;
	movf	SlaveAddr,W
	movwf	DataByte		; load addr to DatByte for transmission
	goto	EndTxmtAddr

SevenBitAddr:
	movf	SlaveAddr,W
	movwf	DataByte		; load addr to DatByte for transmission
	bcf	DataByte,LSB
	btfsc	_Slave_RW		; if skip then write operation
	bsf	DataByte,LSB		; Read Operation

EndTxmtAddr:
	call	SendData		; send 8 bits of address
	btfsc	_Txmt_Progress		; can perform other jobs here
	goto	$-1

;
; if successfully transmitted, expect an ACK bit
;
	btfss	_Txmt_Success		; skip if successful
	goto	_AddrSendFail
	clrwdt
	retlw	TRUE
;
_AddrSendFail:
	clrwdt
	btfss	_ACK_Error
	retlw	FALSE			; Addr Txmt Unsuccessful, so return 0
;
; Address Not Acknowledged, so send STOP bit
;					
	call	TxmtStopBit
	bsf	_Abort			; Transmission aborted
	retlw	FALSE			; Addr Txmt Unsuccessful, so return 0
;
;*********************************************************************************************************
;				Transmit A Byte Of Data
;
; The data to be transmitted must be loaded into DataByte Reg
; Clock stretching is allowed by slave. If the slave pulls the clock low, then, the stretch is detected
; and INT Interrupt on Rising edge is enabled and also RTCC timeout interrupt is enabled
;	The clock stretching slows down the transmit rate because all checking is done in
;	software. However, if the system has fast slaves and needs no clock stretching, then
;	this feature can be disabled during Assembly time by setting
;	_CLOCK_STRETCH_ENABLED must be set to FALSE.
;
;*********************************************************************************************************
SendData:
    #if _CLOCK_STRETCH_ENABLED

	movlw	(RetAddr1 & 0xff)
	movwf	RetAddrPtrLo
	movlw	( (RetAddr1 >> 8) & 0xff)
	movwf	RetAddrPtrHi
	goto	TxmtByte
RetAddr1
	return
    #else

	goto	TxmtByte

    #endif

TxmtByte:
		movf	DataByte,w
                movwf	DataByteCopy	; make copy of DataByte 
		bsf	_Txmt_Progress	; set Bus status for txmt progress
		bcf	_Txmt_Success	; reset status bit
		bsf	_Bus_Busy	; Bus Is Busy

  #if _CLOCK_STRETCH_ENABLED
		movlw	0x08
		movwf	BitCount

;	 set RTCC to INT CLK timeout for 1 mSec and INT for rising edge
;	 do not disturb user's selection of RPUB in OPTION Register
;
		bsf	_rp0
		movf	_option,w
		andlw	_OPTION_INIT	; defined in I2C.H header file
		movwf	_option		
;
		bcf	_gie
		bsf	_inte		; enable INT interrupt
		bsf	_rtie		; enable RTCC Timeout Interrupt

		bcf	_rp0
		clrf	_rtcc
		clrwdt
                bcf	_rtif
		bsf	_intf		; force an interrupt
		retfie

TxmtNextBit:				; INTERRUPT Service Routine will call this
		bcf	_intf		; clear INT flag
                bcf	_rtif		; clear RTCC INT Flag
		bsf	_rp0		; select page 1 to access TRIS Regs
		bcf	_SCL		; set SCL LOW
		rlf     DataByteCopy	; MSB first
                bcf	_SDA
		btfsc	_c
		bsf	_SDA
		call	Delay47uSec	; guareentee min LOW TIME tLOW & Setup time
		bsf	_SCL
;
; Clock is now set high , but may be still pulled low by a slave, so
; wait for Interrupt
;
		bcf	_rp0
		clrf	_rtcc		; reset RTCC timer, so far I2C Bus normal
		clrwdt			; clear WDT
;
		decfsz	BitCount	; decrement bit count
		return			; wait for next interrupt (INT or RTCC)
;
		bsf	_rp0		; select page 1 for TRIS access
		bcf	_inte		; txmt over, disable INT Interrupt
		bcf	_rtie		; I2C Bus normal, disable RTCC Timeout Interrupt
;
; Test For ACK Bit
;
		call	Delay40uSec	; guareentee min HIGH TIME tHIGH
		bcf	_SCL		; reset clock
		bsf	_SDA		; Release SDA line for Slave to pull down
		call	Delay47uSec	; guareentee min LOW TIME tLOW & Setup time
		bsf	_SCL		; clock for slave to ACK
		call	Delay40uSec	; guareentee min HIGH TIME tHIGH	
		bcf	_rp0		; select PAGE 0 to test PortB pin SDA
		btfsc	_SDA		; SDA should be pulled low by slave if OK
		goto	_TxmtErrorAck
;
_EndTxmtByte:
		bsf	_rp0
		bcf	_SCL		; reset clock

		bsf	_Txmt_Success	; transmission successful
		bcf	_ACK_Error	; ACK From Slave
		bcf	_Txmt_Progress	; reset TXMT bit in Bus Status, readu for another byte
		bcf	_Bus_Busy	; Bus Is Free
		goto    RestoreRet	; transmission of a byte over
_TxmtErrorAck:
		bcf	_Txmt_Success	; transmission NOT successful
		bsf	_ACK_Error	; No ACK From Slave
		ReleaseBus
		bcf	_Txmt_Progress	; reset TXMT bit in Bus Status, readu for another byte
		goto    RestoreRet	; transmission of a byte over
  #else
		movlw	0x08
		movwf	BitCount
		bsf	_rp0
TxmtNextBit:
		clrwdt			; clear WDT, set for 18 mSec
		bcf	_SCL
		rlf     DataByteCopy	; MSB first, Note DataByte Is Lost
                bcf	_SDA
		btfsc	_c
		bsf	_SDA
		call	Delay47uSec	; guareentee min LOW TIME tLOW & Setup time
		bsf	_SCL		; set clock high , assume clock not stretched
		call	Delay40uSec	; guareentee min HIGH TIME tHIGH
		decfsz	BitCount
		goto	TxmtNextBit
;
; Check For Acknowledge
;
		bcf	_SCL		; reset clock
		bsf	_SDA		; Release SDA line for Slave to pull down
		call	Delay47uSec	; guareentee min LOW TIME tLOW & Setup time
		bsf	_SCL		; clock for slave to ACK
		call	Delay40uSec	; guareentee min HIGH TIME tHIGH	
		bcf	_rp0		; select PAGE 0 to test PortB pin SDA
		btfsc	_SDA		; SDA should be pulled low by slave if OK
                goto	_TxmtErrorAck
;
		bsf	_rp0
		bcf	_SCL		; reset clock

		bcf	_Txmt_Progress	; reset TXMT bit in Bus Status
		bsf	_Txmt_Success	; transmission successful
		bcf	_ACK_Error	; ACK OK
		bcf	_Bus_Busy	; Bus Is Free
		return
_TxmtErrorAck:
		ReleaseBus
		bcf	_Txmt_Progress	; reset TXMT bit in Bus Status
		bcf	_Txmt_Success	; transmission NOT successful
		bsf	_ACK_Error	; No ACK From Slave
		return
  #endif
;
;*********************************************************************************************************
;
;				Receive  A Byte Of Data From Slave
;
;  assume address is already sent
;  if last byte to be received, do not acknowledge slave (last byte is testted from
;  _Last_Byte_Rcv bit of control reg)
;  Data Received on successful reception is in DataReg register
;
;  Currently No Clock Stretching is used ( TEMP ????)
;	
;*********************************************************************************************************
;

GetData:
		goto	RcvByte

;
RcvByte:
		
		bsf	_Rcv_Progress	; set Bus status for txmt progress
		bcf	_Rcv_Success	; reset status bit
		bsf	_Bus_Busy	; Bus Is Busy

		movlw	0x08
		movwf	BitCount
RcvNextBit:
		clrwdt			; clear WDT, set for 18 mSec
		bsf	_rp0		; page 1 for TRIS manipulation
		bcf	_SCL
		bsf	_SDA		; can be removed from loop
		call	Delay47uSec	; guareentee min LOW TIME tLOW & Setup time
		bsf	_SCL		; clock high, data sent by slave
		call	Delay40uSec	; guareentee min HIGH TIME tHIGH
		bcf	_rp0		; select page 0 to read Ports
		bcf	_c
		btfsc	_SDA
		bsf	_c
;					; TEMP ???? DO 2 out of 3 Majority detect 
                rlf	DataByte	; left shift data ( MSB first)
		decfsz	BitCount
		goto	RcvNextBit
;
; Generate ACK bit if not last byte to be read,
; if last byte Gennerate NACK ; do not send ACK on last byte, main routine will send a STOP bit
;
		bsf	_rp0
		bcf	_SCL
		bcf	_SDA		; ACK by pulling SDA low
		btfsc	_Last_Byte_Rcv
		bsf	_SDA		; if last byte, send NACK by setting SDA high
		call	Delay47uSec	; guareentee min LOW TIME tLOW & Setup time
		bsf	_SCL
		call	Delay40uSec	; guareentee min HIGH TIME tHIGH
RcvEnd:
		bcf	_SCL		; reset clock

		bcf	_Rcv_Progress	; reset TXMT bit in Bus Status
		bsf	_Rcv_Success	; transmission successful
		bcf	_ACK_Error	; ACK OK
		bcf	_Bus_Busy	; Bus Is Free

		return

;*********************************************************************************************************
;				Fatal Error On I2C Bus
;
;  Slave pulling clock for too long or if SCL Line is stuck low.
;  This occurs if during Transmission, SCL is stuck low for period longer than appox 1mS
;   and RTCC times out ( appox 4096 cycles : 256 * 16 -- prescaler of 16).
;
;*********************************************************************************************************

Bus_Fatal_Error:

	ReleaseBus
;
;  Set the Bus_Status Bits appropriately
;
	bsf	_Abort			; transmission was aborted
	bsf	_Fatal_Error		; FATAL Error occured
	bsf	_Txmt_Progress		; Transmission Is Not in Progress 
	bcf	_Txmt_Success		; Transmission Unsuccesful
;
; diable INT & RTCC Interrupts
;
	bcf	_intf			; unnecessary : for debug only
        bcf	_rtif			; unnecessary : for debug only
	bcf	_inte			; disable INT interrupts, until next TXMT try
	bcf	_rtie			; disable RTCC interrupts, until next TXMT try
;
	return
;
;*********************************************************************************************************
;				Device Reset Routine
;	Force WDT timeout by not clearing WDT
;	Can be used for software reset when reset() command is used
;*********************************************************************************************************

SoftReset:

	goto	$
;
;*********************************************************************************************************

;*********************************************************************************************************
;			General Purpose Delay Routines
;
;  Delay4uS	is wait loop for 4.0 uSec
;  Delay47uS	is wait loop for 4.7 uSec
;
;*********************************************************************************************************
;

Delay50uSec:
	movlw	((_50uS_Delay-5)/3 + 1)
DlyK
	movwf	DelayCount
	decfsz	DelayCount
	goto	$-1
	return		
;
Delay47uSec:
	movlw	((_47uS_Delay-8)/3 + 1)
        goto	DlyK
;
Delay40uSec:
	movlw	((_40uS_Delay-8)/3 + 1)
	goto	DlyK
;
;*********************************************************************************************************
;				Long Goto Routine
;
; Branch to Address as pointed to by 16 bit pointer stored in RetAddrPtrHi
;
;*********************************************************************************************************

RestoreRet:
	movf	RetAddrPtrHi,w
	movwf	_pclath			; load high byte of jump address
	movf	RetAddrPtrLo,w		
	movwf	_pcl			; load low byte of jump address & branch

;*********************************************************************************************************


