	Title		"I2C Slave Mode Implemetation"
        SubTitle        "Rev 0.2	:  04 April 1997"



;***********************************************************************************************
;
;			I2C Slave Mode Using Software Polling
;
;  Start Bit Is detected by connecting SDA to RB0/INT Pin
;  Other bits, including STOP & Repeated Start Conditions are Software Polled 
;
;    The software is implemented using PIC16C84 & thus can be ported to all
;	Enhanced core PIC16CXX products
;
;	RB1 is SCL		(Any I/O Pin May Be used instead)
;	RB0/INT is SDA		(Must use this pin for START bit detect when in idle mode)
;
;   Since Slave Mode is always Application Dependent, as a Test Program, PIC16C84 is used to
;    emulate partial functionality of Microchip's 24Cxx Serial EEPROMs
;
;
;       Program:          I2C_TEST.ASM 
;       Revision Date:    Rev 0.1	:  01 Mar 1993
;                         4-04-97      Compatibility with MPASMWIN 1.40
;
;***********************************************************************************************
;
#define	_MY_ADDRESS	0xD6	; This slave's address
;
#define	AtoD_Slave	1
#define	EE_Slave	0		

;

		LIST    p = 16C71
                ERRORLEVEL  - 302

		Radix	DEC
		EXPAND

_ClkIn		equ	16000000    	
_ClkOut		equ	(_ClkIn >> 2)
;
;
ControlByte     EQU     0x20
#define    _STOP    ControlByte, 0
#define    _START   ControlByte, 1
#define    _RW      ControlByte, 2
#define    _ACK     ControlByte, 3
SlvStatus       EQU     0x21
;
;

		include		"p16c71.inc"

_eedata	set	0x08
_eeadr	set	0x09

_eecon1	set	0x08
_eecon2	set	0x09		; bank 1

#define		_rd	_eecon1,0
#define		_wr	_eecon1,1
#define		_wren	_eecon1,2
#define		_wrerr	_eecon1,3
#define		_eeif	_eecon1,4

LSB	equ	0
MSB	equ	7

		include		"i2c.h"


		CBLOCK	_End_I2C_Ram
			SaveStatus		; copy Of STATUS Reg 
                        SaveWReg		; copy of WREG
			byteCount
			HoldData						
		ENDC

		CBLOCK	0x20
                	DataBegin	; Data to be read or written is stored here
                ENDC

;***********************************************************************************************

		ORG	0x00

		goto	Start
;
		ORG	0x04
;*********************************************************************************************************
;				Interrupt Service Routine
;
;   For I2C Slave routines, only Rb0/INT interrupt is used for START Bit Detect From
;	Idle Mode
;
;*********************************************************************************************************

Interrupt:
;
;  At first check if START Bit Detect (currently only INT pin interrupt is enabled, if other
;	Interrupts are enabled, then check for other interrupts)
;
	btfss	INTCON,INTF
        retfie				; other interrupt, simply return & enable GIE
;
; Save Status
;
	movwf	SaveWReg
	swapf	STATUS,W		; affects no STATUS bits : Only way OUT to save STATUS Reg ?????
	movwf	SaveStatus
;
	bcf	STATUS,RP0
	btfss	_SCL			; Most likely a START Bit, Hold Time of START Bit must be valid from an INT to here
	goto	RestoreStatus		; Not a Valid START Bit
	btfsc	_SDA			; If a valid Falling Edge on SDA when SCL is high, SDA must now be Low
	goto	RestoreStatus		; Not a Valid START Bit
        goto	StartBitDetect		; A Valid Start Bit Is Detected, process & then Branch to "RestoreStatus"
;
; Restore Status
;
RestoreStatus:

	swapf	SaveStatus,W
	movwf	STATUS			; restore STATUS Reg
	swapf	SaveWReg, F		; save WREG
	swapf	SaveWReg,W		; restore WREG
;
	bcf	INTCON,INTF
	retfie
;

;*********************************************************************************************************
;
;*********************************************************************************************************

Start:

	call	Init_I2C_Slave		; Initialize I2C Bus for Slave Mode, wait for START Bit detect
;
	bsf	INTCON,GIE			; Enable Interrupts
wait:
	clrwdt				; User can write code here, will be interrupted by Falling Edge on INT
	goto	wait			; Loop until Start Bit Detect 
	

;
;*********************************************************************************************************
;

Init_I2C_Slave:

                clrf	ControlByte
		bsf	STATUS,RP0
		movlw	_OPTION_INIT
		movwf	OPTION_REG		; initially set INT for Falling Edge INT
		bsf	_SCL
		bsf	_SDA		; set TRIS of SCL & SDA to inputs, pulled up by external resistors

		bcf	STATUS,RP0
                movf	PORTB,W
		andlw	0xFC		; do not use BSF & BCF on Port Pins
		movwf	PORTB		; set SDA & SCL to zero. From Now on, simply play with tris

		clrf	SubAddr		; Set Sub Address to Zero
;
; Initialize A/D Setup or EEPROM Control Regs
;

;#if AtoD_Slave
;		movlw	0x01          	; Select Channel 0, and turn on A/D Module with Fosc/2 Sampling Rate
;		movwf	_adcon0		; already in Page 0, STATUS,RP0 = 0
;		bsf	STATUS,RP0
;                clrf	_adcon1		; 4 analog channels with internal Vref
;#else
		clrf	_eeadr		; already in Page 0, init EEPROM Addr = 0
		bsf	STATUS,RP0
                clrf	_eecon1		; clear err flag
;#endif
;
		clrf	INTCON
                bsf	INTCON,INTE		; Enable Falling Edge Interrupt On SDA (connected on RB0/INT pin)
		return

;
;*********************************************************************************************************
;		       In-Line Code For I2C-Slave
;  Returns to detect next Start Bit after a STOP Bit Is Detected
;  This implementation is very in-efficient if the Master is constantly sending a Repeated START Condition
;	
;*********************************************************************************************************

StartBitDetect:
		call	RcvByte		; 1 st byte received in DataByte
                clrwdt
		btfsc	_STOP
		goto	i2c_start_wait	; STOP bit Detected, wait for another START
		btfsc	_START
		goto	StartBitDetect	; a Repeated START condition
;
; The received byte in DataByte contains R/W & 7 Addressthe address
; If address match send ACK bit else NACK
;
		bcf	_RW
		btfsc	DataByte,LSB
		bsf	_RW			; LSB of the 1st byte received contains R/W info
		bcf	DataByte,LSB

		movf	DataByte,W
		xorlw	_MY_ADDRESS		; if match, then Z bit is set
		btfss	STATUS,Z
		goto	SendNACK		; No Address Match, NACK
;
; Address Match Occured, send ACK
		
		bsf	STATUS,C			; SendAck routine sends ACK if Carry == 1
		call	SendAck
		btfsc	_RW
		goto	SendReqData		; read operation, so send current Data
		goto	RcvReqData		; receive 2 bytes of data, sub-addr & data byte
;
;*******************************************************************************************************
;
SendNACK:
		bcf	STATUS,C			; SendAck routine sends NACK if Carry == 0
                call	SendAck			; address not us, so wait for another start bit		
;
i2c_start_wait:
		clrwdt
		bsf	STATUS,RP0
		bcf	INTCON,INTF
		bsf	_SCL			; release CLK line, may be held in low by us
		bsf	_SDA			; release SDA
		goto RestoreStatus
;
;*******************************************************************************************************
;				Receive A Byte Of Data
;*******************************************************************************************************

RcvByte:
		clrf	SlvStatus

		movlw	0x08
		movwf	BitCount
;
		bcf	STATUS,RP0
		btfsc	_SCL		
		goto	$-1		; wait until SCL is low and then Read the Control Byte
;
		bsf	STATUS,RP0
		bsf	_SCL		; release CLK, possibly held low
                bcf	STATUS,RP0
;
RcvNextBit:
		btfss	_SCL
		goto	$-1		; wait until clock is high, for valid data
		btfsc	_SDA
		goto	Rcvd_1
Rcvd_0:
		bcf	STATUS,C
		rlf	DataByte, F	; left shift data ( MSB first)
_WaitClkLo1:
		btfss	_SCL
		goto	next1
		btfss	_SDA		; SDA must still be low when CLK is high, else may be a STOP
		goto	_WaitClkLo1
		bsf	_STOP
                return
;
Rcvd_1:
		bsf	STATUS,C
		rlf	DataByte, F
_WaitClkLo2:
		btfss	_SCL
                goto	next1         	; CLK went low, process next bit
		btfsc	_SDA		; SDA must still be high when CLK is high, else may be a Repeated START
		goto	_WaitClkLo2
		bsf	_START
                return
;
next1:
		decfsz	BitCount, F
		goto	RcvNextBit
;
; A complete Byte Received, HOLD Master's Clock Line Low to force a wait state
;
		bsf	STATUS,RP0
                bcf	_SCL		; force SCL Low for wait state
                return
;*******************************************************************************************************
;  			Write Operation Requested
;
; Read Sub Address and A Data Byte & Acknowledge, if no errors
; Currently Only One Byte Is Programmed At A Time, Buffering scheme is unimplemented
;
;*******************************************************************************************************

RcvReqData
		call	RcvByte		;  Sub-Address Byte Received in DataByte, store in Sub-Addr
		movf	DataByte,W
		movwf	SubAddr
		clrwdt
		btfsc	_STOP
		goto	i2c_start_wait	; STOP bit Detected, wait for another START
		btfsc	_START
		goto	StartBitDetect	; a Repeated START condition
		bsf	STATUS,C			; SendAck routine sends ACK if Carry == 1
		call	SendAck			; Sub-Addr Received, Send an ACK
;
; Receive Data Byte to Write To EEPROM
;
		call	RcvByte		;  Sub-Address Byte Received in DataByte, store in Sub-Addr
		clrwdt
		btfsc	_STOP
		goto	i2c_start_wait	; STOP bit Detected, wait for another START
		btfsc	_START
		goto	StartBitDetect	; a Repeated START condition
		bsf	STATUS,C		; SendAck routine sends ACK if Carry == 1
		call	SendAck		; Sub-Addr Received, Send an ACK
                call	EEpromWrite	; Program EEPROM
		goto	RestoreStatus	; STOP bit Detected, wait for another START 

;*******************************************************************************************************
;				 Read Operation Requested
;
; Send A/D Data Of Required Channel until NACK By Master
; 
;*******************************************************************************************************

SendReqData:

                bcf	STATUS,RP0
		movf	SubAddr,W
		movwf	_eeadr		; Load Sub_Addr Of EEPROM

;
SendNextByte:
		bsf	STATUS,RP0
		bsf	_rd		; read EEPROM
		bcf	STATUS,RP0
		movf	_eedata,W	 
		movwf	DataByte	; DataByte = EEPROM(addr)
		incf	_eeadr, F	; auto-increment sub-address

		call	TxmtByte	; send out EEPROM Data
                clrwdt
		btfsc	_START			; check for abnormal START
		goto	i2c_start_wait
		call	ReadACK
                clrwdt
		btfss	_ACK			; _ACK == 1 if +ve ACK Rcvd              
		goto	i2c_start_wait		; NACK Received, a START or STOP condition may occur
		goto	SendNextByte		; continue to send until NACK
;
;*******************************************************************************************************

TxmtByte:
		movf	DataByte,W
                movwf	DataByteCopy	; make copy of DataByte 
		movlw	0x08
		movwf	BitCount
		clrf	SlvStatus
;
TxmtNextBit:
		bsf	STATUS,RP0
		rlf	DataByteCopy, F	; MSB First

		btfsc	STATUS,C
		goto	Txmt_1
Txmt_0
		bcf	_SDA
		nop
		bsf	_SCL		; release clock line, let master pull it high
		bcf	STATUS,RP0
		btfss	_SCL
		goto	$-1		; wait until clk goes high
		btfsc	_SCL
		goto	$-1		; wait until clk goes low
		bsf	STATUS,RP0
                bcf	_SCL		; clk went low, continue to hold it low 
                goto	Q_TxmtNextBit


Txmt_1
		bsf	_SDA
		nop
		bsf	_SCL		; release clock line, let master pull it high
		bcf	STATUS,RP0
		btfss	_SCL
		goto	$-1		; wait until clk goes high
_IsClkLo_1
		btfss	_SDA
                goto	MayBeErr_Txmt	; must never come here, illegal Repeated Start 
		btfsc	_SCL
		goto	_IsClkLo_1		; wait until clk goes low

		bsf	STATUS,RP0
                bcf	_SCL		; clk went low, continue to hold it low 
;
Q_TxmtNextBit
		decfsz	BitCount, F
		goto	TxmtNextBit
		bsf	_SDA		; release SDA for Master's ACK
                return
;
MayBeErr_Txmt:
		bsf	_START		; illegal Repeated START condition during a byte transfer
                return
;
;*******************************************************************************************************
;				Send ACK/NACK to Master
;
;  Prior to calling this routine, set CARRY bit to 1 for sending +ve ACK & set CARRY = 0, for NACK
;
;*******************************************************************************************************

SendAck:
		bsf	STATUS,RP0
                btfsc	STATUS,C		; Carry bit == 1 for ACK else NACK
                bcf	_SDA		; pull SDA low to send +ve ACK
		bsf	_SCL		; release CLK line, let master clk it
                bcf	STATUS,RP0
		btfss	_SCL
		goto	$-1		; loop until Clk High
		btfsc	_SCL
		goto	$-1		; loop until Clk is Low, ACK bit sent
		bsf  	STATUS,RP0
		bcf	_SCL		; HOLD CLK Line LOW
		bsf	_SDA		; ACK over, release SDA line for Master control		
;
;
		return
;
;*******************************************************************************************************
;               			Read ACK Sent By Master
;
; If +ve ACK then set _ACK bit in SlaveStatus Reg, else 0
;
;*******************************************************************************************************

ReadACK:
		bsf	STATUS,RP0
		bsf	_SCL		; release clock
		bcf	STATUS,RP0
		btfss	_SCL
		goto	$-1		; wait until clock is high (9 the bit:ACK)
		bsf	_ACK		; expecting a +ve ACK
		btfsc	_SDA
		bcf	_ACK		; NACK rcvd, stop transmission
		btfsc	_SCL
		goto	$-1		; wait until Clock is low
		bsf   	STATUS,RP0
                bcf	_SCL		; force Clock low
		return
;*******************************************************************************************************
;  			Write One Byte Of Data To EEPROM Array at Sub-Addr
;
;*******************************************************************************************************
EEpromWrite:
		bcf	STATUS,RP0
		movf	SubAddr,W
		movwf	_eeadr		; load sub-address
		movf	DataByte,W
		movwf	_eedata		; load data to be written into EEDATA
		bsf	STATUS,RP0
		bsf	_wren		; enable write operation
		bcf	_wrerr		; clear any previous error flags
		movlw	0x55
		movwf	_eecon2
		movlw	0xAA
		movwf	_eecon2
		bsf	_wr		; start Write Operation
;
		bsf	_SCL			; release CLK line, may be held in low by us
		bsf	_SDA			; release SDA
;
		clrwdt
		btfsc	_wr		; Poll until WR Over
		goto	$-2
;
		bcf	_wren		; disable write operations
		clrwdt
		bcf	INTCON,INTF
;
		return
;*******************************************************************************************************


    end
