#include "p18f2321.inc"
#include "PIC_config.inc"

#include "32bit_math_18.h"

; firmware version
#define	vermajor	3
#define	verminor	2

SYSCLK		equ	D'8000000'	; in Hertzes
TMR0RATE	equ	D'10000'	; in Hertzes
BAUDRATE	equ	D'19200'	; in bits per second

TXBUFLEN	equ	D'80'		; in bytes
PARAMLEN	equ	0x40		; in bytes

; calculating SPBRG for BRG16=0 and BRGH=1
SPBRG_value = (SYSCLK/BAUDRATE/D'16') - 1

; timer0 reload value for desired rate
tmr0r	equ	D'256' - (SYSCLK/TMR0RATE/D'4') + 1 
t100r	equ	(TMR0RATE/.100)
t10r	equ	.10

; initial temperature and limits 
;tnormal	equ	.2600
mintemp	equ	.1000
maxtemp	equ	.4500


; LCD module wiring definitions
#define		LCD_E_TRIS		TRISC,5
#define		LCD_RS_TRIS		TRISC,4
#define		LCD_DB7_TRIS	TRISB,0
#define		LCD_DB6_TRIS	TRISB,1
#define		LCD_DB5_TRIS	TRISB,2
#define		LCD_DB4_TRIS	TRISB,3

#define		LCD_E			PORTC,5
#define		LCD_RS			PORTC,4
#define		LCD_DB7			PORTB,0
#define		LCD_DB6			PORTB,1
#define		LCD_DB5			PORTB,2
#define		LCD_DB4			PORTB,3

; HD44780 LCD routines
#include "HD_LCD_18.INC"

; Keyboard wiring definitions
#define		KBD_Key1		PORTB,4
#define		KBD_Key2		PORTB,5
#define		KBD_Key3		PORTB,6
#define		KBD_Key4		PORTB,7

; PSON control output
#define		PSON			PORTC,3

; LCD organization (some 1x16ch modules act as if they were 2x8ch)
#define	LCD_CHARS	D'16'

	udata_acs
; bitwide variables
flags	res	1

; interrupt context save area
FSR0L_TEMP	res	1
FSR0H_TEMP	res	1
BSR_TEMP	res	1
W_TEMP		res	1
STATUS_TEMP	res	1

t10cnt	res	1
t100cnt	res	1
cnt1	res	1
cnt2	res	1
lcd_dly	res	1
lcd_ptr	res	1
tmp		res	1
ttmp	res	1
htmp	res	1
tcnt	res	1
tcnt1	res	1
txptri	res	1
txptro	res	1

ctrltype		res	1	; controller type
ctrlchar		res	1	; controller type character
heat_lo			res	1	; actual heating power
heat_hi			res	1	; actual heating power
txtptr_lo		res	1	; text pointer low
txtptr_hi		res	1	; text pointer high
target_lo		res	1	; needed temperature low
target_hi		res	1	; needed temperature high
temp_lo			res	1	; actual temperature low
temp_hi			res	1	; actual temperature high
kbd				res	1	; keyboard data
okbd			res	1	; old keyboard data
mode			res	1	; operating mode
dmodecnt		res	1	; display mode counter
lcd_break		res	1	; to solve 2x8 displays

; time counter
tmcnt0			res	1
tmcnt1			res	1
tmcnt2			res	1
tmcnt3			res	1

ref_lo			res	1	; A/D external reference (2.5V) value low
ref_hi			res	1	; A/D external reference (2.5V) value high
res_lo			res	1	; A/D result low
res_hi			res	1	; A/D result high
resf_lo			res	1	; filtered A/D result low
resf_hi			res	1	; filtered A/D result high

; xv0,xv1 are 32-bit regs
xv00			res	1
xv01			res	1
xv02			res	1
xv03			res	1
xv10			res	1
xv11			res	1
xv12			res	1
xv13			res	1
; yv1, yv2 are 32-bit regs
yv10			res	1
yv11			res	1
yv12			res	1
yv13			res	1
yv20			res	1
yv21			res	1
yv22			res	1
yv23			res	1
; sss is a 32-bit temporary reg
sss0			res	1
sss1			res	1
sss2			res	1
sss3			res	1
; p type controller data register
ppp0			res	1
ppp1			res	1
ppp2			res	1
ppp3			res	1
; sum register for i type controller
iii0			res	1
iii1			res	1
iii2			res	1
iii3			res	1
; register for r type controller
rrr0			res	1
rrr1			res	1
rrr2			res	1
rrr3			res	1

; variables used in setup
par_min			res	1		; minimum value
par_max			res	1		; maximum value
par_disp_u		res	1		; display routine's address upper
par_disp_h		res	1		; display routine's address high
par_disp_l		res	1		; display routine's address low


; banked data: puffers
	udata
lcd_buf	res	LCD_CHARS
par_buf	res	PARAMLEN
tx_buf	res	TXBUFLEN



; bit variable definitions
#define	lcd_phase	flags,0
#define	tick		flags,1
#define	dmode		flags,2
#define	getpar		flags,3


; macros
unpack32	MACRO	Var,Address ;Var = 32 bit literal to be unpacked
	movlw	Var & H'FF'
	movwf	Address+0
	movlw	Var >>D'08' & H'FF'
	movwf	Address+1
	movlw	Var >>D'16' & H'FF'
	movwf	Address+2
	movlw	Var >>D'24' & H'FF'
	movwf	Address+3
	ENDM


;MOVLW DATA_EE_ADDR ;
;MOVWF EEADR ; Data Memory Address to read
;BCF EECON1, EEPGD ; Point to DATA memory
;BCF EECON1, CFGS ; Access EEPROM
;BSF EECON1, RD ; EEPROM Read
;MOVF EEDATA, W ; W = EEDATA
get16eeprom	MACRO	Dest	; get a 16-bit value to Dest from EEPROM
	movlw	LOW Dest
	movwf	FSR0L
	movlw	HIGH Dest
	movwf	FSR0H

	bsf		EECON1,RD
	movf	EEDATA,w
	movwf	POSTINC0

	incf	EEADR,f
	bsf		EECON1,RD
	movf	EEDATA,w
	movwf	POSTINC0

	incf	EEADR,f
	clrf	WREG
;	btfsc	INDF,7		; extending sign to 32-bit
;	movlw	0xff
	movwf	POSTINC0
	movwf	POSTINC0
	ENDM

data16	MACRO	Data
	de	low(Data),High(Data)
	ENDM




;******************************************************
;************* Beginning of the program ***************
;******************************************************

resvec	code	0x0
;resvec	code	0x0400
	goto	main_start

intvec	code	0x8
;intvec	code	0x0408
	goto	int_handle


	code

int_handle
	MOVWF	W_TEMP ; W_TEMP is in virtual bank
	MOVFF	STATUS, STATUS_TEMP ; STATUS_TEMP located anywhere
	MOVFF	BSR, BSR_TEMP ; BSR_TMEP located anywhere
	movff	FSR0H,FSR0H_TEMP
	movff	FSR0L,FSR0L_TEMP

	btfss	INTCON,T0IF
	goto	no_int_tmr0

; timer0 interrupt
	bcf		INTCON,T0IF

	movlw	tmr0r+2
	addwf	TMR0L,f

	movf	lcd_dly,f
	bz		do_lcd
	decf	lcd_dly,f
	goto	tmr0_end

do_lcd
	movlw	low(lcd_buf)
	addlw	LCD_CHARS
	xorwf	lcd_ptr,w
	bnz		no_lcd_end

	movlw	LCD_HOME
	call	LCD_SEND_CMD_NODLY
	movlw	D'30'
	movwf	lcd_dly
	movlw	low(lcd_buf)
	movwf	lcd_ptr
	bcf		lcd_phase
	goto	tmr0_end

no_lcd_end
	movlw	low(lcd_buf)
	addwf	lcd_break,w
	xorwf	lcd_ptr,w
	bnz		do_lcd_chrout

	btfsc	lcd_phase
	goto	do_lcd_chrout

	bsf		lcd_phase
	movlw	LCD_DDRAM+0x40
	call	LCD_SEND_CMD_NODLY
	goto	tmr0_end

do_lcd_chrout
	movf	lcd_ptr,w
	movwf	FSR0L
	movlw	high(lcd_buf)
	movwf	FSR0H
	movf	INDF0,w
	call	LCD_SEND_CHAR_NODLY
	incf	lcd_ptr,f

tmr0_end
	decfsz	t100cnt
	goto	no_int_tmr0
	movlw	t100r
	movwf	t100cnt

	decfsz	t10cnt
	goto	no_int_tmr0
	movlw	t10r
	movwf	t10cnt
	bsf		tick

no_int_tmr0

	btfss	PIR1,TXIF
	goto	no_int_tx

; TX interrupt
	movf	txptro,w
	xorwf	txptri,w
	bnz		tx_ok	; -> buffer is not empty
	; no more chars to xmit
	bcf		PIE1,TXIE
	goto	tx_int_end

tx_ok
	movf	txptro,w
	addlw	LOW tx_buf
	movwf	FSR0L

	clrf	FSR0H
	movlw	HIGH tx_buf
	addwfc	FSR0H,f

	movf	INDF0,w
	movwf	TXREG

	incf	txptro,w
	xorlw	TXBUFLEN
	btfss	STATUS,Z
	xorlw	TXBUFLEN
	movwf	txptro	

tx_int_end

no_int_tx

	movff	FSR0L_TEMP,FSR0L
	movff	FSR0H_TEMP,FSR0H
	MOVFF	BSR_TEMP, BSR ; Restore BSR
	MOVF	W_TEMP, W ; Restore WREG
	MOVFF	STATUS_TEMP, STATUS ; Restore STATUS
	retfie




; main program entry point
main_start

;
; set up variables - initial values
;
	; tmcmt
	clrf	tmcnt0
	clrf	tmcnt1
	clrf	tmcnt2
	clrf	tmcnt3
	; yv1, yv2 are 24-bit regs
	clrf	xv00
	clrf	xv01
	clrf	xv02
	clrf	xv03
	clrf	xv10
	clrf	xv11
	clrf	xv12
	clrf	xv13
	; yv1, yv2 are 24-bit regs
	clrf	yv10
	clrf	yv11
	clrf	yv12
	clrf	yv13
	clrf	yv20
	clrf	yv21
	clrf	yv22
	clrf	yv23
	; sss is a 32-bit reg
	clrf	sss0
	clrf	sss1
	clrf	sss2
	clrf	sss3
	; integrator sum
	clrf	iii0
	clrf	iii1
	clrf	iii2
	clrf	iii3
	; proportional reg
	clrf	ppp0
	clrf	ppp1
	clrf	ppp2
	clrf	ppp3

	clrf	okbd

	movlw	.30
	movwf	dmodecnt

	movlw	.1
	movwf	t100cnt
	movwf	t10cnt
	bcf		tick


; EEPROM access
	BCF		EECON1,EEPGD ; Point to DATA memory
	BCF		EECON1,CFGS ; Access EEPROM

	movlw	lcd_buf
	movwf	lcd_ptr
	bcf		lcd_phase
	bcf		tick

	movlw	.1
	movwf	t100cnt
	movwf	t10cnt

; 8MHz internal clock
	movlw	B'01110000'
	movwf	OSCCON

; PORTC as digital output
	CLRF	PORTC
	MOVLW	0x80
	MOVWF	TRISC

	bsf		PORTC,.3	; Yellow LED

; PORTB as digital input/output
	CLRF	PORTB
	MOVLW	07h
	MOVWF	ADCON1
	MOVLW	B'11110000'
	MOVWF	TRISB

; turn on pull-ups on PORTB
	bcf		INTCON2,RBPU

; PWM1 setup

;1. Set the PWM period by writing to the PR2
;register.
	movlw	0xFE
	movwf	PR2,A
;2. Set the PWM duty cycle by writing to the
;CCPRxL register and CCPxCON<5:4> bits.
	movlw	0xFF
	movwf	CCPR1L,A
;3. Make the CCPx pin an output by clearing the
;appropriate TRIS bit.
	bcf		TRISC,CCP1,A
;4. Set the TMR2 prescale value, then enable
;Timer2 by writing to T2CON.
	movlw	B'00000011'
	movwf	T2CON,A
	bsf		T2CON,TMR2ON,A
;5. Configure the CCPx module for PWM operation.
	movlw	B'00001100'
	movwf	CCP1CON,A

; PWM2 setup

;2. Set the PWM duty cycle by writing to the
;CCPRxL register and CCPxCON<5:4> bits.
	movlw	0x80
	movwf	CCPR2L,A
;3. Make the CCPx pin an output by clearing the
;appropriate TRIS bit.
	bcf		TRISC,CCP2_PORTC,A
;5. Configure the CCPx module for PWM operation.
	movlw	B'00001100'
	movwf	CCP2CON,A


; set up A/D-s

	movlw	B'00001011'		; AN0-AN2 are analog inputs
	movwf	ADCON1
	movlw	B'00000001'		; A/D on, CH0 selected
	movwf	ADCON0


; read parameter table from EEPROM
	movlw	low EEPARS_START
	movwf	EEADR
	movlw	EEPARS_END-EEPARS_START
	movwf	tcnt
	lfsr	0,par_buf

ee_rd
	bsf		EECON1,RD
	movff	EEDATA,POSTINC0
	incf	EEADR,f
	decfsz	tcnt
	bra		ee_rd

; set up initial values from parameter table

; active or sleep mode
	clrf	mode
	lfsr	0,par_buf+(EE_instant_on-EEPARS_START)
	btfsc	INDF0,0
	incf	mode,f		; for instant poweron

; starting mode
	lfsr	0,par_buf+(EE_start_mode-EEPARS_START)
	movff	INDF0,ctrltype

; initial temperature
	lfsr	0,par_buf+(EE_init_temp-EEPARS_START)
	movf	INDF0,w
	addlw	D'10'
	movwf	REGA0
	clrf	REGA1
	clrf	REGA2
	clrf	REGA3

;	multiply by 100
	unpack32	D'100',REGB0
	call	multiply

	movff	REGA0,target_lo
	movff	REGA1,target_hi




; LCD parameter modifying by keypress during poweron
	lfsr	0,par_buf+(EE_lcd_mode-EEPARS_START)
	btfss	PORTB,5
	bcf		INDF0,0
	btfss	PORTB,6
	bsf		INDF0,0

; start parameter modifying by keypress during poweron
	btfss	PORTB,4
	clrf	mode


; LCD init

; initializing LCD
	movlw	.200
	call	LCD_DELAY	;	20ms

; 1-line display
	movlw	LCD_CHARS
	movwf	lcd_break
	movlw	0x20

	lfsr	0,par_buf+(EE_lcd_mode-EEPARS_START)
	btfss	INDF0,0
	bra		startlcd

; 2-line display
	rrncf	lcd_break,f
	movlw	0x28
	

startlcd
	CALL	LCD_HARD_INIT

	MOVLW   LCD_CGRAM
	CALL    LCD_SEND_CMD

	movlw	B'11100'
	CALL	LCD_SEND_CHAR
	movlw	B'10111'
	CALL	LCD_SEND_CHAR
	movlw	B'11100'
	CALL	LCD_SEND_CHAR
	movlw	B'01000'
	CALL	LCD_SEND_CHAR
	movlw	B'01000'
	CALL	LCD_SEND_CHAR
	movlw	B'01000'
	CALL	LCD_SEND_CHAR
	movlw	B'00111'
	CALL	LCD_SEND_CHAR
	movlw	B'00000'
	CALL	LCD_SEND_CHAR

	MOVLW   LCD_HOME
	CALL    LCD_SEND_CMD

	call	disp_clr


; set up TMR0
	movlw	tmr0r+2
	movwf	TMR0L
	bcf		T0CON,T0CS

; set up TMR0 interrupt
	bcf		INTCON,T0IF
	bsf		INTCON,T0IE
	bsf		INTCON,GIE


; set up serial port
	clrf	txptro
	clrf	txptri

	bsf		TRISC,7
	bsf		TRISC,6

	movlw	SPBRG_value
	movwf	SPBRG

	movlw	B'00000000'		; normal polarities, BRG16=0
	movwf	BAUDCON

	movlw	B'00100100'		; Transmit enable, BRGH=1
	movwf	TXSTA

	movlw	B'10010000'		; USART enable, receiver enable
	movwf	RCSTA

	bsf		INTCON,PEIE		; prepare TX int.


;
; power-on greeting
;
	movlw	low(txt_hello)
	movwf	txtptr_lo
	movlw	high(txt_hello)
	movwf	txtptr_hi
	call	txtout

	call	delay1s

	movlw	low(txt_RS232_init)
	movwf	txtptr_lo
	movlw	high(txt_RS232_init)
	movwf	txtptr_hi
	call	RS232_txtout

wkbd
	movf	PORTB,w
	andlw	B'11110000'
	xorlw	B'11110000'
	bnz		wkbd

main_loop
	btfss	tick
	goto	main_loop
	bcf		tick

	; 10 per second
	CALL    disp_home

	bcf		dmode		; normal display mode
	movf	dmodecnt,f
	bz		nospecial
	bsf		dmode		; special display mode
	decf	dmodecnt,f

nospecial
	movlw	0
	xorwf	mode,w
	bz		mode_sleep

	goto		mode_normal

; sleep mode
mode_sleep
	bsf		PSON

	clrf	heat_lo
	clrf	heat_hi
	call	SETHEAT

	movlw	low(txt_sleep)
	movwf	txtptr_lo
	movlw	high(txt_sleep)
	movwf	txtptr_hi
	call	txtout
	goto	keyhandle

;normal	mode
mode_normal
	bcf		PSON

	; measure actual temperature of iron
	call	gettemp
	movf	REGA0,w
	movwf	temp_lo
	movf	REGA1,w
	movwf	temp_hi

	swapf	ctrltype,w
	andlw	B'11110000'
	movwf	EEADR

	bsf		EECON1,RD
	movf	EEDATA,w
	incf	EEADR,f
	
	movwf	ctrlchar

; modified PID-type controller

	movf	target_lo,w
	movwf	REGA0
	movf	target_hi,w
	movwf	REGA1
	clrf	REGA2
	clrf	REGA3

	unpack32	D'10',REGB0
	call	multiply
	unpack32	-D'46919',REGB0
	call	add
	movf	REGA0,w
	movwf	rrr0
	movf	REGA1,w
	movwf	rrr1
	movf	REGA2,w
	movwf	rrr2
	movf	REGA3,w
	movwf	rrr3

	movf	target_lo,w
	movwf	REGA0
	movf	target_hi,w
	movwf	REGA1
	clrf	REGA2
	clrf	REGA3
	
	movf	temp_lo,w
	subwf	REGA0,f
	btfss	STATUS,C
	decf	REGA1,f
	movf	temp_hi,w
	subwf	REGA1,f
	btfsc	REGA1,7
	decf	REGA2,f
	movf	REGA2,w
	movwf	REGA3
	; REGA = target - temp

	call	rega_sss
	; sss = target - temp

	call	absa
	; Km - maximum error signal
	get16eeprom	REGB0
	call	subtract
	btfsc	REGA3,7
	goto	lower

	movf	REGB0,w
	movwf	REGA0
	movf	REGB1,w
	movwf	REGA1
	movf	REGB2,w
	movwf	REGA2
	movf	REGB3,w
	movwf	REGA3
	btfss	sss3,7
	call	negatea
	goto	calc_iii

lower
	call	sss_rega

calc_iii
	; Ki
	get16eeprom	REGB0
;	unpack32	D'100',REGB0
	call	multiply

	movf	iii0,w
	movwf	REGB0
	movf	iii1,w
	movwf	REGB1
	movf	iii2,w
	movwf	REGB2
	movf	iii3,w
	movwf	REGB3
	call	add
	call	trunc24

	movf	REGA0,w
	movwf	iii0
	movf	REGA1,w
	movwf	iii1
	movf	REGA2,w
	movwf	iii2
	movf	REGA3,w
	movwf	iii3
	; iii = (sum of Ki*(target - temp))

	; Kp
	get16eeprom	REGB0
;	unpack32	D'160',REGB0
	call	sss_rega
	call	multiply
	call	trunc24

	movf	REGA0,w
	movwf	ppp0
	movf	REGA1,w
	movwf	ppp1
	movf	REGA2,w
	movwf	ppp2
	movf	REGA3,w
	movwf	ppp3
	; ppp = Kp*(target - temp) - offset

	movf	sss0,w
	movwf	REGA0
	movf	sss1,w
	movwf	REGA1
	movf	sss2,w
	movwf	REGA2
	movf	sss3,w
	movwf	REGA3
	call	absa

	get16eeprom	REGB0
;	unpack32	D'200',REGB0
	call	subtract

	btfsc	REGA3,.7
	goto	noneg_PI

	clrf	iii0
	clrf	iii1
	clrf	iii2
	clrf	iii3

noneg_PI

	; iii/256
	movf	iii1,w
	movwf	REGA0
	movf	iii2,w
	movwf	REGA1
	movf	iii3,w
	movwf	REGA2
	clrf	REGA3
	btfsc	REGA2,7
	decf	REGA3,f

	; add P value
	movf	ppp0,w
	movwf	REGB0
	movf	ppp1,w
	movwf	REGB1
	movf	ppp2,w
	movwf	REGB2
	movf	ppp3,w
	movwf	REGB3
	call	add

	bsf		EECON1,RD
	movf	EEDATA,w
	incf	EEADR,f

	iorlw	0x00
	bz		end_PID

	; add R value
	movf	rrr0,w
	movwf	REGB0
	movf	rrr1,w
	movwf	REGB1
	movf	rrr2,w
	movwf	REGB2
	movf	rrr3,w
	movwf	REGB3
	call	add

end_PID
	call	trunc16
	movf	REGA1,w
	; restoring offset
	addlw	0x80
	movwf	heat_hi
	movf	REGA0,w
	movwf	heat_lo
	
goon
	call	SETHEAT

	btfss	dmode
	goto	goon1

	movlw	low(txt_mode)
	movwf	txtptr_lo
	movlw	high(txt_mode)
	movwf	txtptr_hi
	call	txtout

	movlw	.6
	movwf	tcnt

mode_l1
	bsf		EECON1,RD
	movf	EEDATA,w
	incf	EEADR,f

	call	disp_out
	decfsz	tcnt
	goto	mode_l1

	movlw	' '
	call	disp_out
	movlw	' '
	call	disp_out

goon1
	movf	ctrlchar,w
	btfss	dmode
	call	disp_out
	movf	ctrlchar,w
	call	RS232_send

	movf	tmcnt0,w
	movwf	REGA0
	movf	tmcnt1,w
	movwf	REGA1
	movf	tmcnt2,w
	movwf	REGA2
	movf	tmcnt3,w
	movwf	REGA3
	call	bin2dec

	movlw	','
	call	RS232_send
	movlw	LOW DIGIT3
	movwf	FSR0L
	movlw	HIGH DIGIT3
	movwf	FSR0H
	movlw	.8
	call	RS232_sendnum
	
	movf	temp_lo,w
	movwf	REGA0
	movf	temp_hi,w
	movwf	REGA1
	clrf	WREG
	btfsc	REGA1,7
	decf	WREG
	movwf	REGA2
	movwf	REGA3
	unpack32	D'5',REGB0
	call	add
	unpack32	D'10',REGB0
	call	divide

	call	bin2dec
	movlw	.3	; minimum digits to display
	btfss	dmode
	call	writelcd
;	movlw	'C'
	movlw	0x00
	btfss	dmode
	call	disp_out

	; send calculated temperature to RS232
	movlw	','
	call	RS232_send
	movlw	LOW DIGIT8
	movwf	FSR0L
	movlw	HIGH DIGIT8
	movwf	FSR0H
	movlw	.3
	call	RS232_sendnum

	; send ppp reg to RS232
	movf	ppp0,w
	movwf	REGA0
	movf	ppp1,w
	movwf	REGA1
	movf	ppp2,w
	movwf	REGA2
	movf	ppp3,w
	movwf	REGA3
	call	bin2dec

	movlw	','
	call	RS232_send
	movlw	LOW DIGIT3
	movwf	FSR0L
	movlw	HIGH DIGIT3
	movwf	FSR0H
	movlw	.8
	call	RS232_sendnum

	; send iii reg to rs232
	movf	iii0,w
	movwf	REGA0
	movf	iii1,w
	movwf	REGA1
	movf	iii2,w
	movwf	REGA2
	movf	iii3,w
	movwf	REGA3
	call	bin2dec

	movlw	','
	call	RS232_send
	movlw	LOW DIGIT3
	movwf	FSR0L
	movlw	HIGH DIGIT3
	movwf	FSR0H
	movlw	.8
	call	RS232_sendnum

	; send rrr reg to RS232
	movf	rrr0,w
	movwf	REGA0
	movf	rrr1,w
	movwf	REGA1
	movf	rrr2,w
	movwf	REGA2
	movf	rrr3,w
	movwf	REGA3
	call	bin2dec

	movlw	','
	call	RS232_send
	movlw	LOW DIGIT3
	movwf	FSR0L
	movlw	HIGH DIGIT3
	movwf	FSR0H
	movlw	.8
	call	RS232_sendnum

	movlw	' '
	btfss	dmode
	call	disp_out

	; calculating heating power
	movf	heat_lo,w
	andlw	B'11000000'
	movwf	REGA0
	movf	heat_hi,w
	movwf	REGA1
	clrf	REGA2
	clrf	REGA3

	unpack32	D'100',REGB0
	call	multiply

	unpack32	H'FFC0',REGB0
	call	divide
	call	bin2dec
	movlw	.3	; minimum digits to display
	btfss	dmode
	call	writelcd
	movlw	'%'
	btfss	dmode
	call	disp_out

	; send calculated heating power to RS232
	movlw	','
	call	RS232_send
	movlw	LOW DIGIT7
	movwf	FSR0L
	movlw	HIGH DIGIT7
	movwf	FSR0H
	movlw	.4
	call	RS232_sendnum

	movlw	' '
	btfss	dmode
	call	disp_out
	movlw	0x7e
	btfss	dmode
	call	disp_out

	movf	target_lo,w
	movwf	REGA0
	movf	target_hi,w
	movwf	REGA1
	clrf	REGA2
	clrf	REGA3
	unpack32	D'10',REGB0
	call	divide
	call	bin2dec
	movlw	.3	; minimum digits to display
	btfss	dmode
	call	writelcd
;	movlw	'C'
	movlw	0x00
	btfss	dmode
	call	disp_out

	; send needed temperature to RS232
	movlw	','
	call	RS232_send
	movlw	LOW DIGIT8
	movwf	FSR0L
	movlw	HIGH DIGIT8
	movwf	FSR0H
	movlw	.3
	call	RS232_sendnum

	; send CRLF result to RS232
	movlw	0x0d
	call	RS232_send
	movlw	0x0a
	call	RS232_send

	goto	keyhandle

keyhandle
	call	getkey

	movwf	kbd
	iorlw	0
	bz		endkbd
	movf	okbd,f
	bnz		endkbd

	movlw	.0
	xorwf	mode,w
	bnz		nomode0

; keyboard in sleep mode
	movlw	.1
	xorwf	kbd,w
	bnz		nokbd1_sleep

	; key 1 pressed in sleep mode
	movlw	.1
	movwf	mode
	goto	endkbd

nokbd1_sleep
	movlw	.12
	xorwf	kbd,w
	bnz		endkbd

	; key 4 and 8 are pressed in sleep mode
	goto	setup

nomode0
; keyboard in normal mode
	movlw	.1
	xorwf	kbd,w
	bnz		nokbd1
	
	; key '1' pressed, switch operating mode
	movlw	0
	movwf	mode
	goto	endkbd

nokbd1
	movlw	.2
	xorwf	kbd,w
	bnz		nokbd2
	; key '2' pressed, switch controller type
	movlw	.10
	movwf	dmodecnt

	incf	ctrltype,f

	swapf	ctrltype,w
	andlw	B'11110000'

	movwf	EEADR
	bsf		EECON1,RD
	movf	EEDATA,w

	iorlw	0x00
	btfsc	STATUS,Z
	clrf	ctrltype

	clrf	iii0
	clrf	iii1
	clrf	iii2
	clrf	iii3
	goto	endkbd

nokbd2
	movlw	.4
	xorwf	kbd,w
	bnz		nokbd4
	; key '4' pressed, decrement temperature by 10 degrees
	movlw	high(mintemp)
	subwf	target_hi,w
	bnz		tmpdec1
	movlw	low(mintemp)
	subwf	target_lo,w
	bz		endkbd
tmpdec1
	movlw	low(-.100)
	addwf	target_lo,f
	btfsc	STATUS,C
	incf	target_hi,f
	movlw	high(-.100)
	addwf	target_hi,f
	goto	endkbd

nokbd4
	movlw	.8
	xorwf	kbd,w
	bnz		nokbd8
	; key '8' pressed, increment temperature by 10 degrees
	movlw	high(maxtemp)
	subwf	target_hi,w
	bnz		tmpinc1
	movlw	low(maxtemp)
	subwf	target_lo,w
	bz		endkbd

tmpinc1
	movlw	low(.100)
	addwf	target_lo,f
	btfsc	STATUS,C
	incf	target_hi,f
	movlw	high(.100)
	addwf	target_hi,f
	goto	endkbd

nokbd8
	goto	endkbd

endkbd
	movf	kbd,w
	movwf	okbd
	goto	main_loop


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;    setup    ;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; setup mode
setup
	CALL    disp_home
	movlw	low(txt_setup)
	movwf	txtptr_lo
	movlw	high(txt_setup)
	movwf	txtptr_hi
	call	txtout

	call	delay1s

setup_loop0
	movlw	low setup_items
	movwf	TBLPTRL
	movlw	high setup_items
	movwf	TBLPTRH
	movlw	upper setup_items
	movwf	TBLPTRU
	bsf		getpar

setup_loop
	btfss	tick
	goto	setup_loop
	bcf		tick

	; 10 per second
	CALL    disp_home
	btfss	getpar
	goto	nogetpar

	; get parameter description from table
	; print its name to display
	; get its address to FSR2
	; get its minimum value to par_min
	; get its minimum value to par_max
	; get its display routime's address to par_disp_u, par_disp_h, par_disp_l
	bcf		getpar
	movlw	.8
	movwf	cnt1
gp1
	tblrd	*+
	movff	TABLAT,POSTINC1
	decfsz	cnt1,f
	bra		gp1

	lfsr	2,par_buf
	tblrd	*+
	movf	TABLAT,w
	addwf	FSR2L,f
	clrf	WREG
	addwfc	FSR2H,f

	tblrd	*+
	movff	TABLAT,par_min
	tblrd	*+
	movff	TABLAT,par_max

	tblrd	*+
	movff	TABLAT,par_disp_u
	tblrd	*+
	movff	TABLAT,par_disp_h
	tblrd	*+
	movff	TABLAT,par_disp_l

	goto	setup_showpar

	; skip first character positions
nogetpar
	movlw	.8
	movwf	cnt1
ngp1
	movf	POSTINC1,w
	decfsz	cnt1,f
	bra		ngp1

setup_showpar
	; INDF2 points to par to display
	call	setup_call_show

setup_keyhandle
	call	getkey
	movwf	kbd

	movf	okbd,w
	movff	kbd,okbd
	bnz		endkbd_setup

	movf	kbd,w
	bz		endkbd_setup

; keyboard in setup mode
	movlw	.1
	xorwf	kbd,w
	bnz		nokbd1_setup

	; key 1 pressed in setup mode - exit setup

	; write parameter table to EEPROM
	BCF		INTCON, GIE ; Disable Interrupts
	BCF		EECON1, EEPGD ; Point to DATA memory
	BCF		EECON1, CFGS ; Access EEPROM
	BSF		EECON1, WREN ; Enable writes

	movlw	low EEPARS_START
	movwf	EEADR
	movlw	EEPARS_END-EEPARS_START
	movwf	tcnt
	lfsr	0,par_buf

ee_wr
	MOVFF	POSTINC0,EEDATA ; Data Memory Value to write

	MOVLW	0x55 ; Required Sequence
	MOVWF	EECON2 ; Write 55h
	MOVLW	0xAA ;
	MOVWF	EECON2 ; Write 0AAh
	BSF		EECON1, WR ; Set WR bit to begin write

w_ee_wr
	btfsc	EECON1,WR
	bra		w_ee_wr

	incf	EEADR,f
	decfsz	tcnt
	bra		ee_wr

w_nokey
	call	getkey
	iorlw	0
	bnz		w_nokey

	reset

nokbd1_setup
	movlw	.2
	xorwf	kbd,w
	bnz		nokbd2_setup

	; key 2 pressed in setup mode - next item
	tblrd	*
	movf	TABLAT,w
	bz		setup_loop0
	bsf		getpar
	goto	endkbd_setup

nokbd2_setup
	movlw	.4
	xorwf	kbd,w
	bnz		nokbd4_setup

	; key 4 pressed in setup mode - decrement item
	movf	INDF2,w
	xorwf	par_min,w
	btfss	STATUS,Z
	decf	INDF2,f
	goto	endkbd_setup

nokbd4_setup
	movlw	.8
	xorwf	kbd,w
	bnz		nokbd8_setup

	; key 8 pressed in setup mode - increment item
	movf	INDF2,w
	xorwf	par_max,w
	btfss	STATUS,Z
	incf	INDF2,f
	goto	endkbd_setup

nokbd8_setup

endkbd_setup
	goto	setup_loop

	return

setup_call_show
	movff	par_disp_u,PCLATU
	movff	par_disp_h,PCLATH
	movf	par_disp_l,w
	movwf	PCL


setup_disp_int
	movf	INDF2,w
	movwf	REGA0
	clrf	REGA1
	clrf	REGA2
	clrf	REGA3
	call	bin2dec
	movlw	.3	; minimum digits to display
	call	writelcd
	return

setup_disp_lcd
	movlw	' '
	call	disp_out
	movlw	' '
	call	disp_out
	movlw	' '
	call	disp_out
	movlw	' '
	call	disp_out

	movf	INDF2,f
	bz		lcd_1x16

	movlw	'2'
	call	disp_out
	movlw	'*'
	call	disp_out
	movlw	'8'
	call	disp_out
	movlw	' '
	call	disp_out
	return

lcd_1x16
	movlw	'1'
	call	disp_out
	movlw	'*'
	call	disp_out
	movlw	'1'
	call	disp_out
	movlw	'6'
	call	disp_out
	return

; display poweon mode
setup_disp_pwon
	movlw	' '
	call	disp_out
	movlw	' '
	call	disp_out

	movf	INDF2,f
	bz		pwon_sleep

	movlw	'A'
	call	disp_out
	movlw	'c'
	call	disp_out
	movlw	't'
	call	disp_out
	movlw	'i'
	call	disp_out
	movlw	'v'
	call	disp_out
	movlw	'e'
	call	disp_out
	return

pwon_sleep
	movlw	'S'
	call	disp_out
	movlw	'l'
	call	disp_out
	movlw	'e'
	call	disp_out
	movlw	'e'
	call	disp_out
	movlw	'p'
	call	disp_out
	movlw	' '
	call	disp_out
	return


; display ambient temperature in setup menu
setup_disp_atemp
	movlw	' '
	call	disp_out
	movlw	' '
	call	disp_out
	movlw	' '
	call	disp_out
	movlw	' '
	call	disp_out

	call	getamb
	unpack32	D'10',REGB0
;	div
	call	divide

	call	bin2dec
	movlw	.3	; minimum digits to display
	call	writelcd

	movlw	0x00
	call	disp_out

	return

; display initial temperature in setup menu
setup_disp_itemp
	movlw	' '
	call	disp_out
	movlw	' '
	call	disp_out
	movlw	' '
	call	disp_out
	movlw	' '
	call	disp_out

	rlncf	INDF2,w
	rlncf	WREG,w
	addwf	INDF2,w
	addlw	D'50'
	movwf	REGA0
	clrf	REGA1
	clrf	REGA2
	clrf	REGA3
	rlcf	REGA0,f
	RLCF	REGA1,f

	call	bin2dec
	movlw	.3	; minimum digits to display
	call	writelcd

	movlw	0x00
	call	disp_out

	return


; display ambient temperature in setup menu
setup_disp_tccoeff
	movlw	' '
	call	disp_out
	movlw	' '
	call	disp_out

	movf	INDF2,w
	movwf	REGA0
	clrf	REGA1
	clrf	REGA2
	clrf	REGA3
	call	bin2dec
	movlw	.2	; minimum digits to display
	call	writelcd

	movlw	'u'
	call	disp_out
	movlw	'V'
	call	disp_out
	movlw	'/'
	call	disp_out
	movlw	0x00
	call	disp_out

	return


; display mode in setup menu
setup_disp_mode
	movlw	' '
	call	disp_out
	movlw	' '
	call	disp_out

	swapf	INDF2,w
	andlw	B'11110000'
	addlw	D'10'

	movwf	EEADR
	movlw	.6
	movwf	tcnt

setup_mode_l1
	bsf		EECON1,RD
	movf	EEDATA,w
	incf	EEADR,f

	call	disp_out
	decfsz	tcnt
	goto	setup_mode_l1

	return



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;; subroutines ;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

delay1s
	; delay 50x20ms = 1 sec
	movlw	.50
	movwf	tcnt
dly2
	movlw	.200
	call	LCD_DELAY	;	20ms
	decfsz	tcnt,f
	goto	dly2
	return


; get key from keyboard
getkey
	clrf	WREG
	btfss	KBD_Key1
	bsf		WREG,0
	btfss	KBD_Key2
	bsf		WREG,1
	btfss	KBD_Key3
	bsf		WREG,2
	btfss	KBD_Key4
	bsf		WREG,3
	return

; truncate REGA to signed 16 bit
trunc16
	call	trunc24

	rlcf	REGA1,w
	rlcf	REGA2,w
	bnc		noneg_A16

	xorlw	0xff
	bz		end_A16
	clrf	REGA0
	movlw	0x80
	movwf	REGA1
	movlw	0xFF
	movwf	REGA2
	goto	end_A16

noneg_A16
	iorlw	0x00
	bz		end_A16
	movlw	0xFF
	movwf	REGA0
	movlw	0x7F
	movwf	REGA1
	clrf	REGA2

end_A16
	return
	

; truncate REGA to signed 24 bit
trunc24
	rlcf	REGA2,w
	rlcf	REGA3,w
	bnc		noneg_A24
	xorlw	0xff
	bz		end_A24
	clrf	REGA0
	clrf	REGA1
	movlw	0x80
	movwf	REGA2
	movlw	0xFF
	movwf	REGA3
	goto	end_A24

noneg_A24
	iorlw	0x00
	bz		end_A24
	movlw	0xFF
	movwf	REGA0
	movwf	REGA1
	movlw	0x7F
	movwf	REGA2
	clrf	REGA3

end_A24
	return

SETHEAT
	btfsc	heat_lo,7
	bcf		CCP1CON,DC1B1
	btfss	heat_lo,7
	bsf		CCP1CON,DC1B1

	btfsc	heat_lo,6
	bcf		CCP1CON,DC1B0
	btfss	heat_lo,6
	bsf		CCP1CON,DC1B0

	movf	heat_hi,w
	xorlw	0xff
	movwf	CCPR1L
	return

; get ambient temperature
getamb
	movlw	B'10101101'		; Right, 8Tad, 16Tosc
	movwf	ADCON2

	movlw	B'00001101'		; A/D on, CH3 selected
	movwf	ADCON0			; measuring 2.5V reference
	bsf		ADCON0,GO
wad3
	btfsc	ADCON0,DONE
	goto	wad3

	bcf		STATUS,C
	rlcf	ADRESL,w
	movwf	ref_lo
	rlcf	ADRESH,w
	movwf	ref_hi

	movlw	B'00000001'		; A/D on, CH0 selected
	movwf	ADCON0
	bsf		ADCON0,GO
wad0
	btfsc	ADCON0,DONE
	goto	wad0

	movf	ADRESL,w
	movwf	REGB0
	movf	ADRESH,w
	movwf	REGB1
	clrf	REGB2
	clrf	REGB3

	unpack32	0x03FF,REGA0
	call	multiply

	movf	ref_lo,w
	movwf	REGB0
	movf	ref_hi,w
	movwf	REGB1
	clrf	REGB2
	clrf	REGB3
	call	divide

	unpack32	D'4073',REGB0	; 0.1uV
	call	multiply
	unpack32	D'2100',REGB0	; uV/deg -> 10* deg. value
	call	divide
	unpack32	D'1270',REGB0
	call	subtract
	call	negatea

	; get offset parameter from setup parameters
	lfsr	0,par_buf
	movlw	EE_temp_cal-EEPARS_START
	addwf	FSR0L,f
	clrf	WREG
	addwfc	FSR0H,f

	; add 10*par to temperature
	rlncf	INDF0,w
	rlncf	WREG,w
	addwf	INDF0,w
	movwf	REGB0
	clrf	REGB1
	clrf	REGB2
	clrf	REGB3
	rlcf	REGB0,f
	rlcf	REGB1,f

	call	add

	return

; gettemp
; measuring temperature of soldering iron
gettemp
	call	getamb

	; store ambient temperature in 1/10th Celsius
	call	rega_sss

	movlw	B'00000101'		; A/D on, CH1 selected
	movwf	ADCON0
	bsf		ADCON0,GO
wad1
	btfsc	ADCON0,DONE
	goto	wad1

	movf	ADRESL,w
	movwf	REGB0
	movf	ADRESH,w
	movwf	REGB1
	clrf	REGB2
	clrf	REGB3


	movlw	B'00001001'		; A/D on, CH2 selected
	movwf	ADCON0
	bsf		ADCON0,GO
wad2
	btfsc	ADCON0,DONE
	goto	wad2

	movf	ADRESL,w
	movwf	REGA0
	movf	ADRESH,w
	movwf	REGA1
	clrf	REGA2
	clrf	REGA3

	call	subtract

	unpack32	0x03FF,REGB0
	call	multiply

	movf	ref_lo,w
	movwf	REGB0
	movf	ref_hi,w
	movwf	REGB1
	clrf	REGB2
	clrf	REGB3
	call	divide

	unpack32	D'48392',REGB0	; 0.001uV
	call	multiply

	; get TC coeff from params
	lfsr	0,par_buf
	movlw	EE_tc_coeff-EEPARS_START
	addwf	FSR0L,f
	clrf	WREG
	addwfc	FSR0H,f
	movff	INDF0,REGB0
	clrf	REGB1
	clrf	REGB2
	clrf	REGB3
	call	divide

	unpack32	D'100',REGB0	; 0.01uV/deg -> 10* deg. value
	call	divide

;	unpack32	D'6800',REGB0	; 0.01uV/deg -> 10* deg. value
;	call	divide
	; 1/10th Celsius differential temperature on thermocouple

	; add ambient temperature
	call	sss_regb
	call	add

	movf	REGA0,w
	movwf	res_lo
	movf	REGA1,w
	movwf	res_hi

;0.1Hz filter
;#define NZEROS 2
;#define NPOLES 2
;#define GAIN   6.701159077e+02
;
;static float xv[NZEROS+1], yv[NPOLES+1];
;
;static void filterloop()
;  { for (;;)
;      { xv[0] = xv[1]; xv[1] = xv[2]; 
;        xv[2] = next input value / GAIN;
;        yv[0] = yv[1]; yv[1] = yv[2]; 
;        yv[2] =   (xv[0] + xv[2]) + 2 * xv[1]
;                     + ( -0.8706834551 * yv[0]) + (  1.8647143385 * yv[1]);
;        next output value = yv[2];
;      }
;  }

;0.5Hz Bessel filter
;#define NZEROS 2
;#define NPOLES 2
;#define GAIN   3.423411789e+01
;
;static float xv[NZEROS+1], yv[NPOLES+1];
;
;static void filterloop()
;  { for (;;)
;      { xv[0] = xv[1]; xv[1] = xv[2]; 
;        xv[2] = next input value / GAIN;
;        yv[0] = yv[1]; yv[1] = yv[2]; 
;        yv[2] =   (xv[0] + xv[2]) + 2 * xv[1]
;                     + ( -0.4977439848 * yv[0]) + (  1.3809014824 * yv[1]);
;        next output value = yv[2];
;      }
;  }


;digital filtering input value
;xs = xv0+xv1+{next input}			; G x value, 16 bit
;
;	{next input} -> rega
	movf	res_lo,w
	movwf	REGA0
	movf	res_hi,w
	movwf	REGA1
	clrf	WREG
	btfsc	REGA1,7
	decf	WREG
	movwf	REGA2
	movwf	REGA3
;	xv0 -> regb
	movf	xv00,w
	movwf	REGB0
	movf	xv01,w
	movwf	REGB1
	movf	xv02,w
	movwf	REGB2
	movf	xv03,w
	movwf	REGB3
;	add
	call	add
;	xv1 -> regb
	movf	xv10,w
	movwf	REGB0
	movf	xv11,w
	movwf	REGB1
	movf	xv12,w
	movwf	REGB2
	movf	xv13,w
	movwf	REGB3
;	add
	call	add
;
;xv0 = xv1/2					; G x value, 16 bit
;
;	shr xv1 -> xv0
	rlcf	xv13,w
	rrcf	xv13,w
	movwf	xv03
	rrcf	xv12,w
	movwf	xv02
	rrcf	xv11,w
	movwf	xv01
	rrcf	xv10,w
	movwf	xv00
;
;xv1 = {next input}*2				; G x value, 16 bit
;
;	shl {next input} -> xv1
	bcf		STATUS,C
	rlcf	res_lo,w
	movwf	xv10
	rlcf	res_hi,w
	movwf	xv11
	clrf	WREG
	btfsc	STATUS,C
	decf	WREG
	movwf	xv12
	movwf	xv13
;
;ys = int(20050*xs -641*yv1 + 1561*yv2)		; 1 000 000 x value, 32 bit
;	/1000					; 1 000 x value, 24 bit
;
;	20050 -> regb
	; 5 tenths reciprocal
	;unpack32	D'150',REGB0
	unpack32	D'2921',REGB0
;	mul
	call	multiply
	unpack32	D'1',REGB0
;	mul
	call	divide

;	rega -> sss
	call	rega_sss
;
;	-641 -> rega
	; 4 tenths
	;unpack32	-D'8707',REGA0
	unpack32	-D'4977',REGA0
;	yv1 -> regb
	movf	yv10,w
	movwf	REGB0
	movf	yv11,w
	movwf	REGB1
	movf	yv12,w
	movwf	REGB2
	movf	yv13,w
	movwf	REGB3
;	mul
	call	multiply
;
;	sss -> regb
	call	sss_regb
;	add
	call	add
;	rega -> sss
	call	rega_sss
;
;	1561 -> rega
	; 4 tenths
	;unpack32	D'18647',REGA0
	unpack32	D'13809',REGA0
;	yv2 -> regb
	movf	yv20,w
	movwf	REGB0
	movf	yv21,w
	movwf	REGB1
	movf	yv22,w
	movwf	REGB2
	movf	yv23,w
	movwf	REGB3
;	mul
	call	multiply
;
;	sss -> regb
	call	sss_regb
;	add
	call	add
;
;	1000 -> regb
	unpack32	D'10000',REGB0
;	div
	call	divide
;
;yv1 = yv2					; 1 000 x value, 24 bit
;yv2 = ys					; 1 000 x value, 24 bit
;
;	yv2 -> yv1
	movf	yv20,w
	movwf	yv10
	movf	yv21,w
	movwf	yv11
	movf	yv22,w
	movwf	yv12
	movf	yv23,w
	movwf	yv13
;	rega -> yv2
	movf	REGA0,w
	movwf	yv20
	movf	REGA1,w
	movwf	yv21
	movf	REGA2,w
	movwf	yv22
	movf	REGA3,w
	movwf	yv23
;
;{next output} = yv[2]/1000
;

	unpack32	D'10',REGB0
;	div
	call	divide

;	rega -> {next output}
	movf	REGA0,w
	movwf	resf_lo
	movf	REGA1,w
	movwf	resf_hi
;;
; xv0,xv1 are 16-bit regs
; sss is a 32-bit reg
; yv1, yv2 are 24-bit regs
;

; filtered A/D result is in (resf_hi,resf_lo)
	return



; utility routines

; copy REGA to sss
rega_sss
	movf	REGA0,w
	movwf	sss0
	movf	REGA1,w
	movwf	sss1
	movf	REGA2,w
	movwf	sss2
	movf	REGA3,w
	movwf	sss3
	return

; copy sss to REGA
sss_rega
	movf	sss0,w
	movwf	REGA0
	movf	sss1,w
	movwf	REGA1
	movf	sss2,w
	movwf	REGA2
	movf	sss3,w
	movwf	REGA3
	return

; copy REGB to sss
regb_sss
	movf	REGB0,w
	movwf	sss0
	movf	REGB1,w
	movwf	sss1
	movf	REGB2,w
	movwf	sss2
	movf	REGB3,w
	movwf	sss3
	return

; copy sss to REGB
sss_regb
	movf	sss0,w
	movwf	REGB0
	movf	sss1,w
	movwf	REGB1
	movf	sss2,w
	movwf	REGB2
	movf	sss3,w
	movwf	REGB3
	return


; display routines
writelcd
	movwf	tcnt1
	movlw	LOW DIGIT1
	movwf	FSR0L
	movlw	HIGH DIGIT1
	movwf	FSR0H

	rrcf	DSIGN,w
	bnc		wlcd1
	movlw	'-'
	call	disp_out
	decf	tcnt1,f

wlcd1
	movlw	.10
	movwf	tcnt
	movf	tcnt1,w
	sublw	.10
	movwf	tcnt1
	bz		wr_loop
	
wr_l1
	movf	INDF0,w
	bnz		wr_loop
	movf	POSTINC0,w
	decf	tcnt,f
	decfsz	tcnt1,f
	goto	wr_l1

wr_loop
	movf	POSTINC0,w
	addlw	'0'
	call	disp_out
	decfsz	tcnt,f
	goto	wr_loop

	return

; clear display
disp_clr
	movlw	high(lcd_buf)
	movwf	FSR1H
	movlw	low(lcd_buf)
	movwf	FSR1L

	movlw	LCD_CHARS
	movwf	cnt1
	movlw	' '
lcd_clr
	movwf	POSTINC1
	decfsz	cnt1,f
	goto	lcd_clr

disp_home
	movlw	high(lcd_buf)
	movwf	FSR1H
	movlw	low(lcd_buf)
	movwf	FSR1L

	return


disp_out
	movwf	tmp

	movlw	low(lcd_buf)
	addlw	LCD_CHARS
	subwf	FSR1L,w

	btfsc	STATUS,Z
	return

	movf	tmp,w
	movwf	POSTINC1

	return

disp_hexout2
	movwf	htmp
	swapf	WREG,w
	call	disp_hexout1
	movf	htmp,w
disp_hexout1
	clrf	TBLPTRH
	clrf	TBLPTRU
	andlw	0x0f
	addlw	LOW hextable
	movwf	TBLPTRL
	movlw	HIGH hextable
	addwfc	TBLPTRH,f
	movlw	UPPER hextable
	addwfc	TBLPTRU,f
	tblrd	*
	movf	TABLAT,w
	goto	disp_out

hextable
	data   "0123456789ABCDEF"


; RS232_send - char output to RS232
RS232_send
; RS232 transmit routine
rs232_tx
	movwf	tmp

	incf	txptri,w
	xorlw	TXBUFLEN
	btfss	STATUS,Z
	xorlw	TXBUFLEN

	xorwf	txptro,w
	bnz		tx_start	; buffer is not full -> begin transmit
	; buffer full
	bsf		STATUS,C
	return
	
tx_start
	movf	txptri,w
	addlw	LOW tx_buf
	movwf	FSR2L

	clrf	FSR2H
	movlw	HIGH tx_buf
	addwfc	FSR2H,f

	movf	tmp,w
	movwf	INDF2

	incf	txptri,w
	xorlw	TXBUFLEN
	btfss	STATUS,Z
	xorlw	TXBUFLEN
	movwf	txptri

	bsf		PIE1,TXIE	; enable tx int
	bcf		STATUS,C
	return


; RS232_txtout - text output to RS232
RS232_txtout
	movf	txtptr_lo,w
	movwf	TBLPTRL
	movf	txtptr_hi,w
	movwf	TBLPTRH
	clrf	TBLPTRU

txtoutl
	tblrd	*+
	movf	TABLAT,w

	iorlw	0
	btfsc	STATUS,Z
	return

	call	RS232_send
	goto	txtoutl

; RS232_sendnum - numeric output to RS232
RS232_sendnum
	movwf	tcnt
	rlcf	DSIGN,w
	andlw	.2
	addlw	'+'
	call	RS232_send
rs232_sn1
	movf	POSTINC0,w
	addlw	'0'
	call	RS232_send
	decfsz	tcnt,f
	goto	rs232_sn1
	return

; txtout - text output to lcd
txtout
	movf	txtptr_lo,w
	movwf	TBLPTRL
	movf	txtptr_hi,w
	movwf	TBLPTRH
	clrf	TBLPTRU

tol
	tblrd	*+
	movf	TABLAT,w

	iorlw	0
	btfsc	STATUS,Z
	return

	call	disp_out
	goto	tol

;*** End of program




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;   ROM data  ;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


packed	code_pack

;*** Text area

txt_RS232_init
	db	0x0d,0x0a,"PIC_SOLD v",'0'+vermajor,".",'0'+verminor,0x0d,0x0a,0

txt_hello
	db	"PIC_SOLD v",'0'+vermajor,".",'0'+verminor,0

txt_sleep
	db	"--- SLEEPING ---",0

txt_setup
	db	"-- SETUP MENU --",0

txt_mode
	db	"  Mode: ",0

; setup item descriptors
setup_items
	db	"LCD mode"
	db	EE_lcd_mode-EEPARS_START
	db	D'0',D'1'
	db	upper setup_disp_lcd,high setup_disp_lcd,low setup_disp_lcd
	db	"PwOnMode"
	db	EE_instant_on-EEPARS_START
	db	0x0,0x1
	db	upper setup_disp_pwon,high setup_disp_pwon,low setup_disp_pwon
	db	"InitMode"
	db	EE_start_mode-EEPARS_START
	db	0x0,0x3
	db	upper setup_disp_mode,high setup_disp_mode,low setup_disp_mode
	db	"InitTemp"
	db	EE_init_temp-EEPARS_START
	db	D'0',D'35'
	db	upper setup_disp_itemp,high setup_disp_itemp,low setup_disp_itemp
	db	"Amb.Temp"
	db	EE_temp_cal-EEPARS_START
	db	D'0',D'40'
	db	upper setup_disp_atemp,high setup_disp_atemp,low setup_disp_atemp
	db	"TC.coeff"
	db	EE_tc_coeff-EEPARS_START
	db	D'10',D'99'
	db	upper setup_disp_tccoeff,high setup_disp_tccoeff,low setup_disp_tccoeff
	db	0




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;; EEPROM data ;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


; EEPROM data
EE_CODE	code_pack	0xF00000

	; Fast
	de		'F'			; character to indicate this setting
	data16	D'10000'	; mmm
	data16	D'160'		; iii
	data16	D'640'		; ppp
	data16	D'50'		; ddd
	de		1			; use R parameter
	de		"Fast  "	; must be 6 bytes long

	; Fine
	de		'P'			; character to indicate this setting
	data16	D'30'		; mmm
	data16	D'120'		; iii
	data16	D'200'		; ppp
	data16	D'300'		; ddd
	de		1			; use R parameter
	de		"Prec  "	; must be 6 bytes long

	; Quick
	de		'Q'			; character to indicate this setting
	data16	D'10000'	; mmm
	data16	D'0'		; iii
	data16	D'200'		; ppp
	data16	D'0'		; ddd
	de		1			; use R parameter
	de		"Quick "	; must be 6 bytes long

	; Switch
	de		'S'			; character to indicate this setting
	data16	D'10000'	; mmm
	data16	D'0'		; iii
	data16	D'65000'	; ppp
	data16	D'0'		; ddd
	de		0			; use R parameter
	de		"Switch"	; must be 6 bytes long

	; last entry
	de		0x00		; to indicate end of list

EEPARS_START

EE_lcd_mode		de	0
EE_temp_cal		de	D'20'
EE_tc_coeff		de	D'68'
EE_start_mode	de	0
EE_init_temp	de	D'16'
EE_instant_on	de	0

EEPARS_END


;	; 2nd entry
;	de		'P'			; character to indicate this setting
;	data16	D'10000'	; mmm
;	data16	D'40'		; iii
;	data16	D'160'		; ppp
;	data16	D'200'		; ddd
;	de		1			; use R parameter
;	de		"Precise  "	; must be 6 bytes long
;
;	; 5th entry
;	de		'I'			; character to indicate this setting
;	data16	D'10000'	; mmm
;	data16	D'2'		; iii
;	data16	D'0'		; ppp
;	data16	D'1000'		; ddd
;	de		1			; use R parameter
;	de		"Integr"	; must be 6 bytes long
;
;	; my entry
;	de		'M'			; character to indicate this setting
;	data16	D'10000'	; mmm
;	data16	D'120'		; iii
;	data16	D'120'		; ppp
;	data16	D'5000'		; ddd
;	de		1			; use R parameter
;	de		"Slow  "	; must be 6 bytes long
;
;	; my new entry
;	de		'N'			; character to indicate this setting
;	data16	D'10000'	; mmm
;	data16	D'120'		; iii
;	data16	D'120'		; ppp
;	data16	D'500'		; ddd
;	de		1			; use R parameter
;	de		"ModM  "	; must be 6 bytes long
;
;	; my new entry
;	de		'G'			; character to indicate this setting
;	data16	D'10000'	; mmm
;	data16	D'160'		; iii
;	data16	D'320'		; ppp
;	data16	D'100'		; ddd
;	de		1			; use R parameter
;	de		"ModF  "	; must be 6 bytes long
;
;	; my new entry
;	de		'Q'			; character to indicate this setting
;	data16	D'10000'	; mmm
;	data16	D'320'		; iii
;	data16	D'320'		; ppp
;	data16	D'100'		; ddd
;	de		1			; use R parameter
;	de		"ModG  "	; must be 6 bytes long
;
;	; my new entry
;	de		'Y'			; character to indicate this setting
;	data16	D'30'		; mmm
;	data16	D'160'		; iii
;	data16	D'200'		; ppp
;	data16	D'300'		; ddd
;	de		1			; use R parameter
;	de		"ModW  "	; must be 6 bytes long
;
;	; last entry
;	de		0x00		; to indicate end of list
;

	end
