; PIC Microcontoller Basic Math Methods 
; 32-bit signed integer math routines.
; add, subtract, multiply, divide, round, sqrt, bin2dec, dec2bin.
; By Peter Hemsley.


; USING THE 32-BIT ROUTINES
; Signed 32-bit numbers have a range of -2147483648 to +2147483647.
; However -2147483648 does not have a positive equivalent which would needed in many cases
; and so the range is limited to 1 2147483647, with -2147483648 being trapped as an error.
;
; Three 32-bit pseudo-registers are defined,
; REGA contains the first operand,
; REGB contains the second operand and
; REGC is used as temporary storage by the routines.
;
; The result is always returned in REGA.
;
; Each pseudo-register consists of 4 consecutive bytes denoted by 0, 1, 2 and 3.
; 0 is the least significant byte and 3 is the most significant byte.
; High level languages have built-in run-time error checking but assembler
; has no such luxuries and the programmer must write his/her own.
; Comprehensive error checking is included in the routines and 
; the Carry will be set on return from the routines if an error is found.
; The code required  for error checking after calling a routine is shown below.
;
; CALL     <function>
;
; SKPNC
;
; GOTO    <errorhandler>

 

; Add:			REGA = REGA + REGB
; Subtract:		REGA = REGA - REGB
; Multiply:		REGA = REGA * REGB
; Divide:		REGA = REGA / REGB
; Sq Root:		REGA = SQRT (REGA)

 

;*** SIGNED 32-BIT INTEGER MATHS ROUTINES FOR PIC16 SERIES BY PETER HEMSLEY ***
;
;Functions:
;	add
;	subtract
;	multiply
;	divide
;	round
;	sqrt
;	bin2dec
;	dec2bin

;Macro definitions 

skpc	macro
	btfss	STATUS,C
	endm

skpnc	macro
	btfsc	STATUS,C
	endm

skpz	macro
	btfss	STATUS,Z
	endm

skpnz	macro
	btfsc	STATUS,Z
	endm

setc	macro
	bsf		STATUS,C
	endm

clrc	macro
	bcf		STATUS,C
	endm

;Variable declarations

	#include <p18F2321.inc>	; Processor specific variable definitions

	GLOBAL	REGA0,REGA1,REGA2,REGA3
	GLOBAL	REGB0,REGB1,REGB2,REGB3
	GLOBAL	REGC0,REGC1,REGC2,REGC3
	GLOBAL	DSIGN
	GLOBAL	DIGIT1,DIGIT2,DIGIT3,DIGIT4,DIGIT5,DIGIT6,DIGIT7,DIGIT8,DIGIT9,DIGIT10
	GLOBAL	add,subtract,multiply,divide,round,sqrt,bin2dec,dec2bin
	GLOBAL	absa,negatea,negateb

	udata_acs
REGA0			res	1	;lsb
REGA1			res	1	
REGA2			res	1	
REGA3			res	1	;msb

REGB0			res	1	;lsb
REGB1			res	1	
REGB2			res	1	
REGB3			res	1	;msb

REGC0			res	1	;lsb
REGC1			res	1	
REGC2			res	1	
REGC3			res	1	;msb

DSIGN			res	1	;Digit Sign. 0=positive,1=negative
DIGIT1			res	1	;MSD
DIGIT2			res	1	
DIGIT3			res	1	
DIGIT4			res	1	
DIGIT5			res	1	;Decimal (BCD) digits
DIGIT6			res	1	
DIGIT7			res	1	
DIGIT8			res	1	
DIGIT9			res	1	
DIGIT10			res	1	;LSD

MTEMP			res	1	
MCOUNT			res	1	
DCOUNT			res	1
CYSTACK			res	1	

	code
;*** 32 BIT SIGNED SUTRACT ***
;REGA - REGB -> REGA
;Return carry set if overflow

subtract
	call	negateb		;Negate REGB
	skpnc
	return			;Overflow

;*** 32 BIT SIGNED ADD ***
;REGA + REGB -> REGA
;Return carry set if overflow

add	movf	REGA3,w		;Compare signs
	xorwf	REGB3,w
	movwf	MTEMP

	call	addba		;Add REGB to REGA

	clrc			;Check signs
	movf	REGB3,w		;If signs are same
	xorwf	REGA3,w		;so must result sign
	btfss	MTEMP,7		;else overflow
	addlw	0x80
	return

;*** 32 BIT SIGNED MULTIPLY ***
;REGA * REGB -> REGA
;Return carry set if overflow

multiply
	clrf	MTEMP		;Reset sign flag
	call	absa		;Make REGA positive
	skpc
	call	absb		;Make REGB positive
	skpnc
	return			;Overflow

	call	movac		;Move REGA to REGC
	call	clra		;Clear product

	movlw	D'31'		;Loop counter
	movwf	MCOUNT

muloop	call	slac		;Shift left product and multiplicand
	
	rlcf	REGC3,w		;Test MSB of multiplicand
	skpnc			;If multiplicand bit is a 1 then
	call	addba		;add multiplier to product

	skpc			;Check for overflow
	rlcf	REGA3,w
	skpnc
	return

	decfsz	MCOUNT,f	;Next
	goto	muloop

	btfsc	MTEMP,0		;Check result sign
	call	negatea		;Negative
	return


;*** 32 BIT SIGNED DIVIDE ***
;REGA / REGB -> REGA
;Remainder in REGC
;Return carry set if overflow or division by zero

divide	clrf	MTEMP		;Reset sign flag
	movf	REGB0,w		;Trap division by zero
	iorwf	REGB1,w
	iorwf	REGB2,w
	iorwf	REGB3,w
	sublw	0
	skpc
	call	absa		;Make dividend (REGA) positive
	skpc
	call	absb		;Make divisor (REGB) positive
	skpnc
	return			;Overflow

	clrf	REGC0		;Clear remainder
	clrf	REGC1
	clrf	REGC2
	clrf	REGC3
	call	slac		;Purge sign bit

	movlw	D'31'		;Loop counter
	movwf	MCOUNT

dvloop	call	slac		;Shift dividend (REGA) msb into remainder (REGC)

	movf	REGB3,w		;Test if remainder (REGC) >= divisor (REGB)
	subwf	REGC3,w
	skpz
	goto	dtstgt
	movf	REGB2,w
	subwf	REGC2,w
	skpz
	goto	dtstgt
	movf	REGB1,w
	subwf	REGC1,w
	skpz
	goto	dtstgt
	movf	REGB0,w
	subwf	REGC0,w
dtstgt	skpc			;Carry set if remainder >= divisor
	goto	dremlt

	movf	REGB0,w		;Subtract divisor (REGB) from remainder (REGC)
	subwf	REGC0,f
	movf	REGB1,w
	skpc
	incfsz	REGB1,w
	subwf	REGC1,f
	movf	REGB2,w
	skpc
	incfsz	REGB2,w
	subwf	REGC2,f
	movf	REGB3,w
	skpc
	incfsz	REGB3,w
	subwf	REGC3,f
	clrc
	bsf	REGA0,0		;Set quotient bit

dremlt	decfsz	MCOUNT,f	;Next
	goto	dvloop

	btfsc	MTEMP,0		;Check result sign
	call	negatea		;Negative
	return

;*** ROUND RESULT OF DIVISION TO NEAREST INTEGER ***

round	clrf	MTEMP		;Reset sign flag
	call	absa		;Make positive
	clrc
	call	slc		;Multiply remainder by 2
	movf	REGB3,w		;Test if remainder (REGC) >= divisor (REGB)
	subwf	REGC3,w
	skpz
	goto	rtstgt
	movf	REGB2,w
	subwf	REGC2,w
	skpz
	goto	dtstgt
	movf	REGB1,w
	subwf	REGC1,w
	skpz
	goto	rtstgt
	movf	REGB0,w
	subwf	REGC0,w
rtstgt	skpc			;Carry set if remainder >= divisor
	goto	rremlt
	incfsz	REGA0,f		;Add 1 to quotient
	goto	rremlt
	incfsz	REGA1,f
	goto	rremlt
	incfsz	REGA2,f
	goto	rremlt
	incf	REGA3,f
	bnz		rremlt
	clrc
	return			;Overflow,return carry set
rremlt	btfsc	MTEMP,0		;Restore sign
	call	negatea
	return


;*** 32 BIT SQUARE ROOT ***
;sqrt(REGA) -> REGA
;Return carry set if negative

sqrt	rlcf	REGA3,w		;Trap negative values
	skpnc
	return

	call	movac		;Move REGA to REGC
	call	clrba		;Clear remainder (REGB) and root (REGA)

	movlw	D'16'		;Loop counter
	movwf	MCOUNT

sqloop	rlcf	REGC0,f		;Shift two msb's
	rlcf	REGC1,f		;into remainder
	rlcf	REGC2,f
	rlcf	REGC3,f
	rlcf	REGB0,f
	rlcf	REGB1,f
	rlcf	REGB2,f
	rlcf	REGC0,f
	rlcf	REGC1,f
	rlcf	REGC2,f
	rlcf	REGC3,f
	rlcf	REGB0,f
	rlcf	REGB1,f
	rlcf	REGB2,f

	setc			;Add 1 to root
	rlcf	REGA0,f		;Align root
	rlcf	REGA1,f
	rlcf	REGA2,f

	movf	REGA2,w		;Test if remdr (REGB) >= root (REGA)
	subwf	REGB2,w
	skpz
	goto	ststgt
	movf	REGA1,w
	subwf	REGB1,w
	skpz
	goto	ststgt
	movf	REGA0,w
	subwf	REGB0,w
ststgt	skpc			;Carry set if remdr >= root
	goto	sremlt

	movf	REGA0,w		;Subtract root (REGA) from remdr (REGB)
	subwf	REGB0,f
	movf	REGA1,w
	skpc
	incfsz	REGA1,w
	subwf	REGB1,f
	movf	REGA2,w
	skpc
	incfsz	REGA2,w
	subwf	REGB2,f
	bsf	REGA0,1		;Set current root bit

sremlt	bcf	REGA0,0		;Clear test bit
	decfsz	MCOUNT,f	;Next
	goto	sqloop

	clrc
	rrcf	REGA2,f		;Adjust root alignment
	rrcf	REGA1,f
	rrcf	REGA0,f
	return


;*** 32 BIT SIGNED BINARY TO DECIMAL ***
;REGA -> DIGITS 1 (MSD) TO 10 (LSD) & DSIGN
;DSIGN = 0 if REGA is positive, 1 if negative
;Return carry set if overflow
;Uses FSR register

bin2dec	clrf	MTEMP		;Reset sign flag
	call	absa		;Make REGA positive
	skpnc
	return			;Overflow

	call	clrdig		;Clear all digits

	movlw	D'32'		;Loop counter
	movwf	MCOUNT

b2dloop	rlcf	REGA0,f		;Shift msb into carry
	rlcf	REGA1,f
	rlcf	REGA2,f
	rlcf	REGA3,f

	movlw	LOW DIGIT10
	movwf	FSR0L		;Pointer to digits
	movlw	HIGH DIGIT10
	movwf	FSR0H
	movlw	D'10'		;10 digits to do
	movwf	DCOUNT

adjlp	rlcf	INDF0,f		;Shift digit and carry 1 bit left
	movlw	D'10'
	subwf	INDF0,w		;Check and adjust for decimal overflow
	skpnc
	movwf	INDF0

	rrcf	CYSTACK,f
	decf	FSR0L,f		;Next digit
	rlcf	CYSTACK,f
	decfsz	DCOUNT,f
	goto	adjlp

	decfsz	MCOUNT,f	;Next bit
	goto	b2dloop

	btfsc	MTEMP,0		;Check sign
	bsf	DSIGN,0		;Negative
	clrc
	return


;*** 32 BIT SIGNED DECIMAL TO BINARY ***
;Decimal DIGIT1 thro DIGIT(X) & DSIGN -> REGA
;Set DSIGN = 0 for positive, DSIGN = 1 for negative values
;Most significant digit in DIGIT1
;Enter this routine with digit count in w register
;Return carry set if overflow
;Uses FSR register

dec2bin	movwf	MTEMP		;Save digit count

	movlw	D'32'		;Outer bit loop counter
	movwf	MCOUNT

d2blp1	movlw	DIGIT1-1	;Set up pointer to MSD
	movwf	FSR0L
	clrf	FSR0H
	movf	MTEMP,w		;Inner digit loop counter
	movwf	DCOUNT

	movlw	D'10'
	clrc			;Bring in '0' bit into MSD

d2blp2
	rrcf	CYSTACK,f
	incf	FSR0L,f
	rlcf	CYSTACK,f
	skpnc
	addwf	INDF0,f		;Add 10 if '1' bit from prev digit
	rrcf	INDF0,f		;Shift out LSB of digit

	decfsz	DCOUNT,f	;Next L.S. Digit
	goto	d2blp2

	rrcf	REGA3,f		;Shift in carry from digits
	rrcf	REGA2,f
	rrcf	REGA1,f
	rrcf	REGA0,f

	decfsz	MCOUNT,f	;Next bit
	goto	d2blp1

	movf	INDF0,w		;Check for overflow
	addlw	0xFF
	skpc
	rlcf	REGA3,w
	skpnc
	return

	btfsc	DSIGN,0		;Check result sign
	call	negatea		;Negative
	return


;UTILITY ROUTINES


;Add REGB to REGA (Unsigned)
;Used by add, multiply,

addba	movf	REGB0,w		;Add lo byte
	addwf	REGA0,f

	movf	REGB1,w		;Add mid-lo byte
	skpnc			;No carry_in, so just add
	incfsz	REGB1,w		;Add carry_in to REGB
	addwf	REGA1,f		;Add and propagate carry_out

	movf	REGB2,w		;Add mid-hi byte
	skpnc
	incfsz	REGB2,w
	addwf	REGA2,f

	movf	REGB3,w		;Add hi byte
	skpnc
	incfsz	REGB3,w
	addwf	REGA3,f
	return


;Move REGA to REGC
;Used by multiply, sqrt

movac	movf	REGA0,w
	movwf	REGC0
	movf	REGA1,w
	movwf	REGC1
	movf	REGA2,w
	movwf	REGC2
	movf	REGA3,w
	movwf	REGC3
	return


;Clear REGB and REGA
;Used by sqrt

clrba	clrf	REGB0
	clrf	REGB1
	clrf	REGB2
	clrf	REGB3

;Clear REGA
;Used by multiply, sqrt

clra	clrf	REGA0
	clrf	REGA1
	clrf	REGA2
	clrf	REGA3
	return


;Check sign of REGA and convert negative to positive
;Used by multiply, divide, bin2dec, round

absa	rlcf	REGA3,w
	skpc
	return			;Positive

;Negate REGA
;Used by absa, multiply, divide, bin2dec, dec2bin, round

negatea	movf	REGA3,w		;Save sign in w
	andlw	0x80

	comf	REGA0,f		;2's complement
	comf	REGA1,f
	comf	REGA2,f
	comf	REGA3,f
	incfsz	REGA0,f
	goto	nega1
	incfsz	REGA1,f
	goto	nega1
	incfsz	REGA2,f
	goto	nega1
	rrcf	CYSTACK,f
	incf	REGA3,f
	rlcf	CYSTACK,f
nega1
	rrcf	CYSTACK,f
	incf	MTEMP,f		;flip sign flag
	rlcf	CYSTACK,f
	addwf	REGA3,w		;Return carry set if -2147483648
	return


;Check sign of REGB and convert negative to positive
;Used by multiply, divide

absb	rlcf	REGB3,w
	skpc
	return			;Positive

;Negate REGB
;Used by absb, subtract, multiply, divide

negateb	movf	REGB3,w		;Save sign in w
	andlw	0x80

	comf	REGB0,f		;2's complement
	comf	REGB1,f
	comf	REGB2,f
	comf	REGB3,f
	incfsz	REGB0,f
	goto	negb1
	incfsz	REGB1,f
	goto	negb1
	incfsz	REGB2,f
	goto	negb1
	rrcf	CYSTACK,f
	incf	REGB3,f
	rlcf	CYSTACK,f
negb1
	rrcf	CYSTACK,f
	incf	MTEMP,f		;flip sign flag
	rlcf	CYSTACK,f
	addwf	REGB3,w		;Return carry set if -2147483648
	return


;Shift left REGA and REGC
;Used by multiply, divide, round

slac	rlcf	REGA0,f
	rlcf	REGA1,f
	rlcf	REGA2,f
	rlcf	REGA3,f
slc	rlcf	REGC0,f
	rlcf	REGC1,f
	rlcf	REGC2,f
	rlcf	REGC3,f
	return


;Set all digits to 0
;Used by bin2dec

clrdig	clrf	DSIGN
	clrf	DIGIT1
	clrf	DIGIT2
	clrf	DIGIT3
	clrf	DIGIT4
	clrf	DIGIT5
	clrf	DIGIT6
	clrf	DIGIT7
	clrf	DIGIT8
	clrf	DIGIT9
	clrf	DIGIT10
	return

	end

;Comments: 
;spamanthony-Remove- at @spam@unihedron.com
;   Error in round, "GOTO dtstgt" should be "GOTO rtstgt".
;
;Questions: 
;   I tryed to divide 36/360. But instead 0.1 i get 0.36.
;   This happens with every combination where the REGA is lower then REGB.
;
;spamrobson2-Remove- at @spam@mts.net asks:
;   Looking for the rule that explains the order in which you multiply, add, subtract and divide.
;   Example 811/5+900x4-60
; James Newton replies:
;   In general, multiply and divide first, then add and subtract.
;   There is an expression evaluator that may help at http://www.sxlist.com/techref/expeval2.asp.
;   For more, search for "operator precedence".+
