; Soldering iron heating control firmware
; v2.5 (new hardware)
; 2007.11
; (c) szilva
;
;
; Board connections:
; 
; PIC16F684
;
;      VDD o1  o VSS
; PSON/RA5 o   o RA0/EN
;   TX/RA4 o   o RA1/RS
;          o   o RA2/DB7
; HEAT/P1A o   o RC0/DB6
;  CON/RC4 o   o RC1/DB5
;   TC/AN7 o   o RC2/DB4
;
;
; LCD module
; ----------
; EN  - RA0
; RS  - RA1
; DB7 - RA2
; DB6 - RC0
; DB5 - RC1
; DB4 - RC2
;
; thermocouple:		RC3/AN7
; LCD contrast PWM:	RC4
; heating switch:	RC5/P1A
; PSON control:		RA5
; RS232 TX:			RA4
;

; firmware version
#define	vermajor	2
#define	verminor	5

#include <p16F684.inc>	; Processor specific variable definitions

; processor configuration
;	__config (_INTRC_OSC_NOCLKOUT & _WDT_OFF & _PWRTE_OFF & _MCLRE_OFF & _CP_OFF & _IESO_OFF & _FCMEN_OFF)
	__config (_INTRC_OSC_NOCLKOUT & _WDT_OFF & _PWRTE_OFF & _MCLRE_ON & _CP_OFF & _IESO_OFF & _FCMEN_OFF)

; system clock to calculate with
#define	SYSCLK		D'8000000'		; system clock in Hzs
#define	BAUDRATE	D'19200'		; RS232 baudrate

; arithmetical routines
#include "32bit_math.h"


; LCD module wiring definitions
#define		PORTPAGE		PORTA
#define		TRISPAGE		TRISA

#define		LCD_E			PORTA,0
#define		LCD_E_TRIS		TRISA,0
#define		LCD_RS			PORTA,1
#define		LCD_RS_TRIS		TRISA,1

#define		LCD_DB7			PORTA,2
#define		LCD_DB7_TRIS	TRISA,2
#define		LCD_DB6			PORTC,0
#define		LCD_DB6_TRIS	TRISC,0
#define		LCD_DB5			PORTC,1
#define		LCD_DB5_TRIS	TRISC,1
#define		LCD_DB4			PORTC,2
#define		LCD_DB4_TRIS	TRISC,2

; HD44780 LCD routines
#include "HD_LCD.INC"


; other definitions reflecting hardware
; thermocouple analog input
AnChNum		equ	ANS7

; LCD contrast bit
#define		CON			PORTC,4
#define		CON_TRIS	TRISC,4

; heating control output
#define		HEAT_TRIS	TRISC,5

; PSON control output
#define		PSON		PORTA,5
#define		PSON_TRIS	TRISA,5

; RS232 TX output bit
#define		TX			PORTA,4
#define		TX_TRIS		TRISA,4


; initial temperature and limits 
tnormal	equ	.2600
mintemp	equ	.1000
maxtemp	equ	.4500

; timer0 reload value for desired RS232 baudrate
tmr0r	equ	D'256' - (SYSCLK/BAUDRATE/D'4') + 1 
t100r	equ	(BAUDRATE/.100)
t10r	equ	.10

; PWM configuration word
PWM_configword	equ		(1<<CCP1M0) + (1<<CCP1M1) + (1<<CCP1M2) + (1<<CCP1M3)


;*** Variable definitions
; shared data, used in interrupt
	udata_shr
flags			res	1	; semaphores
saved_w			res	1	; in interrupt
saved_status	res	1	; in interrupt
tx_byte			res	1	; byte to transmit
tx_cnt			res	1	; rs232 bit count
tx_lo			res	1	; low rs232 bits
tx_hi			res	1	; high rs232 bits
t100cnt			res	1	; counter for 100 per second flag
t10cnt			res	1	; counter for 10 per second flag

	udata
ctrltype		res	1	; controller type
ctrlchar		res	1	; controller type character
heat_lo			res	1	; actual heating power
heat_hi			res	1	; actual heating power
txtptr_lo		res	1	; text pointer low
txtptr_hi		res	1	; text pointer high
res_lo			res	1	; A/D result low
res_hi			res	1	; A/D result high
target_lo		res	1	; needed temperature low
target_hi		res	1	; needed temperature high
temp_lo			res	1	; actual temperature low
temp_hi			res	1	; actual temperature high
tmp				res	1	; temporay data
tcnt			res	1	; temporary counter
tcnt1			res	1	; temporary counter
kbd				res	1	; keyboard data
okbd			res	1	; old keyboard data
mode			res	1	; operating mode
dmodecnt		res	1	; display mode counter
; time counter
tmcnt0			res	1
tmcnt1			res	1
tmcnt2			res	1
tmcnt3			res	1

; xv0,xv1 are 16-bit regs
xv0l			res	1
xv0h			res	1
xv1l			res	1
xv1h			res	1
; yv1, yv2 are 24-bit regs
yv1l			res	1
yv1m			res	1
yv1h			res	1
yv2l			res	1
yv2m			res	1
yv2h			res	1
; sss is a 32-bit temporary reg
sss0			res	1
sss1			res	1
sss2			res	1
sss3			res	1
; p type controller data register
ppp0			res	1
ppp1			res	1
ppp2			res	1
ppp3			res	1
; sum register for i type controller
iii0			res	1
iii1			res	1
iii2			res	1
iii3			res	1
; register for r type controller
rrr0			res	1
rrr1			res	1
rrr2			res	1
rrr3			res	1


#define		RS232_TX	flags,0
#define		tick		flags,1
#define		dmode		flags,2

;
; macro definitions
;

unpack32	MACRO	Var,Address ;Var = 32 bit literal to be unpacked
	BANKSEL	Address 
	movlw	Var & H'FF'
	movwf	Address+0
	movlw	Var >>D'08' & H'FF'
	movwf	Address+1
	movlw	Var >>D'16' & H'FF'
	movwf	Address+2
	movlw	Var >>D'24' & H'FF'
	movwf	Address+3
	ENDM

get16eeprom	MACRO	Dest	; get a 16-bit value to Dest from EEPROM
	banksel	EECON1
	movlw	Dest
	movwf	FSR
	bsf		EECON1,RD
	movf	EEDAT,w
	movwf	INDF
	incf	FSR,f
	incf	EEADR,f
	bsf		EECON1,RD
	movf	EEDAT,w
	movwf	INDF
	incf	FSR,f
	incf	EEADR,f
	clrw
;	btfsc	INDF,7		; extending sign to 32-bit
;	movlw	0xff
	movwf	INDF
	incf	FSR,f
	movwf	INDF
	incf	FSR,f
	banksel	0
	ENDM

data16	MACRO	Data
	de	low(Data),High(Data)
	ENDM


; EEPROM data
EE_CODE	code	0x2100

	; Fast
	de		'F'			; character to indicate this setting
	data16	D'10000'	; mmm
	data16	D'160'		; iii
	data16	D'640'		; ppp
	data16	D'50'		; ddd
	de		1			; use R parameter
	de		"Fast  "	; must be 6 bytes long

	; Fine
	de		'P'			; character to indicate this setting
	data16	D'30'		; mmm
	data16	D'120'		; iii
	data16	D'200'		; ppp
	data16	D'300'		; ddd
	de		1			; use R parameter
	de		"Prec  "	; must be 6 bytes long

	; Quick
	de		'Q'			; character to indicate this setting
	data16	D'10000'	; mmm
	data16	D'0'		; iii
	data16	D'200'		; ppp
	data16	D'0'		; ddd
	de		1			; use R parameter
	de		"Quick "	; must be 6 bytes long

	; Switch
	de		'S'			; character to indicate this setting
	data16	D'10000'	; mmm
	data16	D'0'		; iii
	data16	D'65000'	; ppp
	data16	D'0'		; ddd
	de		0			; use R parameter
	de		"Switch"	; must be 6 bytes long

	; last entry
	de		0x00		; to indicate end of list

	; 2nd entry
	de		'P'			; character to indicate this setting
	data16	D'10000'	; mmm
	data16	D'40'		; iii
	data16	D'160'		; ppp
	data16	D'200'		; ddd
	de		1			; use R parameter
	de		"Precise  "	; must be 6 bytes long

	; 5th entry
	de		'I'			; character to indicate this setting
	data16	D'10000'	; mmm
	data16	D'2'		; iii
	data16	D'0'		; ppp
	data16	D'1000'		; ddd
	de		1			; use R parameter
	de		"Integr"	; must be 6 bytes long

	; my entry
	de		'M'			; character to indicate this setting
	data16	D'10000'	; mmm
	data16	D'120'		; iii
	data16	D'120'		; ppp
	data16	D'5000'		; ddd
	de		1			; use R parameter
	de		"Slow  "	; must be 6 bytes long

	; my new entry
	de		'N'			; character to indicate this setting
	data16	D'10000'	; mmm
	data16	D'120'		; iii
	data16	D'120'		; ppp
	data16	D'500'		; ddd
	de		1			; use R parameter
	de		"ModM  "	; must be 6 bytes long

	; my new entry
	de		'G'			; character to indicate this setting
	data16	D'10000'	; mmm
	data16	D'160'		; iii
	data16	D'320'		; ppp
	data16	D'100'		; ddd
	de		1			; use R parameter
	de		"ModF  "	; must be 6 bytes long

	; my new entry
	de		'Q'			; character to indicate this setting
	data16	D'10000'	; mmm
	data16	D'320'		; iii
	data16	D'320'		; ppp
	data16	D'100'		; ddd
	de		1			; use R parameter
	de		"ModG  "	; must be 6 bytes long

	; my new entry
	de		'Y'			; character to indicate this setting
	data16	D'30'		; mmm
	data16	D'160'		; iii
	data16	D'200'		; ppp
	data16	D'300'		; ddd
	de		1			; use R parameter
	de		"ModW  "	; must be 6 bytes long

	; last entry
	de		0x00		; to indicate end of list

;
;*** Beginning of the program 
;
RESVEC	code	0x000		; Processor reset vector
	GOTO	SOLD_START		; Go to initialization


;*** Interrupt vector
INTVEC	code	0x004		; Interrupt vector location
	GOTO	interrupt_handler


; relocatable code segment
	CODE
interrupt_handler
	movwf	saved_w		; save w reg
	swapf	STATUS,w	; the swapf instruction, unlike the movf, affects no status bits, which is why it is used here.
	movwf	saved_status	; save status reg

	banksel	INTCON
	btfss	INTCON,T0IF
	goto	no_int_tmr0

; timer0 interrupt
	bcf		INTCON,T0IF

	movlw	tmr0r+2
	banksel	TMR0
	addwf	TMR0,f

	banksel	PORTPAGE
	btfsc	CON
	goto	int_1
	bsf		CON
	goto	int_2
int_1
	bcf		CON
int_2

	btfss	RS232_TX
	goto	no_int_tmr0
	; RS232 send
	movf	tx_cnt,f
	skpz
	goto	bit_send
	; begin_send
	clrf	tx_hi
	decf	tx_hi,f
	rlf		tx_byte,w
	movwf	tx_lo
	rlf		tx_hi,f
	bcf		tx_lo,0		; set start bit
	movlw	.15
	movwf	tx_cnt
	
bit_send
	rrf		tx_hi,f
	rrf		tx_lo,f
	skpc
	goto	bit_low
	; set bit to high
	bsf		TX
	goto	bit_end
bit_low
	; set bit to low
	bcf		TX
bit_end
	decfsz	tx_cnt,f
	goto	no_int_tmr0
	; sent last bit
	bcf		RS232_TX
	goto	no_int_tmr0

no_int_tmr0
	decfsz	t100cnt
	goto	int_end
	movlw	t100r
	movwf	t100cnt

	decfsz	t10cnt
	goto	int_end
	movlw	t10r
	movwf	t10cnt

	bsf		tick

	incfsz	tmcnt0,f
	goto	int_end
	incfsz	tmcnt1,f
	goto	int_end
	incfsz	tmcnt2,f
	goto	int_end
	incf	tmcnt3,f

int_end
	swapf	saved_status,w	; get saved status in w 
	movwf	STATUS		; restore status ( and bank )
	swapf	saved_w,f	; reload into self to set status bits
	swapf	saved_w,w	; and restore
	RETFIE						; Return from interrupt





; POWER_ON Reset (Beginning of program)
SOLD_START

	banksel	OSCCON
	bsf		OSCCON,IRCF0	; 8MHz internal clock

;
; set up variables - initial values
;
	; tmcmt
	banksel	xv0l
	clrf	tmcnt0
	clrf	tmcnt1
	clrf	tmcnt2
	clrf	tmcnt3
	; yv1, yv2 are 24-bit regs
	banksel	xv0l
	clrf	xv0l
	clrf	xv0h
	clrf	xv1l
	clrf	xv1h
	; yv1, yv2 are 24-bit regs
	clrf	yv1l
	clrf	yv1m
	clrf	yv1h
	clrf	yv2l
	clrf	yv2m
	clrf	yv2h
	; sss is a 32-bit reg
	clrf	sss0
	clrf	sss1
	clrf	sss2
	clrf	sss3
	; integrator sum
	clrf	iii0
	clrf	iii1
	clrf	iii2
	clrf	iii3
	; proportional reg
	clrf	ppp0
	clrf	ppp1
	clrf	ppp2
	clrf	ppp3

	clrf	tx_cnt
	clrf	okbd
	clrf	mode
	incf	mode,f		; for instant poweron

	movlw	.30
	movwf	dmodecnt

	movlw	low(tnormal)
	movwf	target_lo
	movlw	high(tnormal)
	movwf	target_hi

	movlw	.1
	movwf	t100cnt
	movwf	t10cnt
	bcf		tick


;
; set up hardware
;

; initializing PORTA and PORTC as digital I/O on 16F684
; based on Microchip 16F684 manual
	BCF		STATUS,RP0 ;Bank 0
	CLRF	PORTA ;Init PORTA
	CLRF	PORTC ;Init PORTC
	MOVLW	07h ;Set RA<2:0> and RC<4,1:0> to
	MOVWF	CMCON0 ;digital I/O
	BSF		STATUS,RP0 ;Bank 1
	CLRF	ANSEL ;digital I/O
	BCF		STATUS,RP0 ;Bank 0


; setting up PWM module via method based on
; Microchip 16F684 manual "11.3.7 SETUP FOR PWM OPERATION" 
	banksel	TRISPAGE
	bsf		HEAT_TRIS

	banksel	PR2
	movlw	0xFE
	movwf	PR2

	banksel	CCP1CON
	movlw	PWM_configword
	movwf	CCP1CON

	banksel	PIR1
	bcf		PIR1,TMR2IF

	banksel	T2CON
	clrf	T2CON
	movlw	B'00000011'
	movwf	T2CON

	bsf		T2CON,TMR2ON

	banksel	PIR1
wait_tmr2
	btfss	PIR1,TMR2IF
	goto	wait_tmr2

	banksel	TRISPAGE
	bcf		HEAT_TRIS
; end of setting up PWM module

	; turn off heating
	clrf	heat_lo
	clrf	heat_hi
	call	SETHEAT

	; turn off analog inputs but AnChNum
	banksel	ANSEL	
	movlw	(1 << AnChNum)
	movwf	ANSEL

	; set up ADC
	BANKSEL	ADCON1
	MOVLW	B'01110000'	;ADC clock is the slowest
	MOVWF	ADCON1

	;Right justify, Vref=Vdd, AnNum, ADC On
	BANKSEL	ADCON0
	MOVLW	B'10000001' + (AnChNum<<2)
	MOVWF	ADCON0

	; set up PSON and CON control outputs
	banksel	TRISPAGE
	bcf		PSON_TRIS
	bcf		CON_TRIS
	bcf		TX_TRIS

	banksel	PORTPAGE
	bsf		PSON
	bsf		CON
	bcf		TX

	bcf		RS232_TX
	movlw	.0
	movwf	ctrltype	; starting type of control

; set up timer0
	banksel	OPTION_REG
	bcf		OPTION_REG,T0CS
	movlw	tmr0r+2
	banksel	TMR0
	movwf	TMR0

; set up timer0 interrupt
	banksel	PIR1
	bcf		PIR1,TMR2IF
	banksel	INTCON
	bcf		INTCON,T0IF

	banksel	INTCON
	bsf		INTCON,T0IE
	bsf		INTCON,PEIE
	bsf		INTCON,GIE

; initializing LCD
	movlw	.200
	call	LCD_DELAY	;	20ms
	CALL	LCD_HARD_INIT

	MOVLW   LCD_CHARGEN
	CALL    LCD_SEND_CMD

	movlw	B'11100'
	CALL	LCD_SEND_CHAR
	movlw	B'10111'
	CALL	LCD_SEND_CHAR
	movlw	B'11100'
	CALL	LCD_SEND_CHAR
	movlw	B'01000'
	CALL	LCD_SEND_CHAR
	movlw	B'01000'
	CALL	LCD_SEND_CHAR
	movlw	B'01000'
	CALL	LCD_SEND_CHAR
	movlw	B'00111'
	CALL	LCD_SEND_CHAR
	movlw	B'00000'
	CALL	LCD_SEND_CHAR

	MOVLW   LCD_HOME
	CALL    LCD_SEND_CMD


;
; power-on greeting
;
	movlw	low(txt_hello)
	movwf	txtptr_lo
	movlw	high(txt_hello)
	movwf	txtptr_hi
	call	txtout

	; delay 50x20ms = 1 sec
	movlw	.50
	movwf	tcnt
dly2
	movlw	.200
	call	LCD_DELAY	;	20ms
	decfsz	tcnt,f
	goto	dly2

	movlw	low(txt_RS232_init)
	movwf	txtptr_lo
	movlw	high(txt_RS232_init)
	movwf	txtptr_hi
	call	RS232_txtout

;
; main program loop
;
mainloop
	btfss	tick
	goto	mainloop
	bcf		tick

	; 10 per second
	MOVLW   LCD_HOME
	CALL    LCD_SEND_CMD

	bcf		dmode		; normal display mode
	movf	dmodecnt,f
	skpnz
	goto	nospecial
	bsf		dmode		; special display mode
	decf	dmodecnt,f

nospecial
	movlw	0
	xorwf	mode,w
	skpnz
	goto	mode_sleep

	movlw	1
	xorwf	mode,w
	skpnz
	goto	mode_normal

	goto	mode_setup

; sleep mode
mode_sleep
	bsf		PSON

	clrf	heat_lo
	clrf	heat_hi
	call	SETHEAT

	movlw	low(txt_sleep)
	movwf	txtptr_lo
	movlw	high(txt_sleep)
	movwf	txtptr_hi
	call	txtout
	goto	keyhandle

;normal	mode
mode_normal
	bcf		PSON

	; measure actual temperature of iron
	call	gettemp
	movf	REGA0,w
	movwf	temp_lo
	movf	REGA1,w
	movwf	temp_hi

	swapf	ctrltype,w
	andlw	B'11110000'
	banksel	EECON1
	movwf	EEADR

	bsf		EECON1,RD
	movf	EEDAT,w
	incf	EEADR,f
	
	banksel	0
	movwf	ctrlchar

; modified PID-type controller

	movf	target_lo,w
	movwf	REGA0
	movf	target_hi,w
	movwf	REGA1
	clrf	REGA2
	clrf	REGA3

	unpack32	D'10',REGB0
	call	multiply
	unpack32	-D'46919',REGB0
	call	add
	movf	REGA0,w
	movwf	rrr0
	movf	REGA1,w
	movwf	rrr1
	movf	REGA2,w
	movwf	rrr2
	movf	REGA3,w
	movwf	rrr3

	movf	target_lo,w
	movwf	REGA0
	movf	target_hi,w
	movwf	REGA1
	clrf	REGA2
	clrf	REGA3
	
	movf	temp_lo,w
	subwf	REGA0,f
	skpc
	decf	REGA1,f
	movf	temp_hi,w
	subwf	REGA1,f
	btfsc	REGA1,7
	decf	REGA2,f
	movf	REGA2,w
	movwf	REGA3
	; REGA = target - temp

	call	rega_sss
	; sss = target - temp

	call	absa
	; Km - maximum error signal
	get16eeprom	REGB0
	call	subtract
	btfsc	REGA3,7
	goto	lower

	movf	REGB0,w
	movwf	REGA0
	movf	REGB1,w
	movwf	REGA1
	movf	REGB2,w
	movwf	REGA2
	movf	REGB3,w
	movwf	REGA3
	btfss	sss3,7
	call	negatea
	goto	calc_iii

lower
	call	sss_rega

calc_iii
	; Ki
	get16eeprom	REGB0
;	unpack32	D'100',REGB0
	call	multiply

	movf	iii0,w
	movwf	REGB0
	movf	iii1,w
	movwf	REGB1
	movf	iii2,w
	movwf	REGB2
	movf	iii3,w
	movwf	REGB3
	call	add
	call	trunc24

	movf	REGA0,w
	movwf	iii0
	movf	REGA1,w
	movwf	iii1
	movf	REGA2,w
	movwf	iii2
	movf	REGA3,w
	movwf	iii3
	; iii = (sum of Ki*(target - temp))

	; Kp
	get16eeprom	REGB0
;	unpack32	D'160',REGB0
	call	sss_rega
	call	multiply
	call	trunc24

	movf	REGA0,w
	movwf	ppp0
	movf	REGA1,w
	movwf	ppp1
	movf	REGA2,w
	movwf	ppp2
	movf	REGA3,w
	movwf	ppp3
	; ppp = Kp*(target - temp) - offset

	movf	sss0,w
	movwf	REGA0
	movf	sss1,w
	movwf	REGA1
	movf	sss2,w
	movwf	REGA2
	movf	sss3,w
	movwf	REGA3
	call	absa

	get16eeprom	REGB0
;	unpack32	D'200',REGB0
	call	subtract

	btfsc	REGA3,.7
	goto	noneg_PI

	clrf	iii0
	clrf	iii1
	clrf	iii2
	clrf	iii3

noneg_PI

	; iii/256
	movf	iii1,w
	movwf	REGA0
	movf	iii2,w
	movwf	REGA1
	movf	iii3,w
	movwf	REGA2
	clrf	REGA3
	btfsc	REGA2,7
	decf	REGA3,f

	; add P value
	movf	ppp0,w
	movwf	REGB0
	movf	ppp1,w
	movwf	REGB1
	movf	ppp2,w
	movwf	REGB2
	movf	ppp3,w
	movwf	REGB3
	call	add

	banksel	EECON1
	bsf		EECON1,RD
	movf	EEDAT,w
	incf	EEADR,f
	banksel	0

	iorlw	0x00
	skpnz
	goto	end_PID

	; add R value
	movf	rrr0,w
	movwf	REGB0
	movf	rrr1,w
	movwf	REGB1
	movf	rrr2,w
	movwf	REGB2
	movf	rrr3,w
	movwf	REGB3
	call	add

end_PID
	call	trunc16
	movf	REGA1,w
	; restoring offset
	addlw	0x80
	movwf	heat_hi
	movf	REGA0,w
	movwf	heat_lo
	
goon
	call	SETHEAT

	btfss	dmode
	goto	goon1

	movlw	low(txt_mode)
	movwf	txtptr_lo
	movlw	high(txt_mode)
	movwf	txtptr_hi
	call	txtout

	movlw	.6
	movwf	tcnt

mode_l1
	banksel	EECON1
	bsf		EECON1,RD
	movf	EEDAT,w
	incf	EEADR,f
	banksel	0
	call	LCD_SEND_CHAR
	decfsz	tcnt
	goto	mode_l1

	movlw	' '
	call	LCD_SEND_CHAR
	movlw	' '
	call	LCD_SEND_CHAR

goon1
	movf	ctrlchar,w
	btfss	dmode
	call	LCD_SEND_CHAR
	movf	ctrlchar,w
	call	RS232_send

	movf	tmcnt0,w
	movwf	REGA0
	movf	tmcnt1,w
	movwf	REGA1
	movf	tmcnt2,w
	movwf	REGA2
	movf	tmcnt3,w
	movwf	REGA3
	call	bin2dec

	movlw	','
	call	RS232_send
	movlw	DIGIT3
	movwf	FSR
	movlw	.8
	call	RS232_sendnum
	
	movf	temp_lo,w
	movwf	REGA0
	movf	temp_hi,w
	movwf	REGA1
	clrf	REGA2
	clrf	REGA3
	unpack32	D'5',REGB0
	call	add
	unpack32	D'10',REGB0
	call	divide

	call	bin2dec
	movlw	.3	; minimum digits to display
	btfss	dmode
	call	writelcd
;	movlw	'C'
	movlw	0x00
	btfss	dmode
	call	LCD_SEND_CHAR

	; send calculated temperature to RS232
	movlw	','
	call	RS232_send
	movlw	DIGIT8
	movwf	FSR
	movlw	.3
	call	RS232_sendnum

	; send ppp reg to RS232
	movf	ppp0,w
	movwf	REGA0
	movf	ppp1,w
	movwf	REGA1
	movf	ppp2,w
	movwf	REGA2
	movf	ppp3,w
	movwf	REGA3
	call	bin2dec

	movlw	','
	call	RS232_send
	movlw	DIGIT3
	movwf	FSR
	movlw	.8
	call	RS232_sendnum

	; send iii reg to rs232
	movf	iii0,w
	movwf	REGA0
	movf	iii1,w
	movwf	REGA1
	movf	iii2,w
	movwf	REGA2
	movf	iii3,w
	movwf	REGA3
	call	bin2dec

	movlw	','
	call	RS232_send
	movlw	DIGIT3
	movwf	FSR
	movlw	.8
	call	RS232_sendnum

	; send rrr reg to RS232
	movf	rrr0,w
	movwf	REGA0
	movf	rrr1,w
	movwf	REGA1
	movf	rrr2,w
	movwf	REGA2
	movf	rrr3,w
	movwf	REGA3
	call	bin2dec

	movlw	','
	call	RS232_send
	movlw	DIGIT3
	movwf	FSR
	movlw	.8
	call	RS232_sendnum

	movlw	' '
	btfss	dmode
	call	LCD_SEND_CHAR

	; calculating heating power
	movf	heat_lo,w
	andlw	B'11000000'
	movwf	REGA0
	movf	heat_hi,w
	movwf	REGA1
	clrf	REGA2
	clrf	REGA3

	unpack32	D'100',REGB0
	call	multiply

	unpack32	H'FFC0',REGB0
	call	divide
	call	bin2dec
	movlw	.3	; minimum digits to display
	btfss	dmode
	call	writelcd
	movlw	'%'
	btfss	dmode
	call	LCD_SEND_CHAR

	; send calculated heating power to RS232
	movlw	','
	call	RS232_send
	movlw	DIGIT7
	movwf	FSR
	movlw	.4
	call	RS232_sendnum

	movlw	' '
	btfss	dmode
	call	LCD_SEND_CHAR
	movlw	0x7e
	btfss	dmode
	call	LCD_SEND_CHAR

	movf	target_lo,w
	movwf	REGA0
	movf	target_hi,w
	movwf	REGA1
	clrf	REGA2
	clrf	REGA3
	unpack32	D'10',REGB0
	call	divide
	call	bin2dec
	movlw	.3	; minimum digits to display
	btfss	dmode
	call	writelcd
;	movlw	'C'
	movlw	0x00
	btfss	dmode
	call	LCD_SEND_CHAR

	; send needed temperature to RS232
	movlw	','
	call	RS232_send
	movlw	DIGIT8
	movwf	FSR
	movlw	.3
	call	RS232_sendnum

	; send CRLF result to RS232
	movlw	0x0d
	call	RS232_send
	movlw	0x0a
	call	RS232_send

	goto	keyhandle

; setup mode
mode_setup
	movlw	low(txt_setup)
	movwf	txtptr_lo
	movlw	high(txt_setup)
	movwf	txtptr_hi
	call	txtout

	goto	keyhandle

keyhandle
	call	LCD_READ_KBD
	movwf	kbd
	skpnz
	goto	endkbd
	movf	okbd,f
	skpz
	goto	endkbd

	movlw	.0
	xorwf	mode,w
	skpz
	goto	nomode0

; keyboard in sleep mode
	movlw	.1
	xorwf	kbd,w
	skpz
	goto	nokbd1_sleep

	; key 1 pressed in sleep mode
	movlw	.1
	movwf	mode
	goto	endkbd

nokbd1_sleep
	movlw	.12
	xorwf	kbd,w
	skpz
	goto	endkbd

	; key 4 and 8 are pressed in sleep mode
	movlw	.2
	movwf	mode
	goto	endkbd

nomode0
	movlw	1
	xorwf	mode,w
	skpz
	goto	nomode1

; keyboard in normal mode
	movlw	.1
	xorwf	kbd,w
	skpz
	goto	nokbd1
	
	; key '1' pressed, switch operating mode
	movlw	0
	movwf	mode
	goto	endkbd

nokbd1
	movlw	.2
	xorwf	kbd,w
	skpz
	goto	nokbd2
	; key '2' pressed, switch controller type
	movlw	.10
	movwf	dmodecnt

	incf	ctrltype,f

	swapf	ctrltype,w
	andlw	B'11110000'

	banksel	EECON1
	movwf	EEADR
	bsf		EECON1,RD
	movf	EEDAT,w
	banksel	0

	iorlw	0x00
	skpnz
	clrf	ctrltype

	clrf	iii0
	clrf	iii1
	clrf	iii2
	clrf	iii3
	goto	endkbd

nokbd2
	movlw	.4
	xorwf	kbd,w
	skpz
	goto	nokbd4
	; key '4' pressed, decrement temperature by 10 degrees
	movlw	high(mintemp)
	subwf	target_hi,w
	skpz
	goto	tmpdec1
	movlw	low(mintemp)
	subwf	target_lo,w
	skpnz
	goto	endkbd
tmpdec1
	movlw	low(-.100)
	addwf	target_lo,f
	skpnc
	incf	target_hi,f
	movlw	high(-.100)
	addwf	target_hi,f
	goto	endkbd

nokbd4
	movlw	.8
	xorwf	kbd,w
	skpz
	goto	nokbd8
	; key '8' pressed, increment temperature by 10 degrees
	movlw	high(maxtemp)
	subwf	target_hi,w
	skpz
	goto	tmpinc1
	movlw	low(maxtemp)
	subwf	target_lo,w
	skpnz
	goto	endkbd
tmpinc1
	movlw	low(.100)
	addwf	target_lo,f
	skpnc
	incf	target_hi,f
	movlw	high(.100)
	addwf	target_hi,f
	goto	endkbd

nokbd8
	goto	endkbd

; keyboard in setup mode
nomode1
	movlw	.1
	xorwf	kbd,w
	skpz
	goto	nokbd1_sleep

	; key 1 pressed in setup mode
	movlw	.0
	movwf	mode
	goto	endkbd

endkbd
	movf	kbd,w
	movwf	okbd
	goto	mainloop

; subroutines

; truncate REGA to signed 16 bit
trunc16
	call	trunc24

	rlf		REGA1,w
	rlf		REGA2,w
	skpc
	goto	noneg_A16
	xorlw	0xff
	skpnz
	goto	end_A16
	clrf	REGA0
	movlw	0x80
	movwf	REGA1
	movlw	0xFF
	movwf	REGA2
	goto	end_A16

noneg_A16
	iorlw	0x00
	skpnz
	goto	end_A16
	movlw	0xFF
	movwf	REGA0
	movlw	0x7F
	movwf	REGA1
	clrf	REGA2

end_A16
	return
	

; truncate REGA to signed 24 bit
trunc24
	rlf		REGA2,w
	rlf		REGA3,w
	skpc
	goto	noneg_A24
	xorlw	0xff
	skpnz
	goto	end_A24
	clrf	REGA0
	clrf	REGA1
	movlw	0x80
	movwf	REGA2
	movlw	0xFF
	movwf	REGA3
	goto	end_A24

noneg_A24
	iorlw	0x00
	skpnz
	goto	end_A24
	movlw	0xFF
	movwf	REGA0
	movwf	REGA1
	movlw	0x7F
	movwf	REGA2
	clrf	REGA3

end_A24
	return

SETHEAT
	rrf		heat_lo,f
	rrf		heat_lo,w
	rlf		heat_lo,f
	andlw	B'00110000'
	iorlw	PWM_configword

	banksel	CCP1CON
	movwf	CCP1CON

	banksel	heat_lo
	movf	heat_hi,w
	banksel	CCPR1L
	movwf	CCPR1L
	banksel	heat_lo
	return

writelcd
	movwf	tmp
	movlw	DIGIT1
	movwf	FSR

	movlw	.10
	movwf	tcnt
	movf	tmp,w
	sublw	.10
	movwf	tcnt1
	skpnz
	goto	wr_loop
	
wr_l1
	movf	INDF,w
	skpz
	goto	wr_loop
	incf	FSR,f
	decf	tcnt,f
	decfsz	tcnt1,f
	goto	wr_l1

wr_loop
	movf	INDF,w
	addlw	'0'
	call	LCD_SEND_CHAR
	incf	FSR,f
	decfsz	tcnt,f
	goto	wr_loop

	return

; copy REGA to sss
rega_sss
	movf	REGA0,w
	movwf	sss0
	movf	REGA1,w
	movwf	sss1
	movf	REGA2,w
	movwf	sss2
	movf	REGA3,w
	movwf	sss3
	return

; copy sss to REGA
sss_rega
	movf	sss0,w
	movwf	REGA0
	movf	sss1,w
	movwf	REGA1
	movf	sss2,w
	movwf	REGA2
	movf	sss3,w
	movwf	REGA3
	return

; copy REGB to sss
regb_sss
	movf	REGB0,w
	movwf	sss0
	movf	REGB1,w
	movwf	sss1
	movf	REGB2,w
	movwf	sss2
	movf	REGB3,w
	movwf	sss3
	return

; copy sss to REGB
sss_regb
	movf	sss0,w
	movwf	REGB0
	movf	sss1,w
	movwf	REGB1
	movf	sss2,w
	movwf	REGB2
	movf	sss3,w
	movwf	REGB3
	return

; measuring temperature of soldering iron
gettemp
	banksel	ADCON0
	BSF		ADCON0,GO	;Start conversion

waitadcr
	BTFSC	ADCON0,GO	;Is conversion done?
	GOTO	waitadcr	;No, test again

	BANKSEL	ADRESL
	movf	ADRESL,w
	BANKSEL	res_lo
	movwf	res_lo
	BANKSEL	ADRESH
	movf	ADRESH,w
	BANKSEL	res_hi
	movwf	res_hi

;digital filtering input value
;xs = xv0+xv1+{next input}			; G x value, 16 bit
;
;	{next input} -> rega
	movf	res_lo,w
	movwf	REGA0
	movf	res_hi,w
	movwf	REGA1
	clrf	REGA2
	clrf	REGA3
;	xv0 -> regb
	movf	xv0l,w
	movwf	REGB0
	movf	xv0h,w
	movwf	REGB1
	clrf	REGB2
	clrf	REGB3
;	add
	call	add
;	xv1 -> regb
	movf	xv1l,w
	movwf	REGB0
	movf	xv1h,w
	movwf	REGB1
;	add
	call	add
;
;xv0 = xv1/2					; G x value, 16 bit
;
;	shr xv1 -> xv0
	clrc
	rrf		xv1h,w
	movwf	xv0h
	rrf		xv1l,w
	movwf	xv0l
;
;xv1 = {next input}*2				; G x value, 16 bit
;
;	shl {next input} -> xv1
	clrc
	rlf		res_lo,w
	movwf	xv1l
	rlf		res_hi,w
	movwf	xv1h
;
;ys = int(20050*xs -641*yv1 + 1561*yv2)		; 1 000 000 x value, 32 bit
;	/1000					; 1 000 x value, 24 bit
;
;	20050 -> regb
	unpack32	D'20050',REGB0
;	mul
	call	multiply
;	rega -> sss
	call	rega_sss
;
;	-641 -> rega
	unpack32	-D'641',REGA0
;	yv1 -> regb
	movf	yv1l,w
	movwf	REGB0
	movf	yv1m,w
	movwf	REGB1
	movf	yv1h,w
	movwf	REGB2
	clrf	REGB3
	btfsc	REGB2,7		; sign extension to 32 bit
	decf	REGB3,f
;	mul
	call	multiply
;
;	sss -> regb
	call	sss_regb
;	add
	call	add
;	rega -> sss
	call	rega_sss
;
;	1561 -> rega
	unpack32	D'1561',REGA0
;	yv2 -> regb
	movf	yv2l,w
	movwf	REGB0
	movf	yv2m,w
	movwf	REGB1
	movf	yv2h,w
	movwf	REGB2
	clrf	REGB3
	btfsc	REGB2,7		; sign extension to 32 bit
	decf	REGB3,f
;	mul
	call	multiply
;
;	sss -> regb
	call	sss_regb
;	add
	call	add
;
;	1000 -> regb
	unpack32	D'1000',REGB0
;	div
	call	divide
;
;yv1 = yv2					; 1 000 x value, 24 bit
;yv2 = ys					; 1 000 x value, 24 bit
;
;	yv2 -> yv1
	movf	yv2l,w
	movwf	yv1l
	movf	yv2m,w
	movwf	yv1m
	movf	yv2h,w
	movwf	yv1h
;	rega -> yv2
	movf	REGA0,w
	movwf	yv2l
	movf	REGA1,w
	movwf	yv2m
	movf	REGA2,w
	movwf	yv2h
;
;{next output} = yv[2]/1000 - degree resolution
;{next output} = yv[2]/100 - 1/10 degree resolution
;
;	100 -> regb
	unpack32	D'100',REGB0
;	div
	call	divide
;	rega -> {next output}
	movf	REGA0,w
	movwf	res_lo
	movf	REGA1,w
	movwf	res_hi
;;
; xv0,xv1 are 16-bit regs
; sss is a 32-bit reg
; yv1, yv2 are 24-bit regs
;

; filtered A/D result is in (res_hi,res_lo)

; calculating temperature from filtered A/D result
;
;	multiply by 51165
	unpack32	D'51165',REGB0
	call	multiply
;	divide by 100000
	unpack32	D'100000',REGB0
	call	divide
;	add  500 as correction temperature - 1/10 degree resolution
	unpack32	D'500',REGB0
	call	add

	btfss	REGA3,7
	return

	clrf	REGA0
	clrf	REGA1
	clrf	REGA2
	clrf	REGA3
	return

; RS232_send - char output to RS232
RS232_send
	movwf	tx_byte
	bsf		RS232_TX
w_rs232_1
	btfsc	RS232_TX
	goto	w_rs232_1
	return

; RS232_txtout - text output to RS232
RS232_txtout
	call	gettabtxt
	iorlw	0
	skpnz
	return

	call	RS232_send
	incfsz	txtptr_lo,f
	goto	RS232_txtout
	incf	txtptr_hi,f
	goto	RS232_txtout

; RS232_sendnum - numeric output to RS232
RS232_sendnum
	movwf	tcnt
	rlf		DSIGN,w
	andlw	.2
	addlw	'+'
	call	RS232_send
rs232_sn1
	movf	INDF,w
	addlw	'0'
	call	RS232_send
	incf	FSR,f
	decfsz	tcnt,f
	goto	rs232_sn1
	return

; txtout - text output to lcd
txtout
	call	gettabtxt
	iorlw	0
	skpnz
	return

	call	LCD_SEND_CHAR
	incfsz	txtptr_lo,f
	goto	txtout
	incf	txtptr_hi,f
	goto	txtout

gettabtxt
	movf	txtptr_hi,w
	movwf	PCLATH
	movf	txtptr_lo,w
	movwf	PCL

;*** End of program

;*** Text area
txt_RS232_init
	dt	0x0d,0x0a,"PIC_SOLD v
	dt	'0'+vermajor,".",'0'+verminor,0x0d,0x0a,0

txt_hello
	dt	"PIC_SOLD v",'0'+vermajor,".",'0'+verminor,0

txt_sleep
	dt	"--- SLEEPING ---",0

txt_setup
	dt	"-- SETUP MENU --",0

txt_mode
	dt	"  Mode: ",0

	END                     ; Directive 'end of program'

