;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; onhold (DARMICON prototype), with 16x2 character LCD display
; Revision 2
; A Simple 8051 based ATAPI CDROM audio controller
; Copyright (C) 2000,2002 Jason Nunn <jsno@arafura.net.au>
; August 2000
;
; This software comes under the GNU public license agreement. See the COPYING
; file in this distribution.
;
; If you use this code (or parts) of it, then please credit me.
;
; This program is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; This program is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
;   WWW: http//jsno.arafuraconnect.com.au
;
; If you use this code (or parts) of it, then please credit me.
;
; coded to parse by LA51 assembler (ASM51)
; current target: - DARMICON development board (SAB80C537 @ 12Mhz)
;                 (by Felino MOLINA & Howard PULLEN- www.ntu.edu.au)
;                 - 16x2 LM016L compatable LCD display
;
; ====================================================================
;
; designed for one device only (set to master)
;
; design references:
;
; - ATA/ATAPI-5 (T13/1321D R3) (www.t10.org)
; - ATA packet interface for CDROM's (SFF-8020i) (fission.dt.wdc.com)
;
; other references used:
;
; - linux 2.2.x source code (/usr/src/linux/drivers/ide*)
;   (had to be referred to seens that the SFF-8020i spec is so shitty)
;
; - ATAPI references & Black magic in ATA/ATAPI
;   by Constantine Sapuntzakis <csapuntz@stanford.edu>
;   http://www.stanford.edu/~csapuntz/ide.html
;   http://www.stanford.edu/~csapuntz/blackmagic.html
;
; - 8052 Basic project page (IDE logger section)
;   by Sergey Zorin <zorinsn@rs-pribor.ru>
;   http://www.nomad.ee/micros/8052bas.html
;   (nb/ very very helpful)
;
; - General ATAPI info (including some source code)
;   Hale Landis's ATAPI pages http://www.ata-atapi.com
;   (a good introduction to it all, as i knew nothing when i first started
;   this project)
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
CODE_BASE = 8000h
STACK_START = 7fh

;;delete me;;;;;;;;;;;;;;;;;;;;;;;;;;
prhex           code  0142h
hexasc          code  0136h
writechar       code  0133h
lprint          code  0148h
prcrlf          code  013fh
;;delete me;;;;;;;;;;;;;;;;;;;;;;;;;;

;ctrl bus
;
;  7   6   5   4   3   2   1   0
;+DA2+DA1+DA0+CS1+CS0+---+---+---+
;| X | X | X | X | X |   |   |   |
;+---+---+---+---+---+---+---+---+
;
ATA_ADDR        data  p1

ATA_nDIOR       bit   p3.0 ;read strobe
ATA_nDIOW       bit   p3.1 ;write strobe
ATA_nRESET      bit   p3.2
ATA_INTRQ       bit   p3.3 ;irq

;data bus
ATA_DATA_LS     data  p5
ATA_DATA_MS     data  p4

; variable data
;
packet_size     data  50h
start_track     data  51h
end_track       data  52h
current_track   data  53h
next_track      data  54h
toc_track       data  55h
toc_M           data  56h
toc_S           data  57h
toc_F           data  58h
toc_M_next      data  59h
toc_S_next      data  5ah
toc_F_next      data  5bh
start_M         data  5ch
start_S         data  5dh
start_F         data  5eh
end_M           data  5fh
end_S           data  60h
end_F           data  61h
next_M          data  62h
next_S          data  63h
next_F          data  64h
audio_status    data  65h

packet_tab      idata 70h
packet_cmd      data  70h
packet_1        data  71h
packet_2        data  72h
packet_3        data  73h
packet_4        data  74h
packet_5        data  75h
packet_6        data  76h
packet_7        data  77h
packet_8        data  78h
packet_9        data  79h
packet_10       data  7ah
packet_11       data  7bh
packet_12       data  7ch
packet_13       data  7dh
packet_14       data  7eh
packet_15       data  7fh

; LCD ctrl and data lines
;
; nb/ some of the lines on by development board have had the 
; "smoke let out of them" (ie- blown up)... hence the uncontingious
; choice of lines used ;)
;
LCD_E           bit p3.5
LCD_RS          bit p1.7
LCD_174_DATA    bit p1.7
LCD_174_CLK     bit p1.6

org CODE_BASE
  ljmp  start

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; delay routines
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
delay_100000us:
  mov   r7,#0c8h
delay_100000us_l0:
  mov   r6,#0f8h
delay_100000us_l1:
  djnz  r6,delay_100000us_l1
  nop
  djnz  r7,delay_100000us_l0
  ret

;r5 = seconds
delay:
  acall delay_100000us
  acall delay_100000us
  acall delay_100000us
  acall delay_100000us
  acall delay_100000us
  acall delay_100000us
  acall delay_100000us
  acall delay_100000us
  acall delay_100000us
  acall delay_100000us
  djnz  r5,delay
  ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; LCD routines
;
; all latencies have been exaggerated.. as there are no read ops due to 
; the limitations of the electronics to keep things nice, simple and cheap.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
lcd_load_174:
  mov  r7,#8
lcd_load_174_l1:
  rlc  a
  mov  LCD_174_DATA,c
  setb LCD_174_CLK
  nop
  nop
  nop
  nop
  nop
  nop
  nop
  nop
  clr  LCD_174_CLK
  djnz r7,lcd_load_174_l1
  ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
lcd_write_cmd:
  acall lcd_load_174
  clr   LCD_RS
  setb  LCD_E
  clr   LCD_E
  mov   r7,#0
lcd_write_cmd_l:
  nop
  nop
  nop
  nop
  nop
  nop
  nop
  nop
  djnz  r7,lcd_write_cmd_l
  ret

lcd_write_data:
  acall lcd_load_174
  setb  LCD_RS
  setb  LCD_E
  clr   LCD_E
  mov   r7,#0
lcd_write_data_l:
  nop
  nop
  nop
  nop
  nop
  nop
  nop
  nop
  djnz  r7,lcd_write_data_l
  ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
lcd_clear:
  mov   a,#00000001b
  acall lcd_write_cmd
  ret

lcd_home1:
  mov   a,#10000000b
  acall lcd_write_cmd
  ret

lcd_home2:
  mov   a,#11000000b
  acall lcd_write_cmd
  ret

lcd_init:
  clr   LCD_RS
  clr   LCD_174_CLK
  clr   LCD_E

  mov   a,#00110000b
  acall lcd_load_174

  setb  LCD_E
  clr   LCD_E
  setb  LCD_E
  clr   LCD_E
  mov   r5,#1
  acall delay
  setb  LCD_E
  clr   LCD_E
  setb  LCD_E
  clr   LCD_E
  mov   r7,#0
  djnz  r7,$
  setb  LCD_E
  clr   LCD_E
  setb  LCD_E
  clr   LCD_E

; set for 8 bits & 2 line display
;
  mov   a,#00111000b
  acall lcd_write_cmd

  mov   r5,#1
  acall delay

; display on
;
  mov   a,#00001100b
  acall lcd_write_cmd

;  Entry Mode Set- Inc. address + right shift cursor on write
;
  mov   a,#00000110b
  acall lcd_write_cmd

  acall lcd_clear

  ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
lcd_lprint:
  pop  dph
  pop  dpl
lcd_lprint_l0:
  clr  a
  movc a,@a+dptr
  inc  dptr
  jnz  lcd_lprint_l1
  jmp  @a+dptr
lcd_lprint_l1:
  acall lcd_write_data
  sjmp  lcd_lprint_l0

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; atomic routines
;
; Nb/ code assumes 1us clock cycle (12Mhz xtal clock)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
read_cycle:
  mov   ATA_DATA_LS,#0ffh
  mov   ATA_DATA_MS,#0ffh
  clr   ATA_nDIOR
  mov   a,ATA_DATA_LS
  mov   b,ATA_DATA_MS
  setb  ATA_nDIOR
  ret

read_cycle_ls:
  mov   ATA_DATA_LS,#0ffh
  clr   ATA_nDIOR
  mov   a,ATA_DATA_LS
  setb  ATA_nDIOR
  ret

write_cycle:
  mov   ATA_DATA_LS,a
  mov   ATA_DATA_MS,b
  clr   ATA_nDIOW
  nop
  setb  ATA_nDIOW
  ret

write_cycle_ls:
  mov   ATA_DATA_LS,a
  clr   ATA_nDIOW
  nop
  setb  ATA_nDIOW
  ret

wait_irq:
  setb  ATA_INTRQ
  jnb   ATA_INTRQ,wait_irq
  ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; cylinder register
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
get_cyl_high_register:
  mov   ATA_ADDR,#10110000b
  acall read_cycle_ls
  ret

get_cyl_low_register:
  mov   ATA_ADDR,#10010000b
  acall read_cycle_ls
  ret

set_cyl_high_register:
  mov   ATA_ADDR,#10110000b
  acall write_cycle_ls
  ret

set_cyl_low_register:
  mov   ATA_ADDR,#10010000b
  acall write_cycle_ls
  ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; a = cmd
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
set_cmd_register:
  mov   ATA_ADDR,#11110000b
  acall write_cycle_ls
  ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; data register
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; ret: a = ls  b = ms
get_data:
  mov   ATA_ADDR,#00010000b
  acall read_cycle
  ret

; a = ls  b = ms
set_data:
  mov   ATA_ADDR,#00010000b
  acall write_cycle
  ret

; a = ls
set_data_ls:
  mov   ATA_ADDR,#00010000b
  acall write_cycle_ls
  ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; device/head register
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
get_devhead_register:
  mov   ATA_ADDR,#11010000b
  acall read_cycle_ls
  ret

set_devhead_register:
  mov   ATA_ADDR,#11010000b
  acall write_cycle_ls
  ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; error register
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
ERROR_REG_AMNF  bit acc.0
ERROR_REG_TK0NF bit acc.1
ERROR_REG_ABRT  bit acc.2
ERROR_REG_MCR   bit acc.3
ERROR_REG_IDNF  bit acc.4
ERROR_REG_MC    bit acc.5
ERROR_REG_UNC   bit acc.6
ERROR_REG_RES   bit acc.7

get_error_register:
  mov   ATA_ADDR,#00110000b
  acall read_cycle_ls
  ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; sector register's
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
get_sector_c_register:
  mov   ATA_ADDR,#01010000b
  acall read_cycle_ls
  ret

get_sector_n_register:
  mov   ATA_ADDR,#01110000b
  acall read_cycle_ls
  ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; status register
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
STATUS_REG_ERR  bit acc.0
STATUS_REG_IDX  bit acc.1
STATUS_REG_CORR bit acc.2
STATUS_REG_DRQ  bit acc.3
STATUS_REG_DSC  bit acc.4
STATUS_REG_DF   bit acc.5
STATUS_REG_DRDY bit acc.6
STATUS_REG_BSY  bit acc.7

get_status_register:
  mov   ATA_ADDR,#11110000b
  acall read_cycle_ls
  ret

bsy_wait:
  nop
  nop
  nop
  nop
  acall get_status_register
  jb    STATUS_REG_BSY,bsy_wait
  ret

bsy_drdy_wait:
  nop
  nop
  nop
  nop
  acall delay_100000us
  acall get_status_register
  jb    STATUS_REG_BSY,bsy_drdy_wait
  jnb   STATUS_REG_DRDY,bsy_drdy_wait
  ret

drq_wait:
  nop
  nop
  nop
  nop
  acall get_status_register
  jb    STATUS_REG_BSY,drq_wait
  jnb   STATUS_REG_DRQ,drq_wait
  ret

ndrq_wait:
  nop
  nop
  nop
  nop
  acall get_status_register
  jb    STATUS_REG_BSY,drq_wait
  jb    STATUS_REG_DRQ,drq_wait
  ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; main routines
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; misc
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; prints a decimal number
; a = val
print_dec:
  push  0
  push  b
  mov   dptr,#print_dec_tab
  mov   r0,#3
print_dec_l:
  push  acc
  mov   a,r0
  movc  a,@a+dptr
  mov   b,a
  pop   acc
  div   ab
  lcall hexasc
  lcall writechar
  mov   a,b
  djnz  r0,print_dec_l
  pop   b
  pop   0
  ret
print_dec_tab:
  db    0,1,10,100

; prints a decimal number
; a = val
print_dec_lcd:
  push  0
  push  b
  mov   dptr,#print_dec_tab
  mov   r0,#3
print_dec_lcd_l:
  push  acc
  mov   a,r0
  movc  a,@a+dptr
  mov   b,a
  pop   acc
  div   ab
  lcall hexasc
  lcall lcd_write_data
  mov   a,b
  djnz  r0,print_dec_lcd_l
  pop   b
  pop   0
  ret

print_done:
  lcall lprint
  db    'done.',13,10,0
  ret

print_yes:
  lcall lprint
  db    'yes.',13,10,0
  ret

print_no:
  lcall lprint
  db    'no.',13,10,0
  ret

; clocks data out of buffer until all done (DRQ goes low)
;
skip_rest_of_packet:
  acall get_status_register
  jnb   STATUS_REG_DRQ,skip_rest_of_packet_d
  acall get_data
  sjmp  skip_rest_of_packet
skip_rest_of_packet_d:
  ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; MSF routines
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
get_MSF_field:
  acall get_data
  mov   toc_M,b
  acall get_data
  mov   toc_S,a
  mov   toc_F,b
  ret

print_MSF_field:
  mov   a,toc_M
  lcall print_dec
  mov   a,#' '
  lcall writechar

  mov   a,toc_S
  lcall print_dec
  mov   a,#' '
  lcall writechar

  mov   a,toc_F
  lcall print_dec
  lcall prcrlf
  ret

print_MSF_field_lcd:
  mov   a,toc_M
  lcall print_dec
  mov   a,toc_M
  lcall print_dec_lcd
  mov   a,#' '
  lcall writechar
  mov   a,#' '
  lcall lcd_write_data

  mov   a,toc_S
  lcall print_dec
  mov   a,toc_S
  lcall print_dec_lcd
  mov   a,#' '
  lcall writechar
  mov   a,#' '
  lcall lcd_write_data

  mov   a,toc_F
  lcall print_dec
  mov   a,toc_F
  lcall print_dec_lcd
  lcall prcrlf
  ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; init ATA device routines..
;
; also detects the device during the process of initing it.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
init_ata_device:
  lcall lprint
  db    'Initing device...',13,10,0

  lcall lcd_clear
  lcall lcd_lprint
  db    'Initing',0

;*
;* i've noticed that when using this on one of my CDROM devices (a swank
;* little DVD unit-E608 (manufactured by ASUSTek Computer Inc) that the 10k
;* pulldown resister (recommended in T13/1321D R3- see P11) pulls DD7 down
;* too well, hence causing data loss on this line. i've therefore had to
;* remove the resister from the controller design and comment out this
;* code. anyone else noticed the same thing????.
;*
;* Nb/ try uncommenting the code, and put a 10k pulldown on DD7 and see
;* what happens.
;*
;*; sense that a device is connected
;*; if device disconnected, then DD7 is pulled down
;*;
;*  lcall lprint
;*  db    'Device connected?..',0
;*
;*  mov   ATA_DATA_LS,#0ffh
;*  mov   a,ATA_DATA_LS
;*  jb    acc.7,init_device_c0
;*  acall print_no
;*  sjmp  $
;*init_device_c0:
;*  acall print_yes

; do hard reset..
;
; nb/ you have a wait a bit after pulling the reset back high. if you
; don't, then bsy flag will falsey pass. i haven't figured out a way
; yet to determine when the device controller is busy during hard reset
;
  lcall lprint
  db    'Hard Reset..',0

  clr   ATA_nRESET
  acall delay_100000us
  setb  ATA_nRESET
  mov   r5,#2
  acall delay
  acall bsy_wait
  acall print_done

;select device 0
;
;the device will now respond to commands now
;
  lcall lprint
  db    'selecting device 0..',0
  acall get_devhead_register
  clr   acc.4                  ;set device to 0
  acall set_devhead_register
  acall print_done

; test for device signature to determine if it's 'packet command' device
; if it's not a packet command, then it's probably a harddisk or
; something, and we can't continue.
;
; we have to check this inorder to know if we can issue the 'identify
; packet device' ATA command.
;
  lcall lprint
  db    'ATA Packet device?..',0

  acall  bsy_wait

  acall get_cyl_low_register
  cjne  a,#14h,init_device_s_error
  acall get_cyl_high_register
  cjne  a,#0ebh,init_device_s_error

;*  acall get_sector_c_register        ;code not used due to inconsistencies
;*  cjne  a,#01h,init_device_s_error   ;among devices.
;*  acall get_sector_n_register        ;(refer to Black magic in ATA/ATAPI)
;*  cjne  a,#01h,init_device_s_error
;*  acall get_devhead_register
;*  jz    init_device_s_done
;*  cjne  a,#10h,init_device_s_error

  sjmp  init_device_s_done
init_device_s_error:
  acall print_no
  sjmp  $
init_device_s_done:
  acall print_yes

; running self diagonistics..
; nb/ this is the first ATA command we send
;
  lcall lprint
  db    'Passed ATA self diagnositics?..',0

  mov   a,#90h
  acall set_cmd_register

  acall bsy_wait
  acall get_error_register
  jb    acc.0,init_device_c1
  acall print_no
  sjmp  $
init_device_c1:
  acall print_yes
  ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; ATAPI identify packet device
;
; this ATA/ATAPI command is mandatory for CDROM (and like) devices, as it
; will enable the DRDY flag. without this command, we won't be able to
; issue any commands (ATA or packet).
;
; nb/ older ATA standards, DRDY goes high immediately after BSY goes low
; during reset. .. not so for CDROM devices ;)
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
skip_device_fields:
  acall get_data
  djnz  r7,skip_device_fields
  ret

print_device_ansi:
  acall get_data
  xch   a,b
  lcall writechar
  mov   a,b
  lcall writechar
  djnz  r6,print_device_ansi
  ret

print_device_ansi_lcd:
  acall get_data
  xch   a,b
  lcall writechar
  lcall lcd_write_data
  mov   a,b
  lcall writechar
  lcall lcd_write_data
  djnz  r6,print_device_ansi_lcd
  ret

identify_packet_device:
  lcall lprint
  db    'Identifying ATAPI device:',13,10,0

  lcall lcd_clear
  lcall lcd_lprint
  db    'Identifying',0
  lcall lcd_home2

;identify device cmd
;
  mov   a,#2h                 ;set byte count registers
  acall set_cyl_high_register ;maximum transfer length = 0200h
  clr   a                     ;nb/ on the few devices i've tested, this
  acall set_cyl_low_register  ;cmd automatically sets these registers.
                              ;but i'll set it for consistency ;)
  mov   a,#0a1h
  acall set_cmd_register
  acall wait_irq

; get general config word
;
  acall get_data
  jnb   b.7,identify_packet_device_c0
  jb    b.6,identify_packet_device_c0
  lcall lprint
  db    9,'Unit is an ATAPI device',13,10,0
identify_packet_device_c0:

  jnb   acc.7,identify_packet_device_c1
  lcall lprint
  db    9,'Unit has removable media',13,10,0
identify_packet_device_c1:

  lcall lprint
  db    9,'packet size (words): ',0
  jb    acc.0,identify_packet_device_c2
  jb    acc.1,identify_packet_device_c2
  mov   packet_size,#6
  sjmp  identify_packet_device_c3
identify_packet_device_c2:

  jb    acc.0,identify_packet_device_c3
  jnb   acc.1,identify_packet_device_c3
  mov   packet_size,#8
identify_packet_device_c3:
  mov   a,packet_size
  lcall prhex
  lcall prcrlf

  lcall lprint
  db    9,'type:',9,0
  mov   a,b
  anl   a,#0fh
  lcall prhex
  lcall prcrlf

; skip next 9 words
;
  mov   r7,#9
  acall skip_device_fields

; get serial number
;
  lcall lprint
  db    9,'serial:',9,0
  mov   r6,#10
  acall print_device_ansi
  lcall prcrlf

; skip next 3 words
;
  mov   r7,#3
  acall skip_device_fields

; get firmware revision
;
  lcall lprint
  db    9,'rev:',9,0
  mov   r6,#4
  acall print_device_ansi
  lcall prcrlf

; get model number
;
  lcall lprint
  db    9,'model:',9,0
  mov   r6,#20
  acall print_device_ansi_lcd
  lcall prcrlf

; skip next 33 words
;
  mov   r7,#33
  acall skip_device_fields

; get major version
;
  lcall lprint
  db    9,'major:',9,0
  acall get_data  
  xch   a,b
  lcall prhex
  mov   a,b
  lcall prhex
  lcall prcrlf

; get minor version
;
  lcall lprint
  db    9,'minor:',9,0
  acall get_data  
  xch   a,b
  lcall prhex
  mov   a,b
  lcall prhex
  lcall prcrlf

; skip next 46 words
;
  mov   r7,#46
  acall skip_device_fields

; get security word
;
  lcall lprint
  db    9,'secure:',9,0
  acall get_data
  xch   a,b
  lcall prhex
  mov   a,b
  lcall prhex
  lcall prcrlf

  acall skip_rest_of_packet

  acall bsy_drdy_wait
  lcall lprint
  db    'ATAPI device now has READY status.',13,10,0
  ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
do_packet_cmd:

; clr packet table
;
  mov   r0,#packet_tab
  mov   r7,#16
init_pck_l:
  mov   @r0,#0
  inc   r0
  djnz  r7,init_pck_l

; send packet command
;
  acall ndrq_wait
  mov   a,#0a0h
  acall set_cmd_register
  ret


send_packet:
  acall drq_wait

  mov   r0,#packet_tab
  mov   r7,packet_size
send_packet_l:
  mov   a,@r0
  inc   r0
  push  acc
  mov   a,@r0
  inc   r0
  mov   b,a
  pop   acc
  acall set_data
  djnz  r7,send_packet_l

  acall wait_irq
  acall get_status_register
  ret

print_packet_error:
  lcall lprint
  db    '...Failed: ',0
  acall get_error_register
  lcall prhex
  lcall prcrlf
  ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; test unit ready ATAPI command
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
test_unit_ready_cmd:
  acall do_packet_cmd
  acall send_packet
  jnb   STATUS_REG_ERR,test_unit_ready_d
  clr   c
  ret
test_unit_ready_d:
  setb  c
  ret

wait_for_ready_state:
  lcall lprint
  db    'Waiting for Unit READY state (insert CD. if not inserted).',13,10,0
  lcall lcd_clear
  lcall lcd_lprint
  db    'Insert CD',0
  lcall lcd_home2
  lcall lcd_lprint
  db    'if not inserted',0

wait_for_ready_state_restart:
  mov   r2,#2
wait_for_ready_state_l:
  acall delay_100000us
  acall delay_100000us
  acall delay_100000us
  acall delay_100000us
  acall delay_100000us
  acall test_unit_ready_cmd
  jnc   wait_for_ready_state_restart
  dec   r2
  cjne  r2,#0,wait_for_ready_state_l
  ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; read sub channel cmd
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
read_subchan_cmd:
  lcall lprint
  db    'Read Sub-Channel cmd..',13,10,0

  mov   a,#0ffh               ;set byte counter register
  acall set_cyl_low_register  ;(set it to full on.. it doesn't matter coz
  acall set_cyl_high_register ;DRQ controls the feed)

  acall do_packet_cmd
  mov   packet_cmd,#42h
  mov   packet_1,#00000010b  ;MSF=1
  mov   packet_2,#01000000b  ;SubQ=1
  mov   packet_3,#01h        ;format=01  (current position)
  mov   packet_7,#0ffh       ;allocation length
  mov   packet_8,#0ffh       ;(set it to full on)
  acall send_packet
  jb    STATUS_REG_ERR,read_subchan_cmd_e
  jb    STATUS_REG_DRQ,read_subchan_cmd_r
  ret
read_subchan_cmd_e:
  mov   audio_status,#0ffh
  acall print_packet_error
  ret

;read sub-channel header
;
read_subchan_cmd_r:
  acall get_data   ;get audio status
  lcall lprint
  db    9,'Audio Status:',9,0
  mov   audio_status,b
  mov   a,b
  lcall prhex
  lcall prcrlf

  acall get_data   ;sub channel data length (will be zero).. ignore
  acall get_data   ;format code, ADR, cntl (ignore)

  lcall lprint     ;track number, index number
  db    9,'Track number:',9,0
  acall get_data
  mov   current_track,a
  lcall print_dec
  lcall prcrlf

  mov   a,current_track
  inc   a
  cjne  a,end_track,calc_next_track_1
  sjmp  calc_next_track_2
calc_next_track_1:
  jc    calc_next_track_2
  mov   a,start_track
calc_next_track_2:
  mov   next_track,a

  lcall lprint
  db    9,'Next track:',9,0
  mov   a,next_track
  lcall print_dec
  lcall prcrlf

  lcall lcd_home1
  lcall lcd_lprint
  db    'Trk:',0
  mov   a,current_track
  lcall print_dec_lcd

  lcall lcd_home2
  lcall lcd_lprint
  db    'Pos:',0

  lcall lprint     ;absolete CDROM address
  db    9,'Abs address:',9,0
  acall get_MSF_field
  acall print_MSF_field

  lcall lprint     ;relative CDROM address
  db    9,'Rel address:',9,0
  acall get_MSF_field
  acall print_MSF_field_lcd

  acall skip_rest_of_packet
  ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Table of Contents command
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
read_toc_cmd:
  lcall lprint
  db    27,'[2J',0

  lcall lprint
  db    'Read TOC cmd..',13,10,0

  lcall lcd_clear
  lcall lcd_lprint
  db    'Reading TOC',0

  mov   a,#0ffh               ;set byte counter register
  acall set_cyl_low_register  ;(set it to full on.. it doesn't matter coz
  acall set_cyl_high_register ;DRQ controls the feed)

  acall do_packet_cmd
  mov   packet_cmd,#43h
  mov   packet_1,#00000010b   ;MSF=1
  mov   packet_7,#0ffh        ;allocation length
  mov   packet_8,#0ffh        ;(again, set it to full on)
  acall send_packet
  jb    STATUS_REG_ERR,read_toc_cmd_e
  jb    STATUS_REG_DRQ,read_toc_cmd_r
  setb  c
  ret
read_toc_cmd_e:
  acall print_packet_error
  clr   c
  ret

; read header
;
read_toc_cmd_r:
  acall get_data       ;data length

  acall get_data       ;get track limits
  mov   start_track,a
  mov   end_track,b

  lcall lprint
  db    9,'Start Trk:',9,0
  mov   a,start_track
  lcall print_dec
  lcall prcrlf

  lcall lprint
  db    9,'End Trk:',9,0
  mov   a,end_track
  lcall print_dec
  lcall prcrlf

  lcall lcd_home2
  lcall lcd_lprint
  db    'TrkRange:',0
  mov   a,start_track
  lcall print_dec_lcd
  lcall lcd_lprint
  db    '-',0
  mov   a,end_track
  lcall print_dec_lcd

; read track descriptors
;
  lcall lprint
  db    9,'Tracks-',13,10
  db    9,9,'ADR/Ctl',9,'Trk',9,'MSF',13,10
  db    0
read_toc_cmd_l2:
  lcall lprint
  db    9,9,0

  acall get_data   ;get ADR and control fields
  mov   a,b
  lcall prhex
  mov   a,#9
  lcall writechar

  acall get_data   ;get track number
  mov   toc_track,a
  lcall print_dec
  mov   a,#9
  lcall writechar

  acall get_MSF_field
  acall print_MSF_field

; if start track, then record start position
;
  mov   a,toc_track
  cjne  a,start_track,read_toc_cmd_i0
  mov   start_M,toc_M
  mov   start_S,toc_S
  mov   start_F,toc_F
read_toc_cmd_i0:

;if the last track, then record as end position
;
  cjne  a,#0aah,read_toc_cmd_i1
  mov   end_M,toc_M
  mov   end_S,toc_S
  mov   end_F,toc_F
read_toc_cmd_i1:

;if next track, then record next position
;
  cjne  a,next_track,read_toc_cmd_i2
  mov   next_M,toc_M
  mov   next_S,toc_S
  mov   next_F,toc_F
read_toc_cmd_i2:

  acall bsy_wait
  jb    STATUS_REG_DRQ,read_toc_cmd_l2
  setb  c
  ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; play MSF cd cmd
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
play_entire_cd_cmd:
  lcall lprint
  db    'Play Entire CD (Play Audio MSF cmd)..',13,10,0

  lcall lprint
  db    13,10,9,'Start trk:',9,0
  mov   toc_M,start_M
  mov   toc_S,start_S
  mov   toc_F,start_F
  lcall print_MSF_field

  lcall lprint
  db    9,'End trk:',9,0
  mov   toc_M,end_M
  mov   toc_S,end_S
  mov   toc_F,end_F
  lcall print_MSF_field

  lcall lcd_clear
  lcall lcd_lprint
  db    'Playing Trk ',0
  mov   a,next_track
  lcall print_dec_lcd

  lcall do_packet_cmd
  mov   packet_cmd,#47h
  mov   packet_3,start_M
  mov   packet_4,start_S
  mov   packet_5,start_F
  mov   packet_6,end_M
  mov   packet_7,end_S
  mov   packet_8,end_F
  lcall send_packet
  jnb   STATUS_REG_ERR,play_audio_cmd_d
  lcall print_packet_error
  clr   c
  ret
play_audio_cmd_d:
  setb  c
  ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; stop/play command
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
stop_play_cmd:
  lcall lcd_clear
  lcall lprint
  db    'Stop play cmd..',13,10,0

  lcall do_packet_cmd
  mov   packet_cmd,#4eh
  lcall send_packet
  jnb   STATUS_REG_ERR,stop_play_cmd_d
  lcall print_packet_error
stop_play_cmd_d:
  ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; program entry point
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
start:
  mov   sp,#STACK_START
  mov   next_track,#1
  mov   toc_M_next,#0
  mov   toc_S_next,#0
  mov   toc_F_next,#0

  lcall prcrlf
  lcall lprint
  db    'onhold LCD (DARMICON prototype) R2',13,10
  db    'Copyright (C) 2000, 2002 Jason Nunn <jsno@arafura.net.au>',13,10
  db    13,10
  db    'This software comes under the GNU public license agreement.',13,10
  db    'See the COPYING file in this distribution.',13,10
  db    13,10
  db    'If you use this code (or parts) of it, then please credit me.',13,10
  db    13,10,0

  lcall lprint
  db    'initing LCD display..',0
  lcall lcd_init
  lcall lprint
  db    'done.',13,10,0

  lcall lcd_lprint
  db    '-= Onhold R2 =-',0
  lcall lcd_home2
  lcall lcd_lprint
  db    'By Jason Nunn',0

  mov   r5,#2
  lcall delay
  lcall lcd_clear

start_over:
  lcall init_ata_device
  lcall identify_packet_device
  mov   r5,#1
  lcall delay 

main_loop:
  lcall wait_for_ready_state

  lcall read_toc_cmd
  jnc   main_loop

main_play:
  mov   r5,#2
  lcall delay

  lcall play_entire_cd_cmd
  jnc   main_loop
  mov   r5,#2
  lcall delay

  lcall lprint
  db    27,'[2J',0
  lcall lcd_clear
play_loop:
  lcall lprint
  db    27,'[0;0H',0

  lcall read_subchan_cmd
  mov   a,audio_status
  cjne  a,#11h,play_stopped  ;if at end then loop

  mov   a,toc_M
  cjne  a,toc_M_next,play_loop_end_cmp
  mov   a,toc_S
  cjne  a,toc_S_next,play_loop_end_cmp
  mov   a,toc_F
  cjne  a,toc_F_next,play_loop_end_cmp
  sjmp  play_stopped
play_loop_end_cmp:
  mov   toc_M_next,toc_M
  mov   toc_S_next,toc_S
  mov   toc_F_next,toc_F

  lcall delay_100000us

  mov   p8,#0ffh
  mov   a,p8
  jb    acc.0,play_next_track
  sjmp  play_loop

play_stopped:
  mov   next_track,#1

play_next_track:
  acall stop_play_cmd
  lcall read_toc_cmd
  jnc   main_loop
  mov   start_M,next_M
  mov   start_S,next_S
  mov   start_F,next_F
  sjmp  main_play
end
