/****************************************************************************
 * ONE-TOUCH TURN SIGNAL MODULE
 *
 * Copyright 2008 revtor <e31wiki.org@revtor.be>
 *
 * Version: 2.0
 * Documentation: http://e31wiki.org/wiki/One-touch_turn_signal_module
 * Logic: Atmel ATtiny2313
 *
 * May be copied or modified under the terms of the Creative Commons
 * Attribution-Noncommercial 3.0 Unported License. To view a copy of this
 * license, visit http://creativecommons.org/licenses/by-nc/3.0/
 ****************************************************************************/

#define LOOP_WAIT_TIME_MS           5           // Time to wait between each main process loop (time in milliseconds) [Default: 5]
#define DEBOUNCE_LOOPS              10          // Number of loops to wait for debounce (LOOP_WAIT_TIME_MS * DEBOUNCE_LOOPS = time in milliseconds) [Default: 10]
#define COMFORTBLINKS_SHORT         3           // Number of blinks in short mode [Default: 3]
#define COMFORTBLINKS_MEDIUM        4           // Number of blinks in medium mode [Default: 4]
#define COMFORTBLINKS_LONG          5           // Number of blinks in long mode [Default: 5]
#define CANCELCOMFORT_LOOPS_SHORT   100         // Number of loops before comfort mode is disabled when the turn signal lever is held down in short mode (LOOP_WAIT_TIME_MS * CANCELCOMFORT_LOOPS_SHORT = time in milliseconds) [Default: 100]
#define CANCELCOMFORT_LOOPS_MEDIUM  150         // Number of loops before comfort mode is disabled when the turn signal lever is held down in medium mode (LOOP_WAIT_TIME_MS * CANCELCOMFORT_LOOPS_MEDIUM = time in milliseconds) [Default: 150]
#define CANCELCOMFORT_LOOPS_LONG    200         // Number of loops before comfort mode is disabled when the turn signal lever is held down in long mode (LOOP_WAIT_TIME_MS * CANCELCOMFORT_LOOPS_LONG = time in milliseconds) [Default: 200]

#define F_CPU                       1000000UL   // CPU clock frequency (frequency in Hertz) [Default: 1000000UL]

#define bool unsigned char

#include <avr/io.h>
#include <util/delay.h>

// Left turn signal input from lever
#define IN_LEFT                     PD5
#define IN_LEFT_PORT                PORTD
#define IN_LEFT_PIN                 PIND

// Right turn signal input from lever
#define IN_RIGHT                    PB7
#define IN_RIGHT_PORT               PORTB
#define IN_RIGHT_PIN                PINB

// Left turn signal from indicators
#define IN_LEFT_INDICATOR           PD0
#define IN_LEFT_INDICATOR_PORT      PORTD
#define IN_LEFT_INDICATOR_PIN       PIND

// Right turn signal from indicators
#define IN_RIGHT_INDICATOR          PB1
#define IN_RIGHT_INDICATOR_PORT     PORTB
#define IN_RIGHT_INDICATOR_PIN      PINB

// Left turn signal output to flasher relay
#define OUT_LEFT                    PA1
#define OUT_LEFT_PORT               PORTA
#define OUT_LEFT_PIN                PINA
#define OUT_LEFT_DDR                DDRA

// Right turn signal output to flasher relay
#define OUT_RIGHT                   PB2
#define OUT_RIGHT_PORT              PORTB
#define OUT_RIGHT_PIN               PINB
#define OUT_RIGHT_DDR               DDRB

// Comfort mode DIP switch 0 (DIP1)
#define IN_DIP_COMFORT0             PB3
#define IN_DIP_COMFORT0_PORT        PORTB
#define IN_DIP_COMFORT0_PIN         PINB

// Comfort mode DIP switch 1 (DIP2)
#define IN_DIP_COMFORT1             PB4
#define IN_DIP_COMFORT1_PORT        PORTB
#define IN_DIP_COMFORT1_PIN         PINB

// Cancel comfort mode DIP switch 0 (DIP3)
#define IN_DIP_CANCELCOMFORT0       PB5
#define IN_DIP_CANCELCOMFORT0_PORT  PORTB
#define IN_DIP_CANCELCOMFORT0_PIN   PINB

// Cancel comfort mode DIP switch 1 (DIP4)
#define IN_DIP_CANCELCOMFORT1       PB6
#define IN_DIP_CANCELCOMFORT1_PORT  PORTB
#define IN_DIP_CANCELCOMFORT1_PIN   PINB


#define _bit(m)                     (1 << (m))
#define _isset(p,m)                 ((p) & _bit((m)))
#define _set(p,m)                   ((p) |= _bit((m)))
#define _clear(p,m)                 ((p) &= ~_bit((m)))
#define _toggle(p,m)                ((p) ^= _bit((m)))


/****************************************************************************
 * INITIALIZE
 ****************************************************************************/
void initialize()
{
    // Set outputs
    _set(OUT_LEFT_DDR, OUT_LEFT);
    _set(OUT_RIGHT_DDR, OUT_RIGHT);
    // Set pull-up resistors on inputs
    _set(IN_LEFT_PORT, IN_LEFT);
    _set(IN_RIGHT_PORT, IN_RIGHT);
    _set(IN_LEFT_INDICATOR_PORT, IN_LEFT_INDICATOR);
    _set(IN_RIGHT_INDICATOR_PORT, IN_RIGHT_INDICATOR);
    _set(IN_DIP_COMFORT0_PORT, IN_DIP_COMFORT0);
    _set(IN_DIP_COMFORT1_PORT, IN_DIP_COMFORT1);
    _set(IN_DIP_CANCELCOMFORT0_PORT, IN_DIP_CANCELCOMFORT0);
    _set(IN_DIP_CANCELCOMFORT1_PORT, IN_DIP_CANCELCOMFORT1);
}


/****************************************************************************
 * MAIN
 ****************************************************************************/
int main(void)
{
    initialize();

    bool in_left_debounced = 0, in_right_debounced = 0, in_left_indicator_debounced = 0, in_right_indicator_debounced = 0, comfortmode = 0, laststate = 0, allow_comfortmode = 0, allow_cancelcomfortmode = 0;
    uint8_t debounce_in_left_loop = 0, debounce_in_right_loop = 0, debounce_in_left_indicator_loop = 0, debounce_in_right_indicator_loop = 0, comfortblink = 0, comfortblinks = 0, cancelcomfort_loop = 0, cancelcomfort_loops = 0;

    while (1)
    {
        // Read DIP switches
        if (!_isset(IN_DIP_COMFORT0_PIN, IN_DIP_COMFORT0) || !_isset(IN_DIP_COMFORT1_PIN, IN_DIP_COMFORT1))
        {
            allow_comfortmode = 1;
            if (_isset(IN_DIP_COMFORT0_PIN, IN_DIP_COMFORT0) && !_isset(IN_DIP_COMFORT1_PIN, IN_DIP_COMFORT1))
            {
                comfortblinks = COMFORTBLINKS_SHORT;
            }
            else if (!_isset(IN_DIP_COMFORT0_PIN, IN_DIP_COMFORT0) && _isset(IN_DIP_COMFORT1_PIN, IN_DIP_COMFORT1))
            {
                comfortblinks = COMFORTBLINKS_MEDIUM;
            }
            else
            {
                comfortblinks = COMFORTBLINKS_LONG;
            }
        }
        else
        {
            allow_comfortmode = 0;
        }
        if (!_isset(IN_DIP_CANCELCOMFORT0_PIN, IN_DIP_CANCELCOMFORT0) || !_isset(IN_DIP_CANCELCOMFORT1_PIN, IN_DIP_CANCELCOMFORT1))
        {
            allow_cancelcomfortmode = 1;
            if (_isset(IN_DIP_CANCELCOMFORT0_PIN, IN_DIP_CANCELCOMFORT0) && !_isset(IN_DIP_CANCELCOMFORT1_PIN, IN_DIP_CANCELCOMFORT1))
            {
                cancelcomfort_loops = CANCELCOMFORT_LOOPS_SHORT;
            }
            else if (!_isset(IN_DIP_CANCELCOMFORT0_PIN, IN_DIP_CANCELCOMFORT0) && _isset(IN_DIP_CANCELCOMFORT1_PIN, IN_DIP_CANCELCOMFORT1))
            {
                cancelcomfort_loops = CANCELCOMFORT_LOOPS_MEDIUM;
            }
            else
            {
                cancelcomfort_loops = CANCELCOMFORT_LOOPS_LONG;
            }
        }
        else
        {
            allow_cancelcomfortmode = 0;
        }

        // Debounce left turn signal from lever
        if (in_left_debounced)
        {
            if (debounce_in_left_loop == 0)
            {
                if (_isset(IN_LEFT_PIN, IN_LEFT))
                {
                    debounce_in_left_loop++;
                }
            }
            else if (debounce_in_left_loop == DEBOUNCE_LOOPS)
            {
                debounce_in_left_loop = 0;
                if (_isset(IN_LEFT_PIN, IN_LEFT))
                {
                    in_left_debounced = 0;
                }
            }
            else
            {
                debounce_in_left_loop++;
            }
        }
        else
        {
            if (debounce_in_left_loop == 0)
            {
                if (!_isset(IN_LEFT_PIN, IN_LEFT))
                {
                    debounce_in_left_loop++;
                }
            }
            else if (debounce_in_left_loop == DEBOUNCE_LOOPS)
            {
                debounce_in_left_loop = 0;
                if (!_isset(IN_LEFT_PIN, IN_LEFT))
                {
                    in_left_debounced = 1;
                }
            }
            else
            {
                debounce_in_left_loop++;
            }
        }

        // Debounce right turn signal from lever
        if (in_right_debounced)
        {
            if (debounce_in_right_loop == 0)
            {
                if (_isset(IN_RIGHT_PIN, IN_RIGHT))
                {
                    debounce_in_right_loop++;
                }
            }
            else if (debounce_in_right_loop == DEBOUNCE_LOOPS)
            {
                debounce_in_right_loop = 0;
                if (_isset(IN_RIGHT_PIN, IN_RIGHT))
                {
                    in_right_debounced = 0;
                }
            }
            else
            {
                debounce_in_right_loop++;
            }
        }
        else
        {
            if (debounce_in_right_loop == 0)
            {
                if (!_isset(IN_RIGHT_PIN, IN_RIGHT))
                {
                    debounce_in_right_loop++;
                }
            }
            else if (debounce_in_right_loop == DEBOUNCE_LOOPS)
            {
                debounce_in_right_loop = 0;
                if (!_isset(IN_RIGHT_PIN, IN_RIGHT))
                {
                    in_right_debounced = 1;
                }
            }
            else
            {
                debounce_in_right_loop++;
            }
        }

        // Debounce left turn signal from indicators
        if (in_left_indicator_debounced)
        {
            if (debounce_in_left_indicator_loop == 0)
            {
                if (_isset(IN_LEFT_INDICATOR_PIN, IN_LEFT_INDICATOR))
                {
                    debounce_in_left_indicator_loop++;
                }
            }
            else if (debounce_in_left_indicator_loop == DEBOUNCE_LOOPS)
            {
                debounce_in_left_indicator_loop = 0;
                if (_isset(IN_LEFT_INDICATOR_PIN, IN_LEFT_INDICATOR))
                {
                    in_left_indicator_debounced = 0;
                }
            }
            else
            {
                debounce_in_left_indicator_loop++;
            }
        }
        else
        {
            if (debounce_in_left_indicator_loop == 0)
            {
                if (!_isset(IN_LEFT_INDICATOR_PIN, IN_LEFT_INDICATOR))
                {
                    debounce_in_left_indicator_loop++;
                }
            }
            else if (debounce_in_left_indicator_loop == DEBOUNCE_LOOPS)
            {
                debounce_in_left_indicator_loop = 0;
                if (!_isset(IN_LEFT_INDICATOR_PIN, IN_LEFT_INDICATOR))
                {
                    in_left_indicator_debounced = 1;
                }
            }
            else
            {
                debounce_in_left_indicator_loop++;
            }
        }

        // Debounce right turn signal from indicators
        if (in_right_indicator_debounced)
        {
            if (debounce_in_right_indicator_loop == 0)
            {
                if (_isset(IN_RIGHT_INDICATOR_PIN, IN_RIGHT_INDICATOR))
                {
                    debounce_in_right_indicator_loop++;
                }
            }
            else if (debounce_in_right_indicator_loop == DEBOUNCE_LOOPS)
            {
                debounce_in_right_indicator_loop = 0;
                if (_isset(IN_RIGHT_INDICATOR_PIN, IN_RIGHT_INDICATOR))
                {
                    in_right_indicator_debounced = 0;
                }
            }
            else
            {
                debounce_in_right_indicator_loop++;
            }
        }
        else
        {
            if (debounce_in_right_indicator_loop == 0)
            {
                if (!_isset(IN_RIGHT_INDICATOR_PIN, IN_RIGHT_INDICATOR))
                {
                    debounce_in_right_indicator_loop++;
                }
            }
            else if (debounce_in_right_indicator_loop == DEBOUNCE_LOOPS)
            {
                debounce_in_right_indicator_loop = 0;
                if (!_isset(IN_RIGHT_INDICATOR_PIN, IN_RIGHT_INDICATOR))
                {
                    in_right_indicator_debounced = 1;
                }
            }
            else
            {
                debounce_in_right_indicator_loop++;
            }
        }

        if (_isset(OUT_LEFT_PORT, OUT_LEFT) && !in_right_debounced)
        {
            // Left turn signal active
            if (comfortmode)
            {
                if (in_left_debounced && allow_cancelcomfortmode)
                {
                    cancelcomfort_loop++;
                    if (cancelcomfort_loop == cancelcomfort_loops)
                    {
                        comfortmode = 0;
                    }
                }
                if (comfortmode)
                {
                    if (laststate != in_left_indicator_debounced)
                    {
                        laststate = in_left_indicator_debounced;
                        if (laststate)
                        {
                            comfortblink++;
                            if (comfortblink == comfortblinks)
                            {
                                comfortmode = 0;
                            }
                        }
                    }
                }
            }
            else
            {
                if (!in_left_debounced && !in_left_indicator_debounced)
                {
                    _clear(OUT_LEFT_PORT, OUT_LEFT);
                }
            }
        }
        else if (_isset(OUT_RIGHT_PORT, OUT_RIGHT) && !in_left_debounced)
        {
            // Right turn signal active
            if (comfortmode)
            {
                if (in_right_debounced && allow_cancelcomfortmode)
                {
                    cancelcomfort_loop++;
                    if (cancelcomfort_loop == cancelcomfort_loops)
                    {
                        comfortmode = 0;
                    }
                }
                if (comfortmode)
                {
                    if (laststate != in_right_indicator_debounced)
                    {
                        laststate = in_right_indicator_debounced;
                        if (laststate)
                        {
                            comfortblink++;
                            if (comfortblink == comfortblinks)
                            {
                                comfortmode = 0;
                            }
                        }
                    }
                }
            }
            else
            {
                if (!in_right_debounced && !in_right_indicator_debounced)
                {
                    _clear(OUT_RIGHT_PORT, OUT_RIGHT);
                }
            }
        }
        else if (in_left_debounced)
        {
            // Left turn signal triggered
            if (_isset(OUT_RIGHT_PORT, OUT_RIGHT))
            {
                _clear(OUT_RIGHT_PORT, OUT_RIGHT);
                comfortmode = 0;
            }
            if (!_isset(OUT_LEFT_PORT, OUT_LEFT))
            {
                _set(OUT_LEFT_PORT, OUT_LEFT);
            }
            if (allow_comfortmode)
            {
                comfortmode = 1;
                laststate = 0;
                comfortblink = 0;
                cancelcomfort_loop = 0;
            }
        }
        else if (in_right_debounced)
        {
            // Right turn signal triggered
            if (_isset(OUT_LEFT_PORT, OUT_LEFT))
            {
                _clear(OUT_LEFT_PORT, OUT_LEFT);
                comfortmode = 0;
            }
            if (!_isset(OUT_RIGHT_PORT, OUT_RIGHT))
            {
                _set(OUT_RIGHT_PORT, OUT_RIGHT);
            }
            if (allow_comfortmode)
            {
                comfortmode = 1;
                laststate = 0;
                comfortblink = 0;
                cancelcomfort_loop = 0;
            }
        }

        _delay_ms(LOOP_WAIT_TIME_MS);
    } 
}
