;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; oscilloscope functions
;
.include "p30fxxxx.inc"
.include "oscope.inc"
.include "glcd.inc"
.include "twid.inc"
.include "logish.inc"
.include "hann.inc"



.bss
MaxPoint:	.space 2
ButtCount:	.space 2
FFTbits:	.space 2
SampleTimeout: .space 2	;figure out if waiting for trigger has timed out
SampleRate: .space 2	;indicates the current sampe rate
FullScale:	.space 2	;indicates the current gain of input stage
mode:		.space 2	;indicates the mode of operation (trig, no trig, etc...)
sDiv: .space 2			;slower sampling rates setting bits
digits:	.space 8		;space for digits
trigVal: .space 2		;trigger level
slope: .space 2			;trigger slope
iSamp: .space 2			;index of sample (used by interrupt sr)
index: .space 2
trig:	.space 2		;indicate if scope was triggered
newWave: .space 192		;256 bytes for samples
oldWave: .space 192

.section samples ymemory
data: .space 1024




.text
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; oscMain
;
; starts the oscilloscope function, exits never!
; 
; Registers:	none
;
; Locals:
;
; Globals:
;
; Resources:	none
;
; Functions:	oscClear, oscSample, oscDispWave,
;				oscData, Delay
;
.global oscMain
oscMain:
	call oscClear		;clear oscilloscope memory
;	call oscButtonBounce
oscMainOSC:
	call oscSample
	btsc FFTbits, #0
	goto oscMainOSC2
	call oscDispWave
	call oscData
	call Delay
	btss FFTbits, #0
	goto oscMainOSC
oscMainOSC2:
	call clearLCD
	call oscClear
oscMainFFT:
	call oscSample		;get some samples
	btss FFTbits, #0
	goto oscMainFFT2		;if osc, go to osc
	call oscFFT
	call oscFFTDispWave	;display the wave aq'd
	call oscFFTData		;display some data about the data
	call Delay			;delay so display doesn't flicker
	btsc FFTbits, #0
	goto oscMainFFT
oscMainFFT2:
	call clearLCD
	call oscClear
	goto oscMainOSC
return


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; oscButtonBounce
;
; set up timer1 to collect button presses for mode switching
;

oscButtonBounce:
	mov #30000, w0
	mov w0, PR1					;set period of timer1 to 30000
	clr TMR1					;clear the timer count! w00!
	mov #0b0010000000000000, w0	;a value for timer 1, no prescale
	mov w0, T1CON				;
	
	bclr IFS0, #T1IF				;clear a timer 1 int req
	bclr IPC0, #T1IP2
	bset IPC0, #T1IP1
	bset IPC0, #T1IP0				;set priority of 3
	bset IEC0, #T1IE				;enable the timer 1 interrupt

	bset T1CON, #TON				;turn on the timer
	clr FFTbits
	clr ButtCount
	bset ModeTris, #ModeTrisX		;set the pin as input
return




oscFFT:
	mov #data, w0			;
	mov w0, w1				;
	mov #510, w2			;
	add w2, w0, w0			;w0 is last data point
	mov #1022, w2
	add w2, w1, w1			;w1 is last space in buffer
	
	mov #tblpage(HannWindow), w3
	mov w3, TBLPAG
	mov #tbloffset(HannWindow), w3
	mov #510, w4
	add w4, w3, w3			;last entry in hann table for windowing
	bclr CORCON, #IF		;do fractional multiplies
	do #255, oscFFTarrange	;
oscFFTarrangeStart:
	mov #0, w2
	mov w2, [w1--]			;im part is 0
	mov [w0--], w2			;move data point to w2
	asr w2, #1, w2			;scale properly
	tblrdl [w3--], w4
	mov w2, w5				;mov data to a mpy register
	mpy w4*w5, A			;do the windowing
	sac A, w2				;windowed data!
	mov w2, [w1--]			;mov the data into the new position
oscFFTarrange:
	nop
	mov #8, w0
	mov #data, w1
	mov #psvpage(TwidTab), w3	;
	mov #psvoffset(TwidTab), w2	;
	.extern _FFTComplexIP
	call _FFTComplexIP
	mov w0, w1
	mov #8, w0
	.extern _BitReverseComplex
	call _BitReverseComplex
	mov w0, w1
	mov w0, w2
	mov #256, w0
	.extern _SquareMagnitudeCplx
	call _SquareMagnitudeCplx

oscFFTmoveData:
	clr w4
	mov #data, w0
	mov #newWave, w1
	mov #oldWave, w2
	mov #0, w5			;w5 acts as max
	do #95, oscFFTmoveDataDone
	mov [w1], w3		;get new old point
	mov w3, [w2++]		;and store in oldWave
	mov [w0++], w3		;get fresh data
	cp w3, w5			;if w3>w5, w3 is new w5
	bra ltu, oscFFTn1	;
	mov w4, MaxPoint	;save index of new max
	mov w3, w5			;save the new max
oscFFTn1:
	mov w3, [w1++]		;and store in newWave
	inc w4, w4			;keep track of index
oscFFTmoveDataDone:
	nop
	
	mov #tblpage(LogTable), w0
	mov w0, TBLPAG
	mov #tbloffset(LogTable), w4
oscFFTlogish:			;find the log-ish of the amplitudes
	cp0 w5
	bra z, oscFFTdone
	mov #65535, w2
	repeat #17
	div.u w2, w5
	mov #newWave, w1
	do #95, oscFFTdone
	mov [w1], w2		;get a point
	mul.uu w0, w2, w2	;make it bigger
	lsr w2, #5, w2		;shift it
	bclr w2, #0
	add w2, w4, w6		;scaled point plus table offset, store in w6
	tblrdl [w6], w2		;get logish value
	mov w2, [w1++]		;put it back
	
oscFFTdone:
	nop


return





;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; oscFFTDispWave
;
; Display the current waveform.  Clears the previous
; wave as the new one is displayed.
;
; Registers:	w0 used, not restored
;				w1 used, not restored
;				w2 used, not restored
;				w3 used, not restored
;				w4 used, not restored
;				w8 used, not restored
;
; Locals:		index, modified
;
; Globals:		oldWave, read only
;				newWave, read only
;
; Resources:	none
;
; Functions:	
;	
oscFFTDispWave:
	;call oscClearCursor
	mov #31, w1			;x coord of vert line
	mov #0, w2			;y coord of vert line
	mov #31, w3			;x coord of vert line
	mov #63, w4			;y coord of vert line
	call drawLine		;draw a line separating data and wave
	clr index			;start at beginning of data
	mov #96, w8

oscFFTDloop:
	mov #oldWave, w0		;get location of current displayed wave
	add index, WREG			;wreg has current oldpoint to erase
	mov [w0], w2			;get the screen coord
	mov index, w1			;
	lsr w1, w1				;get actual index instead of offset
	mov #Xoffset, w0
	add w1, w0, w1			;shift starting position
	mov w1, w3				;bottom of line
	mov #63, w4				;
	call clearLine
oscFFTdrawNext:
	mov #newWave, w0		;get pointer to just obtained data
	add index, WREG			;wreg has current oldpoint to erase
	mov [w0], w2			;get the screen coord
	mov index, w1			;
	lsr w1, w1				;get actual index instead of offset
	mov #Xoffset, w0
	add w1, w0, w1			;shift starting position
	mov w1, w3
	mov #63, w4
	call drawLine	
	inc2 index
	dec w8, w8				;loop check
	cp0 w8
	bra nz, oscFFTDloop

	;call oscCursor

	return




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; oscFFTData
;
; Display data on the left of screen
;
; Registers:	w0 used, not restored
;				w1 used, not restored
;				w2 used, not restored
;				w3 used, not restored
;				w4 used, not restored
;
; Locals:		
;
oscFFTData:
	mov #0, w0
	mov #0, w1
	call textStartPos

	mov #'F', w1
	call textPutChar
	mov #'U', w1
	call textPutChar
	mov #'N', w1
	call textPutChar
	mov #'D', w1
	call textPutChar
	mov #':', w1
	call textPutChar

	;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
	; calculate and display fundamental here
	mov #1, w0
	mov #0, w1
	call textStartPos
	
	mov #digits, w2			;pointer to storage for digits
	call CalcFunda

	mov #'0', w0			;ascii base value for digits
	add w0, [w2--], w1		;get ascii value
	push w2					;preserve w2
	call textPutChar		;put the digit on the display
	pop w2					;restore w2
	mov #'.', w1			;and again
	push w2
	call textPutChar
	pop w2
	mov #'0', w0			;
	add w0, [w2--], w1
	push w2
	call textPutChar
	pop w2
	mov #'e', w1			;
	push w2
	call textPutChar
	pop w2
	mov #'0', w0			;
	add w0, [w2--], w1
	call textPutChar
return




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; CalcFunda
;	pass in w2 a pointer to where to store digits
;
CalcFunda:
	btsc SampleRate, #0		;750k sample rate
	goto Calc0
	btsc SampleRate, #1		;250k
	goto Calc1
	btsc SampleRate, #2		;100k
	goto Calc2
	btsc SampleRate, #3		;50k
	goto Calc3
	btsc SampleRate, #4		;10k
	goto Calc4
	btsc SampleRate, #5		;5k
	goto Calc5
	btsc SampleRate, #6		;1k
	goto Calc6
	goto Calc7				;.5k
Calc0:
	mov #3, w0
	mov w0, [w2]			;store an e value
	mov MaxPoint, w3		;
	mov #29, w4				;
	mul.uu w3, w4, w4		;store result in w4:w5
	goto CalcE

Calc1:
	mov #2, w0
	mov w0, [w2]
	mov MaxPoint, w3
	mov #98, w4
	mul.uu w3, w4, w4
	goto CalcE

Calc2:
	mov #2, w0
	mov w0, [w2]
	mov MaxPoint, w3
	mov #39, w4
	mul.uu w3, w4, w4
	goto CalcE

Calc3:
	mov #2, w0
	mov w0, [w2]
	mov MaxPoint, w3
	mov #20, w4
	mul.uu w3, w4, w4
	goto CalcE

Calc4:
	mov #1, w0
	mov w0, [w2]
	mov MaxPoint, w3
	mov #39, w4
	mul.uu w3, w4, w4
	goto CalcE

Calc5:
	mov #1, w0
	mov w0, [w2]
	mov MaxPoint, w3
	mov #20, w4
	mul.uu w3, w4, w4
	goto CalcE

Calc6:
	mov #0, w0
	mov w0, [w2]
	mov MaxPoint, w3
	mov #39, w4
	mul.uu w3, w4, w4
	goto CalcE

Calc7:
	mov #0, w0
	mov w0, [w2]
	mov MaxPoint, w3
	mov #20, w4
	mul.uu w3, w4, w4
	goto CalcE

CalcE:
	mov #100, w5
	cp w4, w5			;see if freq > 100
	bra ltu, CalcDP
	inc [w2], [w2]			;increase e
	mov #10, w5
	repeat #17
	div.u w4, w5
	mov w0, w4
	goto CalcE

CalcDP:
	mov #10, w5
	repeat #17
	div.u w4, w5
	mov w1, [++w2]
	mov w0, [++w2]

return
	



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; oscDispWave
;
; Display the current waveform.  Clears the previous
; wave as the new one is displayed.
;
; Registers:	w0 used, not restored
;				w1 used, not restored
;				w2 used, not restored
;				w3 used, not restored
;				w4 used, not restored
;				w8 used, not restored
;
; Locals:		index, modified
;
; Globals:		oldWave, read only
;				newWave, read only
;
; Resources:	none
;
; Functions:	
;	
oscDispWave:
	;call oscClearCursor
	mov #31, w1			;x coord of vert line
	mov #0, w2			;y coord of vert line
	mov #31, w3			;x coord of vert line
	mov #63, w4			;y coord of vert line
	call drawLine		;draw a line separating data and wave
	clr index			;start at beginning of data
	mov #96, w8

oscDloop:
	mov #oldWave, w0		;get location of current displayed wave
	add index, WREG			;wreg has current oldpoint to erase
	mov [w0], w2			;get the 10 bit adc value
	lsr w2, #4, w2			;shift left four times to get 6 bit value
	mov #63, w1				;
	sub w1, w2, w2			;adjust point to lcd coords (63-point)
	mov index, w1			;
	lsr w1, w1				;get actual index instead of offset
	mov #Xoffset, w0
	add w1, w0, w1			;shift starting position
	bclr w4, #0				;clear w4 to indicate pixel clear command
	call drawXY				;clear the pixel

	;inc2 w0, w0			;move to next point
	;mov [w0], w4			;get the next point
	;lsr w4, #4, w4			;10 bit to 6 bit
	;mov #63, w3			;
	;sub w3, w4, w4			;adjust for lcd coordinates
	;mov index, w3			;get the x value of point
	;lsr w3, w3				;adjust to number instead of word offset
	;inc w3, w3				;index is still current point, so change to next point
	;call clearLine			;clear the old line

	mov #newWave, w0		;get pointer to just obtained data
	add index, WREG			;wreg has current oldpoint to erase
	mov [w0], w2			;get the 10 bit adc value
	lsr w2, #4, w2			;shift left four times to get 6 bit value
	mov #63, w1
	sub w1, w2, w2			;adjust point to lcd coords
	mov index, w1			;
	lsr w1, w1				;get actual index instead of offset
	mov #Xoffset, w0
	add w1, w0, w1			;shift starting position
	bset w4, #0
	call drawXY

	;inc2 w0, w0			;move to next point
	;mov [w0], w4			;get the next point
	;lsr w4, #4, w4			;10 bit to 6 bit
	;mov #63, w3			;
	;sub w3, w4, w4			;adjust for lcd coordinates
	;mov index, w3			;get the x value of point
	;lsr w3, w3				;adjust to number instead of word offset
	;inc w3, w3				;index is still current point, so change to next point
	;call drawLine	

	inc2 index
	dec w8, w8				;loop check
	cp0 w8
	bra nz, oscDloop

	;call oscCursor

	return



 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; oscData
;
; Display data on the left of screen
;
; Registers:	w0 used, not restored
;				w1 used, not restored
;				w2 used, not restored
;				w3 used, not restored
;				w4 used, not restored
;
; Locals:		
;
.global oscData
oscData:
	mov #0, w0
	mov #0, w1
	call textStartPos

	mov #'R', w1
	call textPutChar
	mov #'M', w1
	call textPutChar
	mov #'S', w1
	call textPutChar
	mov #':', w1
	call textPutChar
	mov #' ', w1
	call textPutChar

	;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
	; calculate and display amplitude here
	mov #1, w0				;
	mov #0, w1				;
	call textStartPos		;set text to start at line 1, position 0

	call oscRMS				;calculate the RMS value of the waveform
	btsc FullScale, #1
	lsr w0, #1, w0
	btsc FullScale, #2
	lsr w0, #2, w0
	btsc FullScale, #3
	lsr w0, #3, w0
	btsc FullScale, #4
	lsr w0, #4, w0
	btsc FullScale, #5
	lsr w0, #5, w0

	mov #327, w2			;
	repeat #17				;
	div.u w0, w2			;divide w0 by 131 to get volts * 100
	mov w0, w4				;move volts*100 to w4

	mov #10, w3				;
	repeat #17				;
	div.u w4, w3			;divide by 10 to get remainder

	mov #digits, w2			;
	mov w1, [w2++]			;store one's digit
	mov w0, w4				;and result of divide
	mov #10, w3				;
	repeat #17				;	
	div.u w4, w3			;get 10's digit
	mov w1, [w2++]			;store 10's digit
	mov w0, w4				;
	mov #10, w3				;
	repeat #17				;
	div.u w4, w3			;divide by 10
	mov w1, [w2]			;get 100's digit

	mov #'0', w0			;ascii base value for digits
	add w0, [w2--], w1		;get ascii value
	push w2					;preserve w2
	call textPutChar		;put the digit on the display
	pop w2					;restore w2
	mov #'0', w0			;and again
	add w0, [w2--], w1
	push w2
	call textPutChar
	mov #'.', w1			;put a decimal point
	call textPutChar
	pop w2
	mov #'0', w0			;and the final digit
	add w0, [w2], w1
	call textPutChar
	mov #'v', w1			;with a v for good measure
	call textPutChar
return



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; oscCursor
;
; put a cursor on the screen, displaying the voltage
; at that point
;
oscCursor:
	mov #128, w0
	mov #newWave, w1
	add w0, w1, w1
	mov [w1], w2		;get desired point
	lsr w2, #4, w2		;shift it to get y position
	mov #63, w1			;
	sub w1, w2, w2		;transfer to lcd coord
	lsr w0, #1, w0		;
	mov #Xoffset, w1
	add w0, w1, w0		;x coord

	push w0
	push w2

	mov w0, w1
	mov #0, w2
	mov w0, w3
	mov #63, w4
	call drawLine

	pop w0
	push w0

	mov #32, w1
	mov w0, w2
	mov #127, w3
	mov w0, w4
	call drawLine

	pop w2		;pop the y coord
	pop w1		;popo the x coord
	bclr w4, #0	;turn off the pixel
	call drawXY
return




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; oscClearCursor
;
; put a cursor on the screen, displaying the voltage
; at that point
;
oscClearCursor:
	mov #128, w0
	mov #oldWave, w1
	add w0, w1, w1
	mov [w1], w2		;get desired point
	lsr w2, #4, w2		;shift it to get y position
	mov #63, w1			;
	sub w1, w2, w2		;transfer to lcd coord
	lsr w0, #1, w0		;
	mov #Xoffset, w1
	add w0, w1, w0		;x coord

	push w0
	push w2

	mov w0, w1
	mov #0, w2
	mov w0, w3
	mov #63, w4
	call clearLine

	pop w0
	push w0

	mov #32, w1
	mov w0, w2
	mov #127, w3
	mov w0, w4
	call clearLine

	pop w2		;pop the y coord
	pop w1		;popo the x coord
	bclr w4, #0	;turn off the pixel
	call drawXY
return






;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; oscSample
;
; sets up the adc module to get all 128 samples,
; pass the sampling type desired in w0
; 
;
oscSample:
	clr SampleTimeout	;clear the timeout counter
	clr iSamp 			;set index to 0
	bclr trig, #0		;set scope to not triggered
	bclr slope, #0		;set bit 0 for positive slope
	mov #31, w0			;default trigger (0v or so)
	mov w0, trigVal		;save trigger value	

	call oscGetMode
	call oscGetVdiv
	call oscGetTdiv
	
	
	btsc mode, #0
	bset trig, #0
	
oscIE:
	bclr IFS0, #ADIF	;clear a/d interrupt flag
	bset IEC0, #ADIE	;enable a/d interrupt
	bset ADCON1, #ADON	;turn on the a/d comverter

oscDone:
	btsc ADCON1, #ADON	;keep going until interrupt turns a/d off
	goto oscDone

	bclr T3CON, #TON		;turn off timer 3 if it's running
	bclr IEC0, #ADIE	;disable the a/d interrupt
oscAdjData:
	btsc FFTbits, #0
	goto oscMoveDataDone
	btss trig, #0
	goto oscMoveData
	mov #data, w0
	btsc FullScale, #0
	mov #ADJUST0, w1
	btsc FullScale, #1
	mov #ADJUST1, w1
	btsc FullScale, #2
	mov #ADJUST2, w1
	btsc FullScale, #3
	mov #ADJUST3, w1
	btsc FullScale, #4
	mov #ADJUST4, w1
	btsc FullScale, #5
	mov #ADJUST5, w1
	do #511, oscAdjDone
	mov [w0], w2
	add w2, w1, w3
	btsc w3, #10
	mov #1023, w3
	mov w3, [w0++]
oscAdjDone:
	nop
oscMoveData:
	mov #data, w0
	mov #newWave, w1
	mov #oldWave, w2
	do #95, oscMoveDataDone
	mov [w1], w3		;get new old point
	mov w3, [w2++]		;and store in oldWave
	mov [w0++], w3		;get fresh data
	mov w3, [w1++]		;and store in newWave
oscMoveDataDone:
	nop

return	




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; oscGetMode
;
oscGetMode:
	mov #ADCON1_POTval, w1
	mov w1, ADCON1
	mov #ADCON2_POTval, w1
	mov w1, ADCON2
	mov #ADCON3_POTval, w1
	mov w1, ADCON3
	mov #ADCHS_mPOTval, w1
	mov w1, ADCHS
	mov #ANPORTS, w1
	mov w1, ADPCFG
	bset ADCON1, #ADON		;turn on the a/d converter
oscMWait:
	btss ADCON1, #DONE
	goto oscMWait
	mov ADCBUF0, w0			;get the a/d value of pot
	bclr ADCON1, #ADON		;turn off the a/d converter
	clr FFTbits
	clr mode
			;
	btsc w0, #9				;512 bit
	bset FFTbits, #0			;if
	
	btsc w0, #8
	bset mode, #0

	btsc FFTbits, #0
	bset mode, #0

return


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; oscGetVdiv
; 
; read the pot for setting the pga gain 
;
oscGetVdiv:
	mov #3, w0
	mov #0, w1
	call textStartPos
	mov #'F', w1
	call textPutChar
	mov #'S', w1
	call textPutChar
	mov #'V', w1
	call textPutChar
	mov #':', w1
	call textPutChar
	mov #' ', w1
	call textPutChar

	mov #ADCON1_POTval, w1
	mov w1, ADCON1
	mov #ADCON2_POTval, w1
	mov w1, ADCON2
	mov #ADCON3_POTval, w1
	mov w1, ADCON3
	mov #ADCHS_vPOTval, w1
	mov w1, ADCHS
	mov #ANPORTS, w1
	mov w1, ADPCFG
	bset ADCON1, #ADON		;turn on the a/d converter
oscVWait:
	btss ADCON1, #DONE
	goto oscVWait

	bclr ADCON1, #ADON		;turn off the a/d converter
	mov ADCBUF0, w0			;get the a/d value of pot

	bclr PGA_TRIS, #PGA_SSTRIS	;set the pga pin to output 
	mov #SPI1CON_PGA, w1		;set up the SPI1 for pga
	mov w1, SPI1CON				;
	bclr SPI1STAT, #SPIROV		;clear the SPI1 overflow bit
	bset SPI1STAT, #SPIEN		;enable the SPI1 module
	clr FullScale				;clear variable indicating pga gain
osc125check:
	mov #853, w1
	cp w0, w1
	bra lt, osc25check
	bclr PGA_PORT, #PGA_SS	;take slave select low
	nop
	mov #gain125, w1
	mov w1, SPI1BUF		;send serial data
	bset FullScale, #0
	mov #4, w0
	mov #0, w1
	call textStartPos
	mov #'4', w1
	call textPutChar
	mov #'0', w1
	call textPutChar
	mov #'v', w1
	call textPutChar
	mov #' ', w1
	call textPutChar
	mov #' ', w1
	call textPutChar
	goto oscVDone
osc25check:
	mov #682, w1
	cp w0, w1
	bra lt, osc5check
	bclr PGA_PORT, #PGA_SS
	nop
	mov #gain25, w1
	mov w1, SPI1BUF
	bset FullScale, #1
	mov #4, w0
	mov #0, w1
	call textStartPos
	mov #'2', w1
	call textPutChar
	mov #'0', w1
	call textPutChar
	mov #'v', w1
	call textPutChar
	mov #' ', w1
	call textPutChar
	mov #' ', w1
	call textPutChar
	goto oscVDone
osc5check:
	mov #511, w1
	cp w0, w1
	bra lt, osc1check
	bclr PGA_PORT, #PGA_SS
	nop
	mov #gain5, w1
	mov w1, SPI1BUF
	bset FullScale, #2
	mov #4, w0
	mov #0, w1
	call textStartPos
	mov #'1', w1
	call textPutChar
	mov #'0', w1
	call textPutChar
	mov #'v', w1
	call textPutChar
	mov #' ', w1
	call textPutChar
	mov #' ', w1
	call textPutChar
	goto oscVDone
osc1check:
	mov #340, w1
	cp w0, w1
	bra lt, osc2check
	bclr PGA_PORT, #PGA_SS
	nop
	mov #gain1, w1
	mov w1, SPI1BUF
	bset FullScale, #3
	mov #4, w0
	mov #0, w1
	call textStartPos
	mov #'5', w1
	call textPutChar
	mov #'v', w1
	call textPutChar
	mov #' ', w1
	call textPutChar
	mov #' ', w1
	call textPutChar
	mov #' ', w1
	call textPutChar
	goto oscVDone
osc2check:
	mov #169, w1
	cp w0, w1
	bra lt, osc4check
	bclr PGA_PORT, #PGA_SS
	nop
	mov #gain2, w1
	mov w1, SPI1BUF
	bset FullScale, #4
	mov #4, w0
	mov #0, w1
	call textStartPos
	mov #'2', w1
	call textPutChar
	mov #'.', w1
	call textPutChar
	mov #'5', w1
	call textPutChar
	mov #'v', w1
	call textPutChar
	mov #' ', w1
	call textPutChar
	goto oscVDone
osc4check:
	bclr PGA_PORT, #PGA_SS
	nop
	mov #gain4, w1
	mov w1, SPI1BUF
	bset FullScale, #5
	mov #4, w0
	mov #0, w1
	call textStartPos
	mov #'1', w1
	call textPutChar
	mov #'.', w1
	call textPutChar
	mov #'2', w1
	call textPutChar
	mov #'5', w1
	call textPutChar
	mov #'v', w1
	call textPutChar
	;goto oscVDone
oscVDone:
	btss SPI1STAT, #SPIRBF
	goto oscVDone
	bclr SPI1STAT, #SPIRBF
	bset PGA_PORT, #PGA_SS		;raise slave select on PGA
	nop
	return

	


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; oscGetTdiv
;
; read the potentiometer on an7, return in w0
;
oscGetTdiv:
	mov #ADCON1_POTval, w1
	mov w1, ADCON1
	mov #ADCON2_POTval, w1
	mov w1, ADCON2
	mov #ADCON3_POTval, w1
	mov w1, ADCON3
	mov #ADCHS_tPOTval, w1
	mov w1, ADCHS
	mov #ANPORTS, w1
	mov w1, ADPCFG
	mov #0b0000000000000000, w1
	mov w1, ADCSSL

	bset ADCON1, #ADON		;turn on the a/d converter

oscPotWait:
	btss ADCON1, #DONE		;check if an a/d result is available
	goto oscPotWait
	mov ADCBUF0, w0			;get the pot position
	bclr ADCON1, #ADON		;turn off the a/d converter
	
	push w0
	mov #6, w0
	mov #0, w1
	call textStartPos
	pop w0

	clr SampleRate			;clear variable that stores current sample rate
oscFScheck:
	mov #896, w1
	cp w0, w1
	bra lt, oscHScheck
	call oscFullSpeed
	
	bset SampleRate, #0
	mov #'7', w1
	call textPutChar
	mov #'5', w1
	call textPutChar
	mov #'0', w1
	call textPutChar
	mov #'.', w1
	call textPutChar
	mov #'0', w1
	call textPutChar
	goto oscPotDone

oscHScheck:
	mov #768, w1
	cp w0, w1
	bra lt, oscQScheck
	call oscHalfSpeed

	bset SampleRate, #1
	mov #'2', w1
	call textPutChar
	mov #'5', w1
	call textPutChar
	mov #'0', w1
	call textPutChar
	mov #'.', w1
	call textPutChar
	mov #'0', w1
	call textPutChar
	goto oscPotDone

oscQScheck:
	mov #640, w1
	cp w0, w1
	bra lt, oscEScheck
	call oscQuarterSpeed

	bset SampleRate, #2
	mov #'1', w1
	call textPutChar
	mov #'0', w1
	call textPutChar
	mov #'0', w1
	call textPutChar
	mov #'.', w1
	call textPutChar
	mov #'0', w1
	call textPutChar
	goto oscPotDone

oscEScheck:
	mov #512, w1
	cp w0, w1
	bra lt, oscSScheck
	call oscEighthSpeed

	bset SampleRate, #3
	mov #'5', w1
	call textPutChar
	mov #'0', w1
	call textPutChar
	mov #'.', w1
	call textPutChar
	mov #'0', w1
	call textPutChar
	mov #'0', w1
	call textPutChar
	goto oscPotDone

oscSScheck:
	mov #384, w1
	cp w0, w1
	bra lt, osc32check
	call oscSixteenthSpeed

	bset SampleRate, #4
	mov #'1', w1
	call textPutChar
	mov #'0', w1
	call textPutChar
	mov #'.', w1
	call textPutChar
	mov #'0', w1
	call textPutChar
	mov #'0', w1
	call textPutChar
	goto oscPotDone

osc32check:
	mov #256, w1
	cp w0, w1
	bra lt, osc64check
	call osc32ndSpeed

	bset SampleRate, #5
	mov #'5', w1
	call textPutChar
	mov #'.', w1
	call textPutChar
	mov #'0', w1
	call textPutChar
	mov #'0', w1
	call textPutChar
	mov #'0', w1
	call textPutChar
	goto oscPotDone

osc64check:
	mov #128, w1
	cp w0, w1
	bra lt, osc128check
	call osc64Speed

	bset SampleRate, #6
	mov #'1', w1
	call textPutChar
	mov #'.', w1
	call textPutChar
	mov #'0', w1
	call textPutChar
	mov #'0', w1
	call textPutChar
	mov #'0', w1
	call textPutChar
	goto oscPotDone

osc128check:
	call osc128Speed

	bset SampleRate, #7
	mov #'0', w1
	call textPutChar
	mov #'.', w1
	call textPutChar
	mov #'5', w1
	call textPutChar
	mov #'0', w1
	call textPutChar
	mov #'0', w1
	call textPutChar
	;goto oscPotDone

oscPotDone:
	mov #7, w0
	mov #0, w1
	call textStartPos
	
	mov #'k', w1
	call textPutChar
	mov #'s', w1
	call textPutChar
	mov #'p', w1
	call textPutChar
	mov #'s', w1
	call textPutChar
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;fft
	btsc FFTbits, #0
	bset ADCON1, #FORM0
	btsc FFTbits, #0
	bset ADCON1, #FORM1
return




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; oscFullSpeed
;
; set the adc module to sample at 750ksps
oscFullSpeed:
	mov #0b0010000011100100, w1
	mov w1, ADCON1
	mov #0b0110000000111100, w1
	mov w1, ADCON2
	mov #0b0000001000000101, w1
	mov w1, ADCON3
	mov #0b0000000000100011, w1
	mov w1, ADCHS
	mov #ANPORTS, w1
	mov w1, ADPCFG
	mov #0b0000000000001000, w1
	mov w1, ADCSSL
return




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; oscHalfSpeed
;
; set the adc module to sample at 250ksps
oscHalfSpeed:
	mov #0b0010000001000100, w1
	mov w1, ADCON1
	mov #0b0110000000111100, w1
	mov w1, ADCON2
	mov #0b0000001000000101, w1
	mov w1, ADCON3
	mov #0b0000000000100011, w1
	mov w1, ADCHS
	mov #ANPORTS, w1
	mov w1, ADPCFG
	mov #0b0000000000001000, w1
	mov w1, ADCSSL
	
	mov #0b0010000000000000, w1
	mov w1, T3CON
	clr TMR3
	mov #120, w1
	mov w1, PR3
	bset T3CON, #TON		;turn on the timer
	
return




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; oscQuarterSpeed
;
; set the adc module to sample at 100ksps
oscQuarterSpeed:
	mov #0b0010000001000100, w1
	mov w1, ADCON1
	mov #0b0110000000111100, w1
	mov w1, ADCON2
	mov #0b0000001000000101, w1
	mov w1, ADCON3
	mov #0b0000000000100011, w1
	mov w1, ADCHS
	mov #ANPORTS, w1
	mov w1, ADPCFG
	mov #0b0000000000001000, w1
	mov w1, ADCSSL

	mov #0b0010000000000000, w1
	mov w1, T3CON
	clr TMR3
	mov #300, w1
	mov w1, PR3
	bset T3CON, #TON		;turn on the timer
return




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; oscEighthSpeed
;
; set the adc module to sample at 50ksps
oscEighthSpeed:
	mov #0b0010000001000100, w1
	mov w1, ADCON1
	mov #0b0110000000111100, w1
	mov w1, ADCON2
	mov #0b0000001000000101, w1
	mov w1, ADCON3
	mov #0b0000000000100011, w1
	mov w1, ADCHS
	mov #ANPORTS, w1
	mov w1, ADPCFG
	mov #0b0000000000001000, w1
	mov w1, ADCSSL

	mov #0b0010000000000000, w1
	mov w1, T3CON
	clr TMR3
	mov #600, w1
	mov w1, PR3
	bset T3CON, #TON		;turn on the timer
return




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; oscSixteenthSpeed
;
; set the adc module to sample at 10ksps
oscSixteenthSpeed:
	mov #0b0010000001000100, w1
	mov w1, ADCON1
	mov #0b0110000000111100, w1
	mov w1, ADCON2
	mov #0b0000001000000101, w1
	mov w1, ADCON3
	mov #0b0000000000100011, w1
	mov w1, ADCHS
	mov #ANPORTS, w1
	mov w1, ADPCFG
	mov #0b0000000000001000, w1
	mov w1, ADCSSL

	mov #0b0010000000000000, w1
	mov w1, T3CON
	clr TMR3
	mov #3000, w1
	mov w1, PR3
	bset T3CON, #TON		;turn on the timer
return




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; osc32ndSpeed
;
; set the adc module to sample at 5ksps
osc32ndSpeed:
	mov #0b0010000001000100, w1
	mov w1, ADCON1
	mov #0b0110000000111100, w1
	mov w1, ADCON2
	mov #0b0000001000000101, w1
	mov w1, ADCON3
	mov #0b0000000000100011, w1
	mov w1, ADCHS
	mov #ANPORTS, w1
	mov w1, ADPCFG
	mov #0b0000000000001000, w1
	mov w1, ADCSSL

	mov #0b0010000000000000, w1
	mov w1, T3CON
	clr TMR3
	mov #6000, w1
	mov w1, PR3
	bset T3CON, #TON		;turn on the timer
return




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; osc64Speed
;
; set the adc module to sample at 5ksps
osc64Speed:
	mov #0b0010000001000100, w1
	mov w1, ADCON1
	mov #0b0110000000111100, w1
	mov w1, ADCON2
	mov #0b0000001000000101, w1
	mov w1, ADCON3
	mov #0b0000000000100011, w1
	mov w1, ADCHS
	mov #ANPORTS, w1
	mov w1, ADPCFG
	mov #0b0000000000001000, w1
	mov w1, ADCSSL

	mov #0b0010000000000000, w1
	mov w1, T3CON
	clr TMR3
	mov #30000, w1
	mov w1, PR3
	bset T3CON, #TON		;turn on the timer
return


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; osc128Speed
;
; set the adc module to sample at 5ksps
osc128Speed:
	mov #0b0010000001000100, w1
	mov w1, ADCON1
	mov #0b0110000000111100, w1
	mov w1, ADCON2
	mov #0b0000001000000101, w1
	mov w1, ADCON3
	mov #0b0000000000100011, w1
	mov w1, ADCHS
	mov #ANPORTS, w1
	mov w1, ADPCFG
	mov #0b0000000000001000, w1
	mov w1, ADCSSL

	mov #0b0010000000000000, w1
	mov w1, T3CON
	clr TMR3
	mov #60000, w1
	mov w1, PR3
	bset T3CON, #TON		;turn on the timer
return



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; oscClear
;
; init all oscope memory variables
;
oscClear:
	mov #data, w0
	mov #0, w1
	repeat #DATA_LENGTH-1
	mov w1, [w0++]
	mov #newWave, w0
	mov #0, w1
	repeat #DISP_WIDTH-1
	mov w1, [w0++]
oscOldClear:
	mov #oldWave, w0
	repeat #DISP_WIDTH-1
	mov w1, [w0++]
	mov #1, w0
	clr mode
	mov w0, SampleRate		;clear the sample rate indicator
	mov w0, FullScale		;clear the gain indicator
return




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; oscRMS
;
; calculate the rms value of the input waveform
; uses newWave as the input data.
oscRMS:
	bset CORCON, #IF
	mov #data, w0			;get address of data
	clr w1					;clear w1 to use as storage for offset
	clr A					;

oscRMSsquare:
	mov [w0], w4			;get first data item
	mov #511, w3
	cp w4, w3				;check sample for software rectifying
	bra gt, oscRMSmac		;if it's above gnd, ok!
oscRMSrect:
	mov #1023, w3
	sub w3, w4, w4			;rectify the signal
oscRMSmac:
	mov #512, w3			;shift data down so range is 0-511
	sub w4, w3, w4			;instead of 512-1023
	mac w4*w4, A			;square and add to A
	inc2 w1, w1
	inc2 w0, w0				;move to next data
	mov #512, w2
	cp w1, w2				;check loop counter
	bra lt, oscRMSsquare	;if less than, do it again!

oscRMSsqrt:
	sac A, #-6, w0			;get top 16 bits of result
	sftac A, #-16
	sac A, #-6, w1			;get bottom 16 bits of result
	call sqrt				;get the sqrt of w1
return




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; sqrt
;
; takes 32 bit argument in w0:w1, returns 16-bit sqrt
; in w0 
;
sqrt:
	clr w2				;temp storage for result
	mov #15, w3			;current bit location
sqrtLoop:
	do #15, sqrtLoopDone
	mov #1, w4
	sl w4, w3, w4		;shift the bit in 4 to the correct location
	ior w4, w2, w2		;set the desired bit
	mul.uu w2, w2, w6	;square w2 and store result in w6:w7
	cp w7, w0			;compare result and original
	bra gtu, sqrtNoGood	;if greater than, no good
sqrtEq:
	bra nz, sqrtAlmostDone	;if not greater than and not eq, then less than, cool beans
	cp w6, w1				;now compare lower 16 bits
	bra leu, sqrtAlmostDone	;if less than or equal, almost done!
sqrtNoGood:
	com w4, w4			;invert the bit
	and w4, w2, w2		;and remove it if no good
sqrtAlmostDone:
	dec w3, w3			;decrement w3 (move down a bit)
sqrtLoopDone:
	nop
	mov w2, w0
return




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; OLDsqrt
; 
; find the closest integer to the sq root of a 16 bit
; unsigned value stored in w0.  return this in w1
;
OLDsqrt:
	lsr w0, #1, w1		;use w1/2 + 1 as the start value
	inc w1, w1			;
	mov w0, w2			;w2 holds input value
	mov w1, w3			;
	clr w5				;use w5 as a loop counter
OLDsqrtLoop:
	mov w3, w4			;w4 holds old estimate
	repeat #17
	div.u	w2, w3		;input/estimate
	add w3, w0, w3		
	lsr w3, #1, w3		;get new estimate 
	inc w5, w5
	mov #25, w0
	cp w5, w0
	bra lt, OLDsqrtLoop	;if not equal, iterate
mov w3, w1
return					;otherwise, return with result in w1




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Delay
;
; Registers: none
; Resources: do loop, repeat loop
;
Delay:
	do #400, DelayDone
		repeat #10000
			nop

		nop
		nop
DelayDone:
	nop
	return






.global __ADCInterrupt

__ADCInterrupt:
	bclr IFS0, #ADIF		;clear the interrupt flag	
	push w0
	push w1
	push w2
	push w3
	push w4
	push w5

	mov #data, w0

	mov iSamp, w2
	add w2, w0, w0

adcTrig0:
	btsc trig, #0
	goto adcInt0
	mov ADCBUF0, w3
	lsr w3, #4, w3		;get displayed value
	mov trigVal, w4
	cp w3, w4
	bra nz, adcTrig1
	mov ADCBUF0, w3
	mov ADCBUF1, w5
	cp w3, w5
	btss slope, #0		;check for negative slope
	bra le, adcTrig1	;if bufn < bufn+1, not neg slope
	btsc slope, #0		;check for positive slope
	bra ge, adcTrig1	;if bufn > bufn+1, not pos slope
	bset trig, #0		;if all checks ok, then triggered
adcInt0:
	clr SampleTimeout
	mov ADCBUF0, w2
	mov w2,[w0++]
	inc2 iSamp
	goto adcInt1

adcTrig1:
	mov ADCBUF1, w3
	lsr w3, #4, w3		;get displayed value
	cp w3, w4
	bra nz, adcTrig2
	mov ADCBUF1, w3
	mov ADCBUF2, w5
	cp w3, w5
	btss slope, #0		;check for negative slope
	bra le, adcTrig2	;if bufn < bufn+1, not neg slope
	btsc slope, #0		;check for positive slope
	bra ge, adcTrig2	;if bufn > bufn+1, not pos slope
	bset trig, #0		;if all checks ok, then triggered
adcInt1:	
	clr SampleTimeout
	mov ADCBUF1, w2
	mov w2,[w0++]
	inc2 iSamp
	goto adcInt2

adcTrig2:
	mov ADCBUF2, w3
	lsr w3, #4, w3		;get displayed value
	cp w3, w4
	bra nz, adcTrig3
	mov ADCBUF2, w3
	mov ADCBUF3, w5
	cp w3, w5
	btss slope, #0		;check for negative slope
	bra le, adcTrig3	;if bufn < bufn+1, not neg slope
	btsc slope, #0		;check for positive slope
	bra ge, adcTrig3	;if bufn > bufn+1, not pos slope
	bset trig, #0		;if all checks ok, then triggered
adcInt2:
	clr SampleTimeout
	mov ADCBUF2, w2
	mov w2,[w0++]
	inc2 iSamp
	goto adcInt3

adcTrig3:
	mov ADCBUF3, w3
	lsr w3, #4, w3		;get displayed value
	cp w3, w4
	bra nz, adcTrig4
	mov ADCBUF3, w3
	mov ADCBUF4, w5
	cp w3, w5
	btss slope, #0		;check for negative slope
	bra le, adcTrig4	;if bufn < bufn+1, not neg slope
	btsc slope, #0		;check for positive slope
	bra ge, adcTrig4	;if bufn > bufn+1, not pos slope
	bset trig, #0		;if all checks ok, then triggered
adcInt3:
	clr SampleTimeout
	mov ADCBUF3, w2
	mov w2,[w0++]
	inc2 iSamp
	goto adcInt4

adcTrig4:
	mov ADCBUF4, w3
	lsr w3, #4, w3		;get displayed value
	cp w3, w4
	bra nz, adcTrig5
	mov ADCBUF4, w3
	mov ADCBUF5, w5
	cp w3, w5
	btss slope, #0		;check for negative slope
	bra le, adcTrig5	;if bufn < bufn+1, not neg slope
	btsc slope, #0		;check for positive slope
	bra ge, adcTrig5	;if bufn > bufn+1, not pos slope
	bset trig, #0		;if all checks ok, then triggered
adcInt4:
	clr SampleTimeout
	mov ADCBUF4, w2
	mov w2,[w0++]
	inc2 iSamp
	goto adcInt5

adcTrig5:
	mov ADCBUF5, w3
	lsr w3, #4, w3		;get displayed value
	cp w3, w4
	bra nz, adcTrig6
	mov ADCBUF5, w3
	mov ADCBUF6, w5
	cp w3, w5
	btss slope, #0		;check for negative slope
	bra le, adcTrig6	;if bufn < bufn+1, not neg slope
	btsc slope, #0		;check for positive slope
	bra ge, adcTrig6	;if bufn > bufn+1, not pos slope
	bset trig, #0		;if all checks ok, then triggered
adcInt5:
	clr SampleTimeout
	mov ADCBUF5, w2
	mov w2,[w0++]
	inc2 iSamp
	goto adcInt6

adcTrig6:
	mov ADCBUF6, w3
	lsr w3, #4, w3		;get displayed value
	cp w3, w4
	bra nz, adcTrig7
	mov ADCBUF6, w3
	mov ADCBUF7, w5
	cp w3, w5
	btss slope, #0		;check for negative slope
	bra le, adcTrig7	;if bufn < bufn+1, not neg slope
	btsc slope, #0		;check for positive slope
	bra ge, adcTrig7	;if bufn > bufn+1, not pos slope
	bset trig, #0		;if all checks ok, then triggered
adcInt6:
	clr SampleTimeout
	mov ADCBUF6, w2
	mov w2,[w0++]
	inc2 iSamp
	goto adcInt7

adcTrig7:
	mov ADCBUF7, w3
	lsr w3, #4, w3		;get displayed value
	cp w3, w4
	bra nz, adcTrig8
	mov ADCBUF7, w3
	mov ADCBUF8, w5
	cp w3, w5
	btss slope, #0		;check for negative slope
	bra le, adcTrig8	;if bufn < bufn+1, not neg slope
	btsc slope, #0		;check for positive slope
	bra ge, adcTrig8	;if bufn > bufn+1, not pos slope
	bset trig, #0		;if all checks ok, then triggered
adcInt7:
	clr SampleTimeout
	mov ADCBUF7, w2
	mov w2,[w0++]
	inc2 iSamp
	goto adcInt8

adcTrig8:
	mov ADCBUF8, w3
	lsr w3, #4, w3		;get displayed value
	cp w3, w4
	bra nz, adcTrig9
	mov ADCBUF8, w3
	mov ADCBUF9, w5
	cp w3, w5
	btss slope, #0		;check for negative slope
	bra le, adcTrig9	;if bufn < bufn+1, not neg slope
	btsc slope, #0		;check for positive slope
	bra ge, adcTrig9	;if bufn > bufn+1, not pos slope
	bset trig, #0		;if all checks ok, then triggered
adcInt8:
	clr SampleTimeout
	mov ADCBUF8, w2
	mov w2,[w0++]
	inc2 iSamp
	goto adcInt9

adcTrig9:
	mov ADCBUF9, w3
	lsr w3, #4, w3		;get displayed value
	cp w3, w4
	bra nz, adcTrigA
	mov ADCBUF9, w3
	mov ADCBUFA, w5
	cp w3, w5
	btss slope, #0		;check for negative slope
	bra le, adcTrigA	;if bufn < bufn+1, not neg slope
	btsc slope, #0		;check for positive slope
	bra ge, adcTrigA	;if bufn > bufn+1, not pos slope
	bset trig, #0		;if all checks ok, then triggered
adcInt9:
	clr SampleTimeout
	mov ADCBUF9, w2
	mov w2,[w0++]
	inc2 iSamp
	goto adcIntA

adcTrigA:
	mov ADCBUFA, w3
	lsr w3, #4, w3		;get displayed value
	cp w3, w4
	bra nz, adcTrigB
	mov ADCBUFA, w3
	mov ADCBUFB, w5
	cp w3, w5
	btss slope, #0		;check for negative slope
	bra le, adcTrigB	;if bufn < bufn+1, not neg slope
	btsc slope, #0		;check for positive slope
	bra ge, adcTrigB	;if bufn > bufn+1, not pos slope
	bset trig, #0		;if all checks ok, then triggered
adcIntA:
	clr SampleTimeout
	mov ADCBUFA, w2
	mov w2,[w0++]
	inc2 iSamp
	goto adcIntB

adcTrigB:
	mov ADCBUFB, w3
	lsr w3, #4, w3		;get displayed value
	cp w3, w4
	bra nz, adcTrigC
	mov ADCBUFB, w3
	mov ADCBUFC, w5
	cp w3, w5
	btss slope, #0		;check for negative slope
	bra le, adcTrigC	;if bufn < bufn+1, not neg slope
	btsc slope, #0		;check for positive slope
	bra ge, adcTrigC	;if bufn > bufn+1, not pos slope
	bset trig, #0		;if all checks ok, then triggered
adcIntB:
	clr SampleTimeout
	mov ADCBUFB, w2
	mov w2,[w0++]
	inc2 iSamp
	goto adcIntC

adcTrigC:
	mov ADCBUFC, w3
	lsr w3, #4, w3		;get displayed value
	cp w3, w4
	bra nz, adcTrigD
	mov ADCBUFC, w3
	mov ADCBUFD, w5
	cp w3, w5
	btss slope, #0		;check for negative slope
	bra le, adcTrigD	;if bufn < bufn+1, not neg slope
	btsc slope, #0		;check for positive slope
	bra ge, adcTrigD	;if bufn > bufn+1, not pos slope
	bset trig, #0		;if all checks ok, then triggered
adcIntC:
	clr SampleTimeout
	mov ADCBUFC, w2
	mov w2,[w0++]
	inc2 iSamp
	goto adcIntD

adcTrigD:
	mov ADCBUFD, w3
	lsr w3, #4, w3		;get displayed value
	cp w3, w4
	bra nz, adcTrigE
	mov ADCBUFD, w3
	mov ADCBUFE, w5
	cp w3, w5
	btss slope, #0		;check for negative slope
	bra le, adcTrigE	;if bufn < bufn+1, not neg slope
	btsc slope, #0		;check for positive slope
	bra ge, adcTrigE	;if bufn > bufn+1, not pos slope
	bset trig, #0		;if all checks ok, then triggered
adcIntD:
	clr SampleTimeout
	mov ADCBUFD, w2
	mov w2,[w0++]
	inc2 iSamp
	goto adcIntE

adcTrigE:
	mov ADCBUFE, w3
	lsr w3, #4, w3		;get displayed value
	cp w3, w4
	bra nz, adcTrigF
	mov ADCBUFE, w3
	mov ADCBUFF, w5
	cp w3, w5
	btss slope, #0		;check for negative slope
	bra le, adcTrigF	;if bufn < bufn+1, not neg slope
	btsc slope, #0		;check for positive slope
	bra ge, adcTrigF	;if bufn > bufn+1, not pos slope
	bset trig, #0		;if all checks ok, then triggered
adcIntE:
	clr SampleTimeout
	mov ADCBUFE, w2
	mov w2,[w0++]
	inc2 iSamp
	goto adcIntF

adcTrigF:
	mov ADCBUFF, w3
	lsr w3, #4, w3		;get displayed value
	cp w3, w4
	bra nz, ADCdone
	mov ADCBUFF, w3
	mov ADCBUF0, w5
	cp w3, w5
	btss slope, #0		;check for negative slope
	bra le, ADCdone	;if bufn < bufn+1, not neg slope
	btsc slope, #0		;check for positive slope
	bra ge, ADCdone	;if bufn > bufn+1, not pos slope
	bset trig, #0		;if all checks ok, then triggered
adcIntF:
	clr SampleTimeout
	mov ADCBUFF, w2
	mov w2,[w0++]
	inc2 iSamp

ADCdone:
inc SampleTimeout		;increase the sample timeout, cleared if triggering
btsc SampleTimeout, #4	;if 16, then timeout
bclr ADCON1, #ADON		;so turn off a/d
ADCdone1:
btsc iSamp, #9	;if all the way to end of buffer...
bclr ADCON1, #ADON	;turn off the adc

ADCreturn:
pop w5
pop w4
pop w3
pop w2
pop w1
pop w0

retfie				;done with interrupt




.global __T1Interrupt

__T1Interrupt:
	bclr IFS0, #T1IF				;clear a timer 1 int req

	push w0
	push w1
	push w2
	clr w0
	btss ModePort, #ModeButton		;
	bset w0, #0						;set w0 to 1 if button is down
	
	btss ButtCount, #0
	goto WasUp
	;goto WasDown
WasDown:
	mov #0xFF, w1
	mov ButtCount, w2
	cp w1, w2
	bra z, WasDownWait
WasDown2:
	mov #15, w1
	cp w1, w2
	bra geu, WasDownLongEnough
WasDown3:
	btsc w0, #0
	inc2 ButtCount				;inc the bounce counter
	btss w0, #0
	clr ButtCount				;if it's up, clear the count
	goto WasDone
WasDownLongEnough:
	btss w0, #0
	goto WasDown3				;if it is up, go clear the count
	mov #0xFF, w0
	mov w0, ButtCount		;indicate waiting for button up
	btg FFTbits, #0				;switch modes
	goto WasDone
WasDownWait:
	btss w0, #0
	goto WasDown3				;if it is up, clear the count
	mov #0xFF, w0
	mov w0, ButtCount
	goto WasDone
WasDone:
	pop w2
	pop w1
	pop w0
	retfie

WasUp:
	clr ButtCount
	btsc w0, #0					;if the button is down, set so
	bset ButtCount, #0			;viola
	pop w2
	pop w1
	pop w0
	retfie