﻿/*
 * "Copyright (c) 2008 Robert B. Reese, Bryan A. Jones, J. W. Bruce ("AUTHORS")"
 * All rights reserved.
 * (R. Reese, reese_AT_ece.msstate.edu, Mississippi State University)
 * (B. A. Jones, bjones_AT_ece.msstate.edu, Mississippi State University)
 * (J. W. Bruce, jwbruce_AT_ece.msstate.edu, Mississippi State University)
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement is
 * hereby granted, provided that the above copyright notice, the following
 * two paragraphs and the authors appear in all copies of this software.
 *
 * IN NO EVENT SHALL THE "AUTHORS" BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE "AUTHORS"
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE "AUTHORS" SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE "AUTHORS" HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS."
 *
 * Please maintain this header in its entirety when copying/modifying
 * these files.
 *
 * Hungarian translation: István Cserny, 2009
 */

#include "pic24_all.h"


/** \file
Az ipari szabványnak tekinthető párhuzamos illesztővel
ellátott 4-soros, 4x20 karakateres LCD kijelző használatát mutatja be, 4 bites módban.
Feltételezzük, hogy 5V-os az LCD kijelző, ezért a csak digitális funkciójú kimeneteket használjuk
a 4 bites adatbusz vezérlésére, mivel csak azok tolerálják az 5 V-os jelszintet.

Kipróbálva a Hantronix HDM20416L-M kijelzővel.
*/

#define RS_HIGH()        _LATB9 = 1
#define RS_LOW()         _LATB9 = 0
#define CONFIG_RS()      CONFIG_RB9_AS_DIG_OUTPUT()

#define RW_HIGH()        _LATB13 = 1
#define RW_LOW()         _LATB13 = 0
#define CONFIG_RW()      CONFIG_RB13_AS_DIG_OUTPUT()

#define E_HIGH()         _LATB14 = 1
#define E_LOW()          _LATB14 = 0
#define CONFIG_E()       CONFIG_RB14_AS_DIG_OUTPUT()

#define LCD4O          _LATB5
#define LCD5O          _LATB6
#define LCD6O          _LATB7
#define LCD7O          _LATB8
#define LCD7I          _RB8

#define CONFIG_LCD4_AS_INPUT() CONFIG_RB5_AS_DIG_INPUT()
#define CONFIG_LCD5_AS_INPUT() CONFIG_RB6_AS_DIG_INPUT()
#define CONFIG_LCD6_AS_INPUT() CONFIG_RB7_AS_DIG_INPUT()
#define CONFIG_LCD7_AS_INPUT() CONFIG_RB8_AS_DIG_INPUT()

#define CONFIG_LCD4_AS_OUTPUT() CONFIG_RB5_AS_DIG_OUTPUT()
#define CONFIG_LCD5_AS_OUTPUT() CONFIG_RB6_AS_DIG_OUTPUT()
#define CONFIG_LCD6_AS_OUTPUT() CONFIG_RB7_AS_DIG_OUTPUT()
#define CONFIG_LCD7_AS_OUTPUT() CONFIG_RB8_AS_DIG_OUTPUT()

#define GET_BUSY_FLAG()  LCD7I



/**
 A fenti függvényeket kell átírnia, ha más lábkiosztást használ
*/

///--- Kimenetnek állítja be a 4 bites adatbuszt
void configBusAsOutLCD(void) {
  RW_LOW();                  //RW=0 letiltja az LCD kimenetvezérlését
  CONFIG_LCD4_AS_OUTPUT();   //D4
  CONFIG_LCD5_AS_OUTPUT();   //D5
  CONFIG_LCD6_AS_OUTPUT();   //D6
  CONFIG_LCD7_AS_OUTPUT();   //D7
}

///--- Bemenetnek állítja be a 4 bites adatbuszt
void configBusAsInLCD(void) {
  CONFIG_LCD4_AS_INPUT();   //D4
  CONFIG_LCD5_AS_INPUT();   //D5
  CONFIG_LCD6_AS_INPUT();   //D6
  CONFIG_LCD7_AS_INPUT();   //D7
  RW_HIGH();                // R/W = 1 az olvasáshoz
}

///--- Az u8_c adat alsó 4 bitjét küldi ki az LCD adatvonalaira
void outputToBusLCD(uint8 u8_c) {
  LCD4O = u8_c & 0x01;          //D4
  LCD5O = (u8_c >> 1)& 0x01;    //D5
  LCD6O = (u8_c >> 2)& 0x01;    //D6
  LCD7O = (u8_c >> 3)& 0x01;    //D7
}

///--- Az LCD-t vezérlő vonalakat konfigurálja
void configControlLCD(void) {
  CONFIG_RS();     //RS
  CONFIG_RW();     //RW
  CONFIG_E();      //E
  RW_LOW();
  E_LOW();
  RS_LOW();
}

///--- Pulzálja az E kimenetet, 1 us várakozás mindegyik élnél a beálláshoz/tartáshoz
void pulseE(void) {
  DELAY_US(1);
  E_HIGH();
  DELAY_US(1);
  E_LOW();
  DELAY_US(1);
}

/** Kiküld egy bájtot (u8_Cmd)  az LCD-re.
u8_DataFlag = '1' ha adat, '0' ha parancs;  
u8_CheckBusy = '1' ha írás előtt le kell kérdezni a foglaltság bitet, egyébként szimplán késletetünk írás előtt;  
u8_Send8Bits = '1' ha 8 bitet küldünk, egyébként csak a felső 4-bitet
*/
void writeLCD(uint8 u8_Cmd, uint8 u8_DataFlag,
              uint8 u8_CheckBusy, uint8 u8_Send8Bits) {

   uint8 u8_BusyFlag;
  uint8 u8_wdtState;
  if (u8_CheckBusy) {      // --- Foglaltság ellenőrzése
    RS_LOW();              //RS = 0     
    configBusAsInLCD();    //minden adatvonalat bemenetnek állít
    u8_wdtState = _SWDTEN; //elmenti a Watchdog állapotát
    CLRWDT();  			   //törli a wachdog időzítőt
    _SWDTEN = 1;           //engedélyezi a watchdogot, nehogy a végtelenségig várjunk.
    do {
      E_HIGH();
      DELAY_US(1);         // a 4 felső bit olvasása
      u8_BusyFlag = GET_BUSY_FLAG();
      E_LOW();
      DELAY_US(1);
      pulseE();            //mégegyszer pulzálja E-t az alsó 4-bit miatt
    } while (u8_BusyFlag);
    _SWDTEN = u8_wdtState; //visszaállítja a watchdog állapotát
  } else {
    DELAY_MS(10);          //--- Nem nézzük a foglaltságot, csak késleltetünk
  }
  configBusAsOutLCD();     //adatbusz kimenet legyen
  if (u8_DataFlag) RS_HIGH();   // RS=1, adatküldés
  else    RS_LOW();             // RS=0, parancs küldés
  outputToBusLCD(u8_Cmd >> 4);  // a felső 4 bit küldése
  pulseE();
  if (u8_Send8Bits) {
    outputToBusLCD(u8_Cmd);     // az alsó 4 bit küldése
    pulseE();
  }
}

//-- Ezek a  makrók a Hantronix 20x4 LCD-hez kellenek
#define GOTO_LINE1() writeLCD(0x80,0,1,1)
#define GOTO_LINE2() writeLCD(0xC0,0,1,1)
#define GOTO_LINE3() writeLCD(0x94,0,1,1)
#define GOTO_LINE4() writeLCD(0xD4,0,1,1)

///--- Inicializálja az LCD-t, ezt módosítsa, ha másfajta LCD vezérlőhöz kell alkalmazkodnia! 
void initLCD() {
  DELAY_MS(50);         //vár az eszköz beállására
  writeLCD(0x20,0,0,0); // 4 bites interface
  writeLCD(0x28,0,0,1); // 2 soros display, 5x7 font
  writeLCD(0x28,0,0,1); // mégegyszer
  writeLCD(0x06,0,0,1); // display engedélyezése
  writeLCD(0x0C,0,0,1); // a display bekapcsolása; kurzor és villogás kikapcsolva
  writeLCD(0x01,0,0,1); // képernyő törlése, kurzor alaphelyzetbe állítás
  DELAY_MS(3);
}

//Kiír egy karakterfüzért az LCD-re
void outStringLCD(char *psz_s) {
  while (*psz_s) {
    writeLCD(*psz_s, 1, 1,1);
    psz_s++;
  }
}


int main (void) {
  configBasic(HELLO_MSG);  // Oszcillátor,életjel LED, UART konfigurálása, üdvözlő üzenet kiírása

  configControlLCD();      //az LCD vezérlő vonalainak beállítása
  initLCD();               //az LCD inicializálása
  
  GOTO_LINE1();            // kurzor az 1. sor elejére
  outStringLCD("Line 1");
  GOTO_LINE2();            // kurzor a 2. sor elejére
  outStringLCD("Line 2");
  GOTO_LINE3();            // kurzor a 3. sor elejére
  outStringLCD("Line 3");
  GOTO_LINE4();            // kurzor a 4. sor elejére
  outStringLCD("Line 4");
  while (1) {
    doHeartbeat();
  }
}
