; ********************************************************************************
; RGB Stand-alone driver Functions include file

; Functions
; --------------------------------------------------------
;
;
;_pwmDrive	drive RGB LEDs with a PWM signal to control brightness
;	code loops continually.
;	Input on GPIO,4 determines whether active drive is high or low
;	The input is only read at power-on / reset.
;
; _seqData	Loads next data block for current sequence
;	resets to start at end of data
;	no parameters. contents of W lost
;
; _advSeq	Advance to next sequence.
;	Set Sequence No counter
;	Reset to first seqence when it reaches end
;	of available sequences.
;	no parameters.
;	return W=1, reset to first sequence
;	teturn W=0, not reset to first sequence
;
;_advResetBase	Initialise sequence data pointer variables
; 	no parameters, contents of W lost
;	
;_sleep	If Sequence switch held down for over 1.2S 
;	put PIC to sleep
;	no parameters, PIC resets after return from sleep
;
;_eeWrite	write value in W reg to EEPROM address 0
;	call with W = value to write
;
;_eeRead	read data from EEPROM address 0 and return in W
;	no parameters
;	return with W = value read.
;
;------------------------------------------------
; PWM Driver 
; 
;
_pwmDrive	incfsz	pwmramp,F	
	goto	_pwmEntry

_chkTimer	movfw	driveLevel
	movwf	GPIO
	btfsc	INTCON,T0IF
	call	_timerOut

_pwmEntry	movf	driveLevel,F
	bnz	_pwmLow
_pwmHigh	movfw	pwmramp	 
	addwf	bpwm,W	 
	skpnc		
	bsf	GPIO,blue	   
	
	movfw	pwmramp	
	addwf	gpwm,W	
	skpnc		
	bsf	GPIO,green
		
	movfw	pwmramp	
	addwf	rpwm,W	
	skpnc		
	bsf	GPIO,red	
	goto	_pwmDrive
 
_pwmLow	movfw	pwmramp	 
	addwf	bpwm,W	 
	skpnc		
	bcf	GPIO,blue	   
	
	movfw	pwmramp	
	addwf	gpwm,W	
	skpnc		
	bcf	GPIO,green
		
	movfw	pwmramp	
	addwf	rpwm,W	
	skpnc		
	bcf	GPIO,red	
	goto	_pwmDrive

;----------------------------------------------
; Cycle through single sequence.
;
_seqData	call	_seqLookup	; read data
	movwf	rate	; save in rate
	addlw	.1	; add 1 (test for 255)
	bz	_seqRestart	; if data was 255 reset seq
	movlw	rate	; load W with address of rate var
	movwf	FSR	; write to FSR
	
_seqNxtData	incf	FSR,F	; increment FSR
	incf	seql,F	; increment seq pointer low
	skpnz		; skip if result != 0
	incf	seqh,f	; increment seq pointer high
	
	call	_seqLookup	; read data
	movwf	INDF	; save via Indirection 
	movlw	bnew	; load W with address of blue new var
	xorwf	FSR,W	; test if FSR == address of blue new var
	bnz	_seqNxtData	; if not read next data byte
	incf	seql,F	; otherwise increment seq data pointer
	skpnz		; ready for next
	incf	seqh,F	; call to get seq data
	
	return


_seqRestart	movfw	sbl	; get sequence base low
	movwf	seql	; save to current sequence low
	movfw	sbh	; get sequence base high
	movwf	seqh	; save to current seqence high
	goto	_seqData	; load start of sequence
	
_seqLookup	movfw	seqh	; data table lookup
	movwf	PCLATH
	movfw	seql
	movwf	PCL
	
;----------------------------------------------
; Advance to next sequence (cycle)
;
_advSeq	call	_advLookup	
	addlw	0x01
	bz	_advEnd
	movlw	0x05
	addwf	sbl,f
	skpnc
	incf	sbh,f
	goto	_advSeq
	

_advEnd	incf	currentSeqNo,F
	incf	sbl,f
	skpnz
	incf	sbh,f
	call	_advLookup
	addlw	0x01
	skpnz	
	goto	_advResetBase
	movfw	sbl
	movwf	seql
	movfw	sbh
	movwf	seqh
	clrw		; return with W = 0
	return
	
		
_advResetBase	clrf	currentSeqNo
	movlw	HIGH _base
	movwf	sbh
	movwf	seqh
	movlw	LOW _base
	movwf	sbl
	movwf	seql
	iorlw	0xFF	; return with W != 0
	return

	;-------------------------
	; lookup
_advLookup	movfw	sbh
	movwf	PCLATH
	movfw	sbl
	movwf	PCL

;-------------------------------------------
; Fade Direction
; 

_fadeDir	movfw	hold	; Set hold timer
	movwf	holdTimerHi
	movfw	rate	; Set fade rate timer
	movwf	fadeTimer	
	bnz	_calcFade	; if fade rate == 0
	movfw	rnew	; No fade so load
	movwf	rpwm	; pwm variables
	movfw	gnew	; with new data
	movwf	gpwm	; to get instant
	movfw	bnew	; colour change
	movwf	bpwm	; ...
	clrf	fadeflag	; clear fade flag so code goes straight to hold
	return		; return
	

;-------------------------------------------
; Calculate the difference between current and new
; PWM value for each colour. Set fade direction to +/-1
	
	
_calcFade	movlw	(1<<red | 1<<green | 1<<blue)
	movwf	fadeflag	; preset fade flag
	movlw	0x01	; preset fade direction to +1 for each colour
	movwf	rdir
	movwf	gdir
	movwf	bdir
	
	movfw	rpwm
	subwf	rnew,W	; subtract current from new (W = current - new)
	bc	_rabs	; branch if C=1  (current > new)
	decf	rdir,F	; rdir is 1, decrement twice
	decf	rdir,F	; to make rdir -1.
	sublw	0x00	; sub W from 0 to make W positive
_rabs	movwf	Rdif	; save difference between current and new


	movfw	gpwm
	subwf	gnew,W	; subtract current from new (W = current - new)
	bc	_gabs	; branch if C=1  (current > new)
	decf	gdir,F	; rdir is 1, decrement twice
	decf	gdir,F	; to make rdir -1.
	sublw	0x00	; sub W from 0 to make W positive
_gabs	movwf	Gdif	; save difference between current and new

	movfw	bpwm
	subwf	bnew,W	; subtract current from new (W = current - new)
	bc	_babs	; branch if C=1  (current > new)
	decf	bdir,F	; rdir is 1, decrement twice
	decf	bdir,F	; to make rdir -1.
	sublw	0x00	; sub W from 0 to make W positive
_babs	movwf	Bdif	; save difference between current and new


;-------------------------------------------
; Find largest difference between current and new RGB values
; 	
	
_calcLargest	movfw	Rdif
	subwf	Gdif,W
	bc	_chkGB
	
	movfw	Rdif
	subwf	Bdif,W
	movfw	Bdif
	bc	_done
	movfw	Rdif
	goto	_done
	
_chkGB	movfw	Gdif
	subwf	Bdif,W
	movfw	Bdif
	skpc	
	movfw	Gdif

_done	sublw	0x00
	movwf	CntBase
	movwf	Rcnt
	movwf	Gcnt
	movwf	Bcnt
	
	return
	


;-------------------------------------------
; Put PIC to sleep
; 
_sleep	movfw	currentSeqNo	; get current sequence No
	call	_eeWrite	; save to EERPROM

	movfw	driveLevel
	movwf	GPIO	; turn off all outputs
	setbank1
	bcf	TRISIO,drive	; set the active drive input to
			; output and drive low to reduce
			; weak-pull load while sleeping
	setbank0
	
_waitSwRelSleep	btfss	GPIO,seqSw	; test seqSw
	goto	_waitSwRelSleep	; wait for it to be released
	movlw	.15
	call	_swDelay
		
	movfw	GPIO	; read GPIO port to clear Int-on-change 
	bcf	INTCON,GPIF	; clear int-on-change flag
	sleep		; put PIC to sleep
	nop

	movlw	.90	; approx. 1 second delay
	call	_swDelay	; wait after wake up
	btfsc	GPIO,seqSw	; test switch to see if it is still down
	goto	_waitSwRelSleep	; if not go back to sleep
_waitSwRelWake	btfss	GPIO,seqSw	; wait for sequence switch to be 
	goto	_waitSwRelWake	; released before continuing
	goto	_startup	; After wake-up run startup code
			; as if it was a power-on startup


_swDelay	clrf	TMR0	; delay to debounce the switch
	bcf	INTCON,T0IF	; 
_waitSleep	btfss	INTCON,T0IF	; 
	goto	_waitSleep
	addlw	-.1
	skpz
	goto	_swDelay
	return

; *********************************
; EEPROM write
; call with W reg containg sequence # to save
_eeWrite        setbank1
	movwf           EEDATA	; save W into EEDATA
	clrf	EEADR	; set EEPROM address 0
                bcf	PIR1,EEIF	; clear EE Interrupt flag
                bsf             EECON1,WREN	; initiate EEPROM write sequence
                movlw           0x55
                movwf           EECON2
                movlw           0xAA
                movwf           EECON2
                bsf             EECON1,WR
                setbank0
                return		; return
              
; *********************************
; EEPROM read
; returns with saved sequence # in W reg

_eeRead         setbank1
                clrf           	EEADR	; set EEPROM address 0
                bsf             EECON1,RD	; initiate EEPROM read
                movf            EEDATA,W	; load EEPROM data into W
                setbank0	
                return          	; return   



