/*! \file timer.c \brief System Timer function library. */
//*****************************************************************************
//
// File Name	: 'timer.c'
// Title		: System Timer function library
// Author		: Pascal Stang - Copyright (C) 2000-2002
// Created		: 11/22/2000
// Revised		: 07/09/2003
// Version		: 1.1
// Target MCU	: Atmel AVR Series
// Editor Tabs	: 4
//
// This code is distributed under the GNU Public License
//		which can be found at http://www.gnu.org/licenses/gpl.txt
//
//*****************************************************************************

#include "config.h"
#include <ioavr.h>
#include <ina90.h>

//#include "interrupt.h"
#include "config.h"
#include "types.h"
//#include "global.h"
#include "timer.h"

// Program ROM constants
// the prescale division values stored in 2^n format
// STOP, CLK, CLK/8, CLK/64, CLK/256, CLK/1024
__flash unsigned char TimerPrescaleFactor[] =
{
	0, 0, 3, 6, 8, 10
};

// externs

// Global variables
WORD  TIME;
BYTE  TIMER2;
// time registers

volatile unsigned long Timer0Reg0;        // Future use
volatile					BYTE Timer0Reload;

// delay for a minimum of <us> microseconds
// the time resolution is dependent on the time the loop takes
// e.g. with 4Mhz and 5 cycles per loop, the resolution is 1.25 us
void DelayUs(unsigned short us)
{
	unsigned short delay_loops;
	register unsigned short i;

	delay_loops = (us + 3) / 5 * CYCLES_PER_US; // +3 for rounding up (dirty)

	// one loop takes 5 cpu cycles
	for (i = 0; i < delay_loops; i++)
	{
	};
}

void DelayMs(BYTE ms)
{
	do
	{
		DelayUs(996);
	}
	while (--ms);
}


void timer0Init(BYTE rlv)
{
	// initialize timer 0
	timer0SetPrescaler(TIMER0PRESCALE);	// set prescaler
	outp(rlv, TCNT0);			// reset TCNT0
	sbi(TIMSK, TOIE0);						// enable TCNT0 overflow interrupt
	Timer0Reload = rlv;
	timer0ClearOverflowCount();					// initialize time registers
}


void timer0SetPrescaler(BYTE prescale)
{
	// set prescaler on timer 0
	outp((inp(TCCR0) & ~TIMER_PRESCALE_MASK) | prescale, TCCR0);
}

/*
u32 timerMsToTics(u16 ms)
{
	// calculate the prescaler division rate
	u16 prescaleDiv = 1<<(PRG_RDB(TimerPrescaleFactor+inp(TCCR0)));
	// calculate the number of timer tics in x milliseconds
	return (ms*(F_CPU/(prescaleDiv*256)))/1000;
}

u16 timerTicsToMs(u32 tics)
{
	// calculate the prescaler division rate
	u16 prescaleDiv = 1<<(PRG_RDB(TimerPrescaleFactor+inp(TCCR0)));
	// calculate the number of milliseconds in x timer tics
	return (tics*1000*(prescaleDiv*256))/F_CPU;
}
*/

#pragma inline=forced
void timer0ClearOverflowCount(void)
{
	// clear the timer overflow counter registers
	Timer0Reg0 = 0;	// initialize time registers
}
#pragma inline=forced
long timer0GetOverflowCount(void)
{
	// return the current timer overflow count
	// (this is since the last timer0ClearOverflowCount() command was called)
	return Timer0Reg0;
}
//! Interrupt handler for tcnt0 overflow interrupt

#pragma vector=TIMER0_OVF_vect
__interrupt void TIMER0_Overflow(void)
{
	outp(Timer0Reload, TCNT0);			// reset TCNT0
	Timer0Reg0++;			              // increment low-order counter
	// if a user function is defined, execute it too
	TIME++; 										    // Increment 5 ms counter		
	if (TIMER2 > 0)
	{
		// I2c_Write_byte Timeout timer, input rutin timer
		TIMER2--;
	}		

}

